////////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: rr_scheduler.uc
//
//      Purpose: 	Simple 2 queue round robin scheduler.
//					
//					Schedules packets in round-robin fashion. Supports
//					one port. This scheduler will be replaced with a full-
//					blown version of a DRR scheduler in the next release.
//
///////////////////////////////////////////////////////////////////////////////

#include <dl_system.h>
#include <stdmac.uc>

// Define switch to enable packets in flight processing.

#define	PACKETS_IN_FLIGHT_PROCESSING

// Maximum number of packets in flight.

#define MAX_IN_FLIGHT		120


///////////////////////////////////////////////////////////////////////
//
// Init macro for scheduler
//
///////////////////////////////////////////////////////////////////////

#macro rr_scheduler_init()
.begin

	// Set $$txd_p0 to 0. This read transfer register is updated
	// by the transmit microblock with number of packets transmitted.

	.reg $zero
	.reg zero
	.sig zero_sig

	immed[zero, 0]
	immed[$zero, 0]

	sram[write, $zero, zero, 0, 1], ctx_swap[zero_sig]

	sram[read, $$txd_p0, zero, 0, 1], ctx_swap[zero_sig]

	// Set @enqueue_queue_number and @dequeue_queue_number to 0.

	immed[@enqueue_queue_number, 0]
	immed[@dequeue_queue_number, 0]

	// Set @packet_count to 0.

	immed[@packet_count, 0]

	// Set @packets_scheduled to 0

	alu[@packets_scheduled, --, B, 0]

	// Set @packets_in_flight to 0

	alu[@packets_in_flight, --, B, 0]

	// Set the NN ring empty threshold to 2 words

	.reg ctx_enable

	local_csr_rd[ctx_enables]
	immed[ctx_enable, 0]
	
	alu_shf[ctx_enable, ctx_enable, AND~, 3, <<18]	
	alu_shf[ctx_enable, ctx_enable, OR, 2, <<18]

	// Wait 3 cycles before first use
	
	local_csr_wr[ctx_enables, ctx_enable]	
	local_csr_wr[nn_put, 0]	
	local_csr_wr[nn_get, 0]
	
	alu[--, --, b, 0]
	alu[--, --, b, 0]
	alu[--, --, b, 0]

.end
#endm


///////////////////////////////////////////////////////////////////////
//
// Round robin scheduler
//
///////////////////////////////////////////////////////////////////////

#macro rr_scheduler()
.begin

	.reg enqueue_queue_number
	.reg dequeue_queue_number
	.reg sop_buffer_handle

	// Set the enqueue queue number for this round.

	alu[enqueue_queue_number, @enqueue_queue_number, +, 1]

	// Set the dequeue queue number for this round.

	alu[dequeue_queue_number, @dequeue_queue_number, +, 1]

	// Read the incoming message.

	br_inp_state[NN_EMPTY, NN_RING_EMPTY#]

	alu[--, --, B, *n$index++]

	alu[sop_buffer_handle, --, B, *n$index++]

	alu[--, --, B, *n$index++]

	// Check the outgoing NN ring. If it is full,
	// do not schedule a dequeue request.

	br_inp_state[NN_FULL, ENQUEUE_ONLY#]

	// Check packets in flight. If it exceeds a
	// threshold do not schedule a dequeue request.

	#ifdef PACKETS_IN_FLIGHT_PROCESSING

	// Subtract packets transmitted from packets scheduled.

	alu[@packets_in_flight, @packets_scheduled, - , $$txd_p0]

	// Mask off all but the lowest byte to account for wrap around on 256.

	alu[@packets_in_flight, @packets_in_flight, AND, 0xFF]

	// Check packets in flight.

	alu[--, @packets_in_flight, -, MAX_IN_FLIGHT]

	bge[ENQUEUE_ONLY#]

	#endif

ENQUEUE_AND_DEQUEUE#:

	// Schedule an enqueue and dequeue request on outgoing NN ring.

	alu[*n$index++, enqueue_queue_number, or, dequeue_queue_number, <<16]
	alu[*n$index++, --, B, sop_buffer_handle]
	alu[*n$index++, --, B, 0xFF]

	// Increment packets scheduled.

	alu[@packets_scheduled, @packets_scheduled, + , 1]

	// Set global enqueue and dequeue numbers for next round.

	alu[@enqueue_queue_number, @enqueue_queue_number, +, 1]
	alu[@enqueue_queue_number, @enqueue_queue_number, AND, 0x1]

	alu[@dequeue_queue_number, @dequeue_queue_number, +, 1]
	alu[@dequeue_queue_number, @dequeue_queue_number, AND, 0x1]

	br[SCHEDULE_END#]

	
ENQUEUE_ONLY#:

	br_inp_state[NN_FULL, ENQUEUE_ONLY#]

	// Increment packet count 

	alu[@packet_count, @packet_count, +, 1]

	// Send an enqueue only request on outgoing NN ring.

	alu[*n$index++, --, B, enqueue_queue_number]
	alu[*n$index++, --, B, sop_buffer_handle]
	alu[*n$index++, --, B, 0xFF]

	// Set global enqueue queue number for next iteration.

	alu[@enqueue_queue_number, @enqueue_queue_number, +, 1]
	alu[@enqueue_queue_number, @enqueue_queue_number, AND, 0x1]

	br[SCHEDULE_END#]


NN_RING_EMPTY#:

	// Check packet count. If packets are present, schedule a dequeue request.

	alu[--, --, B, @packet_count]

	beq[SCHEDULE_END#]

	// Check packets in flight. If it exceeds a
	// threshold do not schedule a dequeue request.

	#ifdef PACKETS_IN_FLIGHT_PROCESSING

	// Subtract packets transmitted from packets scheduled.

	alu[@packets_in_flight, @packets_scheduled, - , $$txd_p0]

	// Mask off all but the lowest byte to account for wrap around on 256.

	alu[@packets_in_flight, @packets_in_flight, AND, 0xFF]

	// Check packets in flight.

	alu[--, @packets_in_flight, -, MAX_IN_FLIGHT]

	bge[SCHEDULE_END#]

	#endif


DEQUEUE_ONLY#:

	br_inp_state[NN_FULL, SCHEDULE_END#]

	// Decrement packet count.

	alu[@packet_count, @packet_count, -, 1]

	// Schedule a dequeue request on outgoing NN ring.

	alu[*n$index++, --, B, dequeue_queue_number, <<16]
	alu[*n$index++, --, B, 0]
	alu[*n$index++, --, B, 0]

	// Update packets scheduled.

	alu[@packets_scheduled, @packets_scheduled, + , 1]

	// Set global dequeue number for next round.

	alu[@dequeue_queue_number, @dequeue_queue_number, +, 1]
	alu[@dequeue_queue_number, @dequeue_queue_number, AND, 0x1]


SCHEDULE_END#:

.end
#endm

///////////////////////////////////////////////////////////////////////
//
// Scheduler Main Entry Point
//
///////////////////////////////////////////////////////////////////////

MAIN#:

	// Run with thread 0 only.

	br=ctx[0, SCHEDULER#]

	// All other threads should abort.

	ctx_arb[kill]			

SCHEDULER#:

	.reg @enqueue_queue_number
	.reg @dequeue_queue_number

	.reg @packets_scheduled
	.reg @packets_in_flight
	.reg @packet_count

	.reg visible global $$txd_p0

	rr_scheduler_init()

SCHEDULE_LOOP#:	

		rr_scheduler()

		br[SCHEDULE_LOOP#]