/////////////////////////////////////////////////////////////////////////////////////
//                                                                      
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                 
//
// Change History
// --------------
//
// Date			Description											Whom
// ------------------------------------------------------------------------------------
//
// 11/11/01    	Created for IXP2400								Uday Naik
//
//
/////////////////////////////////////////////////////////////////////////////////////
//
//
// Flow Control Handling Routines
//
//
/////////////////////////////////////////////////////////////////////////////////////

#ifndef __CSIX_SCHEDULER_FC_UC__
#define __CSIX_SCHEDULER_FC_UC__

/////////////////////////////////////////////////////////////////////////////////////

// Offset of the flow control fifo

#define FCIFIFO 0x100

/////////////////////////////////////////////////////////////////////////////////////
//
//
// HandleFlowControl:
//
//	  Handle Flow Control Messages from the Fabric
//
//    This runs in one single thread and handles flow control 
//	  messages coming in from the Fabric
//
//    Each frame has atleast n messages of 1 word each. Additional there is 
//    a payload length (16 bits) and vertical parity (16 bits)
//
//
//  Flow Control Word 1
// 
//  |----------------------------------------------------------------------|
//  |	xxxxxxxx | PayloadLength | xxxx	    | Class   | xxxxxxx | On/Off   |
//  |	(31..24) | (16..23)	     | (15..12)	| (8..11) | (7..1)  | (0)      |
//  |----------------------------------------------------------------------|
//
//
//  Flow Control Word 2
//
//  |------------------------------------------------------------------|
//  |	xxxxxxxx | Port Number   | Next Message (or vertical parity)   |
//  |	(31..22) | (16..21)	     |    (0..15)				     	   |
//  |------------------------------------------------------------------|
//
//
//  Flow Control Word 3..n (last word has vertical parity)
//
//  |------------------------------------------------------------------|
//  |	xxxxxxxx | Port Number   | Next Message (or vertical parity)   |
//  |	(31..22) | (16..21)	     |    (0..15)				     	   |
//  |------------------------------------------------------------------|
//
//  
//  Instruction estimate: 30 cycles
// 
//
/////////////////////////////////////////////////////////////////////////////////////

/*
	Pseudo Code for flow control

HandleFlowControl()
{

	while (1)
	{
	
		// Read the flow control message from the FIFO

		ReadFlowControlFifo(2, msg1, msg2);

		// Extract the number of messages

		numberOfMessages = (msg1 >> 16) / 4;

		while (1)
		{

			HandleMessage(msg1, msg2);

			numberOfMessages--;
		
			msg1 = msg2;

			if (numberOfMessages == 0)
				break;
			else
				ReadFlowControlFifo(1, msg2);
		} 
	}
}

HandleMessage(msg1, msg2)
{	
	
	// find the queue group from 0..31 

	queueGroup = (msg2 >> 17) & 0x1f;

	// find the queue number from 0..31 within a group 

	queueNumber = ((msg1 >> 8) & 0xf) | ((msg2 & 0x10000) >> 12)

	// find the byte position in local memory  with bit vector for queue id
	// read the local memory word
		
	fcVector = ReadLocalMemory(1, (queueGroup << 2) + QUEUE_LM_FC_BASE);

	// check if flow control is on or off. If bit is 
	// is set then flow control is off else on

	if (msg1 & 0x1)
	{
		// flow control is off. So turn it off within the queuegroup

		fcVector |= ( 1 << queueNumber)

		// Set the bit in the parent vector. Actually we only need
		// to do this if it is zero. Rather than check for it, we 
		// we just set it
	
		@rootFcVector |= (1 << queueGroup)
	}
	else
	{
		// flow control is off. SO turn if off within group
		
		fcVector &= ~(1 << queueNumber)

		// If fcVector is 0, clear the bit in parent

		if (!fcVector)
			@rootFcVector &= ~(1 << queueGroup)

	}
		
}

*/

/////////////////////////////////////////////////////////////////////////////////////
//
//
//  This takes ~26 instructions for a single 2 word message and 22 instructions
//  for a aggregated message. It takes 7 instructions every beat if no message is
//  present
//
/////////////////////////////////////////////////////////////////////////////////////



#macro _scheduler_handle_flow_control()

.begin

	.reg 			$msg1, $msg2			// xfer registers for flow control
	.xfer_order 	$msg1, $msg2			// declare the order for xfer registers
	.reg			fcfifo					// stores address of fcfifo
	.reg 			message1				// GPR to store first word of fc message 
	.reg			queue_number			// queue number (0..31) within a group
	.reg			queue_group				// queue group (0..31)
	.reg			queue_number_bitmask	// (1 << queueNumber)
	.reg			queue_group_bitmask		// (1 << queueGroup)
	.reg			number_of_messages		// number of messages in the FC frame
	.reg			lm_offset				// local memory offset of queue group
	.reg			temp					// temporary variable
	.reg			port_number_mask		// bit mask to get lsb bit of port
	.sig			FLOW_CONTROL_SIGNAL		// signal for reading msf

	// load the start address for the ingress flow control fifo 

	immed[fcfifo, FCIFIFO]
	
	// load bit mask to extract lsb bit from port number. We store it in a 
	// register so we dont have to compute it at run time
	
	immed32[port_number_mask, 0x10000]

FC_SWAP_OUT#:

	// Swap out. The first time the thread is called, it swaps out immediately	

	ctx_arb[voluntary] 


CHECK_FCIFIFO#:
	
	//	Check if the FCIFIFO has data

	br_!inp_state[FCI_NOT_EMPTY, FC_SWAP_OUT#]	

READ_IFIFO#:
	
	// Read 2 words from the IFIFO and swap out waiting for it to 
	// get done

 	msf[read, $msg1, fcfifo, 0, 2], ctx_swap [FLOW_CONTROL_SIGNAL]

	// Get xfer register into GPR message1 

	alu[message1, --, B, $msg1]
		
	// Get the number of messages. which is payloadLength/4 

	alu_shf[number_of_messages, 0x3F, AND, message1, >>18]

	// Get the queue group for the queue which is the first 5 bits 
	// of the port number

	alu_shf[queue_group, 0x1f, AND, $msg2, >>17]

	// Set the local memory base to the start of the vector for this
	// queue group. Each queue group takes 16 bytes of local memory
	
	alu_shf[lm_offset, --, B, queue_group, <<4]

	// Get the queue number for the queue which is the class [0..3] and
	// and the lsb bit of the port

	alu_shf[queue_number, 0xf, AND, message1, >>8]

HANDLE_FLOW_CONTROL_MESSAGE#:

	// Get the lsb bit of port number and or it in 

	alu[temp, port_number_mask, AND, $msg2]
	alu_shf[queue_number, queue_number, OR, temp, >>12]

	// check the speed bit. If it is 0, then flow control is on
	// else it is off

	alu[--, 0x1, AND, message1] 

	beq[FLOW_CONTROL_ON#] , defer [3]

	// Set the local memory offset for the queue group bit vector. 2 cycle
	// latency

	localmem_set_address[0, lm_offset , LM_HANDLE_0]		; defer 1
		
	// compute 1 << queueNumber

	alu[--, queue_number, OR, 0]							; defer 2
	alu_shf[queue_number_bitmask, --, B, 1, <<indirect]		; defer 3

	// Calculate (1 << queueGroup)

	alu[--, queue_group, OR, 0]
	alu_shf[queue_group_bitmask, --, B, 1, <<indirect]

	br[FLOW_CONTROL_END#]  , defer [2]

	// Set the bit in flow control vector to 1

	alu[*l$index0[LM_GROUP_FC_VECTOR_INDEX], *l$index0[LM_GROUP_FC_VECTOR_INDEX], \
		OR, queue_number_bitmask]	; defer 1

	// Set the bit in the parent vector. Actually need to set it only
	// if the bit is not set. But rather than check, we will just set it

	alu[@root_flow_control_vector, @root_flow_control_vector, OR,  \
		queue_group_bitmask]		; defer 2



FLOW_CONTROL_ON#:

	// Since local memory csr write has 3 cycle latency

	nop

	// Clear the bit in the flow control vector 

	alu[*l$index0[LM_GROUP_FC_VECTOR_INDEX], *l$index0[LM_GROUP_FC_VECTOR_INDEX], \
		AND~, queue_number_bitmask]
	
	// check if the entire vector is 0 in which we need to clear the bit in the parent

	bne[FLOW_CONTROL_END#] 
		
	// Calculate (1 << queueGroup)

	alu[--, queue_group, OR, 0]
	alu_shf[queue_group_bitmask, --, B, 1, <<indirect]

	// Clear the bit in the parent vector

	alu[@root_flow_control_vector, @root_flow_control_vector, AND~,  \
		queue_group_bitmask]

FLOW_CONTROL_END#:

	alu[number_of_messages, number_of_messages, -, 1]

	beq[CHECK_FCIFIFO#] 
		
	alu[message1, --, B, $msg2]

	// Read 1 word from the IFIFO and swap out waiting for it to 
	// get done. Since we already have message1 we will start to use
	// it

	msf[read, $msg2, fcfifo, 0, 1], ctx_swap[FLOW_CONTROL_SIGNAL] , defer[1]

	alu_shf[queue_number, 0xf, AND, message1, >>8] ; defer 1


	br[HANDLE_FLOW_CONTROL_MESSAGE#] , defer[2]
		
	// Get the queue group for the queue which is the first 5 bits 
	// of the port number

	alu_shf[queue_group, 0x1f, AND, $msg2, >>17]	; defer 1

	// Set the local memory base to the start of the vector for this
	// queue group. Each entry is 16 bytes for a queue group
	
	alu_shf[lm_offset, --, B, queue_group, <<4]		; defer 2

.end

#endm
	
/////////////////////////////////////////////////////////////////////////////////////

#endif	// __CSIX_SCHEDULER_FC_UC__

/////////////////////////////////////////////////////////////////////////////////////