/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a licence agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2001 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2400 Network Processor
 *
 * = LIBRARY
 *      
 *
 * = MODULE
 *      ATM TM4.1 Core Component
 *
 * = FILENAME
 *      ix_cc_atmtm41_private.h
 *
 * = DESCRIPTION
 *
 * = AUTHOR
 *      Jaroslaw Kogut
 *      jaroslaw.kogut@intel.com
 *
 * = AKNOWLEDGEMENTS
 *      
 *
 * = CREATION TIME
 *      
 *
 * = CHANGE HISTORY
 *
 * ============================================================================
 * 
 */

#if !defined(__IX_CC_ATMTM41_PRIVATE_H__)
#define __IX_CC_ATMTM41_PRIVATE_H__

#include "cc/ix_cc_atmsar.h"
#include "cc/ix_cc_atmsar_list.h"

#include "dl_system.h"
#include "atm_tm_defs.h"
#include "qm_atm_defs.h"

#if defined(__cplusplus)
extern "C" {
#endif /* end defined(__cplusplus) */

#define _IX_CC_ATMTM41_LINE_RATE   LINK_RATE_PCR  
#define _IX_CC_ATMTM41_VC_HBR_MAX  HBR_TQ_SRAM_SIZE
#define _IX_CC_ATMTM41_LBR_MIN_HANDLE  HBR_THRESHOLD
#define _IX_CC_ATMTM41_MAX_NUBER_OF_LOW_RATE_VCC   MAX_NUMBER_OF_VCC - MAX_NUBER_OF_HIGH_RATE_VCC
#define _IX_CC_ATMTM41_MIN_HBR_VC_BAND ((_IX_CC_ATMTM41_LINE_RATE)/_IX_CC_ATMTM41_VC_HBR_MAX)
#define _IX_CC_ATMTM41_MAX_LBR_VC_BAND ((_IX_CC_ATMTM41_LINE_RATE)/AGGREGATION)
#define _IX_CC_ATMTM41_PORT_SHAPPING_EMPTY_ENTRY   0x0FFFFFFFF

/* size of buffer for prefix name for symbol patched into microcode */
#define _IX_CC_ATMTM41_SYMBOL_PREFIX_SIZE 20
#define _IX_CC_ATMTM41_SYMBOL_SIZE        64
/**
 * TYPENAME: _ix_cc_atmtm41_symbol_id_t
 * 
 * DESCRIPTION:
 *    This enum defines number of symbols serviced by the core component
 */
typedef enum _ix_cc_atmtm41_e_symbol_id
{
    _IX_CC_ATMTM41_GCRA_PARAM_TABLE_SYMBOL_ID=0,
    _IX_CC_ATMTM41_HBR_TQ_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_HBR_TQ_SRAM_INDICATOR_SYMBOL_ID,
    _IX_CC_ATMTM41_RTQ_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_RTQLEN_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_NRTQ_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_NRTQLEN_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_UBRTQ_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_UBRTQLEN_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_PORTINFO_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_PORTSHAPING_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_DQ_SRAM_BASE_SYMBOL_ID,
    _IX_CC_ATMTM41_MAX_SYMBOL_ID
} _ix_cc_atmtm41_symbol_id_t;

/**
 * TYPENAME: _ix_cc_atmtm41_symbol_type_t
 * 
 * DESCRIPTION:
 *    This enum defines the symbol types.
 */
typedef enum _ix_cc_atmtm41_e_symbol_type
{
    _IX_CC_ATMTM41_AAL5_PORT_SYMBOL_TYPE=0,
    _IX_CC_ATMTM41_AAL5_VC_SYMBOL_TYPE,
    _IX_CC_ATMTM41_AAL5_TQTOT_SYMBOL_TYPE,
    _IX_CC_ATMTM41_AAL5_SINGLE_SYMBOL_TYPE,

#ifdef AAL2_INCLUDED

    _IX_CC_ATMTM41_AAL2_CID_SYMBOL_TYPE,

#endif

    _IX_CC_ATMTM41_SYMBOL_TYPE_LAST

} _ix_cc_atmtm41_e_symbol_type_t;


/**
 * TYPENAME: _ix_cc_atmtm41_port_t
 * 
 * DESCRIPTION:
 *    the structure describes ports serviced by the core component.
 */
typedef struct _ix_cc_atmtm41_s_port
{
#define _IX_CC_ATMTM41_INVALID_PORT_RATE_EXP 0x0FF
    ix_uint32 lrCfg;  /**< link rate read from config of port. Value in CPS.*/
    ix_uint32 lrUsed; /**< link rate used by VCs configured on the port.
                         Value in CPS (cells per seconds. */
    ix_uint8 n;       /**< power of 2 assigned to the port. where line rate
                         assigned to port is equal _IX_CC_ATMTM41_LINE_RATE divided
                         by 2 powered by n */
    ix_uint8 hbrDescr;/**< Id of hbr descriptor. If it is equal 0, it means
                         no hbr VC allowed on the port */
    ix_uint16 vcList; /**< head of list VCs associated with the port */
} _ix_cc_atmtm41_port_t;


/**
 * TYPENAME: _ix_cc_atmtm41_hbr_descr_t
 * 
 * DESCRIPTION:
 *    the structure defines a list description
 */
typedef struct _ix_cc_atmtm41_s_hbr_descr
{
    ix_uint8 first;
    ix_uint8 step;
} _ix_cc_atmtm41_hbr_descr_t;


/**
 * TYPENAME: _ix_cc_atmtm41_update_data_t
 * 
 * DESCRIPTION:
 *    This structure describes all parameters for binding external object to
 *    ATMSAR object.
 */
typedef struct 
{
    union {
        ix_uint32 lw0;
        ix_uint16 value;
    } l2port;

    union {
        ix_uint32 lw1;
        ix_uint8  value;        
    } hdrType;
    
} _ix_cc_atmtm41_update_data_t;

/**
 * TYPENAME: _ix_cc_atmtm41_vc_entry_t
 * 
 * DESCRIPTION:
 *    This structure defines format for vc descriptor 
 */
typedef struct _ix_cc_atmtm41_s_vc_entry
{
    /*
     * the same memory space 'lr' and 'head' is used for two purposes: if
     * 'port' is equal IX_CC_ATMSAR_PORTID_INVALID, the 'head' sub-structure
     * is valid, and the VC is linked into list of free VCs. If 'port' is not
     * equal IX_CC_ATMSAR_PORTID_INVALID, the 'bandwidth' is used and it
     * contains bandwidth allocated for the VC.
     */
    union
    {
        ix_cc_atm_list_entry_t head; /* field usad for list purpose*/
        ix_uint32 bandwidth;         /* in cells per seconds */
    } shared;
    ix_uint16              port; /* IX_CC_ATMSAR_PORTID_INVALID - unused  */
} _ix_cc_atmtm41_vc_entry_t;


/**
 * TYPENAME: _ix_cc_atmtm41_symbol_descr_t
 * 
 * DESCRIPTION:
 *    ME symbol description: channel number and memory type (local, scratch pad, sram)
 */
typedef struct _ix_cc_atmtm41_s_symbol_descr
{
    ix_uint32 channel; /* channel number */
    ix_uint32 memType; /* memory type */
    ix_uint32 memSize; /* memory size in bytes */
    ix_uint32 patchMask;  /* mask of microengines where symbol should be patched. */
} _ix_cc_atmtm41_symbol_descr_t;

/**
 * TYPENAME: _ix_cc_atmtm41_context_t
 * 
 * DESCRIPTION:
 *    This structure defines format for ATM TM4.1 core component context
 *    NOTE: field 'hbrVc'. We define copy of HBR VC table in DRAM and
 *          use the copy for calculating indexes real HBR VC table in SRAM.
 *          We use table in DRAM because only microengines should use SRAM.
 */
typedef struct ix_s_cc__ix_cc_atmtm41_context
{
    ix_cc_handle hAtmtm41Handle;       /* Handle of ATM TM4.1 core component */
    ix_cc_atmsar_instance_t instance;  /* Instance ID of the core component */

    ix_uint32 *pSymbolTable[_IX_CC_ATMTM41_MAX_SYMBOL_ID];
    ix_uint32 *pPhyAddress[_IX_CC_ATMTM41_MAX_SYMBOL_ID];
    _ix_cc_atmtm41_symbol_descr_t symbolDescr[_IX_CC_ATMTM41_MAX_SYMBOL_ID];
    ix_int32  schedulerMEMask;
    ix_int32  shaperMEMask;
    ix_uint32 counters;
    ix_int32  tq_tot;
    ix_uint16 *vcId;
    _ix_cc_atmtm41_port_t *portsTab;
    ix_uint16 portCount;
    ix_uint16 log2MaxAtmPort;
    ix_uint16 log2TqTot;
    ix_uint16 log2Aggregation;
    _ix_cc_atmtm41_hbr_descr_t hbrDescrForPort[_IX_CC_ATMTM41_VC_HBR_MAX+1];
#define _IX_CC_ATMTM41_HBR_VC_EMPTY 0x0FF
    ix_uint8  hbrVc[_IX_CC_ATMTM41_VC_HBR_MAX]; /**< copy of HBR VC table.*/
    ix_uint8  tmpHbrVc[_IX_CC_ATMTM41_VC_HBR_MAX];
    ix_cc_atm_list_descr_t listVc;       /**< descriptor of VC list */
    _ix_cc_atmtm41_vc_entry_t *vc;       /**< description of vc */
    ix_uint16 freeHbrVcs;  /**< head of list of free HBR VC entries */
    ix_uint16 freeLbrVcs;  /**< head of list of free LBR VC entries */
    ix_cc_atmsar_plugin_cfg_params_t pluginCfg; /* ATM SAR configuration */
    char symbolPrefix[_IX_CC_ATMTM41_SYMBOL_PREFIX_SIZE+1]; /* the prefix is added to patched symbols */
} _ix_cc_atmtm41_context_t;


/******************************************************************************
 *                          GLOBAL VARIABLES
 *****************************************************************************/


/******************************************************************************
 *           FUNCTIONS FOR SAR CONTROL PLUGIN INFRASTRUCTURE
 *****************************************************************************/

/**
 * TYPENAME: _ix_cc_atmtm41_plugin_vc_create
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it services "VC create" operation. The called
 *    function should perform the operation on internal structures.
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN  arg_hVC :  handle to created VC.
 * @Param:  IN  arg_pVcDescr : parameters describing requested VC.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - ATM port successfully created
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 *    IX_CC_ATMSAR_ERROR_NO_RESOURCES - no resources to add the entry 
 */
ix_error _ix_cc_atmtm41_plugin_vc_create(
void* arg_pInstance,
const ix_cc_atmsar_vc_handle_t arg_phVC,
const ix_cc_atmsar_vc_descr_t* arg_pVcDescr);

/**
 * NAME: _ix_cc_atmtm41_plugin_vc_update
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it services "VC update" operation. The called
 *    function should perform the operation on internal structures.
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN  arg_hVc :  handle to created VC.
 * @Param:  IN  arg_pVcDescr : parameters describing requested VC.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - ATM port successfully created
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 *    IX_CC_ATMSAR_ERROR_NO_RESOURCES - no resources to add the entry 
 */
ix_error
_ix_cc_atmtm41_plugin_vc_update(
void* arg_pInstance,
const ix_cc_atmsar_vc_handle_t arg_hVc,
const ix_cc_atmsar_update_data_t* arg_pUpdate);

/**
 * TYPENAME: _ix_cc_atmtm41_plugin_vc_remove
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it services "VC remove" operation. The called
 *    function should perform the operation on internal structures.
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN  arg_hVc : handle to VC to be removed
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - vc successfully removed
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 */
ix_error _ix_cc_atmtm41_plugin_vc_remove(
void* arg_pInstance,
ix_cc_atmsar_vc_handle_t arg_hVc);


/**
 * TYPENAME: _ix_cc_atmtm41_plugin_port_create
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it services "PORT CREATE" operation. The called
 *    function should perform the operation on internal structures.
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN  arg_pPortDescr : pointer to data describing parameters
 *                  for requested port
 * @Param:  OUT arg_hPort : ATM port handle
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - ATM port successfully created
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 *    IX_CC_ATMSAR_ERROR_NO_RESOURCES - no resources to add the entry 
 */
ix_error _ix_cc_atmtm41_plugin_port_create(
void* arg_pInstance,
ix_cc_atmsar_port_handle_t arg_hPort,
const ix_cc_atmsar_port_descr_t* arg_pPortDescr);


/**
 * TYPENAME: _ix_cc_atmtm41_plugin_port_remove
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it services "PORT REMOVE" operation. The called
 *    function should perform the operation on internal structures.
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN  arg_port : ATM port to be removed
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - port successfully removed
 *    IX_CC_INVALID_PORT - specified port does not exist
 */
ix_error _ix_cc_atmtm41_plugin_port_remove(
void* arg_pInstance,
ix_cc_atmsar_port_handle_t arg_port);


/**
 * NAME: _ix_cc_atmtm41_plugin_vc_alloc
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it wants to allocate VC for VC creation.
 *    The called function should allocate and return new VC handle.
 *    The 'arg_pVcDescr' agrument is passed to the function because
 *    parameters of VC could influent on assigned handle
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN  arg_pVcDescr : pointer to data describing parameters
 *                  for requested VC
 * @Param:  OUT arg_pHVc : poiter to data where assigned handle is stored
 *                  VC handle
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - ATM port successfully created
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 *    IX_CC_ATMSAR_ERROR_NO_RESOURCES - no resources to add the entry 
 */
ix_error _ix_cc_atmtm41_plugin_vc_alloc(
void* arg_pInstance,
ix_cc_atmsar_vc_handle_t *arg_pHVc,
const ix_cc_atmsar_vc_descr_t *arg_pVcDescr);

/**
 * NAME: _ix_cc_atmtm41_plugin_vc_free
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it wants to free VC for VC destroying.
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN arg_hVc : handle of freeing VC.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - ATM port successfully created
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 */
ix_error _ix_cc_atmtm41_plugin_vc_free(
void* arg_pInstance,
ix_cc_atmsar_vc_handle_t arg_hVc);


/**
 * NAME: _ix_cc_atmtm41_plugin_port_alloc
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it wants to allocate port for port creation.
 *    The called function should allocate and return new port handle.
 *    The 'arg_pPortDescr' agrument is passed to the function because
 *    parameters of port could influent on assigned handle
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN  arg_pPortDescr : pointer to data describing parameters
 *                  for requested port
 * @Param:  OUT arg_pHPort : poiter to data where assigned handle is stored
 *                  ATM port handle
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - ATM port successfully created
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 *    IX_CC_ATMSAR_ERROR_NO_RESOURCES - no resources to add the entry 
 */
ix_error _ix_cc_atmtm41_plugin_port_alloc(
void* arg_pInstance,
ix_cc_atmsar_port_handle_t *arg_pHPort,
const ix_cc_atmsar_port_descr_t *arg_pPortDescr);

/**
 * NAME: _ix_cc_atmtm41_plugin_port_free
 *
 * DESCRIPTION:
 *    This is function for plug-in registering. The ATM SAR
 *    calls the function if it wants to free port for port destroying.
 * 
 * @Param:  IN  arg_pInstance :  pointer to structure describing serviced
 *                  instance.
 * @Param:  IN arg_hPort : handle of freeing ATM port.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    IX_SUCCESS - ATM port successfully created
 *    IX_CC_ATMSAR_ERROR_INVALID_PARAM - invalid input parameters
 */
ix_error _ix_cc_atmtm41_plugin_port_free(
void* arg_pInstance,
ix_cc_atmsar_port_handle_t arg_hPort);


/******************************************************************************
 *                  SUPPORT FOR PORTS SERVICING
 *****************************************************************************/


/**
 * NAME: _ix_cc_atmtm41_port_info_init
 *
 * DESCRIPTION:
 *    The function fills PORT INFO with read static configuration.
 * 
 * @Param:  - IN arg_pContext - pointer to context of current core component.
 *
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *    TBD
 */
ix_error _ix_cc_atmtm41_port_info_init(
_ix_cc_atmtm41_context_t *arg_pContext);


/******************************************************************************
 *                  SUPPORT FOR DEBBUGING AND TRACES
 *****************************************************************************/

extern ix_uint32 g_atmTm41DebugMask;

#define _IX_CC_ATMTM41_DBG(__mask, __debugInfo) \
    if((__mask & g_atmTm41DebugMask)!= 0) ix_ossl_message_log __debugInfo;

#define _IX_CC_ATMTM41_DBG_INIT(__debugInfo) _IX_CC_ATMTM41_DBG(0x01, __debugInfo)
#define _IX_CC_ATMTM41_DBG_PORT(__debugInfo) _IX_CC_ATMTM41_DBG(0x02, __debugInfo)
#define _IX_CC_ATMTM41_DBG_VC(__debugInfo)   _IX_CC_ATMTM41_DBG(0x04, __debugInfo)
#define _IX_CC_ATMTM41_DBG_GCRA(__debugInfo) _IX_CC_ATMTM41_DBG(0x08, __debugInfo)

#if !defined (IX_INCLUDE_REGISTRY)

#define _IX_CC_ATMTM41_DEFAULT_CHANNEL(__instance)   0x0
#define _IX_CC_ATMTM41_DEFAULT_MEMTYPE(__instance)   IX_MEMORY_TYPE_SRAM
#define _IX_CC_ATMTM41_SCHEDULER_DEFAULT_MASK(__instance) 0x20
#define _IX_CC_ATMTM41_SHAPER_DEFAULT_MASK(__instance)    0x10
#define _IX_CC_ATMTM41_SCHEDULER(__instance) 0x1
#define _IX_CC_ATMTM41_SYMBOL_PREFIX(__instance) ""

#endif


#if defined(__cplusplus)  
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__IX_CC_ATMTM41_PRIVATE_H__) */

