///////////////////////////////////////////////////////////////////////
//                                                                   
//                  I N T E L   P R O P R I E T A R Y                
//                                                                   
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS  
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY   
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A 
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER 
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL, 
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT 
//     THE PRIOR WRITTEN PERMISSION OF :                             
//                                                                   
//                        INTEL  CORPORATION                         
//                                                                   
//                     2200 MISSION COLLEGE BLVD                     
//                                                                   
//               SANTA  CLARA,  CALIFORNIA  95052-8119               
//                                                                   
///////////////////////////////////////////////////////////////////////
//                                                                   
//                                                                   
//      Filename: rr_scheduler.uc
//                                                                   
//      Description: This is a round robin scheduler used by the IPv4
//                   AAL5 Egress OC48 and quad OC12 applications.
//                   For a TM 4.1 scheduler, refer to the ATM diffserv
//                   application.
//
//      History:
//
//      07/22/2003  Created                               asv
//
///////////////////////////////////////////////////////////////////////

#ifndef _RR_SCHEDULER_UC_
#define _RR_SCHEDULER_UC_

#include <stdmac.uc>
#include <dl_system.h>
#include <rr_scheduler.h>
#include <rr_scheduler_util.uc>

	;------------------------------------------------------------------
	; Code execution begins here
	;------------------------------------------------------------------

	// Define for detecting outgoing scratch ring full condition.

	#define_eval OUT_SCR_RING_FULL	SCR_RING/**/SCHEDULER_RING_OUT_0/**/_STATUS

	; Define and initialize registers and signals.

	; This signal is used for next thread signaling.

	.sig volatile next_thread_sig

	.addr next_thread_sig INTER_THD_SIGNAL

	; Initialize the signal mask.

	.reg schedule_queue_sigmask

	alu[schedule_queue_sigmask, --, b, 1, <<&next_thread_sig]


	; This register stores one bit per group.indicating if the group
	; is empty (0) or non-empty (1).
	; Initially all groups are empty. Initialize it to 0. 

	.reg @root_vector

	immed[@root_vector, 0] 

	; This register is used to control round robin scheduling across groups.
	; Initialize it to all 1's.

	.reg @root_mask

	alu[@root_mask, --, ~b, 0]


	; Set outgoing scratch ring number. Schedule equests are sent out on this
	; scratch ring.

	.reg out_scr_ring

	alu[out_scr_ring, --, b, SCHEDULER_RING_OUT_0, <<2]


	; Set registers with constants. These are used to save instructions.

	.reg minus_two

	move(minus_two, 0xFFFFFFFE)

	.reg Ox8003FFFF

	move(Ox8003FFFF, 0x8003FFFF)

	.reg Ox7FF

	move(Ox7FF, 0x7FF)

	.reg _group_structure_lm_base

	move(_group_structure_lm_base, GROUP_STRUCTURE_LM_BASE)


	; These registers and signals are used to write the schedule request to the
	; outgoing scratch ring.

	.reg write $schedule_queue_1
	.reg write $schedule_queue_2
	.reg write $schedule_queue_3

	.sig schedule_queue_sig_1
	.sig schedule_queue_sig_2
	.sig schedule_queue_sig_3

init#:

	;------------------------------------------------------------------
	; Initialization phase.
	;------------------------------------------------------------------

	.if (ctx() == 0)

	; Initialize local memory and NN ring.

	_rr_scheduler_init()

	; Thread 0 waits for system initialization signal.

	.sig volatile me_init_sig

	.addr me_init_sig ME_INIT_SIGNAL

	ctx_arb[me_init_sig]

	.else

	; All other threads wait for next thread signal.

	ctx_arb[next_thread_sig]

	.endif


loop#:

	;------------------------------------------------------------------
	; Recurring phase.
	;------------------------------------------------------------------

	.io_completed schedule_queue_sig_1 schedule_queue_sig_2 schedule_queue_sig_3

	.reg enqueue_message
	.reg dequeue_message

	.reg enqueue_queue_number
	.reg dequeue_queue_number


	; Signal next thread as early as possible. The next thread will execute only
	; after this threads ctx_arb[]'s out.

	local_csr_wr[same_me_signal, (0x80 | (&next_thread_sig << 3))]


	; Read incoming request from NN ring. If NN ring is empty, jump to schedule_hbr_queue#.

	_rr_scheduler_read_nn_ring(enqueue_message, dequeue_message, schedule_hbr_queue#)


	; NN ring is not empty, so process incoming request.

	; Extract enqueue queue number.

	alu[enqueue_queue_number, 0, +16, enqueue_message]

	; Check if enqueue queue number is a high bit rate or low bit rate queue.

	alu[--, enqueue_queue_number, -, MAX_HBR_QUEUE]

	ble[handle_hbr_enqueue#]


	; Enqueue queue number is a low bit rate queue.

	; Handle low bit rate enqueue case.

	_rr_scheduler_handle_lbr_enqueue(enqueue_message, schedule_queue_sigmask)

	br[handle_dequeue#]


handle_hbr_enqueue#:

	; Enqueue queue number is a high bit rate queue.

	; Handle high bit rate enqueue case.

	_rr_scheduler_handle_hbr_enqueue(enqueue_queue_number, enqueue_message)


handle_dequeue#:

	; Extract dequeue queue number.

	alu[dequeue_queue_number, 0, +16, dequeue_message]

	; Check if dequeue queue number is a high bit rate or low bit rate queue.

	alu[--, dequeue_queue_number, -, MAX_HBR_QUEUE]

	ble[handle_hbr_dequeue#]


	; Dequeue queue number is a low bit rate queue.

	; Handle low bit rate dequeue case.

	_rr_scheduler_handle_lbr_dequeue(dequeue_queue_number, dequeue_message, schedule_queue_sigmask)

	br[schedule_hbr_queue#]


handle_hbr_dequeue#:

	; Handle high bit rate dequeue case.

	_rr_scheduler_handle_hbr_dequeue(dequeue_queue_number, dequeue_message)


schedule_hbr_queue#:

	; Schedule a high bit rate queue if available.

	_rr_scheduler_schedule_hbr_queue(schedule_queue_sigmask)


wait_on_signals#:

	; Wait on all signals and reset signal mask.

	ctx_arb[--], br[loop#], defer[2]

	local_csr_wr[active_ctx_wakeup_events, schedule_queue_sigmask]

	alu[schedule_queue_sigmask, --, b, 1, <<&next_thread_sig]


#endif // _RR_SCHEDULER_UC_