/**
 * ============================================================================
 *  COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2X00 Network Processor, Release 5
 *
 * = FILENAME
 *      ix_cc_eth_mhlp.c
 *
 * = DESCRIPTION
 *      This file contains level-1 only messaging interface APIs and 
 *      related internal async callback functions of Pos Rx CC.
 *
 *
 * = CHANGE HISTORY
 *      11/28/2002 - Created.
 *
 * ============================================================================
 * $Id: ix_cc_eth_rx_msg_hlp.c,v 1.10 2003/03/08 01:41:41 ktseng Exp $
 */

/**
 * System defined include files required.
 */

/**
 * User defined include files required.
 */
#include "ix_cc_error.h"
#include "ix_types.h"
#include "ix_error.h"
#include "ix_rm.h"
#include "ix_ossl.h"
#include "ix_cc.h"

#include "cc/ix_cc_msup.h"
#include "cc/ix_cc_eth_rx.h"
#include "cc/internal/ix_cc_eth_rx_internal.h"
#include "bindings.h"




/**
 * Variable declarations global to this file only.  Extern are followed
 * by static variables.
 */
extern ix_cc_eth_rx_ctl_blk *g_pCcEthRxCtlBlk; /* control block */



/**
 * Pre-processor symbols and macros used in this file.
 */



/**
 * Extern function prototypes and variables
 */



/**
 * Static function prototypes 
 */
PRIVATE ix_error _ix_cc_eth_rx_icb_get_statistics_info(ix_error, void *, void *, ix_uint32);
PRIVATE ix_error _ix_cc_eth_rx_icb_get_interface_state(ix_error, void *, void *, ix_uint32);
PRIVATE ix_error _ix_cc_eth_rx_icb_general(ix_error arg_Result,
                               void *arg_pContext,
                               void *arg_pMsg,
                               ix_uint32 arg_MsgLength);


PRIVATE ix_error _ix_cc_eth_rx_icb_lookup_port(
                              ix_error arg_Result,
                              void* arg_pContext,
                              void* arg_pMsg,
                              ix_uint32 arg_MsgLength);






/**
 * function definitions.
 */



/**
 * NAME: ix_cc_eth_rx_async_get_statistics_info
 * 
 * DESCRIPTION: This function retrieves statistics (counters) information of 
 *              the Ethernet Rx Core Component.
 * 
 * @Param: arg_pMgInfoContext - IN - pointer to statistics information
 *                              context structure
 * @Param: arg_Callback - IN user callback, we define the signature, user implements it, 
 * 			we call it when we are ready with the reply
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_get_statistics_info(
    ix_cc_eth_rx_statistics_info_context *arg_pMgInfoCtx, 
    ix_cc_eth_rx_cb_get_statistics_info arg_Callback)
{

    ix_cc_eth_rx_cb_ctx_entry *entry;
    ix_error err = IX_SUCCESS;


    /* Check any NULL pointer argument */
    if ((arg_pMgInfoCtx == NULL) || (arg_Callback == NULL))
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL argument"));
    }

    /*
     * Check the ranges of the arg_Entity, where arg_Entity stores 
     * the counter number. 
     */
    if (arg_pMgInfoCtx->entity >= IX_CC_ETH_RX_COUNTER_LAST)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_RANGE, 
                              ("Invalid counter type"));
    }


    /*
     * Enter critical section.  It is possible that the mutex can not 
     * be obtained if component has been destroyed and the mutex was 
     * freed by the fini function.)
     */

    
    IX_ERROR_CRT(ix_ossl_mutex_lock(g_pCcEthRxCtlBlk->mid, IX_OSSL_WAIT_FOREVER),
                 IX_CC_ETH_RX_ERROR_OSSL_MUTEX_LOCK, 
                 IX_ERROR_LEVEL_WARNING);
     
    /*
     * Get a free callback context entry.
     */
    IX_CC_ETH_RX_ALLOC_ENTRY(g_pCcEthRxCtlBlk->pCbCtxTblGetStatInfo->pTable, 
                              g_pCcEthRxCtlBlk->pCbCtxTblGetStatInfo->index, 
                              ETH_RX_NUM_ENTRIES_CB_CTX_TBL_GET_STAT,
                              ix_cc_eth_rx_cb_ctx_entry,
                              entry);

    if (entry == NULL)
    {
		/* Exit critical section */
        ix_ossl_mutex_unlock(g_pCcEthRxCtlBlk->mid);
        return IX_ERROR_WARNING(IX_CC_ERROR_OOM_SYSTEM,
                               ("Out of memory for callback context entry."));
    }

    /* Store the user callback and context into the entry */
    entry->pUserCtx = arg_pMgInfoCtx->pUserContext;
    entry->pUserCb = (void *)arg_Callback;;

    /* Based on the counters requested, send message to appropriate Eth CC */
    switch (arg_pMgInfoCtx->entity)
	{
#ifdef MICROBLOCK_COUNTERS
		case IX_CC_ETH_RX_PACKETS:
		case IX_CC_ETH_RX_DROP_PACKETS:
		case IX_CC_ETH_RX_EXCEPTION_PACKETS:
		case IX_CC_ETH_RX_BYTES:
		case IX_CC_ETH_RX_ALL_MICROBLOCK_COUNTERS:
            /* Call msup to send the message */ 
			IX_ERROR_CGT(ix_cc_msup_send_async_msg(
                               IX_CC_ETH_RX_MSG_INPUT,
                               _ix_cc_eth_rx_icb_get_statistics_info,
                               (void *)entry, 
                               IX_CC_ETH_RX_MSG_GET_STATISTICS_INFO,
                               (void *)arg_pMgInfoCtx,
                               sizeof(ix_cc_eth_rx_statistics_info_context)),
                 err,
                 IX_CC_ETH_RX_ERROR_MSUP_SEND_MSG, 
                 IX_ERROR_LEVEL_WARNING,
                 cleanup);			


		break;
#endif /* MICROBLOCK_COUNTERS */
        
        default:

        /* Call msup to send the message */ 
        IX_ERROR_CGT(ix_cc_msup_send_async_msg(
                 IX_CC_ETH_RX_MSG_OUTPUT,
                 _ix_cc_eth_rx_icb_get_statistics_info,
                 (void *)entry, 
                 IX_CC_ETH_RX_MSG_GET_STATISTICS_INFO,
                 (void *)arg_pMgInfoCtx,
                 sizeof(ix_cc_eth_rx_statistics_info_context)),
                 err,
                 IX_CC_ETH_RX_ERROR_MSUP_SEND_MSG, 
                 IX_ERROR_LEVEL_WARNING,
                 cleanup);	

        break;

	}

cleanup:

    /* if the result is not success then we need to free the entry */ 
    if (err != IX_SUCCESS)
    {
        if (entry != NULL)
        {
            IX_CC_ETH_RX_FREE_ENTRY(entry);
        }
    }

    /* Exit critical section */
    ix_ossl_mutex_unlock(g_pCcEthRxCtlBlk->mid);


    return(err);

} /* ix_cc_eth_rx_async_get_statistics_info */



/**
 * NAME: ix_cc_eth_rx_async_get_interface_state
 * 
 * DESCRIPTION: This function retrieves the interface state of a port.
 * 
 * @Param: arg_pMgInfoContext - IN - Pointer to interface state context
 *                                structure
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_get_interface_state(
    ix_cc_eth_rx_if_state_context *arg_pStateContext, 
    ix_cc_eth_rx_cb_get_interface_state arg_Callback)
{

    ix_cc_eth_rx_cb_ctx_entry *entry;
    ix_error err = IX_SUCCESS;



    /* Check any NULL pointer argument */
    if ((arg_pStateContext == NULL) || (arg_Callback == NULL))
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL argument"));
    }

    /*
     * Enter critical section.  It is possible that the mutex can not 
     * be obtained if component has been destroyed and the mutex was 
     * freed by the fini function.)
     */
    IX_ERROR_CRT(ix_ossl_mutex_lock(g_pCcEthRxCtlBlk->mid, 
                                    IX_OSSL_WAIT_FOREVER),
                 IX_CC_ETH_RX_ERROR_OSSL_MUTEX_LOCK, 
                 IX_ERROR_LEVEL_WARNING);

    /*
     * Get a free callback context entry.
     */
    IX_CC_ETH_RX_ALLOC_ENTRY(g_pCcEthRxCtlBlk->pCbCtxTblGetIfState->pTable, 
                              g_pCcEthRxCtlBlk->pCbCtxTblGetIfState->index, 
                              ETH_RX_NUM_ENTRIES_CB_CTX_TBL_GET_IF,
                              ix_cc_eth_rx_cb_ctx_entry,
                              entry);

    if (entry == NULL)
    {
        /* Exit critical section */
        ix_ossl_mutex_unlock(g_pCcEthRxCtlBlk->mid);

        return IX_ERROR_WARNING(IX_CC_ERROR_OOM_SYSTEM,
                               ("Out of memory for callback context entry"));
    }

    /* Store the user callback and context into the entry */
    entry->pUserCtx = arg_pStateContext->pUserContext;
    entry->pUserCb = (void *)arg_Callback;;

    /* Call msup to send the message */
    IX_ERROR_CT(ix_cc_msup_send_async_msg(
               IX_CC_ETH_RX_MSG_INPUT,
               _ix_cc_eth_rx_icb_get_interface_state,
               (void *)entry,
               IX_CC_ETH_RX_MSG_GET_INTERFACE_STATE,
               (void *)arg_pStateContext,
               sizeof(ix_cc_eth_rx_if_state_context)),
                err,
                IX_CC_ETH_RX_ERROR_MSUP_SEND_MSG, 
                IX_ERROR_LEVEL_WARNING);

    if (err != IX_SUCCESS)
    {
        IX_CC_ETH_RX_FREE_ENTRY(entry);
    }

    /* Exit critical section */
    ix_ossl_mutex_unlock(g_pCcEthRxCtlBlk->mid);

    return(err);

} /* ix_cc_eth_rx_async_get_interface_state */


/**** Additional async calls for MAC Filtering support ****/
/**
 * NAME: ix_cc_eth_rx_async_add_mac_addr
 * 
 * DESCRIPTION: This function adds MAC address to SRAM MAC Filter table.
 * This SRAM MAC filter table is used by Packet Rx microblock for MAC
 * filtering.
 * 
 * @Param: arg_destMac - IN -  MAC address to be added in the table
 * @Param: arg_portNum - IN - destination port number to be added in table
 * @Param: arg_pUserContext:IN - pointer to client defined context
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_add_mac_addr(
    ix_uint8 *arg_destMac,
    ix_uint32 arg_portNum,
    ix_cc_eth_rx_cb_mac_addr_op arg_Callback,
    void *arg_pUserContext)
{

    ix_cc_eth_rx_general_callback_data *pCbContext;
    ix_cc_eth_rx_add_mac_addr_data addMac;
    ix_error err = IX_SUCCESS;

    if (arg_pUserContext == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL context argument"));
    
    }

    /* Check any NULL pointer argument */
    if (arg_Callback == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL callback argument"));
    }

    if (arg_destMac == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                ("NULL destMac argument"));

    }

    pCbContext = (ix_cc_eth_rx_general_callback_data*) ix_ossl_malloc(sizeof(ix_cc_eth_rx_general_callback_data));    
    if(pCbContext == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_OOM,("No memory for context"));
    }/* end if no memory for context */


    ix_ossl_memcpy(addMac.destMac,arg_destMac,6 * sizeof(ix_uint8));
    addMac.portNum = arg_portNum;     
    
    /* Store the user callback and context into pCbContext */
    pCbContext->pUserContext = arg_pUserContext;
    pCbContext->pUserCallback = arg_Callback;;

    /* Call msup to send the message */
    IX_ERROR_CGT(ix_cc_msup_send_async_msg(
               IX_CC_ETH_RX_MSG_INPUT,
               _ix_cc_eth_rx_icb_general,
               (void *)pCbContext,
               IX_CC_ETH_RX_MSG_ADD_MAC_ADDR,
               (void *)&addMac,
               sizeof(ix_cc_eth_rx_add_mac_addr_data)),
                err,
                IX_CC_ETH_RX_ERROR_MSUP_SEND_MSG, 
                IX_ERROR_LEVEL_WARNING,
                cleanup_context);
    return err;
cleanup_context:
    if(pCbContext != NULL)
    {
        ix_ossl_free(pCbContext);
    }/* end if pCbContext != NULL */
    return err;


} /* ix_cc_eth_rx_async_add_mac_addr */




/**
 * NAME: ix_cc_eth_rx_async_delete_mac_addr
 * 
 * DESCRIPTION: This function helps client to delete MAC address
 *  from SRAM MAC Filter table. This SRAM MAC filter table is used
 *  by Packet Rx microblock for MAC filtering.
 * 
 * @Param: arg_destMac - IN - MAC address to be deleted from the table
 * @Param: arg_pUserContext:IN- pointer to client defined context
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_delete_mac_addr(
    ix_uint8 *arg_destMac,
    ix_cc_eth_rx_cb_mac_addr_op arg_Callback,
    void *arg_pUserContext)
{

    ix_cc_eth_rx_general_callback_data *pCbContext;
    ix_cc_eth_rx_delete_mac_addr_data deleteMac;
    ix_error err = IX_SUCCESS;



    /* Check any NULL pointer argument */
    if (arg_Callback == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL argument"));
    }


    if (arg_pUserContext == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL argument"));
    
    }

    if (arg_destMac == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
            ("NULL destMac argument"));

    }
    pCbContext = (ix_cc_eth_rx_general_callback_data*) ix_ossl_malloc(sizeof(ix_cc_eth_rx_general_callback_data));    
    if(pCbContext == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_OOM,("No memory for context"));
    }/* end if no memory for context */


    ix_ossl_memcpy(deleteMac.destMac, arg_destMac, 6 * sizeof(ix_uint8));
          
    
    /* Store the user callback and context into pCbContext */
    pCbContext->pUserContext = arg_pUserContext;
    pCbContext->pUserCallback = arg_Callback;;

    /* Call msup to send the message */
    IX_ERROR_CGT(ix_cc_msup_send_async_msg(
               IX_CC_ETH_RX_MSG_INPUT,
               _ix_cc_eth_rx_icb_general,
               (void *)pCbContext,
               IX_CC_ETH_RX_MSG_DELETE_MAC_ADDR,
               (void *)&deleteMac,
               sizeof(ix_cc_eth_rx_delete_mac_addr_data)),
                err,
                IX_CC_ETH_RX_ERROR_MSUP_SEND_MSG, 
                IX_ERROR_LEVEL_WARNING,
                cleanup_context);

    return err;
cleanup_context:
    if(pCbContext != NULL)
    {
        ix_ossl_free(pCbContext);
    }/* end if pCbContext != NULL */
    return err;


} /* ix_cc_eth_rx_async_delete_mac_addr */


/**
 * NAME: ix_cc_eth_rx_async_lookup_port
 * 
 * DESCRIPTION: This function helps client to lookup
 *  SRAM MAC Filter table based on MAC address and find
 *  the forwarding port number. This SRAM MAC filter table is used
 *  by Packet Rx microblock for MAC filtering.
 * 
 * @Param: arg_destMac - IN - MAC address to be looked up in the table
 * @Param: arg_pUserContext:IN - pointer to client defined context
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_lookup_port(
    ix_uint8 *arg_destMac,
    ix_cc_eth_rx_cb_lookup_port arg_Callback,
    void *arg_pUserContext)
{

    ix_cc_eth_rx_lookup_callback_port_data *pCbContext;
    ix_cc_eth_rx_lookup_port_data lookupMac;
    ix_error err = IX_SUCCESS;
    
    /* Check for Null user context */
    if (arg_pUserContext == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL user argument"));
    
    }

    /* Check for NULL callback argument */
    if (arg_Callback == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                              ("NULL callback argument"));
    }

    /* Check for Null dest Mac argument */
    if (arg_destMac == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
               ("NULL destMac argument"));

    }

    pCbContext = (ix_cc_eth_rx_lookup_callback_port_data*) ix_ossl_malloc(sizeof(ix_cc_eth_rx_lookup_callback_port_data));    
    if(pCbContext == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_OOM,("No memory for context"));
    }/* end if no memory for context */

    ix_ossl_memcpy(lookupMac.destMac ,arg_destMac, 6 *sizeof(ix_uint8));
       
    
    /* Store the user callback and context into pCbContext */
    pCbContext->pUserContext = arg_pUserContext;
    pCbContext->pUserCallback = arg_Callback;

    /* Call msup to send the message */
    IX_ERROR_CGT(ix_cc_msup_send_async_msg(
               IX_CC_ETH_RX_MSG_INPUT,
               _ix_cc_eth_rx_icb_lookup_port,
               (void *)pCbContext,
               IX_CC_ETH_RX_MSG_LOOKUP_PORT,
               (void *)&lookupMac,
               sizeof(ix_cc_eth_rx_lookup_port_data)),
                err,
                IX_CC_ETH_RX_ERROR_MSUP_SEND_MSG, 
                IX_ERROR_LEVEL_WARNING,
                cleanup_context);

    return err;
cleanup_context:
    if(pCbContext != NULL)
    {
        ix_ossl_free(pCbContext);
    }/* end if pCbContext != NULL */
    return err;


} /* ix_cc_eth_rx_async_lookup_port */



/**********************
 * Internal functions *
 **********************/


/**
 * NAME: _ix_cc_eth_rx_icb_get_statistics_info
 * 
 * DESCRIPTION: This function is an internal callback for the async 
 *              messaging API, ix_cc_eth_rx_async_get_statistics_info().
 * 
 * @Param: arg_Result - IN - result of call
 * @Param: arg_pContext - IN - retrieved user context number
 * @Param: arg_Msg - IN - message
 * @Param: arg_MsgLen - IN - length of message
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
PRIVATE ix_error _ix_cc_eth_rx_icb_get_statistics_info(
    ix_error arg_Result,
    void *arg_pContext,
    void *arg_Msg,
    ix_uint32 arg_MsgLen)
{
    ix_cc_eth_rx_cb_get_statistics_info userCb;
    void *userCtx;

    /* Retrieve the user callback and context from the context pointer */
    userCb = ((ix_cc_eth_rx_cb_ctx_entry*) arg_pContext)->pUserCb;
    userCtx = ((ix_cc_eth_rx_cb_ctx_entry *)arg_pContext)->pUserCtx;

    /*
     * Enter critical section.  It is possible that the mutex can not 
     * be obtained if component has been destroyed and the mutex was 
     * freed by the fini function.
     */
    IX_ERROR_CRT(ix_ossl_mutex_lock(g_pCcEthRxCtlBlk->mid, IX_OSSL_WAIT_FOREVER),
                 IX_CC_ETH_RX_ERROR_OSSL_MUTEX_LOCK, 
                 IX_ERROR_LEVEL_WARNING);

    /* Free the callback context entry */
    IX_CC_ETH_RX_FREE_ENTRY((ix_cc_eth_rx_cb_ctx_entry *)arg_pContext);

    /* Exit the critical section */
    ix_ossl_mutex_unlock(g_pCcEthRxCtlBlk->mid);

    /* Invoke the user callback to pass back the result and data */
    IX_ERROR_CRT((userCb)(arg_Result, userCtx, (ix_uint64 *)arg_Msg, arg_MsgLen), 
		IX_CC_ETH_RX_ERROR_USER_CB, IX_ERROR_LEVEL_WARNING);

    return IX_SUCCESS;

} /* _ix_cc_eth_rx_icb_get_statistics_info */


/**
 * NAME: _ix_cc_eth_rx_icb_get_interface_state
 * 
 * DESCRIPTION: This function is an internal callback for the async 
 *              messaging API, ix_cc_eth_rx_async_get_interface_state().
 * 
 * @Param: arg_Result - IN - result of call
 * @Param: arg_pContext - IN - retrieved user context number
 * @Param: arg_Msg - IN - message
 * @Param: arg_MsgLen - IN - length of message
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error _ix_cc_eth_rx_icb_get_interface_state(
    ix_error arg_Result,
    void *arg_pContext,
    void *arg_Msg,
    ix_uint32 arg_MsgLen)
{

    ix_cc_eth_rx_cb_get_interface_state userCb;
    void *userCtx;



    /* Retrieve the user callback and context from the context pointer */
    userCb = ((ix_cc_eth_rx_cb_ctx_entry *)arg_pContext)->pUserCb;
    userCtx = ((ix_cc_eth_rx_cb_ctx_entry *)arg_pContext)->pUserCtx;

    /*
     * Enter critical section.  It is possible that the mutex can not 
     * be obtained if component has been destroyed and the mutex was 
     * freed by the fini function.
     */
    IX_ERROR_CRT(ix_ossl_mutex_lock(g_pCcEthRxCtlBlk->mid, 
                                    IX_OSSL_WAIT_FOREVER),
                 IX_CC_ETH_RX_ERROR_OSSL_MUTEX_LOCK, 
                 IX_ERROR_LEVEL_WARNING);

    /* Free the callback context entry */
    IX_CC_ETH_RX_FREE_ENTRY(((ix_cc_eth_rx_cb_ctx_entry *)arg_pContext));

    /* Exit the critical section */
    ix_ossl_mutex_unlock(g_pCcEthRxCtlBlk->mid);

    /* Invoke the user callback to pass back the result and data */
    IX_ERROR_CRT((userCb)(arg_Result, 
                          userCtx, 
                          (ix_cc_eth_rx_if_state *)arg_Msg),
                 IX_CC_ETH_RX_ERROR_USER_CB, 
                 IX_ERROR_LEVEL_WARNING);

    return(IX_SUCCESS);

} /* _ix_cc_eth_rx_icb_get_interface_state */



/**
 * NAME: _ix_cc_eth_rx_icb_genaral
 *
 * DESCRIPTION: This is a general callback function. Message support library 
 * calls this function to return the status of the requested asynchronous operation 
 * @Param:  - IN  arg_Result: result of the asynchronous operation
 * @Param:  - IN  arg_pContext: context provided by caller
 * @Param:  - IN  arg_pMsg: return message
 * @Param:  - IN  arg_MsgLength: length of the message
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 */

PRIVATE ix_error _ix_cc_eth_rx_icb_general(ix_error arg_Result,
                               void *arg_pContext,
                               void *arg_pMsg,
                               ix_uint32 arg_MsgLength)
{

    ix_error err = IX_SUCCESS;
 
    ix_cc_eth_rx_general_callback_data *pCbData = NULL;
 
    pCbData = (ix_cc_eth_rx_general_callback_data*)arg_pContext;
   
    if(pCbData != NULL)
    {
        err = pCbData->pUserCallback(arg_Result,pCbData->pUserContext);
        ix_ossl_free(arg_pContext);
        if(err != IX_SUCCESS)
        {
            return IX_ERROR_WARNING(IX_CC_ETH_RX_ERROR_USER_CB,
                                      ("general callback failed"));
        }/* end if (err != IX_SUCCESS) */
    }/* end if(pCbData != NULL) */
       
    return IX_SUCCESS;


}

/**
 * NAME: _ix_cc_eth_rx_icb_lookup_port
 *
 * DESCRIPTION: This is a callback function for lookup of SRAM filter table
 * and to get forwarding port number. 
 * Message support library calls this function to return the status
 * of the requested asynchronous operation 
 * @Param:  - IN  arg_Result: result of the asynchronous operation
 * @Param:  - IN  arg_pContext: context provided by caller
 * @Param:  - IN  arg_pMsg: return message
 * @Param:  - IN  arg_MsgLength: length of the message
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 */
PRIVATE ix_error _ix_cc_eth_rx_icb_lookup_port(
                              ix_error arg_Result,
                              void *arg_pContext,
                              void *arg_pMsg,
                              ix_uint32 arg_MsgLength)
{

    ix_error err = IX_SUCCESS;
 
    ix_cc_eth_rx_lookup_callback_port_data *pCbData = NULL;
    ix_uint32 *pPort = NULL;



    pCbData = (ix_cc_eth_rx_lookup_callback_port_data*)arg_pContext;
    pPort = (ix_uint32*)arg_pMsg;
    
   
    if(pCbData != NULL)
    {
        err = pCbData->pUserCallback(arg_Result,pCbData->pUserContext,pPort);
        ix_ossl_free(arg_pContext);
        if(err != IX_SUCCESS)
        {
            return IX_ERROR_WARNING(IX_CC_ETH_RX_ERROR_USER_CB,
                                      ("lookup callback failed"));
        }/* end if (err != IX_SUCCESS) */
    }/* end if(pCbData != NULL)*/
       
    return IX_SUCCESS;

}
