/**
 * ============================================================================
 *  COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 *
 * = FILENAME
 *      ix_cc_eth_rx.h
 *
 * = DESCRIPTION
 *      This file will describe the common definitions that will be exported 
 *      by the Ethernet Rx Core Component.
 *
 *
 *
 * = CHANGE HISTORY
 *      11/15/2002 - Created.
 *
 * ============================================================================
 * $Id: ix_cc_eth_rx.h,v 1.24 2003/09/20 01:20:50 ktseng Exp $
 */

#if !defined(__IX_CC_ETH_RX_H__)
#define __IX_CC_ETH_RX_H__


/**
 * System defined include files required.
 */
#include "ix_error.h"
#include "ix_types.h"
#include "ix_cc.h"

#ifndef IX_EXCLUDE_CCI
#include "ix_cci.h"
#endif

#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */

/*#define MICROBLOCK_COUNTERS*/

/**
 * The following defines the symbols to be patched into the microblock.
 */
#ifdef MICROBLOCK_COUNTERS
#define ETH_RX_NUM_SYMBOLS_PATCHED	1
#endif

#define L2_DECAP_NUM_SYMBOLS_PATCHED	2

/* base for the counters maintained between microblock and Core Component */
#define SYMBOL_ETH_RX_COUNTERS_BASE    "PACKET_COUNTERS_SRAM_BASE"

/*This is the base address for the port mode bit map in SRAM.
 Each bit represents one Ethernet port mode.
 Bit definitions are :0 - the port is defined as unicast mode 
1 - the port is defined as promiscuous mode
*/
#define SYMBOL_ETH_RX_PORT_BIT_MAP_BASE	"SRAM_PORT_BIT_MAP_BASE"

/*MAC filtering table base address */
#define SYMBOL_ETH_RX_FLT_TABLE_BASE	"SRAM_FLT_TABLE_BASE"

/**
 * Various definitions for maximum number of ports, SRAM channel 
 * number for the context memory, ME number.
 */
#if (_IX_HARDWARE_TYPE_ == _IX_HW_2800_)

#define IX_CC_ETH_RX_MAX_NUM_ETH_PORTS     32 /* max. number of ETH ports */
#define IX_CC_ETH_RX_MAX_ME_NUM            16  /* max. number of MEv2 */
#define IX_CC_ETH_RX_DEFAULT_SRAM_CH_NUM   2   /* for 32-bit counter */
#define IX_CC_ETH_RX_DEFAULT_SRAM_CH_NUM_DECAP   1 /* for decap tabels */
#define IX_CC_ETH_RX_MAX_NUM_PORTS	   16

#else /* (_IX_HARDWARE_TYPE_ != _IX_HW_2800_) */

#define IX_CC_ETH_RX_MAX_NUM_ETH_PORTS  32 /* max. number of ETH ports */
#define IX_CC_ETH_RX_MAX_ME_NUM     8  /* max. number of MEv2 */
#define IX_CC_ETH_RX_DEFAULT_SRAM_CH_NUM   0
#define IX_CC_ETH_RX_MAX_NUM_PORTS	4

#endif /* (_IX_HARDWARE_TYPE_ == _IX_HW_2800_) */


/* Definitions specific to dumping of MAC Filter Table */
#define IX_CC_ETH_RX_TITLE_STRING                                      \
    "Entry#/ MAC address/ port/ Index\n"
#define IX_CC_ETH_RX_TITLE_SIZE sizeof(IX_CC_ETH_RX_TITLE_STRING)
#define IX_CC_ETH_RX_TERM_STRING "...\n"
#define IX_CC_ETH_RX_TERM_SIZE sizeof(IX_CC_ETH_RX_TERM_STRING)

/*
 * The line size is the number of characters in each line including the \n,
 * but excluding the null terminator.
 * 
 */
#define IX_CC_ETH_RX_LINE_SIZE 35

/**
 * MAC filtering table entry structure. (All structure members are 
 * in network byte order) 
 */
typedef struct ix_s_cc_eth_rx_mac_filter_table_entry
{
    ix_uint8 macAddr[6];    /* MAC address */
    ix_uint16 portId:6,     /* 6-bit output port id in bit[15:10] */
              index:10;     /* 10-bit index in bit[9:0] */
} ix_cc_eth_rx_mac_filter_table_entry;



/*Blade Id is read from the init context passed by system app; no more a registry entry*/
#ifdef IX_INCLUDE_REGISTRY
/**
 * The following defines the static properties in registry
 */
#define PROP_ETH_RX_MEV2_NUM_MASK        "ETH_RX/ETH_RX_MEV2_NUM_MASK"

#define PROP_ETH_RX_FLT_TABLE_BASE       "ETH_RX/ETH_RX_FLT_TABLE_SIZE"

#define PROP_L2_DECAP_MEV2_NUM_MASK      "ETH_RX/L2_DECAP_MEV2_NUM_MASK"

#define PROP_L2_DECAP_PORT_MODE_BIT_MAP    "ETH_RX/L2_DECAP_PORT_MODE_BIT_MAP"

#else
/**
 * The following defines static configuration data.
 * ETH Rx uses these data when system repository is not included
 * in the system.
 */
#if (_IX_HARDWARE_TYPE_ == _IX_HW_2800_)

#define ETH_RX_MEV2_NUM_MASK                    0xC00 /* MEs 0x12 and 0x13 */
#define L2_DECAP_MEV2_NUM_MASK                  0x3007 /* MEs 0, 1, 2, 12, 13 */

#else /* (_IX_HARDWARE_TYPE_ != _IX_HW_2800_) */

#define ETH_RX_MEV2_NUM_MASK            0x00000001 /* default ME0 Note: This is a bitmask used to to 
													 go in a loop across all the uEs with left shift. This gives
													 us an index that is mapped to the g_... global array to get the 
													 actual ME no*/
/* Microengine numbers for L2 Decap microblock */
#define L2_DECAP_MEV2_NUM_MASK			0x66

#endif /* (_IX_HARDWARE_TYPE_ == _IX_HW_2800_) */


/* define for default size of SRAM filter table */
#define IX_CC_ETH_RX_NUM_ENTRIES_MAC_FILTER_TABLE_BASE_BUCKET 256
#define IX_CC_ETH_RX_NUM_ENTRIES_MAC_FILTER_TABLE_OV_BUCKET   IX_CC_ETH_RX_NUM_ENTRIES_MAC_FILTER_TABLE_BASE_BUCKET 
#define IX_CC_ETH_RX_TOTAL_NUM_ENTRIES_MAC_FILTER_TABLE       (IX_CC_ETH_RX_NUM_ENTRIES_MAC_FILTER_TABLE_BASE_BUCKET + IX_CC_ETH_RX_NUM_ENTRIES_MAC_FILTER_TABLE_OV_BUCKET)
#define ETH_RX_SIZE_SRAM_FILTER_TBL	                     (sizeof(ix_cc_eth_rx_mac_filter_table_entry) * IX_CC_ETH_RX_TOTAL_NUM_ENTRIES_MAC_FILTER_TABLE)

/* This sets all ports in promiscuous mode */
#define L2_DECAP_PORT_MODE_BIT_MAP      0xffffffff;

#endif

/* define for the SRAM_PORT_BIT_MAP_BASE size */
#define ETH_RX_PORT_BIT_MAP_SIZE	sizeof(ix_uint32)


/**
 * Various feature options
 */


/*Enable this to get verbose debug information*/
/*#define IX_CC_ETH_RX_DEBUG_ENABLE*/



/*error codes*/
enum  {
IX_CC_ETH_RX_ERROR_USER_CB = IX_ERROR_MAKE_GROUP(CC_ETH_RX), 
IX_CC_ETH_RX_ERROR_CCI_REG_HDLR, 
IX_CC_ETH_RX_ERROR_CCI_REM_HDLR,
IX_CC_ETH_RX_ERROR_RM_REG_HDLR, 
IX_CC_ETH_RX_ERROR_RM_REM_HDLR,
IX_CC_ETH_RX_ERROR_RM_PATCH_SYMBOL, 
IX_CC_ETH_RX_ERROR_RM_GET_PHYS_OFFSET, 
IX_CC_ETH_RX_ERROR_RM_FREE_BUF, 
IX_CC_ETH_RX_ERROR_RM_GET_64BIT_COUNTER,
IX_CC_ETH_RX_ERROR_INVALID_IF_STATE,
IX_CC_ETH_RX_ERROR_MSUP_SEND_REPLY,
IX_CC_ETH_RX_ERROR_MSUP_SEND_MSG,
IX_CC_ETH_RX_ERROR_MSUP_EXTRACT_MSG, 
IX_CC_ETH_RX_ERROR_INVALID_BLADE_ID,
IX_CC_ETH_RX_ERROR_INVALID_RANGE,
IX_CC_ETH_RX_ERROR_OSSL_MUTEX_INIT,
IX_CC_ETH_RX_ERROR_OSSL_MUTEX_FINI,
IX_CC_ETH_RX_ERROR_OSSL_MUTEX_LOCK,
IX_CC_ETH_RX_ERROR_OSSL
};




/* 32-bit counter overflow time */
#define CC_ETH_RX_32BIT_COUNTER_OVERFLOW_TIME  13000 
/* For IXP2400, calculation is for byte counter (packet counter can be read less frequently beacause they overflow 
 * less frequently)
 * 13 sec <= 2^32 / (2.488Gbps/8)*/ 

/* Statistics information context structure */
typedef struct ix_s_cc_eth_rx_statistics_info_context
{
    /* entity specifies the type of statistics information
    requested*/
    ix_cc_eth_rx_statistics_info entity; 
    /* index specifies the port for which statistics 
    is requested */
    ix_uint32 index;
    /* pUserContext is pointer to user provided context
    structure,which will be used by the CC in the callback 
    function for the caller to identify the instance of
    the request */   
    void *pUserContext;
} ix_cc_eth_rx_statistics_info_context;

/* ix_cc_eth_rx_statistics_info_data is used by
 * ix_cc_eth_rx_get_statistics_info library API
 * in which the requested entity is stored 
 */
typedef struct ix_s_eth_rx_cc_statistics_info_data
{
    ix_uint32 dataLength; /* length in bytes of the data buffer */
    void *pDataBuffer;    /* buffer to store the statistics data */
} ix_cc_eth_rx_statistics_info_data;

/* Interface state context structure */
typedef struct ix_s_cc_eth_rx_if_state_context
{
    ix_uint32 portId;
    void *pUserContext;
} ix_cc_eth_rx_if_state_context;


/* Enumeration for interface state */
typedef enum ix_e_cc_eth_rx_if_state
{
    IX_CC_ETH_RX_IF_STATE_DOWN = 0,
    IX_CC_ETH_RX_IF_STATE_UP
} ix_cc_eth_rx_if_state;



/**
 * prototypes for async messaging user callbacks
 */

typedef ix_error (*ix_cc_eth_rx_cb_get_statistics_info)(
    ix_error arg_Result,
    void *arg_pContext,
    ix_uint64 *arg_pBuffer,
    ix_uint32 arg_MsgLen);


typedef ix_error (*ix_cc_eth_rx_cb_get_interface_state)(
    ix_error arg_Result,
    void *arg_pContext,
    ix_cc_eth_rx_if_state *arg_pState);


/**
 * TYPENAME: ix_cc_eth_rx_add_mac_addr_data
 * 
 * DESCRIPTION: This structure defines format for add MAC addr
 *  message helper.
 *
 */
typedef struct ix_s_cc_eth_rx_add_mac_addr_data
{
    ix_uint8 destMac[6]; /*  destination MAC addr */
    ix_uint32 portNum; /* destination port number */
} ix_cc_eth_rx_add_mac_addr_data;

/**
 * TYPENAME: ix_cc_eth_rx_delete_mac_addr_data
 * 
 * DESCRIPTION: This structure defines format for delete 
 * MAC address message helper.
 *
 */
typedef struct ix_s_cc_eth_rx_delete_mac_addr_data
{
    ix_uint8 destMac[6]; /* Mac address to be deleted */
    
} ix_cc_eth_rx_delete_mac_addr_data;


/**
 * TYPENAME: ix_cc_eth_rx_lookup_port_data
 * 
 * DESCRIPTION: This structure defines format for lookup 
 * port information message helper.
 *
 */
typedef struct ix_s_cc_eth_rx_lookup_port_data
{
    ix_uint8  destMac[6]; /* destination MAC address to be looked up */
    
} ix_cc_eth_rx_lookup_port_data;




/**
 * TYPENAME: ix_cc_eth_rx_cb_mac_addr_op
 * 
 * DESCRIPTION: Type definition for add/delete MAC addr Message callbacks.
 *
 */

typedef ix_error (*ix_cc_eth_rx_cb_mac_addr_op)(
                ix_error arg_Result,
                void *arg_pContext);

/**
 * TYPENAME: ix_cc_eth_rx_general_callback_data
 * 
 * DESCRIPTION: This structure defines format for general callback data.
 *
 */
typedef struct ix_s_cc_eth_rx_general_callback_data
{
    ix_cc_eth_rx_cb_mac_addr_op pUserCallback;
    void *pUserContext;
} ix_cc_eth_rx_general_callback_data;


/**
 * TYPENAME: ix_cc_eth_rx_cb_lookup_port
 * 
 * DESCRIPTION: Type definition for port lookup callback
 *
 */

typedef ix_error (*ix_cc_eth_rx_cb_lookup_port)(
                           ix_error arg_Result,
                           void *arg_pContext,
                           ix_uint32 *arg_pPortNum);


/**
 * TYPENAME: ix_cc_eth_rx_lookup_callback_port_data
 * 
 * DESCRIPTION: This structure defines format for lookup callback data.
 *
 */
typedef struct ix_s_cc_eth_rx_lookup_callback_port_data
{
    ix_cc_eth_rx_cb_lookup_port pUserCallback;
    void* pUserContext;
} ix_cc_eth_rx_lookup_callback_port_data;




/**
 * Prototypes for interface functions.
 */
/**
 * NAME: ix_cc_eth_rx_init
 * 
 * DESCRIPTION: This function initializes the Ethernet rx core component.This 
 *				primitive shall be called and returned successfully before 
 *				requesting any service from Ethernet Rx core component. 
 *				This primitive should be called only once to initialize 
 *              Ethernet Rx core component.This function performs allocation
 *				of memory for symbols to be patched, creation of 64 bit 
 *				counters, registration of packet and message handlers,
 *              and allocation and initialization of internal data structures
 *              
 * 
 * @Param: arg_hCc - IN -  handle to Ethernet Rx core component, created by
 *		core component infrastructure; 
 *		this shall be used later to get other services from core 
 *		component infrastructure.
 *
 * @Param: arg_ppContext - IN & OUT  - We return this to CCI which uses this 
 *	 while calling our msg/pkt hdlr. This is a pointer to our (internal) CtrlBlk
 *   Upon successfull completion, it represents the location where the pointer
 *	 to the control block allocated by Ethernet Rx core component will be stored.
 *	 The control block is internal to Ethernet Rx core component and it contains
 *   information about Ethernet Rx internal data structures, allocated memory 
 *   and other relevant information. Later this pointer will be passed into  
 *   ix_cc_eth_rx_fini function for freeing memory when Ethernet Rx Core 
 *   Component is being destroyed.
 *  
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_init(
    ix_cc_handle arg_CcHandle, 
    void **arg_ppContext);

/**
 * NAME: ix_cc_eth_rx_fini
 * 
 * DESCRIPTION: This function terminates the eth rx core component.
 * 
 * @Param: arg_hCc - IN - handle to Ethernet Rx core component, created by
 *		core component infrastructure
 * @Param: arg_pContext - IN - pointer to the control block memory allocated 
 * earlier in ix_cc_eth_rx_init function. 
 * 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_fini(
    ix_cc_handle arg_CcHandle, 
    void *arg_pContext);

/**
 * NAME: ix_cc_eth_rx_msg_handler
 * 
 * DESCRIPTION: This function is the common message handler routine for
 *              Ethernet Rx CC. Ethernet Rx Core component receives messages from
 *				other core components through this message handler function and 
 *				internally it calls appropriate library function
 * 				to process the message. This message handler shall be used 
 *				to update dynamic properties.
 * 
 * @Param: arg_Msg - IN - buffer handle embedding information for the message
 * passed in arg_UserData.
 * @Param: arg_UserData - IN - message type
 * @Param: arg_pContext - IN - pointer to Ethernet Rx core component context 
 * that will be passed to the core component when a message arrives. This context was defined
 * by the core component and passed to core components infrastructure through 
 * ix_cc_eth_rx_init function.
 * 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_msg_handler(
	ix_buffer_handle arg_Msg, 
    ix_uint32 arg_UserData,
    void *arg_pContext);


/**
 * NAME: ix_cc_eth_rx_high_priority_pkt_handler
 *
 * DESCRIPTION: This is the registered function to receive high-priority exception packets
 *				from Ethernet Rx microblock. This function sends the ARP packet to Ethernet
 *				Tx Core Component over PCI.
 * 
 * 
 * @Param:  - IN arg_hDataToken: handle to a buffer which contains exception packets
 *  from Ethernet Rx microblock
 * @Param: - IN arg_ExceptionCode: set to relevant value (ARP packets can be received
 *           as exceptions). Valid exception code is IX_ARP_PACKET and is defined in 
 *           ix_cc_microengines_bindings.h
 * @Param: - IN arg_pContext: pointer to Ethernet Rx core component context that 
 * will be passed to the core component when a packet arrives. This context was defined by
 * the core component and passed to core components infrastructure through 
 * the ix_cc_eth_rx_init function.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_SEND_FAIL
 * 
 */

ix_error ix_cc_eth_rx_high_priority_pkt_handler(
								  ix_buffer_handle arg_hDataBuffer,  
                                  ix_uint32 arg_ExceptionCode,
                                  void* arg_pContext
                                  );


/**
 * NAME: ix_cc_eth_rx_low_priority_pkt_handler
 *
 * DESCRIPTION: This is the registered function to receive low priority exception packets 
 *				from Ethernet Rx microblock. This function sends the non-IP packets to
 *				IX_ETH_RX_OTHER_OUTPUT endpoint.
 * 
 * 
 * @Param:  - IN arg_hDataToken: handle to a buffer which contains exception packets
 *  from Ethernet Rx microblock
 * @Param: - IN arg_ExceptionCode: set to relevant value (non-IP packets
 *           can be received as exceptions).Valid exception code is 
 *           IX_NON_IP_PACKET and is defined in ix_cc_microengines_bindings.h
 * @Param: - IN arg_pContext: pointer to Ethernet Rx core component context that 
 * will be passed to the core component when a packet arrives. This context was defined by
 * the core component and passed to core components infrastructure through 
 * the ix_cc_eth_rx_init function.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_SEND_FAIL
 * 
 */
ix_error ix_cc_eth_rx_low_priority_pkt_handler(
								  ix_buffer_handle arg_hDataBuffer,  
                                  ix_uint32 arg_ExceptionCode,
                                  void* arg_pContext
                                  );

/**
 * NAME: ix_cc_eth_rx_async_get_statistics_info
 * 
 * DESCRIPTION: This function retrieves statistics (counters) information from 
 *              the Ethernet Rx Core Component.
 * 
 * @Param: arg_pMgInfoContext - IN - pointer to statistics information
 *                              context structure
 * @Param: arg_Callback - IN- (user callback)- we define the signature, user
 *                    implements it, 
 * 			we call it when we are ready with the reply
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */

ix_error ix_cc_eth_rx_async_get_statistics_info(
	ix_cc_eth_rx_statistics_info_context *arg_pMgInfoContext, 
	ix_cc_eth_rx_cb_get_statistics_info arg_Callback);

/**
 * NAME: ix_cc_eth_rx_async_get_interface_state
 * 
 * DESCRIPTION: This function retrieves the interface state of a port.
 * 
 * @Param: arg_pMgInfoContext - IN - Pointer to interface state context
 *                                structure
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_get_interface_state(
    ix_cc_eth_rx_if_state_context *arg_pStateContext, 
    ix_cc_eth_rx_cb_get_interface_state arg_Callback);


/**
 * NAME: ix_cc_eth_rx_async_add_mac_addr
 * 
 * DESCRIPTION: This function adds MAC address to SRAM MAC Filter table.
 * This SRAM MAC filter table is used by Packet Rx microblock for MAC
 * filtering.
 * 
 * @Param: arg_destMac - IN -  MAC address to be added in the table
 * @Param: arg_portNum - IN - destination port number to be added in table
 * @Param: arg_pUserContext: pointer to client defined context
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_add_mac_addr(
    ix_uint8 *arg_destMac,
    ix_uint32 arg_portNum,
    ix_cc_eth_rx_cb_mac_addr_op arg_Callback,
    void *arg_pUserContext);



/**
 * NAME: ix_cc_eth_rx_async_delete_mac_addr
 * 
 * DESCRIPTION: This function helps client to delete MAC address
 *  from SRAM MAC Filter table. This SRAM MAC filter table is used
 *  by Packet Rx microblock for MAC filtering.
 * 
 * @Param: arg_destMac - IN - MAC address to be deleted from the table
 * @Param: arg_pUserContext: pointer to client defined context
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_delete_mac_addr(
    ix_uint8 *arg_destMac,
    ix_cc_eth_rx_cb_mac_addr_op arg_Callback,
    void *arg_pUserContext);

/**
 * NAME: ix_cc_eth_rx_async_lookup_port
 * 
 * DESCRIPTION: This function helps client to lookup
 *  SRAM MAC Filter table based on MAC address and find
 *  the forwarding port number. This SRAM MAC filter table is used
 *  by Packet Rx microblock for MAC filtering.
 * 
 * @Param: arg_destMac - IN - MAC address to be looked up in the table
 * @Param: arg_pUserContext: pointer to client defined context
 * @Param: arg_Callback - IN - user callback 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_async_lookup_port(
    ix_uint8 *arg_destMac,
    ix_cc_eth_rx_cb_lookup_port arg_Callback,
    void *arg_pUserContext);


/****** Library APIs ********/
/**
 * NAME: ix_cc_eth_rx_get_statistics_info
 * 
 * DESCRIPTION: This function retrieves statistics/counters value of 
 *              Ethernet Rx CC.
 * 
 * @Param: arg_Entity - IN - type of statistics information
 * @Param: arg_Index - IN - port for which statistics is requested
 * @Param: arg_pBuffer - IN - pointer to ix_cc_eth_rx_statistics_info_data 
 *	structure where the entity requested will be stored.
 * @Param: arg_pContext - IN pointer to the component context allocated
 *                            in the component's init function. 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_get_statistics_info(
    ix_cc_eth_rx_statistics_info arg_Entity,
    ix_uint32 arg_Index,
    ix_cc_eth_rx_statistics_info_data *arg_pBuffer,
    void *arg_pContext);

/**
 * NAME: ix_cc_eth_rx_get_interface_state
 * 
 * DESCRIPTION: This function retrieves the link status of a Ethernet
 *              interface port.
 * 
 * @Param: arg_PortId - IN: port id
 * @Param: arg_pIfState - OUT: location where the interface state will be 
 *                            stored.
 * @Param: arg_pContext - IN: pointer to the component context allocated
 *                            in the component's init function. 
 * 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_get_interface_state(
    ix_uint32 arg_PortId, 
    ix_cc_eth_rx_if_state *arg_pIfState,
    void *arg_pContext);

/**
 * NAME: ix_cc_eth_rx_set_property
 * 
 * DESCRIPTION: This function sets the properties (ony sets the IF_STATE property 
 * (but can be extended).
 * 
 * @Param: arg_PropId - IN: property id
 * @Param: arg_pProperty - IN: This is pointer to the property structure passed in
 * @Param: arg_pContext - IN: pointer to the component context allocated
 *                            in the component's init function. 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */

ix_error ix_cc_eth_rx_set_property(
    ix_uint32 arg_PropId,
    ix_cc_properties *arg_pProperty,
    void *arg_pContext);


/**
 * NAME: ix_cc_eth_rx_add_mac_addr
 * 
 * DESCRIPTION: This function adds MAC address to Filter table
 * 
 * @Param: arg_pDestMac - IN: Mac address to be added to Filter table
 * @Param: arg_PortNum - IN: Port Number to be added to Filter table
 * @Param: arg_pContext - IN: pointer to the component context allocated
 *                            in the component's init function. 
 * 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_add_mac_addr(
    ix_uint8 *arg_pDestMac, 
    ix_uint32 arg_PortNum,
    void *arg_pContext);

/**
 * NAME: ix_cc_eth_rx_del_mac_addr
 * 
 * DESCRIPTION: This function deletes MAC address information
 * from Filter table
 * 
 * @Param: arg_pDestMac - IN: Mac address to be deleted from Filter table
 * @Param: arg_pContext - IN: pointer to the component context allocated
 *                            in the component's init function. 
 * 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_del_mac_addr(
    ix_uint8 *arg_pDestMac, 
    void *arg_pContext);

/**
 * NAME: ix_cc_eth_rx_lookup_port
 * 
 * DESCRIPTION: This function retrieves port information
 * from Filter table
 * 
 * @Param: arg_pDestMac - IN: Mac address to be looked up in
 *  Filter table
 * @Param: arg_pPortNum - OUT:Port information returned from lookup 
 * in Filter table
 * @Param: arg_pContext - IN: pointer to the component context allocated
 *                            in the component's init function. 
 * 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_lookup_port(
    ix_uint8 *arg_pDestMac, 
    ix_uint32 *arg_pPortNum,
    void *arg_pContext);


/**
 * NAME: ix_cc_eth_rx_dmp_mac_filter_table
 * 
 * DESCRIPTION: This function dumps each entry from filter table
 *             into a buffer
 * 
 * @Param: arg_pContext - IN: pointer to the component context allocated
 *                            in the component's init function. 
 * 
 * @Param: arg_pBuffer - OUT:pointer to a block of memory where the routes are
 *                      to be stored; human-readable ASCII-Z string.
 * @Param: arg_BufferSize - IN: size in bytes of the buffer; host byte order.
 * 
 * 
 * @Return: IX_SUCCESS if successful or a valid ix_error for failure.
 */
ix_error ix_cc_eth_rx_dump_mac_filter_table(void *arg_pContext, 
                                            char *arg_pBuffer,
                                            ix_uint32 arg_BufferSize);

#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__IX_CC_ETH_RX_H__) */

