///////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
///////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: qm_shaper.uc
//
//      Purpose: Performs initialization of two combined functionalities, Queue
//               Manager and AAL5 Shaper.
//				 
//      History:
//
//
//      Date            Comment                         By
//      ---------------------------------------------------------------------
//
//      01/03/2003      Created                         Jacek Wysoczynski
//
//
/////////////////////////////////////////////////////////////////////////////

#ifndef QM_SHAPER
#define QM_SHAPER

// Because we want to include only main QM loop we have to exclude 
// part of the file containing #begin directive
#define QM_TAKE_LOOP_ONLY

#include "qm_shaper_defs.h"
#include "qm_atm_code.uc"
#include "qm_shaper_init.uc"

#include <atm_tm_shaper_util.uc>

///////////////////////////////////////////////////////////////////////////////
//		DESIGN STRATEGY:
//
//		This building block combines functionality of the Cell-based Queue
//		Manager and ATM TM4.1 AAL5 Shaper. Because of the design and performance 
//		reasons functionalities are combined in one dispatch loop executed in all
//		contexts. Each enqueue/dequeue performed by the QM have to be serviced
//      by the Shaper.
//
//		QM-2-Shaper interface:
//		In classical Queue Manager, communication with Shaper is performed with
//		use of Next-Neighbor register. Because of the organisation of the code:
//      QM prepares enqueue/dequeue <-> Shaper processes, there is a synchronisation 
//		between both components. Therefore and for the performance reasons we can 
//      use shared register to exchagne data between QM and Shaper.
//		There is no problems with blockings, etc. Such an approach causes each pair:
//      enqueue/dequeue message will be consumed at the same run of the loop. 
//
//		Shaper-2-Scheduler interface:
//		This means that Shaper is able to use the NN ring to communicate 
//		Scheduler. This part is not changed.
//		
//		Scheduller-2-QM
//		This part is not changed. Queue Manager receives dequeue messages via
//		the scratch ring.
//
//		Local Memory sharing:		
//		Queue Manager handles LM in the same way as classical QM. Sheper does not
//		use CAM. It uses one predefined location in LM. It starts with 
//      QM_SHAPER_SH_LM_BASE. This approach causes the LM is reloaded each time
//      new enqueue/dequeue message is processed. The starting LM offset vary
//      on the thread number.
//		
//		CAM:
//		Queue Manager uses CAM to store queue parameters and QD entry points.
//		Shaper does not use CAM and writes/reads parameters to/fron SRAM, every 
//		processed enqueue/dequeue.
//
//		SCRIPTS:
//		To avoid changes in script files, both modules are initialized by the
//		original scripts:
//		* QM     - qm_cell_init.ind
//		* Shaper - atm_tm_init.ind (whole ATM TM 4.1 functionality 
//				   initialization)
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// _qm_shaper
//	 	Description: This is the main macro of the combined Queue Manager and
//		Shaper functionality
//
//	 	Outputs:
//		Inputs:
//		Size: 
//			TBD
///////////////////////////////////////////////////////////////////////////////	
#macro _qm_shaper()
.begin

///////////////////////////////////////////////////
// Definitions specific for Queue Manager
///////////////////////////////////////////////////

// these transfer registers are used to 
// write to transmit scratch ring
.reg volatile write $deq_q, $qa_message
.xfer_order $deq_q, $qa_message

// allocate transfer register for enqueue message 3 LW
.reg read volatile $$enqueuemessage_0, $$enqueuemessage_1, $$enqueuemessage_2
.xfer_order $$enqueuemessage_0, $$enqueuemessage_1, $$enqueuemessage_2 

// This register is used to send transition messages to scheduler
.reg sched_message_enq, sched_message_deq

// register used for the masking operations (can't be global because of the shifts)
.reg qm_sched_deq_port_mask_reg

// this signal is used to signal between threads
// basically every thread waits on the previous thread
// to signal
.sig volatile sig_prev_thread

// this signal is used when writing to a scratch ring.
.sig volatile sig_scratch_w_done

// these signals are used to wait on scratch read to complete
.sig sig_scratch_deq_r_done , sig_scratch_enq_r_done

// when a deq request is issued this signal is used to 
// wait for the request to complete.
.sig sig_deq_done

// these signals are used to wait on q array read issued for enq and
// deq.
.sig sig_q_array_deq_r_done, sig_q_array_enq_r_done

// this signal is used for any reads and writes to SRAM
.sig sram_done

// when a write timestamp value to the SRAM is issued this signal is used to 
// signal the request is complete. In fact we are not interested in its return
.sig volatile sig_timestamp_wr_done

///////////////////////////////////////////////////
// Definitions specific for Shaper
///////////////////////////////////////////////////

// Signal definition
#define next_thread_sig sig_prev_thread

// CSR next thread definition
#define next_thread_sig_csr_val gl_next_context_sig

//Register definitions
.reg mult_fac 			//multiply factor need to divide to obtain the cell tx slot
// mask registers
.reg mask_upper15
.reg mask_for_port_extraction
.reg mask_for_mul_result
.reg mask_for_thmask_extraction   

	.reg @sched_tstamp			// scheduler timestamp low
	.reg @sched_tslot			// scheduler time in transmission slots
	.reg @ring_delay			// The delay in communication between shaper & scheduler
    .reg @_indirect_read9

	.reg visible global $sync_shap
	.reg global $tq_shap $tslo_shap
	.xfer_order $sync_shap $tq_shap $tslo_shap
	.set $sync_shap $tq_shap $tslo_shap
    .reg _gcra_param_sram_base		//base addr in SRAM for storing traffic descriptors

    // This register stores offset of the GCRA parameters in LM
    .reg gcra_lm_offset

    .reg lm_pointer_shaper, @_qm_shaper_lm_base
	///////////////////////////////////////////////////
	// Initialization implementation starts here
	///////////////////////////////////////////////////

	// Initialization of global and absolute registers, Q_array and local memory
	_qm_shaper_init()

	// Perform all actions needed to prepare QM before it enters main loop
	queue_manager_pre_loop()

	///////////////////////////////////////////////////
	// Main Loop of the code
	///////////////////////////////////////////////////
qm_shaper_loop#:

	// In case of Queue Manager and Shaper synchronization we need this additional signal
	_qm_signal_next_ctx(gl_next_context_sig)

	// Here is where the Queue Manager code is executed
	queue_manager_loop()

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;; SHAPER main macro starts here
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	// we call modified macro of the Shaper
	atm_tm_shaper_util(mask_upper15, mult_fac, gcra_lm_offset, @ring_delay, mask_for_thmask_extraction)

	br[qm_shaper_loop#]

.end
#endm

///////////////////////////////////////////////////
// This is where the code begins..................
///////////////////////////////////////////////////
begin#:

	// Main macro of the combined code is entered here
	_qm_shaper()

error#:
	nop ; place holder will only come here if there is an error


#endif //QM_SHAPER
