#ifndef QM_ATM_MESSAGE_UC
#define QM_ATM_MESSAGE_UC

/*******************************************************************************
                             Intel Proprietary

 Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.  
 No part of this program or publication may be reproduced, transmited,
 transcribed, stored in a retrieval system, or translated into any language
 or computer language in any form or by any means, electronic, mechanical,
 magnetic, optical, chemical, manual, or otherwise, without the prior
 written permission of:
                         Intel Corporation
                         2200 Mission College Blvd.
                         Santa Clara, CA  95052-8119
*******************************************************************************/

/*
 *         Change History
 *         --------------
 *
 * Date            Description                                            Whom
 * ------------------------------------------------------------------------------------
 *
 * 03/24/03        Queue Manager message API implementation.            Jacek Wysoczynski
 *
 */                                                                      

#include "qm_atm_include.h"

////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_set_enqhit_message
// Description : This macro checks if enqueue transition message needs to be 
//               sent to scheduler. This is the CAM hit case ie. entry is 
//               already in the Q_Array.
//               If there is a transition SOP bit is also activated. If not
//               SOP bit is cleared.
// Output      : out_message
// Input       : in_enq_queue_num
// Constants   : none
// Size        : 6
// Branches    : 1
////////////////////////////////////////////////////////////////////////////
#macro _qm_set_enqhit_message(in_enq_queue_num, out_message)
.begin

    // Format of the message sent from the Queue Manager to Shaper (Sheduler)
    // In this macro it is an 'out_message'
    //
    //   1    1      1            11              1          17
    // ----------------------------------------------------------------
    // | 1 |ENQ TR| CLP |Cell Count/Pkt Length | SOP |    ENQ Q Num   |  1st Lword (enqueue)
    // ----------------------------------------------------------------
    // set the enqueue queue num in the message. this also sets the MSB
    alu[out_message, @enq_msg2_resv_bit_mask, AND, in_enq_queue_num]

    // read the queue count for a given queue from local memory
    // the index is set during cam_lookup in the previous steps of the phase
    alu[--, --, b, *l$index0]

    // check for enqueue transition - if queue changes its state from empty to non-empty
    //.if(queueCount == 0)
    bne[enq_transition_check_done#], defer[1]

        // write the incremented queue count back to local memory
        alu[*l$index0, *l$index0, +, 1]

    // It's an enqueue transition so set the transition message bit
    // Note that SOP bit is sent always when there is a transition so 
    // mask contains this SOP bit also
    alu[out_message, out_message, OR, @enq_transition_bit_mask]

#ifndef TM4_1    // the message is always sent to scheduler
    // Mark also global transition bit to force sending a message to Shaper
    alu[gl_transition_valid_reg, --, b, 1]
#endif

    // Jump to the end of macro
    br[enq_hit_done#]

enq_transition_check_done#:    

    // No transtion: it can't be SOP - mark it as ZERO
    alu[out_message, out_message, XOR, @enq_deq_sop_bit_mask]

enq_hit_done#:

.end
#endm

////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_set_enqmiss_message
// Description : This macro checks if enqueue transition message needs to be 
//               sent to scheduler. This is the CAM miss case ie. entry was 
//               not in the Q_Array and had to be read in.
//               If there is a transition SOP bit is also activated.
// Output      : out_message
// Input       : in_enq_queue_num
// Constants   : none
// Size        : 5+1(TM4_1)
// Branches   : 1
////////////////////////////////////////////////////////////////////////////
#macro _qm_set_enqmiss_message(in_enq_queue_num, in_qd_array_enq_0, out_message)
.begin

    // Format of the message sent from the Queue Manager to Shaper (Sheduler)
    // In this macro it is an 'out_message'
    //
    //   1    1      1            11              1          17
    // ----------------------------------------------------------------
    // | 1 |ENQ TR| CLP |Cell Count/Pkt Length | SOP |    ENQ Q Num   |  1st Lword (enqueue)
    // ----------------------------------------------------------------

    // write the incremented queue count back to local memory
    // the index is set during cam_lookup in the previous phase.
    alu[*l$index0, 1, +, in_qd_array_enq_0]

    // set the enqueue queue num in the message. this also sets the MSB
    alu[out_message, @enq_msg2_resv_bit_mask, AND, in_enq_queue_num]
    
    // on a CAM miss (in the previous steps of the phase) the read of the 
    // queue descriptor from SRAM into Q_Array returns the
    // queue count and that is the one we are using to compare.
    alu[--, --, b, in_qd_array_enq_0]

    // check for enqueue transition
    //.if( $QDArrayEnq_0 == 0 )
    bne[enq_transition_check_done#]

    // It's an enqueue transition so set the transition message bit
    // Note that SOP bit is sent always when there is a transition so 
    // mask contains this SOP bit also
    alu[out_message, out_message, OR, @enq_transition_bit_mask]

#ifndef TM4_1    // the message is always sent to scheduler
    // Set also global transition    
    alu[gl_transition_valid_reg, --, b, 1]
#endif

    // Jump to the end of macro
    br[enq_miss_end#]

enq_transition_check_done#:

    // No transtion: it can't be SOP - mark it as ZERO
    alu[out_message, out_message, XOR, @enq_deq_sop_bit_mask]

enq_miss_end#:

.end
#endm

////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_dropq_deq_handler
// Description : This macro is used to enq a buffer to the free list. This was
//               originally part of a buffer chain that had to be dropped. The
//               buffer being dequeued from the drop queue is enqueue back to the
//               free list when the cell count goes to zero.
// Output      : 
// Input       : in_deq_queue_num, in_qarray_message
// Constant    : none
// Size        : 10
// Branches    : 3
////////////////////////////////////////////////////////////////////////////
#macro _qm_dropq_deq_handler(in_deq_queue_num, in_qarray_message)
.begin
    .reg cellcount_mask, eopbit_mask

    // check if we have a valid dequeue from the drop Q
    alu[--, --, B, in_qarray_message]
    beq[reset_lm_bit#]

    // now we need to enqueue the dequeued buffer to the free list
    // if the cellcount is zero

    // mask bits other than cell count to zero
    move(cellcount_mask, QM_QAMSG_CELLCNT_MASK)
    alu[--, in_qarray_message, AND, cellcount_mask]
    
    // if the cell count is not zero then buffer has not been evicted from
    // the Q_Array and we have to wait for more dequeue to add the buffer 
    // to the free list
    bne[end_handler#]

    // Add the buffer to the free list
    dl_drop_buffer(in_qarray_message)

    // to decrement packet count stored in local memory we have
    // to check if the buffer has its eop bit set.
    move(eopbit_mask, QM_QAMSG_EOP_MASK)
    alu[--, in_qarray_message, AND, eopbit_mask]
    beq[end_handler#]

reset_lm_bit#:

    //reset the LM bit to 0
    alu[*l$index1, --, b, 0]

end_handler#:

.end
#endm


////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_set_deq_message
// Description : This macro checks if dequeue message needs to be sent to 
//               scheduler. In this macro we also send the transmit message
//               by writing on the scratch ring of the transmit block. The
//               message varies for ingress and egress.
// Output      : out_message
// Input       : in_deq_queue_num, in_qarray_message, in_qarray_entry
// Constant    : none
// Size        : 14
// Branches    : 3
////////////////////////////////////////////////////////////////////////////
#macro _qm_set_deq_message(in_deq_queue_num, in_qarray_message, in_qarray_entry, out_message)
.begin
    .reg eop_cellcount_set, queue_num, temp_deq_q

    // add dequeue queue num to the message
    alu[out_message, --, b, in_deq_queue_num]    

    // this is required to send transmit message on the transmit scratch
    // ring
    alu[$qa_message, --, b, in_qarray_message]

    // move from read to write xfer register to write to scratch ring.
    alu[temp_deq_q, --,b, in_deq_queue_num]

	// SCR6029
	#ifdef ATM_OAM_ENABLED

	// Check if a dummy flag transmit is requested by ATM Tx
	// to send PM cells ? If so then send a FLAG DESCR message
	// Since we are not dequeuing the qarray, there could not be 
	// a transition in this case
			
	br_bclr[*l$index0[3], FLAG_DESCR_TX_BIT_POS, transmit_normal#]

	// write queue count to local memory by decrementing by 1
	alu[*l$index0, *l$index0 , -, 1]

	// Send a dummy transmit request to ATM Tx
	br[just_transmit#], defer[2]

		// Clear the flag descr bit and mark the field as changed
		alu[*l$index0[3], --, B, FLAG_DESCR_CHANGED_BIT]

		// Set buffer handle as the flag descriptor
		alu[$qa_message,--, B, flag_descr_buffer_handle]


transmit_normal#:

	#endif // ATM_OAM_ENABLED

    // get the eop and cell count from the dequeue return message (q_link format)
    //    1     1        6               24
    // -------------------------------------------------
    // | EOP | SOP |   seg_cnt  |   next_pointer       |
    // -------------------------------------------------
    //    1     x    1 1 1 1 1 1  x x x x x ....x x x x 

    // Jump if it is an EOP == 1 and seg_cnt == 0
    //.if( eopCellcountSet == EOPCELLCOUNT_MASK)
    alu[eop_cellcount_set, SOPRESET_MASK, AND, in_qarray_message, >>EOPCELLCOUNT_BITS]
    br!=byte[eop_cellcount_set, 0, EOPCELLCOUNT_MASK, just_transmit#]

    // Because it is a dequeue, and EOP == 1 and seg_cnt == 0, 
    // we have to modify reference counter
    _qm_dec_ref_cnt()

    ////////////////////////////////////////////////////////////////////
    //        ATM SHAPER MESSAGE PREPARE
    ////////////////////////////////////////////////////////////////////

    // Check if there is a transition - moving queue state from non-empty to empty 
    //.if($QDArrayDeq_0 == 1)
    alu[--, *l$index0, -, DEQ_TRANS_CHECK]
    bne[deq_msg_no_trans#], defer[1]
        // write queue count to local memory by decrementing by 1
        // You only do it if cellCount is 1 and eop bit is set
        alu[*l$index0, *l$index0 , -, 1]
        
    // There is a transition
    // set the dequeue transition bit
    alu[out_message, out_message, OR, @deq_transition_bit_mask]

    // Clear SOP bit (always when there is transition it is ZEROed)
    alu[out_message, out_message, XOR, @enq_deq_sop_bit_mask]

    // Set timestamp value in SRAM
    #ifndef QM_SKIP_SETTING_TIMESTAMP
        _qm_set_timestamp(in_deq_queue_num)
    #endif

#ifndef TM4_1    // the message is always sent to scheduler
    // Enable also a global transition bit - causes sending a message to
    // Shaper (Scheduler).
    alu[gl_transition_valid_reg, --, b, 1]
#endif

    // Now, transmit the message
    br[just_transmit#]

deq_msg_no_trans#:
    // There is no transition, we have to make more checks to fully support GFR and
    // SOP setting
    // If seg_cnt is equal ZERO, SOP will be set, in other case it will be cleared

    // Clear SOP bit (always when there is transition it is ZEROed)
    alu[out_message, out_message, XOR, @enq_deq_sop_bit_mask]
    
    // Because eop_cellcount_set contains cell_count and EOP (at the left-most position) 
    // we have to mask EOP and we will have seg_cnt value
    //             24            1     1        6 
    // -------------------------------------------------
    // |                      | EOP | SOP |   seg_cnt  | (eop_cellcount_set format, here)
    // -------------------------------------------------
    //  x x x .... x x x x x x   1     x    0 0 0 0 0 0  ( SOP bit set condition)
    //  0 0 0 .... 0 0 0 0 0 0   1     0    1 1 1 1 1 1  ( mask to retrieve interesting bits)
    alu[eop_cellcount_set, 0xBF, AND, eop_cellcount_set]

    // if ((EOP == 1) && (seg_cnt == 0))
    alu[--, eop_cellcount_set, -, 0x80]
    bne[just_transmit#]

    // Because seg_cnt is equal ZERO and EOP is equal 1 we have to SET SOP bit
    // NOTE: We override SOP set in defer section
    alu[out_message, out_message, OR, @enq_deq_sop_bit_mask]

just_transmit#:

    ////////////////////////////////////////////////////////////////////
    //        ATM TX MESSAGE SEND
    ////////////////////////////////////////////////////////////////////

    // To enable support for multiple ports and CLP bit we have to
    // move port number (11 bits) and CLP bit from dequeue message
    // received from Scheduler to dequeue message sent to ATM TX block
    _qm_copy_port_and_clp(temp_deq_q, $dequeuemessage_0)

    // Because we didn't use the transfer register we have to write its contents now
    alu[$deq_q, --, b, temp_deq_q]

    // send dequeued cell to ATM TX block
    _qm_send_transmit_message($deq_q, in_deq_queue_num)

no_transmit#:

.end

#endm


////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_send_sched_message
// Description : This macro sends the message to the schedular
//               Only enqueue/dequeue transition and invalid dequeue request
//               messages are sent.
// Output      : Nil
// Input       : in_message_enq, in_message_deq
// Constant    : none
// Size        : 4
// Branches    : 2
////////////////////////////////////////////////////////////////////////////
#macro _qm_send_sched_message(in_message_enq, in_message_deq)

#ifdef QM_USE_REG_INTF_TO_SHAPER

    // Instead of writing messages to the NN ring we keep them in registers 
    // used to pass enqueue/dequeue information. These registers can be shared
    // among the QM and i.e. SHAPER
    // Here we just leave the macro blank

#else // not QM_USE_REG_INTF_TO_SHAPER

#ifdef TM4_1
    // Write to Schedular via NN ring
    _qm_write_nn_ring(in_message_enq, in_message_deq) ;2, 1
#else

    // check if message needs to be sent
    alu[--, --, b, gl_transition_valid_reg]
    beq[do_not_send_message#]

    // Write to Schedular via NN ring
    _qm_write_nn_ring(in_message_enq, in_message_deq) ;2, 1

do_not_send_message#:
    
    alu[gl_transition_valid_reg, --,b, 0]

#endif

#endif // QM_USE_REG_INTF_TO_SHAPER
#endm

////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_send_transmit_message
// Description : This macro writes to the transmit scratch ring
//               and also sets the signalmap so to wait on the write signal
// Output      : NIL
// Input       : in_deq_queue_num 
// Size        : 5
// Branches    : 1
////////////////////////////////////////////////////////////////////////////
#macro _qm_send_transmit_message(in_tx_message, in_deq_queue_num)

#if ( defined(TX_PHY_MODE) && ((TX_PHY_MODE == SPHY_4_8) | (TX_PHY_MODE == MPHY_4) ))
    .begin
        .reg jump_value

        // Extract port number
        #define_eval QM_LOCAL_SHIFT (QM_ATM_PORT_BITS_SHIFT - QM_JUMP_SHIFT)

            // Note that we make here a trick (performance). We mask all bits to take only
            // a port number. shift it to the begin, and shift it 2 positions left to place it
            // on the position needed by the jump operation
            alu[jump_value, qm_sched_deq_port_mask_reg, AND, in_deq_queue_num, >>QM_LOCAL_SHIFT]

        #undef QM_LOCAL_SHIFT

        // The write scratch ring macro uses 2 instructions
        jump[jump_value, write_ring0#], targets[write_ring0#, write_ring1#, write_ring2#, write_ring3#]
        
write_ring0#:
        _qm_write_scratch_ring(NUM_WORDS_TX_MESSAGE, in_tx_message, TX_RING_NUMBER_0, QM_RING_OUT_0)
        br[write_done#]
        nop

write_ring1#:
        _qm_write_scratch_ring(NUM_WORDS_TX_MESSAGE, in_tx_message, TX_RING_NUMBER_1, QM_RING_OUT_1)
        br[write_done#]
        nop

write_ring2#:
        _qm_write_scratch_ring(NUM_WORDS_TX_MESSAGE, in_tx_message, TX_RING_NUMBER_2, QM_RING_OUT_2)
        br[write_done#]
        nop

write_ring3#:
        _qm_write_scratch_ring(NUM_WORDS_TX_MESSAGE, in_tx_message, TX_RING_NUMBER_3, QM_RING_OUT_3)
    
write_done#:

    .end

#else //SPHY mode on qm_atm mode

    #ifdef AAL5_TX_EVEN_ODD_PORTS

        // For this operation mode we use two independent AAL5 TX blocks lokated on separate
        // microengines.
        br_bset[in_deq_queue_num, 0, write_odd_queue#]

        _qm_write_scratch_ring(NUM_WORDS_TX_MESSAGE, in_tx_message, TX_RING_NUMBER_1, TX_RING_1)
        br[write_done#]
        nop

    write_odd_queue#:
    
        _qm_write_scratch_ring(NUM_WORDS_TX_MESSAGE, in_tx_message, TX_RING_NUMBER_2, TX_RING_2)
        nop
    
    write_done#:

    #else // AAL5_TX_EVEN_ODD_PORTS

        // write to CSIX TX scrating ring 
        _qm_write_scratch_ring(NUM_WORDS_TX_MESSAGE, in_tx_message, TX_RING_NUMBER, TX_RING)

    #endif // AAL5_TX_EVEN_ODD_PORTS

#endif

#endm

#endif // QM_ATM_MESSAGE_UC