#ifndef QM_ATM_MACRO_UC
#define QM_ATM_MACRO_UC

/*******************************************************************************
                             Intel Proprietary

 Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.  
 No part of this program or publication may be reproduced, transmited,
 transcribed, stored in a retrieval system, or translated into any language
 or computer language in any form or by any means, electronic, mechanical,
 magnetic, optical, chemical, manual, or otherwise, without the prior
 written permission of:
                         Intel Corporation
                         2200 Mission College Blvd.
                         Santa Clara, CA  95052-8119
*******************************************************************************/

/*
 *        Change History
 *         --------------
 *
 * Date            Description                                            Whom
 * ------------------------------------------------------------------------------------
 *
 * 03/24/03        Queue Manager macro implementation file.            Jacek Wysoczynski
 *
 */                                                                      

#include "qm_atm_include.h" ; cel lbased queue manager

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_signal_next_ctx
// Description : Signal next context (thread) in the same ME with the spe-
//               cified signal.
// Output      : Nil
// Input       : in_signal - signal number (0 - 0xF)
// Constant    : Nil
// Size        : 1
///////////////////////////////////////////////////////////////////////////////
#macro _qm_signal_next_ctx(in_signal)

    // WARNING: This macro is used in defer, so do not change its contents!
    local_csr_wr[SAME_ME_SIGNAL, in_signal]        ; Signal the next thread

#endm

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_set_signal_next_ctx_reg
// Description : This Macro sets the next_context_reg which is to be used
//               to signal the next context.
// Output      : Nil
// Input       : Nil
// Constant    : Nil
// Size        : 6
///////////////////////////////////////////////////////////////////////////////
#macro _qm_set_signal_next_ctx_reg()
.begin

    .reg sig_context

    br=ctx[7, ctx_7#]
                
    // for all other context signal next context
    move(sig_context, 0x80)
    alu[gl_next_context_sig, sig_context, OR, &sig_prev_thread, <<3]
    br[sig_reg_setting_done#]
ctx_7#:
    
    // if its ctx 7 then need to signal context 0
    move(sig_context, 0)
    alu[gl_next_context_sig, sig_context, OR, &sig_prev_thread, <<3]

sig_reg_setting_done#:

.end
#endm

////////////////////////////////////////////////////////////////////////
/////////// //////Initialization MACROS  ///////////// //////////////////
/////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Macro Name  :  _qm_nn_init
// Description : Macro to initialize next neighbour index register
// Output      : Nil
// Input       : Nil
// Constant    : Nil
// Size        : 5
///////////////////////////////////////////////////////////////////////////////
#macro _qm_nn_init()
.begin
    
    .reg ctx_enable_data

    // initialization
    // bit 20 in CTX_ENABLE - NN_MODE = 0 : next neighbor register are written
    //                                      by previous ME 
    // bits [19:18] in CTX_ENABLE - NN_RING_EMPTY = 0 : NN_EMPTY asserts when
    //                                                  NN_PUT == NN_GET
    // bits [15:8] CTX enables for contexts 0 to 7
    // initialize local csr CTX_ENABLES to 0xFF00, local csr NN_PUT and NN_GET
    // to 0
    immed[ctx_enable_data, 0xFF00]                ; set low word value
    immed_w1[ctx_enable_data, 0x0000]           ; set high word value

    local_csr_wr[CTX_ENABLES, ctx_enable_data]     ; initialize CTX_ENABLES
    local_csr_wr[NN_PUT, 0]                        ; initialize NN_PUT
    local_csr_wr[NN_GET, 0]    

.end
#endm


///////////////////////////////////////////////////////////////////////////////
// Macro Name  :  _qm_lm_init
// Description : Macro to initialize local memory index register
// Output      : Nil
// Input       : in_num   - active LM address index (0 or 1)
//               in_entry - value to be inserted in LM pointed by LM index
// Size        : 1 
///////////////////////////////////////////////////////////////////////////////
#macro _qm_lm_init(in_num, in_entry)

    local_csr_wr[active_lm_addr_/**/in_num, in_entry]

#endm

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_q_array_init
// Description : Macro to initialize and preload 16 Q_Array entries
// Output      : Nil
// Input       : Nil
// Constant    : Nil
// Size        : 17
///////////////////////////////////////////////////////////////////////////////
#macro _qm_q_array_init()
.begin

    .reg q_num, queue_ptr, qa_cam_num, size, reg, qarray_entry

	// SCR6029 
	#ifdef ATM_OAM_ENABLED
	
	.reg read $qdesc_cell_count, $qdesc_addn_data
	.xfer_order $qdesc_cell_count, $qdesc_addn_data

	#else
    
    .reg read $qdesc_cell_count

	#endif // ATM_OAM_ENABLED

    // set the initial queue number and Q Array entry
    immed[q_num, 0]
    immed[qa_cam_num, QM_QARRAY_BASE]

    // finaly Q Array entry to be initialized
    alu[size, qa_cam_num, +, QM_QARRAY_SIZE]
    immed[reg, 0]

qa_loop#:

    // set the local memory index 0 
    _qm_lm_init(0, reg)
    alu[reg, reg, +, 64]

    // set the channel number
    alu[qarray_entry, gl_channel_num, OR, qa_cam_num, <<SRAM_Q_ARRAY_NUMBER_FIELD]

    .begin 
        .sig  sig_qarray_enq_read_done
        // Read the new queue into Q_Array
        _qm_read_q_array(q_num, qarray_entry, $qdesc_cell_count, sig_qarray_enq_read_done) ; 3, 0

        // wait for read to complete
        ctx_arb[sig_qarray_enq_read_done]
    .end        

    // update CAM with new tag entry
    cam_write[qa_cam_num, q_num, STATUS_VALID]

    // increment queue number
    alu[q_num, q_num, +,1]

    // update local memory with retrieved parameters
    alu[*l$index0[0], --, b, $qdesc_cell_count] ; First word is a cell_count
    alu[*l$index0[2], --,b, QM_REF_CNT]            ; Third is a reference number needed to check whether 
                                                ; update of the QD entry in SRAM should be done
	// SCR6029
	#ifdef ATM_OAM_ENABLED
	alu[*l$index0[3], --, B, $qdesc_addn_data]	; Initialize OAM flags (from addn. data)
	#endif // ATM_OAM_ENABLED

    // increment the QArray entry to be used in next iteration
    alu[qa_cam_num, qa_cam_num, +, 1]

    // Check if it was a last iteration
    alu[--, qa_cam_num, -, size]
    bne[qa_loop#]

go_ahead#:

.end
#endm

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_thread0_init
// Description : Macro to initialize registers, rings and memory used by
//               by Queue Manager. This macro implements only part performed by
//               the first thread,
// Output      : Nil
// Input       : Nil
// Constant    : Nil 
// Size        : 20
///////////////////////////////////////////////////////////////////////////////
#macro _qm_thread0_init()
.begin

     .reg count

    // clear the CAM         
    cam_clear;

    // Initialize absoulte registers that are common to all threads
    // used to store few immediates in global registers for ALU operation
    move(@enq_transition_bit_mask, QM_ENQ_TRANSITION_BIT_MASK)
    move(@enq_deq_sop_bit_mask, QM_ENQ_DEQ_SOP_BIT_MASK)
    move(@invalid_dequeue_bit_mask, QM_INVALID_DEQUEUE_BIT_MASK)
    move(@enq_ind_ref_mask_reg, ENQ_IND_REF_MASK)
    move(@deq_transition_bit_mask, QM_DEQ_TRANSITION_BIT_MASK)
    move(@enq_msg2_resv_bit_mask, QM_ENQ_MSG2_RESV_BIT_MASK)

    // Initialize the NN ring - in LimePoint for OC-12 we are not using NN ring to 
    // exchange data between QM and Scheduler
    _qm_nn_init[]

    // Drop Queue Initialization 
    // Initialize local memory for drop Q
    .begin
        .reg drop_queue_num, qa_cam_num, qarray_entry

		// SCR6029
		#ifdef ATM_OAM_ENABLED
		
		.reg read $qdesc_cell_count, $qdesc_addn_data
		.xfer_order $qdesc_cell_count, $qdesc_addn_data
		
		#else
        
        .reg read $qdesc_cell_count
        .xfer_order $qdesc_cell_count

		#endif // ATM_OAM_ENABLED

        immed32(qa_cam_num,QM_DROP_QUEUE_ENTRY)
        move(drop_queue_num, QM_DROP_QUEUE)

        // set the channel number
        alu[qarray_entry, gl_channel_num, OR, qa_cam_num, <<SRAM_Q_ARRAY_NUMBER_FIELD]
        
        .begin

            .sig sig_qarray_enq_read_done
            // Read the new queue into Q_Array Entry reserved for Drop Queue in 
            _qm_read_q_array(drop_queue_num, qarray_entry, $qdesc_cell_count, sig_qarray_enq_read_done)

            // wait for read to complete
            ctx_arb[sig_qarray_enq_read_done]

        .end
        // update local  memory
        alu[*l$index1, --, b, 0]

    .end
        
.end
#endm


///////////////////////////////////////////////////////////////////////////////
// Macro Name  : qm_one_time_init
// Description : This macro performs all actions that have to be done only once.
// Output      : Nil
// Input       : Nil 
// Size        : 
///////////////////////////////////////////////////////////////////////////////
#macro _qm_one_time_init()
.begin

    /* This signal is declared in system_init() macro */
    .sig visible sys_init_signal

    // We need this register to support SRAM address calculations
    .begin
        .reg qarray_sram_addr

        // Thread 0 does the initialization
        _qm_thread0_init()

        //  Initialize Q_Array 
        _qm_q_array_init()

    .end

    // this is the signal that all blocks are to 
    // wait on to indicate that system initialization 
    // is done
    ctx_arb[sys_init_signal]

.end
#endm


///////////////////////////////////////////////////////////////////////////////
// Macro Name  : qm_common_init
// Description : This macro performs actions that have to be done for all threads.
// Output      : Nil
// Input       : Nil 
// Size        : 
///////////////////////////////////////////////////////////////////////////////
#macro _qm_common_init()
.begin

    // set the next context signal register
    _qm_set_signal_next_ctx_reg()

    // set the channel number for Q_Array
    immed32(gl_channel_num, QD_SRAM_BASE) 
    alu[gl_channel_num, --,b, gl_channel_num, >>SRAM_CHANNEL_NUMBER_FIELD]
    alu_shf[gl_channel_num, --, B, gl_channel_num, <<SRAM_CHANNEL_NUMBER_FIELD]

    .begin
        .reg sram_base

        // Setting a global base register. Need to shift by 2, 
        // because address used in reading and writing QD 
        // instruction uses LW alighned addresses.
        immed32(sram_base, QD_SRAM_BASE)               ; Inserting base
        alu[sram_base, sram_base, and~, 0xC0, <<24]    ; Mask-out channel number
        alu[gl_qd_sram_base_reg, --, b, sram_base,>>2] ; shifting to get LW-alignment

    .end

    // set the override bit used to override defaults in enqueue operation
    move(gl_override_bit, QM_OVERRIDE_BIT)

    // set the drop queue register
    .begin
        .reg msb_set_reg

        move(msb_set_reg, MSB_SET)
        immed32(gl_qm_drop_queue_reg, QD_TOTAL)
        alu[gl_qm_drop_queue_reg, gl_qm_drop_queue_reg, OR, msb_set_reg]

    .end

    // Initialize few more context specific global registers
    move(zero, 0)
    alu[gl_transition_valid_reg, --, b, 0]
    move(gl_addr_ptr_mask_reg, ADDR_PTR_MASK)
    move(gl_enq_deq_queue_num_mask, ENQ_DEQ_QUEUE_NUM_MASK)
    move(gl_enq_deq_vcqNo_mask, QM_ENQ_DEQ_VCQNO_MASK)
    move(gl_enq_deq_vcq_mask, QM_ENQ_DEQ_VCQ_MASK)
    move(qm_sched_deq_port_mask_reg, QM_SCHED_DEQ_PORT_MASK)

    // Initialize local memory for drop Q
    .begin 
        .reg reg

        move(reg, QM_LM_DROP_Q_BASE)
        _qm_lm_init(1, reg)

    .end

.end
#endm

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : qm_init
// Description : Macro to initialize registers, rings and memory used by
//               by Queue Manager
// Output      : Nil
// Input       : Nil 
// Size        : 
///////////////////////////////////////////////////////////////////////////////
#macro qm_init()

    // Perform actions common for all threads
    _qm_common_init()

    .if(ctx() == 0)

        // This part of initialization is performed only once
        _qm_one_time_init()

    .else

        // wait for previous thread/context signal to wake you up
        ctx_arb[sig_prev_thread]
    
    .endif
    
#endm 

////////////////////////////////////////////////////////////////////////////////
/////////// SCRATCH/NN RING READ/WRITE RELATED MACROS //////////////////////////
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_read_scratch_ring
// Description : Macro to read from scratch ring given the ring number and 
//               number of words.
// Output      : out_message
// Input       : in_num_of_words, in_ring_number
// Size        : 1
// Branches    : 0
////////////////////////////////////////////////////////////////////////////////
#macro _qm_read_scratch_ring(in_num_of_words, out_message, in_ring_number)

    // read "numOfWords" from the scratch ring 
    scratch[get, out_message, 0, in_ring_number, in_num_of_words], sig_done[ sig_scratch_r_done]
    
#endm

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_write_scratch_ring
// Description : Macro to write to scratch ring given the ring number,
//                 message and number of words.
// Output      : 
// Input       : in_num_of_words, in_message, in_ring_number, IN_RING
// Size        : 2
// Branches    : 1
///////////////////////////////////////////////////////////////////////////////
#macro _qm_write_scratch_ring[in_num_of_words, in_message, in_ring_number, IN_RING]

// These define_eval are required. Otherwise the caller cannot have spaces
// in between parameters like init[a, b, c].
#define_eval RING IN_RING

scratch_ring_full#:
    // check if transmit ring is full
    br_inp_state[SCR_RING/**/RING/**/_STATUS, scratch_ring_full#]

    // write "numOfWords" to the scratch ring
    scratch[put, in_message, zero, in_ring_number, in_num_of_words], sig_done[sig_scratch_w_done]

#undef RING

#endm

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_read_nn_ring
// Description : Macro to read from next neighbour registers of the ME.
// Output      : out_dequeueMessage
// Input       : Nil 
// Size        : 1
// Branches    : 0
///////////////////////////////////////////////////////////////////////////////
#macro _qm_read_nn_ring(out_dequeue_message)

    // read one LW from the NN ring 
    alu[out_dequeue_message, 0, b, *n$index++]

#endm

///////////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_write_nn_ring
// Description : Macro to write two LWs to the next neighbor registers.
// Output      : Nil
// Input       : in_message_0, in_pkt_length
// Size        : 3
// Branches    : 1
///////////////////////////////////////////////////////////////////////////////
#macro _qm_write_nn_ring(in_message_0, in_message_1)

    // the message sent to scheduler is 1 LW. The LW is for transition 
    // message.

sched_nn_full#:
    br_inp_state[NN_FULL, sched_nn_full#]
    alu[*n$index++, --, b, in_message_0]
    alu[*n$index++, --, b, in_message_1]

#endm

////////////////////////////////////////////////////////////////////////
//////////////////// Q_ARRAY RELATED MACROS ////////////////////////////
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_write_q_array
// Description : Macro to write the cached QDescriptor in Q_Array  back
//               back to SRAM. .
//               Note that this macro doesn't store timestamp
// Output      : Nil
// Input       : in_queue_ptr, in_qarray_entry 
// Size        : 3
// Constant    : nil
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_write_q_array(in_queue_ptr, in_qarray_entry)
.begin 

    .reg shifted_vcq_no
	.reg sram_addr
	.sig sram_wr_sig

    // Get VCQ# to calculate SRAM address of the QDescriptor
    alu[shifted_vcq_no, gl_enq_deq_vcqNo_mask, AND, in_queue_ptr, <<(QD_LOG_SRAM_SIZE - 2)]

    // Add the starting offset to the address
    alu[qarray_sram_addr, gl_qd_sram_base_reg, +, shifted_vcq_no]

    // Write data from Queue Array to SRAM
    sram[wr_qdesc, --, in_qarray_entry, qarray_sram_addr] 

	// SCR6029
	#ifdef ATM_OAM_ENABLED
	
	// Has the flag descriptor bit changed in local memory since
	// it was read out ? If it has then writback the bit to memory
	br_bclr[*l$index0[3], FLAG_DESCR_CHANGED_BIT_POS, no_writeback#]

	// The bit is stored in the optional data i.e. the 4th LW 
	// (offset 12) in the QD SRAM memory.
	move(sram_addr, QD_SRAM_BASE)
	
	// Store the status of the flag descriptor transmission request
	// in the queue descriptor for this queue (store in optional data)
	// Get the byte address from LW address for this QD entry
	alu[shifted_vcq_no, --, B, shifted_vcq_no, <<2]
	alu[shifted_vcq_no, shifted_vcq_no, +, (LWOFFSET3 * BYTES_PER_LW)]
	br_bclr[*l$index0[3], FLAG_DESCR_SET_BIT_POS, clear_flag#]

	sram[set, $flag_descr_set_bit, sram_addr, shifted_vcq_no], sig_done[sram_wr_sig]
	br[no_writeback#]

clear_flag#:
	sram[clr, $flag_descr_set_bit, sram_addr, shifted_vcq_no], sig_done[sram_wr_sig]

no_writeback#:
	.io_completed sram_wr_sig
	#endif // ATM_OAM_ENABLED

.end
#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_read_q_array
// Description : Macro to read and cache a Queue Descriptor entry from 
//               SRAM to Q_Array. $QDArray is actually packet count.
// Output      : out_qd_array
// Input       : in_queue_ptr, in_qarray_entry, in_qa_signal
// Constant    : nil
// Size        : 4
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_read_q_array(in_queue_ptr, in_qarray_entry, out_qd_array, in_qa_signal)

    .begin

        .reg shifted_vcq_no

        // Get VCQ# to calculate SRAM address of the QDescriptor
        alu[shifted_vcq_no, gl_enq_deq_vcqNo_mask, AND, in_queue_ptr, <<(QD_LOG_SRAM_SIZE - 2)]

        // Add the starting offset to the address
        alu[qarray_sram_addr, gl_qd_sram_base_reg, +, shifted_vcq_no]
    
    .end

	// SCR6029
	#ifndef ATM_OAM_ENABLED
		#define_eval NUM_LW 2	// Read only the cell count and the pointer from QD
	#else // ATM_OAM_ENABLED
		// An addittional bytes is used to store the status of flag descriptor transmit
		// request pending state in the queue descriptor for the queue.
		#define_eval NUM_LW 3	// Also read one LW of optional data also from the QD
	#endif // ATM_OAM_ENABLED

    // read the complete queue descriptor from the sram.
    sram[rd_qdesc_head, out_qd_array, in_qarray_entry, qarray_sram_addr, NUM_LW], sig_done[in_qa_signal]
    sram[rd_qdesc_other, --, in_qarray_entry, qarray_sram_addr]

#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_dequeue
// Description : Macro to issue dequeue requests to the SRAM Controller 
//                 to dequeue a cell from a queue. The queue should be 
//                 cached in the QArray. 
// Output      : out_qarray_message
// Input       : in_qarray_entry 
// Constant    : none
// Size        : 2
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_dequeue(in_qarray_entry, out_qarray_message)
      
    sram[dequeue, out_qarray_message, in_qarray_entry, 0], sig_done[sig_deq_done]

#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_enqueue
// Description : Macro to issue enqueue requests to the SRAM Controller 
//               to enqueue a packet to a queue. The queue should be 
//               cached in the QArray. There are more instructions here
//               which are primarily used to set the override bit using
//               indirect reference.
// Output      : Nil
// Input       : in_enqueueMessage_0,  in_enqueueMessage_1,   in_qarray_entry
// Size        : TBD
// Branches    : TBD
////////////////////////////////////////////////////////////////////////
#macro _qm_enqueue(in_enqueue_message_0, in_enqueue_message_1, in_qarray_entry)
.begin

    .reg indirect_ref_contents
          
    // have to mask of the cell count, eop and sop bit for h/w use
    // qarray_sram_addr stores SOP Buffer handle
    alu[qarray_sram_addr, in_enqueue_message_0, AND, gl_addr_ptr_mask_reg]

    // Prepare indirect reference register
    //        7       1    1     1        6              12
    // -------------------------------------------------------------
    // |     RES   | OV | EOP | SOP |  seg_cnt    |      RES       |
    // -------------------------------------------------------------
    //  0 0 ... 0 0   1    0     0    X X X X X X   0 0 0 ... 0 0 0
    
    // setting up the indirect reference by setting the override bit
    alu[indirect_ref_contents, gl_override_bit, OR, in_enqueue_message_0, >>12]

    // set the non required bits to zero for indirect reference
    alu[--, indirect_ref_contents, AND, @enq_ind_ref_mask_reg]

    // enqueue. we use enqueue tail along with it else will have to use
    // branch. It adds to memory transaction.
    sram[enqueue, --, in_qarray_entry, qarray_sram_addr], indirect_ref

    alu[--, IX_NULL, -, in_enqueue_message_1]
    beq[one_buffer_only#]

    // have to mask off the cell count, eop and sop bit for h/w use
    // qarray_sram_addr stores EOP Buffer handle
    alu[qarray_sram_addr, in_enqueue_message_1, AND, gl_addr_ptr_mask_reg]

    sram[enqueue_tail, --, in_qarray_entry, qarray_sram_addr]

one_buffer_only#:

.end
#endm
 

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_cam_check
// Description : This Macro checks if the requested queue's QDescriptor 
//               is cached in the Q_Array. If it is then returns just the
//               cam_entry. If not then a request is sent to evict an entry
//               from Q_Array if required and read in the required queue's
//               QD into the Q_Array and update the CAM accordingly. This macro
//               also sets the signalmap based on hit or miss which later defines
//               which signals to wait on.
//               If QM_WITH_SHAPER constant is defined, this macro also performs
//               swapping of GCRA_PARAM table entry
// Output      : out_qd_array, inout_qarray_entry [,inout_gcra_param_write, inout_gcra_param_read]
// Input       : in_queue_id,  in_hit_label, inout_qarray_entry, in_qa_signal, in_lmnum
//               [, in_additional_sig1, in_additional_sig2]
// Size        : 13 [+21]
// Branches    : 1
// Worst Case  : 13 [+21]
////////////////////////////////////////////////////////////////////////
#macro _qm_cam_check(in_queue_id, in_hit_label, inout_qarray_entry, out_qd_array, inout_gcra_param_write, inout_gcra_param_read, \
                     in_qa_signal, in_lmnum, in_additional_sig1, in_additional_sig2)
.begin

    .reg cam_result, cam_tag, cam_entry

    // do a cam lookup to see if entry is present
    cam_lookup[cam_result, in_queue_id], lm_addr/**/in_lmnum[0]

    // get the cam entry
    GET_CAM_ENTRY

    // setting the channelnum and q array entry. This is done before the branch
    // since it might be required for enqueue and dequeue operations in hit case.
    alu[inout_qarray_entry, gl_channel_num, OR, cam_entry, <<SRAM_Q_ARRAY_NUMBER_FIELD]

    // Get CAM tag. The reason this instruction has been moved out of branch since it 
    // cannot be the next instruction after br_bset. This instruction is not required
    // in a hit case though. 
    cam_read_tag[cam_tag, cam_entry] ;
    
    // check if result shows hit or miss
    
    // if it is a HIT jump to the 'in_hit_label' outside this macro
    br_bset[cam_result, 7, in_hit_label] 

    ; It is a MISS case
    
    // Write back the queue information 
    _qm_write_q_array(cam_tag, inout_qarray_entry)

	// SCR6029
	#ifdef ATM_OAM_ENABLED

	// Initialize the flag descriptor tx pending bit to 0 in local memory
	// This will be updated with the correct state when the QD is actually
	// loaded and used to initialize the LM.
	alu[*l$index0[3], --, B, 0]

	#endif

    // Read the new queue into Q_Array
    _qm_read_q_array(in_queue_id, inout_qarray_entry, out_qd_array, in_qa_signal)

    #ifdef QM_WITH_SHAPER
        //Swap GCRA param table
        .begin
            .reg gcra_offset, lm_addr, lm_offset

            // Set LM index to shaper LM space
            alu[gcra_offset, --, B, cam_tag, <<GCRA_ENTRY_SIZE_LOG]
            alu[lm_offset, --, B, cam_entry, <<GCRA_ENTRY_SIZE_LOG]
            alu[lm_addr, @_qm_shaper_lm_base, +, lm_offset]
            local_csr_wr[active_lm_addr_/**/in_lmnum, lm_addr]
            nop
            nop
            nop

            // Write GCRA param table entry to SRAM
            alu[inout_gcra_param_write[0], --, B, *l$index/**/in_lmnum[0]]
            alu[inout_gcra_param_write[1], --, B, *l$index/**/in_lmnum[1]]
            alu[inout_gcra_param_write[2], --, B, *l$index/**/in_lmnum[2]]
            alu[inout_gcra_param_write[3], --, B, *l$index/**/in_lmnum[3]]
            alu[inout_gcra_param_write[4], --, B, *l$index/**/in_lmnum[4]]

            sram[write, inout_gcra_param_write[0], _gcra_param_sram_base, gcra_offset, 5], sig_done[in_additional_sig1]

            // Issue read of a new entry from SRAM
            alu[gcra_offset, --, B, in_queue_id, <<GCRA_ENTRY_SIZE_LOG]
            alu[--, --, b, @_indirect_read9]
            sram[read, inout_gcra_param_read[0], _gcra_param_sram_base, gcra_offset, max_9], indirect_ref, sig_done[in_additional_sig2]

            // Restore LM index, so it can be used by queue manager macros
            alu[lm_addr, --, B, cam_entry, <<6]
            local_csr_wr[active_lm_addr_/**/in_lmnum, lm_addr]
            nop
            nop
            nop
        .end
    #endif
        
    // Update CAM with new tag entry
    cam_write[cam_entry, in_queue_id, STATUS_VALID]

    // Set reference counter to the default value
    _qm_set_ref_cnt()

.end
#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_copy_port_and_clp
// Description : Macro used to copy port number and CLP bit value from
//               dequeue message sent from Scheduler to the dequeue message
//               sent to ATM TX component
// Output      : out_atm_tx_message
// Input       : in_sched_message 
// Constant    : none
// Size        : 4
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_copy_port_and_clp(out_atm_tx_message, in_sched_message)
.begin

    // DEQUEUE from Scheduler
    //     1      1         11         1     1            17
    // ----------------------------------------------------------------
    // | Valid | RES |     Port#    | CLP | RES |        VCQ#         |
    // ----------------------------------------------------------------
    //     0      0   1 1 1 ... 1 1    1     0    0 0 0 0 0 0 0 0 0 0   (mask)
    //
    // DEQUEUE to ATM TX
    //     1      1         11         1       2           16
    // ----------------------------------------------------------------
    // | Valid | RES |     Port#    | CLP |   RES   |      VCQ#       |
    // ----------------------------------------------------------------
    //
    // As we can see the only thing is to move 12 bits of data from 
    // one message to another with two bits offset

    .reg deq_sch_msg_port_clp

    // Prepare mask for port number and CLP (it is common for both dequeue messages)
    move(deq_sch_msg_port_clp, DEQ_MSG_PORT_CLP_MASK)

    // Clear bits responsible for carrying Port# and CLP - make sure there are ZEROes
    alu[out_atm_tx_message, out_atm_tx_message, AND~, deq_sch_msg_port_clp]

    // Get port number and CLP bit from dequeue message (from Scheduler)
    alu[deq_sch_msg_port_clp, deq_sch_msg_port_clp, AND, in_sched_message]

    // Store port number and CLP bit in dequeue message (sent to ATM TX)
    alu[out_atm_tx_message, out_atm_tx_message, OR, deq_sch_msg_port_clp]

.end
#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_set_timestamp
// Description : Macro reads value of timestamp and writes it to SRAM.
//               This is an extension needed by WRED module to retrieve
//               time of the last cell departure. Timestamp is set only
//               during transition - queue changes its state from 
//               non-empty to empty.
// Output      : 
// Input       : in_deq_queue_num
// Constant    : none
// Size        : 9
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_set_timestamp(in_deq_queue_num)
.begin
    
    // Retrieve timestamp value and store it in transfer register
    local_csr_rd[TIMESTAMP_LOW]
    immed[$timestamp_0, 0]
    local_csr_rd[TIMESTAMP_HIGH]
    immed[$timestamp_1, 0]

    // Prepare address of the QD in SRAM
    .reg shifted_vcq_no shifted_qd_sram_base_reg

    // Get VCQ# to calculate SRAM address of the QDescriptor
    alu[shifted_vcq_no, gl_enq_deq_vcqNo_mask, AND, in_deq_queue_num, <<QD_LOG_SRAM_SIZE]

    // Add the starting offset to the base address (shifted to get byte-alignment)
    alu[shifted_qd_sram_base_reg, --, b, gl_qd_sram_base_reg, <<2]
    alu[qarray_sram_addr, shifted_vcq_no, +, shifted_qd_sram_base_reg]
    alu[qarray_sram_addr, qarray_sram_addr, +, gl_channel_num]

    // write it to SRAM to the specified QD entry
    sram[write, $timestamp_0, qarray_sram_addr, QM_TIMESTAMP_OFFSET, 2], sig_done[sig_timestamp_wr_done]
    
    // Because we are not interested in operation result we place this ugly .io_complete here
    // Note that to avoid destruction caused by the the signal returning from the SRAM we have
    // to mark this particular signal as volatile. This causes the address of the signal to be 
    // not assigned to any of the operations performed in the uEngine code
    .io_completed sig_timestamp_wr_done
.end
#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_set_signal
// Description : There are branches that does not provide operation that
//               set a signal, however to avoid deadlocks, QM needs the 
//               signal to be set. This macro sets the requested signal.
// Output      : none
// Input       : in_sig_to_be_set
// Constant    : none
// Size        : 4
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_set_signal(in_sig_to_be_set)
.begin
    
    .reg signal_value

    // Get current state of the signals for the microengine
    local_csr_rd[ACTIVE_CTX_SIG_EVENTS]
    immed[signal_value, 0]

    // Change the contents and store new sig state in CSR
    alu_shf[signal_value, signal_value, OR, 0x1, <<in_sig_to_be_set]
    local_csr_wr[ACTIVE_CTX_SIG_EVENTS, signal_value]

.end
#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_delay
// Description : Because of optimization we have to implment macro that
//               will be not optimized and concume requested number of
//               clock cycles.
// Output      : none
// Input       : in_delay_count
// Constant    : none
// Size        : 3
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_delay(in_delay_count)

    // Repeat fake operation for a requested number of times
    #repeat in_delay_count

    // Fake instruction, does nothing and consumes one cycle
    alu[--, --, b, 0]

    #endloop

#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_set_ref_cnt
// Description : This macro sets reference counter to its initial value.
//               Counter is stored in Local Memory. Pointer is set during
//               the CAM_CHECK. 
// Output      : 
// Input       : none
// Constant    : none
// Size        : 1
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_set_ref_cnt()

    // Store initial value of the counter in LM entry 
    alu[*l$index0[2], --, b, QM_REF_CNT]

#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_dec_ref_cnt
// Description : This macro decrements reference counter by one.
//                  Pointer to LM is set during the CAM_CHECK procedure.
// Output      : none
// Input       : none
// Constant    : none
// Size        : 1
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_dec_ref_cnt()

    // Store initial value of the counter in LM entry 
    alu[*l$index0[2], *l$index0[2], -, 0x01]

#endm

////////////////////////////////////////////////////////////////////////
// Macro Name  : _qm_dequeue_flush_q_count_to_sram
// Description : This macro flushes queue count to Queue Descriptor located
//               in SRAM memory. This is made when REF_CNT expires.
//               Remember that qdarray_sram_addr register has to keep SRAM 
//               address in LW-aligned order. However because it is macro
//               called after the dequeue, we have to count this address.
// Output      : 
// Input       : in_qarray_entry         number of queue in QA
// Constant    : none
// Size        : 5
// Branches    : 0
////////////////////////////////////////////////////////////////////////
#macro _qm_dequeue_flush_q_count_to_sram(in_qarray_entry)
.begin

    // Because last QArray operation was DEQUEUE, address stored in qarray_sram_addr
    // does not contain proper value. It was not needed during dequeue operation so
    // we have to calculate it now
    .reg deq_queue_num, shifted_vcq_no

    ; Leave only a queue number (mask all remaining bits except Valid).
    alu[deq_queue_num, gl_enq_deq_queue_num_mask, AND, $dequeuemessage_0]

    // Get VCQ# to calculate SRAM address of the QDescriptor
    alu[shifted_vcq_no, gl_enq_deq_vcqNo_mask, AND, deq_queue_num, <<(QD_LOG_SRAM_SIZE - 2)]

    // Add the starting offset to the address
    alu[qarray_sram_addr, gl_qd_sram_base_reg, +, shifted_vcq_no]
    
    // Store the current queue count in SRAM
    sram[wr_qdesc_count, --, in_qarray_entry, qarray_sram_addr]

    // Because it is flushed, we have to update reference counter
    _qm_set_ref_cnt()

.end
#endm

#ifdef QM_WITH_SHAPER

    ////////////////////////////////////////////////////////////////////////
    // Macro Name  : _qm_store_gcra_param
    // Description : This macro copies GCRA param entry from transfer
    //                  regisrers into LM
    //               
    //               
    //               
    //               
    // Output      : 
    // Input       : in_gcra_param_regs :   prefix of xfers
    //               in_lm_num          :   number of LM CSR to be used
    // Constant    : none
    // Size        : 20
    // Branches    : 0
    ////////////////////////////////////////////////////////////////////////
    #macro  _qm_store_gcra_param(in_gcra_param_regs, in_lm_num)
    .begin
    .reg lm_addr lm_addr_shaper

        // Set LM index so it points to the proper entry in GCRA param table in LM
        local_csr_rd[active_lm_addr_/**/in_lm_num]
        immed[lm_addr, 0]
        alu[lm_addr_shaper, lm_addr, +, @_qm_shaper_lm_base]

        local_csr_wr[active_lm_addr_/**/in_lm_num, lm_addr_shaper]
        nop
        nop
        nop

        alu[*l$index/**/in_lm_num[0], --, B, in_gcra_param_regs[0]]
        alu[*l$index/**/in_lm_num[1], --, B, in_gcra_param_regs[1]]
        alu[*l$index/**/in_lm_num[2], --, B, in_gcra_param_regs[2]]
        alu[*l$index/**/in_lm_num[3], --, B, in_gcra_param_regs[3]]
        alu[*l$index/**/in_lm_num[4], --, B, in_gcra_param_regs[4]]
        alu[*l$index/**/in_lm_num[5], --, B, in_gcra_param_regs[5]]
        alu[*l$index/**/in_lm_num[6], --, B, in_gcra_param_regs[6]]
        alu[*l$index/**/in_lm_num[7], --, B, in_gcra_param_regs[7]]
        alu[*l$index/**/in_lm_num[8], --, B, in_gcra_param_regs[8]]

        // Restore LM index value
        local_csr_wr[active_lm_addr_/**/in_lm_num, lm_addr]
        nop
        nop
        nop

    .end
    #endm

#endif //QM_WITH_SHAPER

#endif //QM_ATM_MACRO_UC