#ifndef QM_ATM_CODE_UC
#define QM_ATM_CODE_UC

/*******************************************************************************
                             Intel Proprietary

 Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.  
 No part of this program or publication may be reproduced, transmited,
 transcribed, stored in a retrieval system, or translated into any language
 or computer language in any form or by any means, electronic, mechanical,
 magnetic, optical, chemical, manual, or otherwise, without the prior
 written permission of:
                         Intel Corporation
                         2200 Mission College Blvd.
                         Santa Clara, CA  95052-8119
*******************************************************************************/

/*
 *		Change History
 * 		--------------
 *
 * Date			Description											Whom
 * ------------------------------------------------------------------------------------
 *
 * 03/24/03    	Queue Manager code                                  Jacek Wysoczynski
 *                                                                      
 */

#include "qm_atm_macro.uc"
#include "qm_atm_message.uc"

/////////////////////////////////////////////////////
// Bunch of global register definitions
/////////////////////////////////////////////////////

// global register used. Most of the registers especially those ending 
// with _reg or _mask are used to store constants. This effort is to reduce an
// instruction to move immediate data into a register. Hopefully the names are
// self explanatory. Also these names will have a corresponding #define with the 
// same name except the "_reg".

.reg	gl_next_context_sig, gl_override_bit, gl_channel_num, zero,  gl_addr_ptr_mask_reg
.reg	gl_qarray_entry_mask_reg , gl_qm_drop_queue_reg, gl_qd_sram_base_reg
.reg	gl_transition_valid_reg, gl_enq_deq_queue_num_mask
.reg	gl_enq_deq_vcqNo_mask gl_enq_deq_vcq_mask

// these are initialized registers used to store 32bitconstants
.reg  	@enq_transition_bit_mask, @deq_transition_bit_mask @enq_msg2_resv_bit_mask
.reg  	@enq_deq_sop_bit_mask, @enq_ind_ref_mask_reg 
.reg	@bits_for_queue_mask, @invalid_dequeue_bit_mask
.reg	@sopreset_mask_reg


//////////////////////////////////////////////////////////////////////////
// Macro Name  : _queue_manager
// Description : This is the main macro where all registers are defined and
//               where main loop of the Queue Manager is called.
// Output      : Nil
// Input       : Nil
// Constant	   : Nil 
// Size        : 82
// Branches    : 19
//////////////////////////////////////////////////////////////////////////
#macro _queue_manager()
.begin

// these transfer registers are used to 
// write to transmit scratch ring
.reg volatile write $deq_q, $qa_message
.xfer_order $deq_q, $qa_message

// allocate transfer register for enqueue message 3 LW
.reg volatile read $$enqueuemessage_0, $$enqueuemessage_1, $$enqueuemessage_2
.xfer_order $$enqueuemessage_0, $$enqueuemessage_1, $$enqueuemessage_2 

// This register is used to send transition messages to scheduler
.reg sched_message_enq, sched_message_deq

// register used for the masking operations (can't be global because of the shifts)
.reg qm_sched_deq_port_mask_reg

// this signal is used to signal between threads
// basically every thread waits on the previous thread
// to signal
.sig volatile sig_prev_thread

// this signal is used when writing to a scratch ring.
.sig volatile sig_scratch_w_done

// these signals are used to wait on scratch read to complete
.sig sig_scratch_deq_r_done
.sig sig_scratch_enq_r_done

// when a deq request is issued this signal is used to 
// wait for the request to complete.
.sig sig_deq_done

// these signals are used to wait on q array read issued for enq and
// deq.
.sig sig_q_array_deq_r_done, sig_q_array_enq_r_done

// this signal is used for any reads and writes to SRAM
.sig sram_done

// When a write timestamp value to the SRAM is issued this signal is used to 
// signal the request is complete. In fact we are not interested in its return
// Note that to avoid destruction caused by the the signal returning from the SRAM we have
// to mark this particular signal as volatile. This causes the address of the signal to be 
// not assigned to any of the operations performed in the uEngine code
// IMPORTANT: Do NOT remove the 'volatile' from the signal definition
.sig volatile sig_timestamp_wr_done

    
	// Initialization of global and absolute registers, Q_array and local memory
	qm_init()

	// There is macro containing actions that have to be performed before
	// entering the main loop of the Queue Manger
	queue_manager_pre_loop() 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;Loop Begins Here ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

queue_manager_loop#:

	// Here is where the Queue Manager code is executed
	queue_manager_loop()

	br[queue_manager_loop#]

.end
#endm // _queue_manager()

//////////////////////////////////////////////////////////////////////////
// Macro Name  : queue_manager_pre_loop
// Description : This macro contains actions that have to be performed 
//               before Queue Manager enters main loop. (signal setting,
//               initializing registers, etc.).
// Output      : Nil
// Input       : Nil
// Constant	   : Nil 
// Size        : 5
// Branches    : 0
//////////////////////////////////////////////////////////////////////////
#macro queue_manager_pre_loop()
.begin

	// The following 6 instr are executed only  
	// once and are to jump start the loop

	// signal the next ctx
	_qm_signal_next_ctx(gl_next_context_sig) ;1,0

	// Read enqueue message 
	scratch[get, $$enqueuemessage_0, zero, ENQ_RING_NUMBER, NUM_WORDS_ENQ_MESSAGE], \
            sig_done[sig_scratch_enq_r_done]

	//  swap out waiting on scratch reads to complete. 
	ctx_arb[sig_scratch_enq_r_done, sig_prev_thread]

	// signal the next ctx. this is added because the first time there may not be an 
	// enqueue message and before swapping out should signal the next thread.  
	_qm_signal_next_ctx(gl_next_context_sig) ;1,0

	// Signal also first scratch put operation - this sends scratch end signal
	_qm_set_signal(&sig_scratch_w_done)

.end
#endm

//////////////////////////////////////////////////////////////////////////
// Macro Name  : queue_manager_loop
// Description : This is the main macro where the queue manager algorithm
//				 is implemented. The QM is designed to issue an enqueue 
//	 			 dequeue request per beat. Based on this the QM then needs
//				 to send messages to scheduler and Transmit blocks.
// Output      : Nil
// Input       : Nil
// Constant	   : Nil 
// Size        : X
// Branches    : X
//////////////////////////////////////////////////////////////////////////
#macro queue_manager_loop()
.begin

// This register is used to determine the qarray entry 
// we are going to act on for enq/deq purposes.
.reg qarray_entry

// for dequeue message read from scratch ring and message received
// from dequeue
.reg volatile read $dequeuemessage_0  
.reg volatile read $qarray_message

// registers required to read in QDescriptor for dequeue
.reg read $$qd_array_deq_0, $$qd_array_deq_1
.xfer_order	$$qd_array_deq_0, $$qd_array_deq_1

// register required to read in QDescriptor for enqueue
.reg read $$qd_array_enq_0, $$qd_array_enq_1
.xfer_order	$$qd_array_enq_0, $$qd_array_enq_1

// registers required to write timestamp value to QDescriptor
.reg write $timestamp_0, $timestamp_1
.xfer_order	$timestamp_0, $timestamp_1

// this register is required to save the sop handle for enqueue messages
.reg sop_handle, enq_queue_num

// SCR6029 
#ifdef ATM_OAM_ENABLED
	.reg 	write $flag_descr_set_bit
	.reg	flag_descr_buffer_handle
	// AAL5 Tx microengine sends a dummy transmit request with buffer
	// handle set to ATM_OAM_FLAG_DESCR to get cell buffered due to FPM
	// cell generation scheduled. Setup registers to identify this request
	move(flag_descr_buffer_handle, ATM_OAM_FLAG_DESCR)
	alu[$flag_descr_set_bit, --, b, FLAG_DESCR_SET_BIT]
#endif // ATM_OAM_ENABLED

// This register holds SRAM address of the recently used QD
.reg qarray_sram_addr

#ifdef  QM_WITH_SHAPER
    .sig    gcra_read_sig
    .sig    gcra_write_sig
#endif

#ifdef TM4_1

// These registers are used to keep queue number
.reg tm4_1_enq_queue_num

#endif // TM4_1

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;; QUEUE MANAGER main macro starts here
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	// This is where the loop begins. Deq CAM miss and Enq CAM miss in a cycle
	// is the worst case for Queue Manager. Therefore, care has been taken
	// that we take minimum branch penalty. Hence as we read on, the code 
	// executes the worst case without taking any branch.

	// Read dequeue message
	scratch[get, $dequeuemessage_0, zero, DEQ_RING_NUMBER, NUM_WORDS_DEQ_MESSAGE], sig_done[ sig_scratch_deq_r_done]

	; check if enqueue ring is empty. the read from enqueue scratch ring
	; was issued in the previous phase.

	//.if( $$enqueueMessage_0 != 0)
	alu[sched_message_enq, 0, - , $$enqueuemessage_0]
	beq[no_enqueue_request#] 			; if there is no request for enq
										; branch to no_enqueue_request# label


	// remove the cell count from the queue number for enqueuing.
	alu[enq_queue_num, gl_enq_deq_queue_num_mask, AND, $$enqueuemessage_2]


    // this check is for verifying if the enqueue request is for a buffer chain
	// to be dropped. If it matches the drop queue ring number then the special
	// drop queue processsing will be done.
	alu[--, enq_queue_num, -, gl_qm_drop_queue_reg]
	beq[process_drop_queue_enqueue#]

	// check if entry is in CAM. if there is a CAM hit then
	// jump to label enq_hit#. if there is a CAM miss then
	// write back the current CAM entry and read in the
	// new entry based on the queue number. 
	#ifdef TM4_1

		; Use only lower 16 bits as queue id.
		ld_field_w_clr[tm4_1_enq_queue_num, 0011, enq_queue_num]
        #ifndef  QM_WITH_SHAPER
    		_qm_cam_check(tm4_1_enq_queue_num, enq_hit#, qarray_entry, $$qd_array_enq_0, --, --, sig_q_array_enq_r_done, 0, SIG_NONE, SIG_NONE)
        #else
        	xbuf_alloc($gcra_param_read, 9, read);
            xbuf_alloc($gcra_param_write, 5, write)
    		_qm_cam_check(tm4_1_enq_queue_num, enq_hit#, qarray_entry, $$qd_array_enq_0, $gcra_param_write, $gcra_param_read, sig_q_array_enq_r_done, 0, gcra_write_sig, gcra_read_sig)
        #endif

	#else

        #ifndef  QM_WITH_SHAPER
	    	_qm_cam_check(enq_queue_num, enq_hit#, qarray_entry, $$qd_array_enq_0, --, --, sig_q_array_enq_r_done, 0, SIG_NONE, SIG_NONE)
        #else
            xbuf_alloc($gcra_param_read, 9, read)
            xbuf_alloc($gcra_param_write, 5, write)
    		_qm_cam_check(enq_queue_num, enq_hit#, qarray_entry, $$qd_array_enq_0, $gcra_param_write, $gcra_param_read, sig_q_array_enq_r_done, 0, gcra_write_sig, gcra_read_sig)
        #endif
	
	#endif // TM4_1

	// SCR6029 
	#ifdef ATM_OAM_ENABLED
	
	// If the enqueue request is a dummy OAM enqueue request, 
	// do not enqueue to the Q-array. 

	alu[--, $$enqueuemessage_0, XOR, flag_descr_buffer_handle]
	bne[continue_enqueue#]

	// This is a dummy flag descriptor enqueue. 
	// Set the flag descr bit and also mark it 
	// as changed so that it will be written back if swapped out

	br[enq_miss_done#], defer[1]
		alu[*l$index0[3], --, B, FLAG_DESCR_SET_CHANGED_MASK]

	// If not a dummy enqueue request, continue and enqueue to Q-Array
	// Store the flag descriptor state as read from memory into LM.

continue_enqueue#:

	#endif // ATM_OAM_ENABLED

	; If we are here it implies that there is a CAM miss.
	; send an enqueue message
	_qm_enqueue($$enqueuemessage_0, $$enqueuemessage_1, qarray_entry) ; 8, 1

enq_miss_done#:

	//save the sop_handle for future use.
	alu[sop_handle, --,b, $$enqueuemessage_0]

	// then wait for an additional signal. This is the miss path so we
	// wait on Q_Array enq read to complete in addition. Also wait for info about the 
	// succcessful stration of QDescriptors optional_info in SRAM
    #ifndef  QM_WITH_SHAPER
    	ctx_arb[sig_q_array_enq_r_done, sig_scratch_deq_r_done, sig_prev_thread]
    #else
    	ctx_arb[sig_q_array_enq_r_done, sig_scratch_deq_r_done, sig_prev_thread, gcra_read_sig, gcra_write_sig]

        xbuf_free($gcra_param_write)

        _qm_store_gcra_param($gcra_param_read, 0)

        xbuf_free($gcra_param_read)
    #endif

	// SCR6029 
	#ifdef ATM_OAM_ENABLED
	// Store the flag descriptor status read from memory into local memory
	alu[*l$index0[3], --, B, $$qd_array_enq_1]
    #endif


#ifdef QM_DEBUG

	.begin
		.reg deb_queue_num

		// remove the cell count from the queue number for enqueuing.
		alu[deb_queue_num, gl_enq_deq_queue_num_mask, AND, $$enqueuemessage_2]

		// write back the queue information 
		_qm_write_q_array(deb_queue_num, qarray_entry)
	.end

#endif // QM_DEBUG

set_enqmiss_message#:

	// Set the enqueue transtion message. Since this is a miss we depend on
	// $qd_array_enq_0 that contains the queue count, got from return value 
	// when a read Q_Array request was issued in the previous phase. 
	_qm_set_enqmiss_message($$enqueuemessage_2, $$qd_array_enq_0, sched_message_enq)

check_dequeue#:

	// Issue read of enqueue message from scratch ring
	scratch[get, $$enqueuemessage_0, zero, ENQ_RING_NUMBER, NUM_WORDS_ENQ_MESSAGE], sig_done[sig_scratch_enq_r_done]

	; signal the next ctx well before
	_qm_signal_next_ctx(gl_next_context_sig)

	;.if( $dequeueMessage_0 != 0) checking for valid dequeue message. The
	; read for dequeueMessage was issued in the previous phase

    ; store contents of the transfer register in local register.
	alu[sched_message_deq, 0, - , $dequeuemessage_0]
	beq[no_dequeue_request#]


	.begin 
	.reg deq_queue_num


	; Leave only a queue number (mask all remaining bits except Valid).
	alu[deq_queue_num, gl_enq_deq_queue_num_mask, AND, $dequeuemessage_0]

    // check if entry is in CAM. if there is a CAM hit then
	// jump to label deq_hit#. if there is a CAM miss then
	// write back the current CAM entry and read in the
	// new entry based on the queue number. 
	#ifdef TM4_1
	
		.reg tm4_1_deq_queue_num

		; Use only lower 16 bits as queue id.
		ld_field_w_clr[tm4_1_deq_queue_num, 0011, deq_queue_num]

        #ifndef  QM_WITH_SHAPER
      		_qm_cam_check(tm4_1_deq_queue_num, deq_hit#, qarray_entry, $$qd_array_deq_0, --, --, sig_q_array_deq_r_done, 0, SIG_NONE, SIG_NONE)
        #else
            xbuf_alloc($gcra_param_read, 10, read_write)
            xbuf_alloc($gcra_param_write, 5, write)
      		_qm_cam_check(tm4_1_deq_queue_num, deq_hit#, qarray_entry, $$qd_array_deq_0, $gcra_param_write,  $gcra_param_read, sig_q_array_deq_r_done, 0, gcra_write_sig, gcra_read_sig)
        #endif

	#else

        #ifndef  QM_WITH_SHAPER
	    	_qm_cam_check(deq_queue_num, deq_hit#, qarray_entry, $$qd_array_deq_0, --, --, sig_q_array_deq_r_done, 0, SIG_NONE, SIG_NONE)
        #else
            xbuf_alloc($gcra_param_read, 10, read_write)
            xbuf_alloc($gcra_param_write, 5, write)
    		_qm_cam_check(deq_queue_num, deq_hit#, qarray_entry, $$qd_array_deq_0, $gcra_param_write,  $gcra_param_read, sig_q_array_deq_r_done, 0, gcra_write_sig, gcra_read_sig)
        #endif

	#endif	

	// SCR6029
	#ifndef ATM_OAM_ENABLED	

    ; If we are here it implies that there is a CAM miss.
	; send a dequeue message
	_qm_dequeue(qarray_entry, $qarray_message) ;1, 0

	// wait on signals and jump to deqmiss_message when thread comes back
	// Also wait for info about the succcessful stration of QDescriptors optional_info in SRAM
    #ifndef  QM_WITH_SHAPER
    	ctx_arb[sig_scratch_enq_r_done, sig_q_array_deq_r_done, sig_deq_done, \
                sig_scratch_w_done, sig_prev_thread]
    #else
    	ctx_arb[sig_scratch_enq_r_done, sig_q_array_deq_r_done, sig_deq_done, \
                sig_scratch_w_done, sig_prev_thread, gcra_write_sig, gcra_read_sig]

        xbuf_free($gcra_param_write)

        _qm_store_gcra_param($gcra_param_read, 0)

        xbuf_free($gcra_param_read)
    #endif

	.end


#ifdef QM_DEBUG

	.begin 
		.reg deb_queue_num

		; Leave only a queue number (mask all remaining bits except Valid).
		alu[deb_queue_num, gl_enq_deq_queue_num_mask, AND, $dequeuemessage_0]

		// write back the queue information 
		_qm_write_q_array(deb_queue_num, qarray_entry)

	.end

#endif // QM_DEBUG

	; check if dequeue was invalid
	//.if( $$qd_array_deq_0 == 0  )
	alu[*l$index0,--, b, $$qd_array_deq_0]
	beq[send_invalid_dequeue_to_sched#]

	#else  // ATM_OAM_ENABLED

	// If we are here it implies that there is a CAM miss.
	// send a dequeue message

	// wait on signals and jump to deqmiss_message when thread comes back
    #ifndef  QM_WITH_SHAPER
    	ctx_arb[sig_scratch_enq_r_done, sig_q_array_deq_r_done, \
                sig_scratch_w_done, sig_prev_thread]
    #else
    	ctx_arb[sig_scratch_enq_r_done, sig_q_array_deq_r_done, \
                sig_scratch_w_done, sig_prev_thread, gcra_write_sig, gcra_read_sig]

        xbuf_free($gcra_param_write)

        _qm_store_gcra_param($gcra_param_read, 0)

        xbuf_free($gcra_param_read)
    #endif

	.end
	
	// If a request for scheduling a empty slot for sending the cell buffered
	// at AAL5 Tx due to FPM cell generation is pending, dont dequeue the 
	// Q array. Generate a dummy transmit request to get the buffered cell
	// scheduled for transmission.

	br_bclr[$$qd_array_deq_1, FLAG_DESCR_SET_BIT_POS, no_flag_descr#]

	// Request transmission of dummy transmit request
	alu[*l$index0[3], --, B, FLAG_DESCR_TX_BIT]

	// Save the count to local memory
	alu[*l$index0[0], --, b, $$qd_array_deq_0]

	br[set_sch_deq#]

no_flag_descr#:

	// There is no empty slot scheduling request for this queue, issue
	// dequeue to Q array to get the transmit message.

	_qm_dequeue[ qarray_entry, $qarray_message] ;1, 0

	ctx_arb[sig_deq_done]

	; check if dequeue was invalid
	//.if( $qd_array_deq_0 == 0  )
	alu[*l$index0,--, b, $$qd_array_deq_0]
	beq[send_invalid_dequeue_to_sched#]

set_sch_deq#:

    #endif  // ATM_OAM_ENABLED

	; Setup the dequeue transtion message 
	_qm_set_deq_message($dequeuemessage_0,  $qarray_message, qarray_entry, sched_message_deq)

	; Send Message to the scheduler if required
	_qm_send_sched_message(sched_message_enq, sched_message_deq)

	// signal the next ctx (macro takes only one cycle)
	_qm_signal_next_ctx(gl_next_context_sig) ;1,0

	// loop back and read the next enqueue/dequeue request
	br[next_iteration#]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;No Enqueue/Dequeue Request on Scratch case handled here
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

no_enqueue_request#:

	// To avoid scratch controler overload we wait if there is no request
	_qm_delay(QM_NO_REQ_DELAY)

	// zero sop handle
	alu[sop_handle, --, b, 0]
	
	//  swap out waiting on scratch reads to complete. 
	ctx_arb[sig_scratch_deq_r_done, sig_prev_thread]

	br[check_dequeue#]

no_dequeue_request#:

	// Check if there is anything to dequeue from the drop queue
	// well we can always get rid of setting the local memory and always issue
	// a dequeue. We can do this optimization for better performance.
	alu[--, --, b, *l$index1]
	bne[process_drop_queue_dequeue#]

    // Swap out waiting for the ENQUEUE scratch read operation
	ctx_arb[sig_scratch_enq_r_done, sig_prev_thread]

	// To avoid scratch controler overload we wait if there is no request
	_qm_delay(QM_NO_REQ_DELAY)

	// Send Message to the scheduler if required
	// basically we check if we got a valid enqueue message
	// in the previous phase
	alu[--, --, b, sop_handle]
	beq[do_not_send_message#]

	_qm_send_sched_message(sched_message_enq, sched_message_deq)

	// signal the next ctx
	_qm_signal_next_ctx(gl_next_context_sig) ;1,0

	br[next_iteration#]


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;Enqueue/Dequeue CAM hit cases handled here;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	// this is the code path when there is a valid enqueue
	// and there is a CAM hit.

enq_hit#:

	// SCR6029 
	#ifdef ATM_OAM_ENABLED
	// If the enqueue request is a dummy OAM enqueue request, 
	// do not enqueue to the Q-array. 

	alu[--, $$enqueuemessage_0, XOR, flag_descr_buffer_handle]
	bne[continue_enqueue_hit#]

	// This is a dummy enqueue request to create a OAM FPM cell
	// transmission slot. Just mark this event in local memory
	// Do not enqueue this buffer into SRAM q-array			
	br[enq_hit_done#], defer[1]
		alu[*l$index0[3], --, B, FLAG_DESCR_SET_CHANGED_MASK]

	#endif // ATM_OAM_ENABLED

continue_enqueue_hit#:

#ifdef TM4_1

		// Check queue number against beeing one of HBR VCs0
		alu[--, tm4_1_enq_queue_num, -, HBR_THRESHOLD]
		ble[tm4_1enqueue#]

		alu[--, *l$index0, -, QUEUE_THRESHOLD]
		bge[do_not_enqueue#]

tm4_1enqueue#:

		// For the HBR VCs we have to check if queue is too long
		alu[--, *l$index0, -, HBR_QUEUE_THRESHOLD]
		bge[do_not_enqueue#]

#endif

	// send an enqueue message
	_qm_enqueue($$enqueuemessage_0, $$enqueuemessage_1, qarray_entry) 

enq_hit_done#:

	//save the sop_handle for future use.
	alu[sop_handle, --,b, $$enqueuemessage_0]

	//  swap out waiting on scratch read for DEQ to complete and the signal
	//  from previous thread. 
	ctx_arb[sig_scratch_deq_r_done, sig_prev_thread]


#ifdef QM_DEBUG

	.begin
		.reg deb_queue_num

		// remove the cell count from the queue number for enqueuing.
		alu[deb_queue_num, gl_enq_deq_queue_num_mask, AND, $$enqueuemessage_2]

		// write back the queue information 
		_qm_write_q_array(deb_queue_num, qarray_entry)

	.end

#endif // QM_DEBUG

	// Set the enqueue transtion message 
	_qm_set_enqhit_message($$enqueuemessage_2, sched_message_enq)

	// enqueue is done. so now check for dequeue
	br[check_dequeue#]

deq_hit#:

	// SCR6029 
	#ifdef ATM_OAM_ENABLED
	
	// Check if this dequeue needs to be bypassed to send out the flag
	// descriptor buffered. If the flag descriptor bit is set it indicates
	// that the dequeue needs to be bypassed and a dummy transmit request
	// needs to be sent to ATM Tx. This transmit request compensates for
	// a transmit timeslot which was used by ATM Tx to send a PM cell by
	// buffering a user cell. The buffered user cell gets flushed out by
	// this dummy transmit request

	br_bclr[*l$index0[3], FLAG_DESCR_SET_BIT_POS, issue_dequeue_hit#]


	alu[*l$index0[3], --, B, FLAG_DESCR_CHANGED_BIT]
	//  swap out waiting on scratch read for to complete, 
	//  and the signal from previous thread. 

	ctx_arb[sig_scratch_enq_r_done, sig_prev_thread]
	
	alu[*l$index0[3], --, B, FLAG_DESCR_TX_BIT]
	br[dequeue_wait_done#]

issue_dequeue_hit#:

	#endif // ATM_OAM_ENABLED

	// send a dequeue message
	_qm_dequeue(qarray_entry, $qarray_message) ;1, 0

	//  swap out waiting on scratch read for ENQ to complete, DEQ to complete
	//  and the signal from previous thread. On returning branch directly to
	// 	to determine dequeue transition message for the CAM hit case.
	ctx_arb[sig_scratch_enq_r_done, sig_deq_done, sig_scratch_w_done, sig_prev_thread]

#ifdef QM_DEBUG

	.begin 
		.reg deb_queue_num

		; Leave only a queue number (mask all remaining bits except Valid).
		alu[deb_queue_num, gl_enq_deq_queue_num_mask, AND, $dequeuemessage_0]

		// write back the queue information 
		_qm_write_q_array(deb_queue_num, qarray_entry)

	.end

#endif // QM_DEBUG

dequeue_wait_done#:
	// To let the assembler know that scratch write 
	// is completed and these transfer registers can be reused

	.io_completed  $$qd_array_deq_0, $deq_q

	// SCR6029
	#ifdef ATM_OAM_ENABLED

	// If the flag descriptor pending bit is set, set a scheduler 
	// message irrespective of queueu array status

	br_bset[*l$index0[3], FLAG_DESCR_TX_BIT_POS, set_sched_deq_hit#]

	#endif  // ATM_OAM_ENABLED

	// check if dequeue was invalid
	//.if( *l$index0 == 0  )
	alu[--,--,b, *l$index0]
	beq[send_invalid_dequeue_to_sched#]

set_sched_deq_hit#:

	// Set the dequeue transtion message 
	_qm_set_deq_message($dequeuemessage_0, $qarray_message, qarray_entry, sched_message_deq)

	// Send Message to the scheduler if required
	_qm_send_sched_message(sched_message_enq, sched_message_deq)

	// During a HIT procedure we have also to check if it is a time to update QD in SRAM 
	.if (*l$index0[2] == 0)

		// Yes - it is equal ZERO. We have to update QD entry in SRAM 
		_qm_dequeue_flush_q_count_to_sram(qarray_entry)

	.endif

	// signal the next ctx
	_qm_signal_next_ctx(gl_next_context_sig)

	// loop back and read the next enqueue/dequeue request
	br[next_iteration#]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;Enqueue and Dequeue Request for drop queue i.e. dropping a 
;;;;;;;;;;multiple buffer packet is processed here.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

// Handling enqueing to the drop queue

process_drop_queue_enqueue#:

	.begin 
		.reg cell_count_masked ;, cell_count_mask_reg

		// mask the cell count, SOP and EOP from the head handle.
		// This will leave only the buffer handle address
		// this is an optimization that will require
		// only  one dequeue to free that buffer.
		;move[cell_count_mask_reg, ADDR_PTR_MASK]
		;gl_addr_ptr_mask_reg
		;alu[cell_count_masked, cell_count_mask_reg, AND, $$enqueuemessage_0]
		alu[cell_count_masked, gl_addr_ptr_mask_reg, AND, $$enqueuemessage_0]

		// set the Qarry Entry where the drop queue descriptor
		// is stored and where the packet will be enqueued
		alu[qarray_entry,--,b, QM_DROP_QUEUE_ENTRY, <<SRAM_Q_ARRAY_NUMBER_FIELD]
		alu[qarray_entry, qarray_entry, OR, gl_channel_num]

		// Enqueue the packet to the drop queue
		_qm_enqueue(cell_count_masked, $$enqueuemessage_1, qarray_entry)

		// Update drop queue cell count 
		alu[*l$index1, *l$index1, +, 1] 

	.end	

	// zero sop handle
	alu[sop_handle, --, b, 0]
		
	// initialize message register. this is the register
	// that is used to send messages to scheduler.
	alu[sched_message_enq,--, b, 0]

	//  swap out waiting on scratch read for DEQ to complete. 
	ctx_arb[sig_scratch_deq_r_done, sig_prev_thread], br[check_dequeue#]


process_drop_queue_dequeue#:

	// set the Qarry Entry where the drop queue descriptor
	// is stored and where the packet will be enqueued
	// we can store this information in a global register and
	// avoid these two instruction every time.
	alu[qarray_entry,--,b, QM_DROP_QUEUE_ENTRY, <<SRAM_Q_ARRAY_NUMBER_FIELD]
	alu[qarray_entry, qarray_entry, OR, gl_channel_num]

	// This implies there is need to dequeue so issue a dequeue request
	_qm_dequeue(qarray_entry, $qarray_message) ;1, 0

	ctx_arb[sig_scratch_enq_r_done, sig_deq_done, sig_prev_thread]

	// Update local memory regarding drop Q queuecount
	// and also enqueue the dequeued buffer to the free list
	// if all cells have been dequeued
	_qm_dropq_deq_handler(deqQueueNum, $qarray_message)

	// Send Message to the scheduler if required
	alu[--, --, b, sop_handle]
	beq[do_not_send_message#]

	_qm_send_sched_message(sched_message_enq, sched_message_deq)

	// signal the next ctx
	_qm_signal_next_ctx(gl_next_context_sig) ;1,0

	br[next_iteration#]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; Invalid dequeue message and null messages for scheduler
;;;;;;;;;; handled here.  ;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#ifdef TM4_1

do_not_enqueue#:

	// initialize message register. this is the register
	// that is used to send messages to scheduler.
	alu[sched_message_enq, --, b, 0]

	alu[--, IX_NULL, -, $$enqueuemessage_1]
	bne[drop_chained_buffer#]


	dl_drop_buffer($$enqueuemessage_0)

	//save the sop_handle for future use.
	alu[sop_handle, --, b, $$enqueuemessage_0]

	//then wait for an additional signal. 
	ctx_arb[sig_scratch_deq_r_done, sig_prev_thread], br[check_dequeue#]
	
drop_chained_buffer#:

// Drop a chained buffer by writing to QM scratch ring with QM_DROP_QUEUE.
.begin
	.reg $sc_temp0, $sc_temp1, $sc_temp2
	.xfer_order $sc_temp0, $sc_temp1, $sc_temp2
	.sig sig_enq_scratch_w_done

	move($sc_temp0, $$enqueuemessage_0)
	move($sc_temp1, $$enqueuemessage_1)
	move($sc_temp2, QM_DROP_QUEUE)

scratch_ring_full#:

#define_eval	RING_NO			QM_RING_IN
	// check if transmit ring is full
	br_inp_state[SCR_RING/**/RING_NO/**/_STATUS, scratch_ring_full#]
#undef	RING_NO

	// write "numOfWords" to the scratch ring
	scratch[put, $sc_temp0, zero, ENQ_RING_NUMBER, 3], sig_done[sig_enq_scratch_w_done]

	//save the sop_handle for future use.
	alu[sop_handle, --, b, $$enqueuemessage_0]

	//then wait for an additional signal. 
	ctx_arb[sig_enq_scratch_w_done, sig_scratch_deq_r_done, sig_prev_thread]
	
	br[check_dequeue#]

.end

#endif

do_not_send_message#:
	
	alu[gl_transition_valid_reg, --, b, 0]

	// signal the next ctx
	_qm_signal_next_ctx(gl_next_context_sig)

	br[next_iteration#]

send_invalid_dequeue_to_sched#:

	// Because it is a branch were there is no scratch put operation we have to set
	// the corresponding signal. If the signal is not set we will have deadlock.
	_qm_set_signal(&sig_scratch_w_done)

	//It is an invalid dequeue
	
	// IMPORTANT: We've changed the design. Because bit 30th of the dequeue message
	// sent to shaper is modified to carry CLP, we cant set this bit to mark
	// dequeue as invalid. Instead of this, we clear all the contents of the
	// message. We do it only if enqueue message is valid.

	// Send Message to the scheduler if required
	// basically we check if we got a valid enqueue message
	// in the previous phase
	alu[--, --, b, sop_handle]
	beq[do_not_send_message#]

	// OLD CONTENTS: alu[sched_message_deq, $dequeuemessage_0, OR, @invalid_dequeue_bit_mask ]
	// NEW CONTENTS:
	alu[sched_message_deq, --, b, 0]

	// Send message to scheduler - there is an enqueue message valid.
	_qm_send_sched_message(sched_message_enq, sched_message_deq)

	// signal the next ctx
	_qm_signal_next_ctx(gl_next_context_sig) ;1,0

next_iteration#:

.end
#endm // queue_manager_loop()

// There is a compilation trick to allow including only a Queue Manager main loop
// and leave the code structure (files organisation, and contents).
#ifndef QM_TAKE_LOOP_ONLY

///////////////////////////////////////////////////
// This is where the code begins..................
///////////////////////////////////////////////////
begin#:

	// Here is where the Queue Manager is implemented
	_queue_manager() 

error#:
	nop ; place holder will only come here if there is an error

#endif // QM_TAKE_LOOP_ONLY

#endif // QM_ATM_CODE_UC