/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2000 Network Processor, Release 5
 *
 *
 *      
 *
 * = MODULE
 *      IP Options Handler module
 *
 * = FILENAME
 *      ip_options.h
 *
 * = DESCRIPTION
 *      This file contains definition of IP options handling
 *      functions
 *
 * 
 *     
 *
 * = CREATION TIME
 *      7/25/2002 11:53:42 AM
 *
 * = CHANGE HISTORY
 *
 *
 * ============================================================================
 * $Id: ip_options.h,v 1.4 2002/12/13 19:45:03 rjpetri Exp $
 */

#if !defined(__IP_OPTIONS_H__)
#define __IP_OPTIONS_H__

#include "ix_cc_error.h"
#include "ix_netmacros.h"
#include "icmp.h"

#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */

/**
 * Preprocessor symbol and macro definitions.
 */
/*********************************************************************
**   IP Options - values for code field for different options 
*********************************************************************/

/* End of Line */
#define IX_CC_IPV4_IPO_EOL			0

/* No OP */
#define IX_CC_IPV4_IPO_NOP			1

/* Record Route */
#define IX_CC_IPV4_IPO_RR			7

/* TimeStamp */
#define IX_CC_IPV4_IPO_TIMESTAMP		0x44

/* Loose Source & Record Route */
#define IX_CC_IPV4_IPO_LSRR		0x83

/* Strict Source & Record Route */
#define IX_CC_IPV4_IPO_SSRR		0x89

/**********
The value in the 4-bit FLAGS field of IP Timestamp option
controls the exact format of the option and tells how routers should 
supply timestamps.

The Flag [4 bits] values are

          0 -- time stamps only, stored in consecutive 32-bit words,

          1 -- each timestamp is preceded with internet address of the
               registering entity,

          3 -- the internet address fields are prespecified.  An IP
               module only registers its timestamp if it matches its own
               address with the next specified internet address.

************/

/** The following three defines are to incorporate what is defined above **/

/* Time Stamp Only */
#define IX_CC_IPV4_IPO_TS_TSONLY	0

/* Time Stamp and Address */
#define IX_CC_IPV4_IPO_TS_TSANDADDR	1

/* PreSpecified Adress in Time Stamp */
#define IX_CC_IPV4_IPO_TS_PRESPEC	3

/** The below is for generation of ICMP message **/
#define IX_CC_IPV4_ICMP_SOURCE_ROUTE_FAILED	5

/**
 * Type definitions.
 */
/** Options Header common to all options **/
/** Option field is variable in length **/
/** It consists of option type octet, option length octet, pointer**/ 
/** and actual option data octet **/
typedef struct ix_s_cc_ipv4_options_hdr {
    ix_uint8	code; /* option type in option header*/
    ix_uint8	len;  /* option length in option header*/
    ix_uint8	ptr;  /* Pointer field in option header*/
}ix_cc_ipv4_options_hdr;


/** The Timestamp option has a slight different format **/
/** type, length, pointer, overflow, flag **/
/* Options Header for Time Stamp. Easier & elegant than having a union **/
/* in ix_options_hdr */
typedef struct ix_s_cc_ipv4_ip_timestamp {
    ix_uint8	code; /* code field in IP timestamp option*/
    ix_uint8	len;  /* length field in IP timestamp option*/
    ix_uint8	ptr;  /* pointer field in IP timestamp option*/
    
#if IX_BYTE_ORDER == IX_LITTLE_ENDIAN
    ix_uint8 flag:4,
        overflow:4; /* Flag and overflow field in IP timestamp option*/
#elif IX_BYTE_ORDER == IX_BIG_ENDIAN
    ix_uint8	overflow:4,
	flag:4; /* Flag and overflow field in timestamp option*/
#endif
}ix_cc_ipv4_ip_timestamp;

/* IP address structure */
typedef struct ix_s_cc_ipv4_addr
{

    ix_uint32 ipAddr;
} ix_cc_ipv4_addr;

/* Time structure used in timestamp option*/
typedef struct ix_s_cc_ipv4_time_val
{
    ix_uint32 milliseconds;
}ix_cc_ipv4_time_val;


/**
 * Prototypes for interface functions.
 */

/**
 * NAME: _ix_cc_ipv4_parse_ip_options
 *
 * DESCRIPTION: This function is called when the packet has IP options
 *              It appropriately modifies the header of the packet
 *              and decides if the packet is to be dropped or forwarded.
 *              The actual dropping and forwarding of the packet 
 *              is caller's responsibility 
 * 
 * @Param:  - IN arg_hBuffer - the actual IP packet 
 *		
 *            IN arg_pIphdr - Pointer to IP header
 * @Param:  - INOUT  - None
 * @Param:  - OUT - arg_pDrop - On return set to 1 if the packet 
 *                               should be dropped
 *                             On return if set to 0, the packet
 *                              should be forwarded  
 *
 * @Return: IX_SUCCESS if successful or a valid ix_error token 
 *          for failure.
 *          arg_pDrop should be set accordingly on return
 */
ix_error _ix_cc_ipv4_parse_ip_options( 
                                  ix_cc_ipv4_header *arg_pIphdr, 
                                  ix_buffer_handle arg_hBuffer,
                                  ix_uint32 *arg_pDrop);



/**
 * NAME: _ix_cc_ipv4_record_route_option
 *
 * DESCRIPTION: This function processes the record route option
 *              It is called from _ix_cc_ipv4_parse_ip_options. 
 *              This function may modify the IP header. It also
 *              decides if the packet has to be dropped or 
 *              forwarded.
 * 
 * @Param:  - IN arg_hBuffer - a buffer handle containing the
 *                           packet with IP header 
 *		
 *            IN arg_pOpt - a pointer to the start of relevant
 *                            option in IP header
 *            IN arg_pIphdr - Pointer to IP header
 * @Param:  - INOUT  - None
 * @Param:  - OUT - arg_pForward - value to tell whether the packet 
 *                     should be dropped or forwarded.On return, if 
 *                     set to 1, the packet should be dropped.
 *                     On return,if set to 0, the packet should
 *                     be forwarded. 
 *                              
 *
 * @Return: IX_SUCCESS if successful or a valid ix_error token for failure.
 *          arg_pForward should be set accordingly on return
 */
ix_error _ix_cc_ipv4_record_route_option( ix_cc_ipv4_options_hdr *arg_pOpt,
                                         ix_cc_ipv4_header *arg_pIphdr,
                                         ix_buffer_handle arg_hBuffer,
                                         ix_uint32 *arg_pForward
                                       );





/**
 * NAME: _ix_cc_ipv4_timestamp_option
 *
 * DESCRIPTION: This function processes the timestamp option
 *              It is called from _ix_cc_ipv4_parse_ip_options.
 *              This function may modify the IP header. 
 *              It also decides if the packet has to
 *              dropped or forwarded.
 * 
 * @Param:  - IN arg_hBuffer - a buffer handle containing the 
 *                            packet with IP header 
 *		
 *            IN arg_pOpt - a pointer to the start of relevant
 *                           option in IP header
 *            IN arg_pIphdr - Pointer to IP header
 * @Param:  - INOUT  - None
 * @Param:  - OUT - arg_pForward - value to tell whether the packet 
 *                       should be dropped or forwarded.
 *                       On return, if set to 1, the packet 
 *                       should be dropped. On return,
 *                       if set to 0, the packet should be forwarded. 
 *                              
 *
 * @Return: IX_SUCCESS if successful or a valid ix_error token for failure.
 *          arg_pForward should be set accordingly on return
 */
ix_error _ix_cc_ipv4_timestamp_option( ix_cc_ipv4_options_hdr *arg_pOpt,
                                         ix_cc_ipv4_header *arg_pIphdr,
                                         ix_buffer_handle arg_hBuffer,
                                         ix_uint32 *arg_pForward
                                       );


/**
 * NAME: _ix_cc_ipv4_source_route_option
 *
 * DESCRIPTION: This function processes the Strict/Loose source 
 *              route option. It is called from
 *              _ix_cc_ipv4_parse_ip_options. This function             
 *              may modify the IP header. It also decides if 
 *              the packet has to dropped or forwarded.
 *		The loose source and record route (LSRR) option
 *              provides a means for the source of an internet 
 *              datagram to supply routing information
 * 		to be used by the gateways in forwarding the 
 *              datagram to the destination, and to record the
 *              route information  
 *
 *
 * @Param:  - IN arg_hBuffer - a buffer handle containing 
 *            the packet with IP header 
 *		
 *            IN arg_pOpt - a pointer to the start of 
 *                        relevant option in IP header
 *            IN arg_pIphdr - Pointer to IP header
 * @Param:  - INOUT  - None
 * @Param:  - OUT - arg_pForward - value to tell whether the packet 
 *                             should be dropped or forwarded.
 *                             On return, if set to 1, the packet
 *                             should be dropped. On return,
 *                             if set to 0, the packet should be
 *                             forwarded. 
 *                              
 *
 * @Return: IX_SUCCESS if successful or a valid ix_error token
 *          for failure.
 *          arg_pForward should be set accordingly on return
 */
ix_error _ix_cc_ipv4_source_route_option( ix_cc_ipv4_options_hdr *arg_pOpt,
                                         ix_cc_ipv4_header *arg_pIphdr,
                                         ix_buffer_handle arg_hBuffer,
                                         ix_uint32 *arg_pForward
                                       );




#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */


#endif /* end !defined(__IP_OPTIONS_H__) */
