/*===========================================================================
 * = COPYRIGHT INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 * ===========================================================================
 */

/* =========================================================================*/
/**
 * @file   sysapp_main.c
 *
 * Main system application code.
 *
 * This file containes the code to initialize, shutdown and restart
 * the system application and all system components.
 **/
/*=========================================================================*/
#define IX_ERROR_FILE_IDENT "$Id: sysapp_main.c,v 1.35 2003/12/12 20:25:49 ttschlue Exp $"



#include "ix_cc_error.h"
#include "ix_ossl.h"
#include "ix_rm.h"
#include "ix_cci.h"

#include "sa/internal/internal_sa.h"
#include "sa/internal/internal_registry.h"
#include "cc/ix_cc_reg_util.h"
#include "ix_cc_macros.h"

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
#include <errno.h>
#endif

#ifdef __KERNEL__
#define SaLog  printk
#else
#define SaLog  printf
#endif

#if (defined IX_PLATFORM_2401) || (defined IX_PLATFORM_2801)
#ifdef USE_BB_DRV_CONFIG
#if 0
#include "baseboard_media_drv.h"
#endif
#include "cc/ix_cc_eth_tx.h"

extern ix_uint32 hwConfigEthPortMask;/* ETH Port Mask */
extern ix_uint32 hwConfigEthMode[IX_CC_ETH_TX_MAX_NUM_PORTS]; /* ETH Port configuration mode value */

#if 0
#ifndef	IX_CONFIGURATION_quad_gbeth_2401
extern ix_uint32 hwConfigPosPortMask;  /* POS Port Mask */
#endif
#endif

#endif /* USE_BB_DRV_CONFIG */
#endif /* IX_PLATFORM_2x01 */

ix_error _ix_sa_entry(void *bladeID);
ix_error _ix_sa_start(ix_uint32 *arg_auto_restart, ix_uint32 arg_bladeID);
ix_error _ix_sa_init(ix_sa_ctrl *arg_sa_ctrl,
                     ix_uint32 *arg_auto_restart,
                     ix_uint32 arg_bladeID);
ix_error _ix_sa_fini(ix_sa_ctrl *arg_sa_ctrl);

#if (defined IX_PLATFORM_2401) || (defined IX_PLATFORM_2801)
#ifdef USE_BB_DRV_CONFIG
ix_error	_ix_set_hwConfig(void);
#endif /* USE_BB_DRV_CONFIG */
#endif /* IX_PLATFORM_2x01 */

ix_error _ix_sa_patch_symbols(ix_sa_ctrl *arg_sa_ctrl);

ix_ossl_mutex_t g_sa_shutdown_mid = 0;
ix_ossl_sem_t g_sa_shutdown_sid = 0;
ix_uint8      g_sa_shutting_down = 0;
ix_uint32     g_sa_auto_restart = 0;

/*  ix_sa_create */
/*  ---------------------------------------- */
/**
 * Primary entry point for the system application.
 *
 * This function creates a new thread for the system application to
 * initialize and run in.  Passing the initialization off to another thread
 * is required becuase VxWorks will fail if the init function has not
 * returned in a certain amount of time.  So this function is called and
 * quickly returns, allowing the child thread to perform most initialization
 * functions.
 *
 * @return This function is called by VxWorks and therefore does not return ix_error
 *   but instead and int containing success or failure flag.
 * @retval 0 Sucessful system application creation.
 * @retval 1 Failure to create system application
 *
 *
**/
int
ix_sa_create(ix_uint32 arg_bladeID)
{
    ix_error err = 0;
    ix_ossl_thread_t tid;

    ix_error_init(64, 256);

    IX_ERROR_CG(ix_ossl_thread_create(_ix_sa_entry, (void*)arg_bladeID, "Exe1", &tid),
                err, l_error);

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
    return 0;
#else
    return OK;
#endif

 l_error:
    if(err){
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
        ix_error_dump(0, err);
        /*errno = EBUSY;*/
        return -1;
#else
        ix_error_dump(stderr, err);
        return ERROR;
#endif
    }
    ix_error_fini();

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
    return 0;
#else
    return OK;
#endif

}

/*  _ix_sa_entry */
/*  ---------------------------------------- */
/**
 * Entry point for system application initialization thread.
 *
 * This function is called as the entry point for the system application
 * initialization thread.  This function initializes the error system,
 * calls _ix_sa_start, is responsible for dumping and handling errors and
 * implements the system re-initialization policy based on the condition
 * of the system after an error.
 *
 * @param arg context passed by thread creator.  Not used.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 *
**/

ix_error
_ix_sa_entry(void *arg_bladeID)
{
    ix_error err;

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
    ix_error_init(63, 2048);
#else
    ix_error_init(64, 2048);
#endif

    /* initialize mutex to protect shutdown process */
    IX_ERROR_CR(ix_ossl_mutex_init(IX_OSSL_MUTEX_UNLOCK, &g_sa_shutdown_mid));
    g_sa_shutting_down = 0;
    g_sa_shutdown_sid  = 0;
    /* set the auto restart policy in case an error occurs
       before the registry is populated */
    g_sa_auto_restart  = 0;

    /* g_sa_auto_restart is recieved fromt he configuration but can be
     * overwriden by the caller of ix_sa_shutdown */
    while(1){
        g_sa_shutting_down = 0;
        err = _ix_sa_start(&g_sa_auto_restart, (ix_uint32)arg_bladeID);
        if(err){
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
            ix_error_dump(0, err);
#else
            ix_error_dump(stderr, err);
#endif
        }

        /* detrmine if system should re restarted */
        if(g_sa_auto_restart == IX_SA_RESTART_NO){
            break;
        }
    }
    /* clean-up */
    if(g_sa_shutdown_mid != 0){
        ix_ossl_mutex_fini(g_sa_shutdown_mid);
        g_sa_shutdown_mid = 0;
    }
    ix_error_fini();
    return IX_SUCCESS;
}
/*  _ix_sa_start */
/*  ---------------------------------------- */
/**
 * Calls _ix_sa_init and _ix_sa_fini and waits for shutdown event.
 *
 * This function calls _ix_sa_init and then blocks on a semaphore, waiting
 * for a system shutdown event.  When shutdown is signaled, _ix_sa_fini is
 * called and the function returns, allowing _ix_sa_entry to apply the
 * system reinitialization policy.
 *
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 *
**/
ix_error
_ix_sa_start(ix_uint32 *arg_auto_restart, ix_uint32 arg_bladeID)
{
    ix_sa_ctrl *sa_ctrl;

    IX_ERROR_CHECK_ARG_NULL(1, arg_auto_restart);

    /* allocate control structure */
    sa_ctrl = (ix_sa_ctrl*)ix_ossl_malloc(sizeof(ix_sa_ctrl));
    if(sa_ctrl == NULL){
        return IX_ERROR_PANIC(IX_CC_ERROR_OOM_SYSTEM,
                       ("Out of memory allocating ix_sa_ctrl struct."));
    }
    ix_ossl_memset(sa_ctrl, 0, sizeof(ix_sa_ctrl));

    /* initialize semaphore */
    IX_ERROR_CR(ix_ossl_sem_init(IX_OSSL_SEM_UNAVAILABLE, &g_sa_shutdown_sid));
    /* initialize system*/
    IX_ERROR_CR(_ix_sa_init(sa_ctrl, arg_auto_restart, arg_bladeID));
    /* Wait for shutdown event */
    IX_ERROR_CR(ix_ossl_sem_take(g_sa_shutdown_sid, IX_OSSL_WAIT_FOREVER));
    IX_ERROR_CR(ix_ossl_sem_fini(g_sa_shutdown_sid));
    IX_ERROR_CR(_ix_sa_fini(sa_ctrl));
    if(sa_ctrl){
        ix_ossl_free(sa_ctrl);
    }
    return IX_SUCCESS;
}



void
ix_sa_fatal_cci_error_func(ix_error arg_FatalException,
		           ix_exe_handle arg_hEngine,
 	                   void* arg_pContext)
{
    ix_error err;

    IX_ERROR_C(arg_FatalException, err);

    if(err != IX_SUCCESS){
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
        ix_error_dump(0, err);
#else
        ix_error_dump(stderr, err);
#endif
    }

    IX_ERROR_C(ix_sa_shutdown(2), err);
    if(err != IX_SUCCESS){
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
        ix_error_dump(0, err);
#else
        ix_error_dump(stderr, err);
#endif
    }
}




/*  _ix_sa_init */
/*  ---------------------------------------- */
/**
 * Main intialization function that starts system.
 *
 * This function makes calls to all of the subsystems that need
 * to be started as part of the initialization phase.
 *
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
 *
**/
ix_error
_ix_sa_init(ix_sa_ctrl *sa_ctrl, ix_uint32 *arg_auto_restart, ix_uint32 arg_bladeID)
{
    char sp_path[64];
    ix_error err;

    ix_configuration_property_handle sp_prop;

#if defined(IX_PLATFORM_2401) || defined(IX_PLATFORM_2801)
#ifdef USE_BB_DRV_CONFIG
#if (_IX_OS_TYPE_ == _IX_OS_VXWORKS_)
#ifdef IX_CONFIGURATION_oc12_pos_gbeth_2401
	Bb_DriverStart("spi0;spi1;sphy;104mhz", "", "spi2;spi3;sphy;104mhz");
#elif IX_CONFIGURATION_quad_gbeth_2401
	Bb_DriverStart("spi0;spi1;sphy;125mhz", "", "spi2;spi3;sphy;125mhz");
#elif IX_CONFIGURATION_pos_ethernet_2801
	Bb_DriverStart("", "", "a2;dev0=pos:even;dev1=spi3:odd;dev2=spi3:odd");
#else
	#warning "Add additional configuration for VxWorks"
#endif /* IX_CONFIGURATION_ */
#endif /* _IX_OS_TYPE_ == _IX_OS_VXWORKS_ */
	if((err = _ix_set_hwConfig()) != IX_SUCCESS)
		return err;
#endif /* USE_BB_DRV_CONFIG */
#endif /* IX_PLATFORM_2x01 */

    IX_ERROR_CHECK_ARG_NULL(1, sa_ctrl);
    IX_ERROR_CHECK_ARG_NULL(2, arg_auto_restart);

    IX_ERROR_CR(_IX_SA_HOOK(ix_sa_init_hook_first()));
#if defined(IX_PLATFORM_2401) || defined(IX_PLATFORM_2801)
    /* Platform spefific
     * We need this reset on IXDP2x01
     */
#if (_IX_OS_TYPE_ == _IX_OS_VXWORKS_)
    IX_ERROR_CR(_ix_sa_media_reset());
#endif
#endif /* IX_PLATFORM_2x01 */

    IX_ERROR_CR(ix_rm_init(0));
    IX_ERROR_CR(ix_cci_init());
    IX_ERROR_CR(ix_cci_register_fatal_error_handler(ix_sa_fatal_cci_error_func,
                                                    0));
    IX_ERROR_CG(_ix_sa_populate_registry(), err, e_cci);

    /* get the auto_restart value from the registry*/
    sprintf(sp_path, "/System_Properties");
    IX_ERROR_CG(ix_cc_reg_get_prop_from_path(sp_path,
                                             &sp_prop), err, e_cci);
    IX_ERROR_CG(_ix_sa_prop_get_uint_value(sp_prop, "AUTO_RESTART",
                                           arg_auto_restart), err, e_cci);
    IX_ERROR_CG(ix_rm_cp_property_close(sp_prop), err, e_cci);

    IX_ERROR_CG(_ix_sa_load_props(sa_ctrl, arg_bladeID), err, e_cci);
    IX_ERROR_CG(_ix_sa_set_microcode(sa_ctrl), err, e_props);
    IX_ERROR_CG(_ix_sa_alloc_freelists(sa_ctrl), err, e_props);
    IX_ERROR_CG(_ix_sa_create_scratch_rings(sa_ctrl), err, e_freelist);

    IX_ERROR_CG(_IX_SA_HOOK(ix_sa_init_hook_pre_ee()), err, e_scratch);

    IX_ERROR_CR(_ix_sa_patch_symbols(sa_ctrl));


    IX_ERROR_CG(_ix_sa_create_default_engine(sa_ctrl), err, e_scratch);
    IX_ERROR_CG(_ix_sa_create_execution_engines(sa_ctrl), err, e_eng);

    IX_ERROR_CG(_IX_SA_HOOK(ix_sa_init_hook_pre_me()),err, e_eng);
    IX_ERROR_CG(_ix_sa_load_microcode(), err, e_eng);
    IX_ERROR_CG(_ix_sa_start_microengines(sa_ctrl), err, e_eng);

    /*XX INSERT PDK INITIALIZATION CODE HERE */


    IX_ERROR_CG(_IX_SA_HOOK(ix_sa_init_hook_last()), err, e_start_me);

    return IX_SUCCESS;

 e_start_me:
    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_first()));
    IX_ERROR_CP(_ix_sa_stop_microengines(sa_ctrl));
    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_post_me()));
 e_eng:
    IX_ERROR_CP(_ix_sa_delete_execution_engines(sa_ctrl));
    IX_ERROR_CP(_ix_sa_delete_default_engine(sa_ctrl));
    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_post_ee()));
 e_scratch:
    IX_ERROR_CP(_ix_sa_delete_scratch_rings(sa_ctrl));
 e_freelist:
    IX_ERROR_CP(_ix_sa_delete_freelists(sa_ctrl));
 e_props:
    IX_ERROR_CP(_ix_sa_delete_props(sa_ctrl));
 e_cci:
    IX_ERROR_CP(ix_cci_fini());
    IX_ERROR_CP(ix_rm_term());
    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_last()));
    return err;

}

/*  _ix_sa_fini */
/*  ---------------------------------------- */
/**
 * Main fini function that stops system.
 *
 * This function makes calls to all of the subsystems that need
 * to be stoped as part of the shutdown phase.
 *
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
 *
 *
**/
ix_error
_ix_sa_fini(ix_sa_ctrl *sa_ctrl)
{
#if ((defined IX_PLATFORM_2401) || (defined IX_PLATFORM_2801))
#ifdef USE_BB_DRV_CONFIG
#if (_IX_OS_TYPE_ == _IX_OS_VXWORKS_)
	Bb_DriverStop();
#endif /* _IX_OS_TYPE_ == _IX_OS_VXWORKS_ */
#endif /* USE_BB_DRV_CONFIG */
#endif /* IX_PLATFORM_2x01 */

    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_first()));
    IX_ERROR_CP(_ix_sa_stop_microengines(sa_ctrl));
    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_post_me()));
    IX_ERROR_CP(_ix_sa_delete_execution_engines(sa_ctrl));
    IX_ERROR_CP(_ix_sa_delete_default_engine(sa_ctrl));

    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_post_ee()));
    IX_ERROR_CP(_ix_sa_delete_scratch_rings(sa_ctrl));
    IX_ERROR_CP(_ix_sa_delete_freelists(sa_ctrl));
    IX_ERROR_CP(_ix_sa_delete_props(sa_ctrl));
    IX_ERROR_CP(ix_cci_fini());
    IX_ERROR_CP(ix_rm_term());
    IX_ERROR_CP(_IX_SA_HOOK(ix_sa_shutdown_hook_last()));

    return IX_SUCCESS;
}


/*  ix_sa_shutdown */
/*  ---------------------------------------- */
/**
 * Signals the system application to shutdown the system.
 *
 * If a special override is provided the caller of ix_sa_shutdown can
 * specify whether the system should automatically restart.
 *
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
 *
**/
ix_error
ix_sa_shutdown(ix_sa_restart_override arg_restart)
{
    /* lock mutex and see if sysapp is already shutting down.*/
    IX_ERROR_CR(ix_ossl_mutex_lock(g_sa_shutdown_mid, IX_OSSL_WAIT_FOREVER));
    if(g_sa_shutting_down){
        IX_ERROR_CR(ix_ossl_mutex_unlock(g_sa_shutdown_mid));
        return IX_ERROR_WARNING(IX_ERROR_SA_ALREADY_SHUTTINGDOWN,
                                ("Already Shutting Down"));
    }
    /* flag shutting down and release mutex */
    g_sa_shutting_down = 1;
    IX_ERROR_CR(ix_ossl_mutex_unlock(g_sa_shutdown_mid));
    /* override system restart setting if arg_restart*/
    if(arg_restart != IX_SA_RESTART_DEFUALT){
        g_sa_auto_restart = arg_restart;
    }
    /* trigger the shutdown process */
    IX_ERROR_CR(ix_ossl_sem_give(g_sa_shutdown_sid));
    return IX_SUCCESS;
}


#if defined(IX_PLATFORM_2401) || defined(IX_PLATFORM_2801)
/*
 * Platform specific - for FPGA conncted to IXF6048 we must set
 * MSF PLL to 104MHz, for other cases we
 */
/* Utility macro to cast pointers to volitile */
#undef PUT32
#define PUT32(address,val) (*(volatile unsigned int *)(address) = (val))

/* Defines for MCCR register */
#define PLL_25MHZ        0x0
#define PLL_50MHZ        0x5
#define PLL_104MHZ       0xA
#define PLL_125MHZ       0xF

#if !defined(IX_CONFIGURATION_oc12_pos_gbeth_2401)

#define MSF_CLKCFG_MEDIA PLL_125MHZ
#define MSF_CLKCFG_SF    PLL_125MHZ

#else /* here 104 MHz */

#define MSF_CLKCFG_MEDIA PLL_104MHZ
#define MSF_CLKCFG_SF    PLL_104MHZ

#endif

#define MSF_CLKCFG_IN    (MSF_CLKCFG_MEDIA | (MSF_CLKCFG_SF << 4))
#define MSF_CLKCFG_EG    (MSF_CLKCFG_SF | (MSF_CLKCFG_MEDIA << 4))

#define MCCR_MASK1       0x0000F0F0 /* bypass and powerdown */
#define MCCR_MASK2       0x000000F0 /* powerdown */
#define MCCR_MASK3       0x00000000


#if (defined IX_PLATFORM_2401) || (defined IX_PLATFORM_2801)
#ifdef USE_BB_DRV_CONFIG
ix_error _ix_set_hwConfig()
{

#if 0

	int		i;
	int		bb_handle;
	char	mezzanine_pos_atm = 0;
#ifdef IX_CONFIGURATION_quad_gbeth_2401
	ix_uint32	hwConfigPosPortMask;
#endif

	hwConfigEthPortMask = 0;
	hwConfigPosPortMask = 0;
	bb_handle = Bb_Open();

	/* Mezzanine in DB0 */
	switch(Bb_GetDevType(bb_handle, MEDIA_CARD1_ID))
	{
		case BB_FAILURE				:	break;
		case POS_ATM_OC312_ID		:   SaLog("Mezzanine %s ", POS_ATM_OC312_STR_ID);
										mezzanine_pos_atm |= 0x1;
										break;
		case POS_ATM_OC48_ID		:   SaLog("Mezzanine %s ", POS_ATM_OC48_STR_ID);
										mezzanine_pos_atm |= 0x1;
										break;
		case GB_ETH_ID				:   SaLog("Mezzanine %s ", GB_ETH_STR_ID);
										break;
		default						:	return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("ERROR!!! Device in DB0 unknown!!!\n"));
	}

	/* MIC1 */
	switch(Bb_GetDevType(bb_handle, MIC1_ID))
	{
		case BB_FAILURE				:	if((mezzanine_pos_atm & 1) != 0)
										{
										 	SaLog("in DB0\n");
											hwConfigPosPortMask |= 1;
										}
										else
											SaLog("No device in DB0\n");
										break;
		case MIC_C4xGB_MEZZ_ID 		:   SaLog("with %s in DB0\n", MIC_C4xGB_MEZZ_STR_ID);
										#if (_IX_BOARD_TYPE_ == _IX_IXDP2401_)
										#if defined IX_CONFIGURATION_oc12_pos_gbeth_2401
										hwConfigEthPortMask |= 0xc0;
										for(i = 6; i < 8; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#elif defined IX_CONFIGURATION_quad_gbeth_2401
										hwConfigEthPortMask |= 0x30;
										for(i = 4; i < 6; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* IX_CONFIGURATION_ */
										#elif (_IX_BOARD_TYPE_ == _IX_IXDP2801_)
										hwConfigEthPortMask |= 0xf;
										for(i = 0; i < 4; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_ODD |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* _IX_BOARD_TYPE_ */
										break;
		case MIC_F4xGB_MEZZ_ID		:   SaLog("with %s in DB0\n", MIC_F4xGB_MEZZ_STR_ID);
										#if (_IX_BOARD_TYPE_ == _IX_IXDP2401_)
										#if defined IX_CONFIGURATION_oc12_pos_gbeth_2401
										hwConfigEthPortMask |= 0xc0;
										for(i = 6; i < 8; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#elif defined IX_CONFIGURATION_quad_gbeth_2401
										hwConfigEthPortMask |= 0x30;
										for(i = 4; i < 6; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* IX_CONFIGURATION_ */
										#elif (_IX_BOARD_TYPE_ == _IX_IXDP2801_)
										hwConfigEthPortMask |= 0xf;
										for(i = 0; i < 4; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_ODD |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* _IX_BOARD_TYPE_ */
										break;
		case MIC_C2xGB_BRD_ID		:   SaLog("%s on baseboard in slot 1\n", MIC_C2xGB_BRD_STR_ID);
										#if (_IX_BOARD_TYPE_ == _IX_IXDP2401_)
										#if defined IX_CONFIGURATION_oc12_pos_gbeth_2401
										hwConfigEthPortMask |= 0xc;
										for(i = 2; i < 4; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
										SaLog("Spi switch to front\n");
										Bb_SwitchSpiMic(bb_handle);
										#elif defined IX_CONFIGURATION_quad_gbeth_2401
										hwConfigEthPortMask |= 0x3;
										for(i = 0; i < 2; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
										#endif /* IX_CONFIGURATION_ */
										#endif /* _IX_BOARD_TYPE_ */
										break;
		case MIC_F4xOC12_MEZZ_ID	:   SaLog("with %s in DB0\n", MIC_F4xOC12_MEZZ_STR_ID);
										hwConfigPosPortMask |= 1;
										break;
		case MIC_F1xOC48_MEZZ_ID	:   SaLog("with %s in DB0\n", MIC_F1xOC48_MEZZ_STR_ID);
										hwConfigPosPortMask |= 1;
										break;
		case MIC_F2xOC12_MEZZ_ID	:   SaLog("with %s in DB0\n", MIC_F2xOC12_MEZZ_STR_ID);
										hwConfigPosPortMask |= 1;
										break;
		default						:	return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("ERROR!!! MIC1 unknown!!!\n"));
	}

	/* Mezzanine in DB1 */
	switch(Bb_GetDevType(bb_handle, MEDIA_CARD2_ID))
	{
		case BB_FAILURE				:	break;
		case POS_ATM_OC312_ID		:   SaLog("Mezzanine %s ", POS_ATM_OC312_STR_ID);
										mezzanine_pos_atm |= 0x2;
										break;
		case POS_ATM_OC48_ID		:   SaLog("Mezzanine %s ", POS_ATM_OC48_STR_ID);
										mezzanine_pos_atm |= 0x2;
										break;
		case GB_ETH_ID				:   SaLog("Mezzanine %s ", GB_ETH_STR_ID);
										break;
		default						:	return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("ERROR!!! Device in DB1 unknown!!!\n"));
	}

	switch(Bb_GetDevType(bb_handle, MIC2_ID))
	{
		case BB_FAILURE				:	if((mezzanine_pos_atm & 2) != 0)
										{
										 	SaLog("in DB1\n");
											hwConfigPosPortMask |= 2;
										}
										else
											SaLog("No device in DB1\n");
										break;
		case MIC_C4xGB_MEZZ_ID 		:   SaLog("with %s in DB1\n", MIC_C4xGB_MEZZ_STR_ID);
										#if (_IX_BOARD_TYPE_ == _IX_IXDP2401_)
										#if defined IX_CONFIGURATION_oc12_pos_gbeth_2401
										hwConfigEthPortMask |= 0xc00;
										for(i = 10; i < 12; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#elif defined IX_CONFIGURATION_quad_gbeth_2401
										hwConfigEthPortMask |= 0x300;
										for(i = 8; i < 10; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* IX_CONFIGURATION_ */
										#elif (_IX_BOARD_TYPE_ == _IX_IXDP2801_)
										hwConfigEthPortMask |= 0xf0;
										for(i = 4; i < 8; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_ODD |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* _IX_BOARD_TYPE_ */
										break;
		case MIC_F4xGB_MEZZ_ID		:   SaLog("with %s in DB1\n", MIC_F4xGB_MEZZ_STR_ID);
										#if (_IX_BOARD_TYPE_ == _IX_IXDP2401_)
										#if defined IX_CONFIGURATION_oc12_pos_gbeth_2401
										hwConfigEthPortMask |= 0xc00;
										for(i = 10; i < 12; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#elif defined IX_CONFIGURATION_quad_gbeth_2401
										hwConfigEthPortMask |= 0x300;
										for(i = 8; i < 10; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* IX_CONFIGURATION_ */
										#elif (_IX_BOARD_TYPE_ == _IX_IXDP2801_)
										hwConfigEthPortMask |= 0xf0;
										for(i = 4; i < 8; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_ODD |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										#endif /* _IX_BOARD_TYPE_ */
										break;
		case MIC_F2xGB_BRD_ID		:   SaLog("%s on baseboard in slot 2\n", MIC_F2xGB_BRD_STR_ID);
										#if (_IX_BOARD_TYPE_ == _IX_IXDP2401_)
										#if defined IX_CONFIGURATION_oc12_pos_gbeth_2401
										hwConfigEthPortMask |= 0xc;
										for(i = 2; i < 4; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
										SaLog("Spi switch to front\n");
										Bb_SwitchSpiMic(bb_handle);
										#elif defined IX_CONFIGURATION_quad_gbeth_2401
										hwConfigEthPortMask |= 0x3;
										for(i = 0; i < 2; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
										#endif /* IX_CONFIGURATION_ */
										#endif /* _IX_BOARD_TYPE_ */
										break;
		case MIC_F4xOC12_MEZZ_ID	:   SaLog("with %s in DB1\n", MIC_F4xOC12_MEZZ_STR_ID);
										hwConfigPosPortMask |= 2;
										break;
		case MIC_F1xOC48_MEZZ_ID	:   SaLog("with %s in DB1\n", MIC_F1xOC48_MEZZ_STR_ID);
										hwConfigPosPortMask |= 2;
										break;
		case MIC_F2xOC12_MEZZ_ID	:   SaLog("with %s in DB1\n", MIC_F2xOC12_MEZZ_STR_ID);
										hwConfigPosPortMask |= 2;
										break;
		default						:	return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("ERROR!!! MIC2 unknown!!!\n"));
	}

	Bb_Close(bb_handle);

#endif /* nizhner -- #if 0 */
	return IX_SUCCESS;
}
#endif /* USE_BB_DRV_CONFIG */
#endif /* IX_PLATFORM_2x01 */


/**
 * This function will initialize the the MSF RBUF and enable it.
 *
 * @param None
 * @return None
 *
 * @retval NONE
 **/
void
_ix_sa_msf_init(void)
{

#if 0
    /* Setting RESET_0 CAP CSR. */
    PUT32(0xc0004a0c,0x208080);
    PUT32(0xc0004a0c,0x208000);
#endif

}


int msf_pll(void)
{

#if 0

    unsigned int temp_val;
    unsigned int freq;

#if (defined IX_PLATFORM_2401)

#ifdef USE_BB_DRV_CONFIG
	int	bb_handle;
	bb_handle = Bb_Open();

	if(((freq = Bb_GetSpiClk(bb_handle, BB_DB0)) == BB_FAILURE) || (freq == BB_CLK_UNKNOWN))
		if(((freq = Bb_GetSpiClk(bb_handle, BB_DB1)) == BB_FAILURE) || (freq == BB_CLK_UNKNOWN))
			if(((freq = Bb_GetSpiClk(bb_handle, BB_VAL)) == BB_FAILURE) || (freq == BB_CLK_UNKNOWN))
        		return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("Freqency for DB0, DB1 or VAL doesn't set!!!\n"));

   switch(freq)
   {
		case	BB_CLK_50MHz	:	freq = (PLL_50MHZ | (PLL_50MHZ << 4));
									SaLog("Msf frequency set to 50MHz\n");
									break;
		case	BB_CLK_104MHz	:	freq = (PLL_104MHZ | (PLL_104MHZ << 4));
									SaLog("Msf frequency set to 104MHz\n");
									break;
		case	BB_CLK_125MHz	:	freq = (PLL_125MHZ | (PLL_125MHZ << 4));
									SaLog("Msf frequency set to 125MHz\n");
									break;
		case	BB_CLK_133MHz	:	/*freq = (PLL_133MHZ | (PLL_133MHZ << 4));*/
									return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("Freqency 133 doesn't known!!!\n"));
									break;
		default					:	return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("Unknown freqency for DB0, DB1 or VAL!!!\n"));
   }
#else /* USE_BB_DRV_CONFIG */
    freq = MSF_CLKCFG_IN;
#endif /* USE_BB_DRV_CONFIG */

#else /*IX_PLATFORM_2x01 */
	/* Orginal code */
	freq = MSF_CLKCFG_IN;
#endif /*IX_PLATFORM_2x01 */

    temp_val = ((unsigned int) MCCR_MASK1 | (freq << 16));
    PUT32(0xC0004A08, temp_val);

    temp_val = ((unsigned int)MCCR_MASK2 | (freq << 16));
    PUT32(0xC0004A08, temp_val);

    temp_val = ((unsigned int)MCCR_MASK3 | (freq << 16));
    PUT32(0xC0004A08, temp_val);

#if (defined IX_PLATFORM_2401)
#ifdef USE_BB_DRV_CONFIG
	Bb_Close(bb_handle);
#endif /*IX_PLATFORM_2x01 */
#endif /* USE_BB_DRV_CONFIG */

#endif /* nizhner -- #if 0 */
    return IX_SUCCESS;
}


/**
 * This routine resets media settings.
 *
 * @param
 * @return IX_SUCCESS, a pass through ix_error, or an ix_error with one
 *         of the following values.
 * @retval NONE
 **/
ix_error
_ix_sa_media_reset(void)
{

#if 0
    int tries, i;
    int r;

    SaLog("Resetting Media\n");

    for (r = 0; r < 3; r++)
    {
        /* Initialize MSF */
        _ix_sa_msf_init();

        /* MSF PLL 125MHz */
        if (msf_pll() != IX_SUCCESS)
        	return IX_ERROR_PANIC(IX_CC_ERROR_UNINIT, ("Setting msf_pll error !!!\n"));

        /* Flush RBUF elements that might contain packets from media card buffer */
        /* Repeat 100 times for now */
        for (tries = 0; tries < 1000; tries++)
        {
            for (i = 0; i < 64; i++ )
            {
                PUT32(0xc8000000 + 0x44, i);
            }
        }
        SaLog("+");
        ix_ossl_sleep(2000);
    }

    PUT32(0xc0004a0c,0x208080);
    PUT32(0xc0004a0c,0x208000);
    PUT32(0xc0004a0c,0x208080);
    PUT32(0xc0004a0c,0x208000);
    SaLog("\n");

#endif

    return IX_SUCCESS;
}

#endif /* IX_PLATFORM_2x01 */


#if defined(IX_PLATFORM_2800)

/**
 * The following functions are required to initialize the MSF on 2800.
 */

#define get_msf(address)       GET32(0xC8000000 + address)
#define set_msf(address, val)   PUT32((0xC8000000+ address), val)

void media_card_10x1gb_init(void)
{

	int gi_stat_reg, gi_train_stat;

	/* Set display clock divider */
	PUT32(0xc0080000, 0x2);

	/*
	Enable LEDs
	LED Control Register (Addr: 0x509)
	This is a Ben Nevis register.
	Turns all the LEDs off
	*/
	PUT32(0xc6001424, 0x2);

	/*
	Turn all Green RX LEDs (middle row) on
	This is a Ben Nevis register.
	Link LED Enable Register (Addr: 0x502)
	*/
	PUT32(0xc6001408, 0x3ff);

	/*
	Set bit 5 to start auto-negotiation one for each port
	This is Ben Nevis register: 0x18
	Diverse Config Register - 0x18
	*/
	PUT32(0xc6000060, 0x116d);
	PUT32(0xc6000260, 0x116d);
	PUT32(0xc6000460, 0x116d);
	PUT32(0xc6000660, 0x116d);
	PUT32(0xc6000860, 0x116d);
	PUT32(0xc6000a60, 0x116d);
	PUT32(0xc6000c60, 0x116d);
	PUT32(0xc6000e60, 0x116d);
	PUT32(0xc6001060, 0x116d);
	PUT32(0xc6001260, 0x116d);

	/*
	Set MAX frame size for each port as 9k
	This is Ben Nevis register: 0xf
	Max Frame Size Register - 0xof
	*/
	PUT32(0xc600003c, 0x240a);
	PUT32(0xc600023c, 0x240a);
	PUT32(0xc600043c, 0x240a);
	PUT32(0xc600063c, 0x240a);
	PUT32(0xc600083c, 0x240a);
	PUT32(0xc6000a3c, 0x240a);
	PUT32(0xc6000c3c, 0x240a);
	PUT32(0xc6000e3c, 0x240a);
	PUT32(0xc600103c, 0x240a);
	PUT32(0xc600123c, 0x240a);

	/* Set periodic training intervals*/
	/*PUT32(0xc6001c04, 0x01ff);*/

	/*
	 Set MaxBurst1 value to 8 in order to allow RX receiving 128 bytes
	 keep the MaxBurst2 4 to avoid length error for 10gx1 pipeline.
	*/
	PUT32(0xc6001c00, 0x00080004);

	/*
	Bits 21 & 20 should be set for valid link: one for each port
	This is a Ben Nevis register.
	RX Config Word Register - 0x16
	*/

	/* Send training to 10x1GB media card */
	SaLog("Sending training to 10x1GB media card.\n");
	set_msf(0xA0, 0x00000100);
	taskDelay(5);

	gi_stat_reg = GET32(0xc6001c08);
	gi_train_stat = (gi_stat_reg >> 12) & 0x1;

	if (gi_train_stat == 1)
	{

		SaLog("10x1GB media card is sending FIFO status to Egress NPU.\n");
		set_msf(0xA0, 0x00000000);
		SaLog("Egress NPU stopped sending training to 10x1GB media card.\n");
		taskDelay(5);
	}
	else
	{
		SaLog("10x1GB media card failed to train.\n");
		SaLog("Please verify that GI has a B0 IXF1110 MAC.\n");
	}
}


void ingress_media_init_1(void)
{
	int clock_control;
	int ixp_reset0;

	/* Slowport fix, CSR SP_WTC2 */
	PUT32(0xC0080008, 0x00000048);
	taskDelay(5);

	/* Setting CSR CLOCK_CONTROL */
	clock_control = GET32(0xC0004A14);
	clock_control &= 0xFF0FFFFF;
	clock_control |= (0x3 << 20);
	PUT32(0xC0004A14, clock_control);
	taskDelay(5);

	/* Setting CSR IXP_RESET0 */
	ixp_reset0 = GET32(0xC0004A0C);
	ixp_reset0 |= 0x00400080;
	ixp_reset0 &= 0xFFFF7FFF;
	PUT32(0xC0004A0C, ixp_reset0);
	taskDelay(5);

	ixp_reset0 = GET32(0xC0004A0C);
	ixp_reset0 |= 0x00408000;
	ixp_reset0 &= 0xFFFFFF7F;
	PUT32(0xC0004A0C, ixp_reset0);
	taskDelay(5);

	/* Setting CSR IXP_RESET1 */
	PUT32(0xC0004A10, 0x00000000);
	taskDelay(5);

	/* Setting CSR MSF_IO_BUF_CNTL */
	/*set_msf(0x8008, 0x000014AF);*********************************/
	set_msf(0x8008, 0x000016A3);
	taskDelay(5);

	/* Setting CSR FC_IO_BUF_CNTL */
	set_msf(0x800C, 0x000014AF);
	taskDelay(5);

	/* Setting CSR MSF_Tx_Control (select transmit clock) */
	/*set_msf(0x04, 0x00100905);*/
	set_msf(0x04, 0x00100805);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable transmit clocks) */
	set_msf(0x2C, 0x00000050);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable transmit sections) */
/*	set_msf(0x2C, 0x00000154);
	taskDelay(5);
*/
}


void egress_media_init_1(void)
{
	int clock_control;
	int ixp_reset0;

	/* Slowport fix, CSR SP_WTC2 */
	/*PUT32(0xC0080008, 0x00000048);
	taskDelay(5);*/

	/* Setting CSR CLOCK_CONTROL */
	clock_control = GET32(0xC0004A14);
	clock_control &= 0xFF0FFFFF;
	clock_control |= (0x4 << 20);
	PUT32(0xC0004A14, clock_control);
	taskDelay(5);

	/* Setting CSR IXP_RESET0 */
	ixp_reset0 = GET32(0xC0004A0C);
	ixp_reset0 |= 0x00400080;
	ixp_reset0 &= 0xFFFF7FFF;
	PUT32(0xC0004A0C, ixp_reset0);
	taskDelay(5);

	ixp_reset0 = GET32(0xC0004A0C);
	ixp_reset0 |= 0x00408000;
	ixp_reset0 &= 0xFFFFFF7F;
	PUT32(0xC0004A0C, ixp_reset0);
	taskDelay(5);

	/* Setting CSR IXP_RESET1 */
	PUT32(0xC0004A10, 0x00000000);
	taskDelay(5);

	/* Setting CSR MSF_IO_BUF_CNTL */
	set_msf(0x8008, 0x000014AF);
	taskDelay(5);

	/* Setting CSR FC_IO_BUF_CNTL */
	set_msf(0x800C, 0x000014AF);
	taskDelay(5);

	/* Setting CSR MSF_Rx_Control (configure receive section for 2 partiton mode/duplex mode) */
	set_msf(0x0, 0x00000005);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive clock select) */
	set_msf(0x2C, 0x00003000);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive clock dll) */
	set_msf(0x2C, 0x00003C00);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive sections) */
	set_msf(0x2C, 0x00003C0A);
	taskDelay(5);

	/* Setting CSR MSF_Tx_Control (select transmit clock) */
	set_msf(0x04, 0x00000904);
	taskDelay(5);

	/* Enabling CSR MSF_Tx_Control */
	set_msf(0x04, 0x20000D04);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable transmit clocks) */
	set_msf(0x2C, 0x00003C5A);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable transmit sections) */
	set_msf(0x2C, 0x00003D5F);
	taskDelay(5);


	#if defined(MEDIA_1X10GB)

	/* Setting CSR TX_CALENDAR_LENGTH */
	set_msf(0x70, 0x00000010);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_0 */
	set_msf(0x1000, 0x00000000);
	taskDelay(5);

	#elif defined(MEDIA_10X1GB)

	/* Setting CSR TX_CALENDAR_LENGTH */
	set_msf(0x70, 0x0000000A);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_0 */
	set_msf(0x1000, 0x00000000);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_1 */
	set_msf(0x1004, 0x00000001);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_2 */
	set_msf(0x1008, 0x00000002);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_3 */
	set_msf(0x100C, 0x00000003);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_4 */
	set_msf(0x1020, 0x00000004);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_5 */
	set_msf(0x1024, 0x00000005);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_6 */
	set_msf(0x1028, 0x00000006);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_7 */
	set_msf(0x102C, 0x00000007);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_8 */
	set_msf(0x1030, 0x00000008);
	taskDelay(5);

	/* Setting CSR TX_CALENDAR_9 */
	set_msf(0x1034, 0x00000009);
	taskDelay(5);

	#endif

#if 0
	/* Setting CSR MSF_Clock_Control (enable receive clock) */
	set_msf(0x2C, 0x00003D7F);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive calender section) */
	set_msf(0x2C, 0x00003F7F);
	taskDelay(5);
#endif

}


void ingress_media_init_2(void)
{
	/* Setting CSR MSF_Rx_Control (configure receive section for 2 partiton mode/duplex mode) */
	set_msf(0x0, 0x00000004);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive clock select) */
	set_msf(0x2C, 0x00003150);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive clock dll) */
	set_msf(0x2C, 0x00003D50);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive sections) */
	set_msf(0x2C, 0x00003D50);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive clock) */
	set_msf(0x2C, 0x00003D70);
	taskDelay(5);

	/* Setting CSR MSF_Clock_Control (enable receive calender section) */
	set_msf(0x2C, 0x00003F7D);
	taskDelay(5);

	/* Enabling CSR MSF_Tx_Control */
	/*set_msf(0x04, 0x80100D05);*/
	set_msf(0x04, 0x80100C05);
	taskDelay(5);

	/* Send data training to Egress */
	SaLog("\nSending continuous data training to Egress.\n");
	/*set_msf(0xa0, 0x00000100); *****************************************************/
	set_msf(0xa0, 0x00008100);
	taskDelay(5);

	SaLog("\n");

}


void egress_media_init_2(void)
{
	int msf_interrupt_status, dip4_err, rec_data_train;

	int attempt = 1;
	int MAX_ATTEMPT = 10;

	/* Wait for received data training MSF_INTERRUPT_STATUS (bit 7 should be set) */

	while(attempt <= MAX_ATTEMPT)
	{

	SaLog("\nAttempt %d to receive data training from Ingress.\n", attempt);

	msf_interrupt_status = get_msf(0x08);
	taskDelay(5);

	SaLog("\tMSF_INTERRUPT_STATUS = 0x%x.\n", msf_interrupt_status);

	dip4_err = (msf_interrupt_status >> 2) & 0x1;
	rec_data_train = (msf_interrupt_status >> 7) & 0x1;

	if (rec_data_train==0)
	{
		SaLog("\tNo data training received.\n");
	}

	if (dip4_err==1)
	{
		SaLog("\tDPI4 parity failed.\n");
		set_msf(0x08, 0x00000004);
		taskDelay(5);
	}

	if (rec_data_train==1 && dip4_err==0)
	{
		SaLog("\tData training received successfully!\n");
		break;
	}

	attempt++;

	}

	if (rec_data_train==0) SaLog("\nFailed to receive data training from Ingress after %d attempts.\n", MAX_ATTEMPT);

	/* Enabling CSR MSF_Rx_Control */
	set_msf(0x00, 0x80000005);
	taskDelay(5);

	/* Setting CSR CSIX_TYPE_MAP */
	set_msf(0x0010, 0x08);

	/* Send flow control training to Ingress */
	SaLog("\nSending flow control training to Ingress.\n");
	set_msf(0xa8, 0x00000004);
	taskDelay(5);

	SaLog("\n");

}


void ingress_media_init_3(void)
{
	int train_flow_control;
	int msf_interrupt_status, rec_flw_ctl_train, parity_err;

	int attempt = 1;
	int MAX_ATTEMPT = 10;

	/* Stop training Egress */
	SaLog("\nStopped sending continuous data training to Egress.\n");
        /*set_msf(0xa0, 0x00000000);******************************************/
	set_msf(0xa0, 0x00008000);
	taskDelay(5);

	/* Start receiving flow control training from Egress */
	train_flow_control = get_msf(0xa8);
	train_flow_control &= 0xFFFFFFFD;
	set_msf(0xa8, train_flow_control);
	taskDelay(5);

	/* Wait for flow control training MSF_INTERRUPT_STATUS (bit 9 should be set) */

	while(attempt <= MAX_ATTEMPT)
	{

	SaLog("\nAttempt %d to receive flow control training from Egress.\n", attempt);

	msf_interrupt_status = get_msf(0x08);
	taskDelay(5);

	SaLog("\tMSF_INTERRUPT_STATUS = 0x%x.\n", msf_interrupt_status);	

	parity_err = ((msf_interrupt_status >> 6) & 0x1) | ((msf_interrupt_status >> 15) & 0x1);
	rec_flw_ctl_train = (msf_interrupt_status >> 9) & 0x1;

	if (rec_flw_ctl_train==0)
	{
		SaLog("\tNo flow control training received.\n");
	}

	if (parity_err==1)
	{
		SaLog("\tParity failed.\n");
		set_msf(0x08, 0x00008040);
		taskDelay(5);
	}

	if (rec_flw_ctl_train==1 && parity_err==0)
	{
		SaLog("\tFlow control training received successfully!\n");
		break;
	}
	
	attempt++;

	}

	if (rec_flw_ctl_train==0) SaLog("\nFailed to receive flow control training from Egress after %d attempts.\n", MAX_ATTEMPT);

	/* Stop receiving flow control training */
	set_msf(0xa8, 0x00000002);
	taskDelay(5);

	/* Setting CSR FC_STATUS_OVERRIDE */
	set_msf(0x28, 0x00000000);
	taskDelay(5);

	SaLog("\n");
}



void egress_media_init_3(void)
{
	int msf_interrupt_status, rec_data_train, data_train_stopped;

	int attempt = 1;
	int MAX_ATTEMPT = 10;

	/* Clear REC_DATA_TRAIN */
	set_msf(0x08, 0x00000080);
	taskDelay(5);

	/* Wait for end of data training MSF_INTERRUPT_STATUS (bit 7 should be reset) */

	while(attempt <= MAX_ATTEMPT)
	{

	SaLog("\nAttempt %d to stop receiving data training from Ingress.\n", attempt);

	msf_interrupt_status = get_msf(0x08);
	taskDelay(5);

	SaLog("\tMSF_INTERRUPT_STATUS = 0x%x.\n", msf_interrupt_status);	

	rec_data_train = (msf_interrupt_status >> 7) & 0x1;
	data_train_stopped = (msf_interrupt_status >> 12) & 0x1;

	if (rec_data_train==1 || data_train_stopped==0)
	{
		SaLog("\tStill receiving data training.\n");
		set_msf(0x08, 0x00000080);
		taskDelay(5);
	}

	if (rec_data_train==0 && data_train_stopped==1)
	{
		SaLog("\tStopped receiving data training!\n");
		break;
	}
	
	attempt++;

	}

	if (rec_data_train==1 || data_train_stopped==0)
		SaLog("\nFailed to stop receiving data training from Ingress after %d attempts.\n", MAX_ATTEMPT);

	SaLog("\n");

	#if defined(MEDIA_1X10GB)
	/* Initialize 1x10GB media card */
	media_card_1x10gb_init();

	#elif defined(MEDIA_10X1GB)
	/* Initialize 10x1GB media card Island */
	media_card_10x1gb_init();
	#endif

	/* Turns training of to 10x1 GB media card */
	set_msf(0xa0, 0x00000000);
	taskDelay(5);

	/* Setting CSR FC_STATUS_OVERRIDE */
	set_msf(0x28, 0x00000000);
	taskDelay(5);

}




void egress_media_init_3a(void)
{
	int msf_interrupt_status, rec_data_train, data_train_stopped;

	int attempt = 1;
	int MAX_ATTEMPT = 10;

	/* Clear REC_DATA_TRAIN */
	set_msf(0x08, 0x00000080);
	taskDelay(5);

	/* Wait for end of data training MSF_INTERRUPT_STATUS (bit 7 should be reset) */

	while(attempt <= MAX_ATTEMPT)
	{

	SaLog("\nAttempt %d to stop receiving data training from Ingress.\n", attempt);

	msf_interrupt_status = get_msf(0x08);
	taskDelay(5);

	SaLog("\tMSF_INTERRUPT_STATUS = 0x%x.\n", msf_interrupt_status);	

	rec_data_train = (msf_interrupt_status >> 7) & 0x1;
	data_train_stopped = (msf_interrupt_status >> 12) & 0x1;

	if (rec_data_train==1 || data_train_stopped==0)
	{
		SaLog("\tStill receiving data training.\n");
		set_msf(0x08, 0x00000080);
		taskDelay(5);
	}

	if (rec_data_train==0 && data_train_stopped==1)
	{
		SaLog("\tStopped receiving data training!\n");
		break;
	}
	
	attempt++;

	}

	if (rec_data_train==1 || data_train_stopped==0)
		SaLog("\nFailed to stop receiving data training from Ingress after %d attempts.\n", MAX_ATTEMPT);

	SaLog("\n");

} /* egress_media_init_3a() */



void egress_media_init_3b(void)
{

	int gi_stat_reg, gi_train_stat;

#if 1
	/* Send training to 10x1 GB media card */
	SaLog("Sending training to 10x1GB media card.\n");
	set_msf(0xA0, 0x00000100);
	taskDelay(5);
	
	gi_stat_reg = GET32(0xc6001c08);
	gi_train_stat = (gi_stat_reg >> 12) & 0x1;
	
	if (gi_train_stat == 1)
	{
			
		SaLog("10x1GB media card is sending FIFO status to Egress NPU.\n");
		set_msf(0xA0, 0x00000000);
		SaLog("Egress NPU stopped sending training to 10x1GB media card.\n");
		taskDelay(5);
	}
	else
	{
		SaLog("10x1GB media card failed to train.\n");
		SaLog("Please verify that GI has a B0 IXF1110 MAC.\n");	
	}


#endif

	/* Turns training of to media card */
	set_msf(0xa0, 0x00000000);
	taskDelay(5);

	/* Setting CSR FC_STATUS_OVERRIDE */
	set_msf(0x28, 0x00000000);
	taskDelay(5);

} /* egress_media_init_3b() */



void ingress_media_init_4(void)
{

	int msf_interrupt_status, dip4_err, rec_data_train, data_train_stopped, train_done_bit;

	int attempt = 1;
	int MAX_ATTEMPT = 10;

	/* Setting CSR MSF_Clock_Control (enable receive clock select) */
/*	set_msf(0x2C, 0x00003155);
	taskDelay(5);*/

	/* Setting CSR MSF_Clock_Control (enable receive clock dll) */
/*	set_msf(0x2C, 0x00003D55);
	taskDelay(5);*/

	/* Setting CSR MSF_Clock_Control (enable receive sections) */
/*	set_msf(0x2C, 0x00003D5F);
	taskDelay(5);*/

	/* Setting CSR MSF_Clock_Control (enable receive clock) */
/*	set_msf(0x2C, 0x00003D7F);
	taskDelay(5);*/

	/* Setting CSR MSF_Clock_Control (enable receive calender section) */
	set_msf(0x2C, 0x00003F7F);
	taskDelay(5);

	/* Enabling CSR MSF_Rx_Control */
	set_msf(0x0, 0x40000004);
	taskDelay(5);

	#if defined(MEDIA_1X10GB)
	/* Setting CSR RX_CALENDAR_LENGTH to a value that 1x10GB media card expects */
	set_msf(0x48, 0x00000010);
	taskDelay(5);

	#elif defined(MEDIA_10X1GB)
	/* Setting CSR RX_CALENDAR_LENGTH to a value that media card expects */
	set_msf(0x48, 0x0000000A);
	taskDelay(5);
	#endif


	/* Wait for received data training MSF_INTERRUPT_STATUS (bit 7 should be set) */
	/* clear dip4 error bit*/
	set_msf(0x08, 0x0000000f);

	while(attempt <= MAX_ATTEMPT)
	{

	SaLog("\nAttempt %d to receive data training and deskew the interface.\n", attempt);

	msf_interrupt_status = get_msf(0x08);
	taskDelay(5);

	SaLog("\tMSF_INTERRUPT_STATUS = 0x%x.\n", msf_interrupt_status);	

	dip4_err = (msf_interrupt_status >> 2) & 0x1;
	rec_data_train = (msf_interrupt_status >> 7) & 0x1;

	if (rec_data_train==0)
	{
		SaLog("\tNo data training received.\n");
	}

	if (dip4_err==1)
	{
		SaLog("\tDPI4 parity failed.\n");
		set_msf(0x08, 0x00000004);
		taskDelay(5);
	}

	if (rec_data_train==1 && dip4_err==0)
	{
		SaLog("\tData training received and interface deskewed successfully!\n");
		break;
	}
	
	attempt++;

	}

	if (rec_data_train==0)
		SaLog("\nFailed to receive data training after %d attempts.\n", MAX_ATTEMPT);
	

	/* Setting RSTAT_EN to tell the media card that Ingress received training */
	SaLog("\n");
	SaLog("\tMSF_INTERRUPT_STATUS before sending status = 0x%x.\n", msf_interrupt_status);
	set_msf(0xa0, 0x00008002);		
	taskDelay(5);

	attempt = 1;
	while (attempt <= MAX_ATTEMPT)
	{
	SaLog("\nAttempt %d to check Ingress NPU train stop bit.\n", attempt);

	msf_interrupt_status = get_msf(0x08);
	taskDelay(5);
	SaLog("\tMSF_INTERRUPT_STATUS = 0x%x.\n", msf_interrupt_status);

	train_done_bit = (msf_interrupt_status >>12) & 0x1;
	if (train_done_bit == 1)
	{
		SaLog("\tIngress NPU detected train stop from IXF1110!\n");
		break;
	}
	if (train_done_bit == 0)
	{
		SaLog("\tIngress NPU never detected train stop from IXF1110!\n");
		taskDelay(5);	
	}
	

	attempt++;
	}	
	if (train_done_bit ==0)
		SaLog("\nIngress NPU failed to deskew SPI-4 interface after %d attempts.\n", MAX_ATTEMPT);

	SaLog("\n");

	/* Wait for end of data training MSF_INTERRUPT_STATUS (bit 12 should be set) */

	/* Clear REC_DATA_TRAIN */
	msf_interrupt_status = get_msf(0x08);
	SaLog("\n");
	SaLog("\tMSF_INTERRUPT_STATUS before clearing data train bit = 0x%x.\n", msf_interrupt_status);	
	set_msf(0x08, 0x00000080);
	msf_interrupt_status = get_msf(0x08);
	SaLog("\tMSF_INTERRUPT_STATUS after clearing data train bit = 0x%x.\n", msf_interrupt_status);	
	SaLog("\n");
	taskDelay(5);

	attempt = 1;

	while(attempt <= MAX_ATTEMPT)
	{

	SaLog("\nAttempt %d to stop receiving data training.\n", attempt);

	msf_interrupt_status = get_msf(0x08);
	taskDelay(5);

	SaLog("\tMSF_INTERRUPT_STATUS = 0x%x.\n", msf_interrupt_status);	

	data_train_stopped = (msf_interrupt_status >> 7) & 0x1; /*change this bit from 12 to 7*/

	if (data_train_stopped==1)
	{
		SaLog("\tStill receiving data training.\n");
		set_msf(0x08, 0x00000080);	/*attempt to clear data train bit*/
	}

	if (data_train_stopped==0)
	{
		SaLog("\tStopped receiving data training!\n");
		break;
	}
	
	attempt++;

	}		

	if (data_train_stopped==1)
		SaLog("\nFailed to stop receiving data training after %d attempts.\n", MAX_ATTEMPT);

	SaLog("\n");

}


void egress_media_init_4(void)
{
	int media_stat, media_rec_FIFO_stat;
	int attempt = 1;
	int MAX_ATTEMPT = 10;

#if defined(MEDIA_1X10GB)
	{
		while(attempt <= MAX_ATTEMPT)
		{

			SaLog("\nAttempt %d by 1x10GB media card to receive FIFO status from Ingress.\n", attempt);

			media_stat = GET32(0xC600C024);
			taskDelay(5);

			if (media_stat==0x5)
			{
				SaLog("\tReceived FIFO status from Ingress! (status = 0x%x)\n", media_stat);
				break;
			}
			else
			{
				SaLog("\tDid not receive FIFO status from Ingress. (status = 0x%x)\n", media_stat);
			}	

			attempt++;
		}

		if (media_stat!=0x5)
		SaLog("\nFailed to receive FIFO status from Ingress after %d attempts.\n", MAX_ATTEMPT);

	}
	
#elif defined(MEDIA_10X1GB)
	SaLog("+++It's 10x1GB media card.\n");
	{
		while(attempt <= MAX_ATTEMPT)
		{

			SaLog("\nAttempt %d by 10x1GB media card to receive FIFO status from Ingress.\n", attempt);
			media_stat = GET32(0xC6001c08);
			taskDelay(5);
			media_rec_FIFO_stat = (media_stat >> 13) & 0x1;

			if (media_rec_FIFO_stat ==0x1)
			{
				SaLog("\tReceived FIFO status from Ingress! (status = 0x%x)\n", media_stat);
				break;
			}
			else
			{
				SaLog("\tDid not receive FIFO status from Ingress. (status = 0x%x)\n", media_stat);
			}

			attempt++;

		}

		if (media_rec_FIFO_stat !=0x1)
		{
			SaLog("\nFailed to receive FIFO status from Ingress after %d attempts.\n", MAX_ATTEMPT);
			SaLog("Please verify that GI has a B0 IXF1110 MAC.\n");
		}

	}
#endif /* (MEDIA_CARD == MEDIA_10X1GB) */

	SaLog("\n");
	/* disable Ben Nevis Periodic training*/
	PUT32(0xc6001c04, 0x0000);

}

void imi1(void)
{	
    ingress_media_init_1();
}

void emi1(void)
{	
    egress_media_init_1();
}

void imi2(void)
{	
    ingress_media_init_2();
}

void emi2(void)
{	
    egress_media_init_2();
}

void imi3(void)
{	
    ingress_media_init_3();
}

void emi3(void)
{	
    egress_media_init_3();
}

void emi3a(void)
{	
    egress_media_init_3a();
}

void emi3b(void)
{	
    egress_media_init_3b();
}

void imi4(void)
{	
    ingress_media_init_4();
}

void emi4(void)
{	
    egress_media_init_4();
}

#endif /* defined(IX_PLATFORM_2800) */

