/*===========================================================================
 * = COPYRIGHT INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 * ===========================================================================
 */

/* =========================================================================*/
/**
 * @file   sysapp_common_linux_ifconfig_user_mode_util.c
 *
 * Code to configure the IPv6 addresses into the kernel
 *
 * This file containes the code to configure the IPv6 address into the kernel.
 **/
/*=========================================================================*/
#define IX_ERROR_FILE_IDENT "$Id: sysapp_common_linux_ifconfig_user_mode_util.c,v 1.4 2003/12/12 00:44:55 ktseng Exp $"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <pthread.h>
#include <unistd.h>
#include "ix_cc_properties.h"
#include "sa/ix_sa_linux_kernel_module.h"
#include "sa/internal/internal_registry.h"
#include "sa/internal/ingress_internal_registry_data_linux.h"

#if defined(IX_PLATFORM_2800)
#define MAX_NUMBER_OF_PORTS 10
#else
#define MAX_NUMBER_OF_PORTS 4
#endif
#define SUCCESS 0
#define FAILURE 1
#define MAX_STRING_LENGTH 1024
#define MAX_IPV6_ADDRESS_LENGTH 256
#define MAX_ITERATION_COUNTER 300
/* Interface is present in registry file */
#define IF_NOT_PRESENT 0
#define IF_PRESENT 1
/* Interface is created in the system */
#define IF_CREATED 2

extern ix_sa_registry_property gSAPropList[];
unsigned char interfaces[MAX_NUMBER_OF_PORTS];
char file_name[] = "temporary_file";

/**
 * NAME: check_if_port_present
 *
 * DESCRIPTION: This function checks if the properties for a
 *              given port number are present in the registry file.
 *
 * @Param:  - IN argPortNumber - Port number
 *
 * @Return: SUCCESS for success and FAILURE for failure
 */

unsigned char check_if_port_present(unsigned char argPortNumber)
{
    unsigned int count;
    char string_to_compare[MAX_STRING_LENGTH];

    /* Form the string to be compared */
    sprintf (string_to_compare, "PORT0%1d\0", argPortNumber);
    for (count = 0;
           !((strcmp(gSAPropList[count].parent_path, "") == 0) && 
	     (strcmp(gSAPropList[count].name, "") == 0)
            );
	 count++)
    {
        if(strcmp (gSAPropList[count].name, string_to_compare) == 0)
        {
            /* Port found, return success */
            return SUCCESS;
        }
    }
    printf ("IPv6 Addr Config: IPv6 Addr Config: Properties for port %d not found in the registry file...\n", argPortNumber);
    return FAILURE;
}

/**
 * NAME: read_number_of_ipv6_addresses
 *
 * DESCRIPTION: This function reads the number of IPv6 addresses
 *              for a given port.
 *
 * @Param:  - IN argPortNumber - Port number
 * @Param:  - OUT argNumberOfIpv6Addresses - Number of IPv6 addresses
 *                                           present for the given port
 *
 * @Return: SUCCESS for success and FAILURE for failure
 */

unsigned char read_number_of_ipv6_addresses(unsigned char argPortNumber,
    unsigned char *argNumberOfIpv6Addresses)
{
    unsigned int count;
    char string_to_compare1[MAX_STRING_LENGTH];
    char string_to_compare2[MAX_STRING_LENGTH];

    /* Form the string to be compared */
    sprintf (string_to_compare1, "/System_Properties/PORT0%1d/IP_VER\0", argPortNumber);
    sprintf (string_to_compare2, "NUM_VIRTUAL_IF_IPV6\0");
    for (count = 0;
            !((strcmp(gSAPropList[count].parent_path, "") == 0) && 
	      (strcmp(gSAPropList[count].name, "") == 0)
             );
	 count++)
    {
        if((strcmp (gSAPropList[count].parent_path, string_to_compare1) == 0) &&
           (strcmp (gSAPropList[count].name, string_to_compare2) == 0)
          )
        {
            /* Check if the property attribute is proper */
            if (gSAPropList[count].hasvalue == 1)
            {
                /* The property is found, check if the type is proper */
                if (gSAPropList[count].type == 1)
                {
                    *argNumberOfIpv6Addresses = gSAPropList[count].uint_val;
                    return SUCCESS;
                }
                else
                {
                    *argNumberOfIpv6Addresses = 0;
                    printf ("IPv6 Addr Config: IPv6 Addr Config: Property type for number of IPv6 addresses is mentioned wrong for port %d in registry file...\n", argPortNumber);
                    return FAILURE;
                }
            }
	    else
            {
                printf ("IPv6 Addr Config: IPv6 Addr Config: Number of IPv6 addresses does not have a value for port %d in registry file...\n", argPortNumber);
                *argNumberOfIpv6Addresses = 0;
                return FAILURE;
            }
        }
    }
    *argNumberOfIpv6Addresses = 0;
    return FAILURE;
}

/**
 * NAME: read_ipv6_address
 *
 * DESCRIPTION: This function reads the nth IPv6 address
 *              for a given port.
 *
 * @Param:  - IN argPortNumber - Port number
 * @Param:  - IN argNumberOfV6addrToBeRead - nth IPv6 address to read
 * @Param:  - OUT argIpv6Addr - IPv6 adddress read
 *
 * @Return: SUCCESS for success and FAILURE for failure
 */

unsigned char read_ipv6_address (unsigned char argPortNumber,
    unsigned char argNumberOfV6addrToBeRead,
    unsigned char *argIpv6Addr)
{
    unsigned int count;
    char string_to_compare1[MAX_STRING_LENGTH];
    char string_to_compare2[MAX_STRING_LENGTH];

    /* Form the string to be compared */
    sprintf (string_to_compare1, "/System_Properties/PORT0%1d/IP_VER\0", argPortNumber);
    sprintf (string_to_compare2, "IPV6_ADDR_%1d\0", argNumberOfV6addrToBeRead);
    for (count = 0;
           !( (strcmp(gSAPropList[count].parent_path, "") == 0) && 
	     (strcmp(gSAPropList[count].name, "") == 0) );
	 count++)
    {
        if((strcmp (gSAPropList[count].parent_path, string_to_compare1) == 0) &&
           (strcmp (gSAPropList[count].name, string_to_compare2) == 0)
          )
        {
            /* Check if the property attribute is proper */
            if (gSAPropList[count].hasvalue == 1)
            {
                /* The property is found, check if the type is proper */
                if (gSAPropList[count].type == 2)
                {
                    strcpy (argIpv6Addr, gSAPropList[count].value);
                    return SUCCESS;
                }
                else
                {
                    printf ("IPv6 Addr Config: Invalid property type for ipv6 addresses for port %d\n", argPortNumber);
                    return FAILURE;
                }
            }
	    else
            {
                printf ("IPv6 Addr Config: String of IPv6 address expected, but not found for port %d\n", argPortNumber);
                return FAILURE;
            }
        }
    }
    printf ("IPv6 Addr Config: The IPv6 address property not found in registry file for port %d and for %dth address...\n", argPortNumber, argNumberOfV6addrToBeRead);
    return FAILURE;
}

/**
 * NAME: read_ipv6_prefixlen
 *
 * DESCRIPTION: This function reads the nth prefix length
 *              for a given port.
 *
 * @Param:  - IN argPortNumber - Port number
 * @Param:  - IN argNumberOfPrefixlenToBeRead - nth prefix length to read
 * @Param:  - OUT argPrefixlen - Prefix length read
 *
 * @Return: SUCCESS for success and FAILURE for failure
 */

unsigned char read_ipv6_prefixlen (unsigned char argPortNumber,
    unsigned char argNumberOfPrefixlenToBeRead,
    unsigned char *argPrefixlen)
{
    unsigned int count;
    char string_to_compare1[MAX_STRING_LENGTH];
    char string_to_compare2[MAX_STRING_LENGTH];

    /* Form the string to be compared */
    sprintf (string_to_compare1, "/System_Properties/PORT0%1d/IP_VER\0", argPortNumber);
    sprintf (string_to_compare2, "IPV6_PREFIX_LEN_%1d\0", argNumberOfPrefixlenToBeRead);
    for (count = 0;
           !( (strcmp(gSAPropList[count].parent_path, "") == 0) && 
	     (strcmp(gSAPropList[count].name, "") == 0) );
	 count++)
    {
        if((strcmp (gSAPropList[count].parent_path, string_to_compare1) == 0) &&
           (strcmp (gSAPropList[count].name, string_to_compare2) == 0)
          )
        {
            /* Check if the property attribute is proper */
            if (gSAPropList[count].hasvalue == 1)
            {
                /* The property is found, check if the type is proper */
                if (gSAPropList[count].type == 1)
                {
                    *argPrefixlen = gSAPropList[count].uint_val;
                    return SUCCESS;
                }
                else
                {
                    printf ("IPv6 Addr Config: Invalid property type for prefixlength for port %d\n",
                        argPortNumber);
                    return FAILURE;
                }
            }
	    else
            {
                printf ("IPv6 Addr Config: Prefix length number expected, but not found for port %d\n",
                    argPortNumber);
                return FAILURE;
            }
        }
    }
    printf ("IPv6 Addr Config: The prefix length property not found in registry file for port %d and for %dth address...\n", argPortNumber, argNumberOfPrefixlenToBeRead);
    return FAILURE;
}

/**
 * NAME: config_ipv6_addresses
 *
 * DESCRIPTION: This is the main function which gets called to
 *              configure the IPv6 addresses.
 *
 * @Return: void
 */

void config_ipv6_addresses()
{
    unsigned char number_of_port_loop;
    unsigned char number_of_v6addr_loop;
    unsigned char number_of_ipv6_addresses;
    unsigned char ipv6_addr[MAX_IPV6_ADDRESS_LENGTH];
    unsigned char string_to_config[MAX_STRING_LENGTH];
    ix_uint8   prefixlen;

    /* Iterate for each port */
    for (number_of_port_loop = 0;
         number_of_port_loop < MAX_NUMBER_OF_PORTS;
         number_of_port_loop++)
    {
        /* Check if this port is present in the registry structure.
         * If this port is not present in the registry then it is assumed
         * that there are no more ports in the system and this function will return
         */
        if (interfaces[number_of_port_loop] == IF_CREATED)
        {
#ifdef IX_DEBUG
            printf ("IPv6 Addr Config: Setting the IPv6 addr for port %d ...\n",
                number_of_port_loop);
#endif
            /* Read the number of IPv6 addresses belonging to this port */
            if (read_number_of_ipv6_addresses(number_of_port_loop, &number_of_ipv6_addresses)
                == SUCCESS)
            {
                /* Iterate for each IPv6 address */
                for (number_of_v6addr_loop = 0;
                     number_of_v6addr_loop < number_of_ipv6_addresses;
                     number_of_v6addr_loop++)
                {
                    /* Check if the IPv6 address being configured exceeds the maximum
                     * number of allowed IPv6 address per interface.
                     */
                    if (number_of_v6addr_loop < PROP_MAX_IPV6_ADDRESSES)
                    {
                        if (read_ipv6_address
                            (number_of_port_loop, number_of_v6addr_loop, ipv6_addr) == SUCCESS)
                        {
                            if (read_ipv6_prefixlen
                                (number_of_port_loop, number_of_v6addr_loop, &prefixlen)
                                == SUCCESS)
                            {
                                /* Form the string to execute */
                                sprintf (string_to_config, "ifconfig ixdev%1d inet6 add %s/%d\0",
                                    number_of_port_loop, ipv6_addr, prefixlen);
#ifdef IX_DEBUG
                                printf ("IPv6 Addr Config: Command being issued %s ...\n",
                                    string_to_config);
#endif
                                /* Everything is success, call "ifconfig" */
                                if (system(string_to_config) == -1)
                                {
                                    printf ("IPv6 Addr Config: Unable to set the IPv6 address to port %d\n", number_of_port_loop);
                                }
                            }
                        }
                    }
		    else
                    {
                        printf ("IPv6 Addr Config: Exceeding the allowed number of IPv6 addresses for port %d, skipping the extra ones...\n", number_of_port_loop);
                        break;
                    }
                }
            }
        }
    }
}

/**
 * NAME: check_if_created
 *
 * DESCRIPTION: This function checks if the interface for a
 *              given port is created.
 *
 * @Param:  - IN argPortNumber - Port number
 *
 * @Return: SUCCESS on success and FAILURE on failure
 */

unsigned char check_if_created(unsigned char argPortNumber)
{
    char command_string[MAX_STRING_LENGTH];
    char read_string[MAX_STRING_LENGTH];
    char port_string[10];
    FILE *file_handle;

#ifdef IX_DEBUG
    printf ("IPv6 Addr Config: Checking if port %d is created...\n", argPortNumber);
#endif
    /* Form the command string */
    sprintf (command_string, "cat /proc/net/dev | grep ixdev%1d>>%s",
        argPortNumber, file_name);
    /* Execute the command */
    system (command_string);

    file_handle = fopen (file_name, "r");
    if (file_handle == NULL)
        return FAILURE;
    if (fread (read_string, sizeof(char), 6, file_handle) == 0)
    {
        fclose (file_handle);
        truncate (file_name, 0);
        return FAILURE;
    }
    fclose (file_handle);
    /* Make the file contents 0 */
    truncate (file_name, 0);
    /* Form the port string */
    sprintf (port_string, "ixdev%1d", argPortNumber);

    if (strcmp(read_string, "") == 0)
        return FAILURE;
    if (strcmp(read_string, port_string) == 0)
    {
        return SUCCESS;
    }
    else
    {
        return FAILURE;
    }
}

/**
 * NAME: thread_check_and_set_ipv6_addresses
 *
 * DESCRIPTION: This is the thread function responsible
 *              for checking if the interface a port is created
 *              and setting IPv6 adress to it.
 *
 *
 * @Return: void
 */

void *thread_check_and_set_ipv6_addresses(void *arg)
{
    unsigned char number_of_port_loop;
    unsigned char flag;
    unsigned int iteration_counter = 0;

    while (1)
    {
        for (number_of_port_loop = 0;
             number_of_port_loop < MAX_NUMBER_OF_PORTS;
             number_of_port_loop++)
        {
            if (interfaces[number_of_port_loop] == IF_PRESENT)
            {
                /* Check if the interface is created */
                if (check_if_created(number_of_port_loop) == SUCCESS)
                {
                    interfaces[number_of_port_loop] = IF_CREATED;
                }
            }
        }
        flag = 0;
        /* Check if all the interfaces are created */
        for (number_of_port_loop = 0;
             number_of_port_loop < MAX_NUMBER_OF_PORTS;
             number_of_port_loop++)
        {
            if (interfaces[number_of_port_loop] != IF_CREATED)
            {
                flag = 1;
            }
        }
        /* NOTE: that even if some of the interfaces are created, none of the ipv6
         *       IPv6 addresses are configured as there is something wrong.
         */
        if ((flag == 1) && (iteration_counter == MAX_ITERATION_COUNTER))
        {
            /* No interfaces are created even after 300 seconds, so exiting */
            printf ("IPv6 Addr Config: Not all interfaces are created, no IPv6 address added to interfaces...\n");
            printf ("IPv6 Addr Config: Increase the iteration counter value...\n");
            pthread_exit(NULL);
        }
        else if (flag != 1)
        {
            /* All the interfaces are created, hence call the function to
             * add the interface addresses
             */
#ifdef IX_DEBUG
            printf ("IPv6 Addr Config: All the interfaces are created, setting the IPv6 addresses, iteration no: %d\n", iteration_counter);
#endif
            config_ipv6_addresses();
            printf ("IPv6 Addr Config: Finished configuring the IPv6 addresses...\n");
            pthread_exit(NULL);
        }
#ifdef IX_DEBUG
        if (flag == 1)
        {
            printf ("IPv6 Addr Config: All the interfaces are not created, iteration no: %d\n",
                iteration_counter);
        }
#endif
        iteration_counter++;
        /* sleep for 1 second */
        sleep (1);
    }
}

/**
 * NAME: set_ipv6_addr
 *
 * DESCRIPTION: This function checks if there are any
 *              ports present and if so spawns a thread to
 *              configure the IPv6 addresses. This function
 *              spawns the thread and returns.
 *
 * @Return: void
 */

void set_ipv6_addr()
{
    unsigned char number_of_port_loop;
    unsigned char ipv6_addr_found;
    int return_val;
    pthread_t thread_id;

#ifdef IX_DEBUG
    printf ("IPv6 Addr Config: set_ipv6_addr called to configure the IPv6 addresses\n");
#endif

    /* Check if there are any IPv6 interfaces present in the registry */
    /* Iterate for each port */
    ipv6_addr_found = 0;
    for (number_of_port_loop = 0;
         number_of_port_loop < MAX_NUMBER_OF_PORTS;
         number_of_port_loop++)
    {
        if (check_if_port_present (number_of_port_loop) == SUCCESS)
        {
            ipv6_addr_found = 1;
            interfaces[number_of_port_loop] = IF_PRESENT;
        }
        else
        {
            interfaces[number_of_port_loop] = IF_NOT_PRESENT;
#ifdef IX_DEBUG
            printf ("IPv6 Addr Config: Port %d not present in the registry...\n",
                number_of_port_loop);
#endif
        }
    }
    if (ipv6_addr_found == 1)
    {
        /* IPv6 addresses are found, so spawn the thread */
        return_val = pthread_create (&thread_id, NULL, thread_check_and_set_ipv6_addresses, NULL);
        if (return_val != 0)
        {
            printf ("IPv6 Addr Config: Unable to spawn the IPv6 address assignment thread, quitting...\n");
        }
    }
    return;
}
