/*===========================================================================
 * = COPYRIGHT INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 * ===========================================================================
 */

/* =========================================================================*/
/**
 * @file   scratch.c
 *
 * Allocate screatch rings
 *
 * Allocate the scratch rings required by the microengines based on 
 * properties in the registry.
 **/
/*=========================================================================*/
#define IX_ERROR_FILE_IDENT "$Id: scratch.c,v 1.15 2003/11/24 21:36:14 ktseng Exp $"



#include "ix_cc_error.h"
#include "ix_rm.h"
#include "ix_ossl.h"
#include "sa/internal/internal_sa.h"
#include "cc/ix_cc_reg_util.h"
#include "ix_cc_microengines_bindings.h"

ix_error _ix_sa_count_scratch_rings(ix_uint32 *arg_count);
ix_error _ix_sa_core_to_ue_ring_pkt_hdlr_cb(ix_buffer_handle arg_hBuffer, ix_uint32 arg_UserData, void* arg_pContext);

/*
ix_hw_ring_handle g_hExceptionRing0 = 0;
ix_hw_ring_handle g_hExceptionRing1 = 0;
*/
/*  _ix_sa_create_scratch_rings */
/*  ---------------------------------------- */
/**
 * Allocate scratch rings.
 *
 * Enumerates the scratch entries in the registry and creates them
 *
 *
 * @param arg_sa_ctrl System App control structure
 * @return IX_SUCCESS upon success or an ix_error with one of the following 
 * values.
 * @retval NONE
**/
ix_error
_ix_sa_create_scratch_rings(ix_sa_ctrl *arg_sa_ctrl)
{
    ix_configuration_property_handle sr_prop;
    ix_configuration_property_handle sr_subprop;
    ix_uint32 element_size;
    ix_uint32 ring_id;
    ix_uint32 ring_chan;
    ix_uint32 idx = 0;
    ix_uint32 sr_count = 0;
    char sr_path[64];

    /* Validate Parameters */
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);

    /* open base scratch ring property */
    sprintf(sr_path, "/SystemApp/SCRATCH_RINGS");
    IX_ERROR_CR(ix_cc_reg_get_prop_from_path(sr_path, 
                                             &sr_prop));

    IX_ERROR_CR(_ix_sa_count_scratch_rings(&sr_count));
    /* No scratch rings to create. */
    if(sr_count == 0){
        arg_sa_ctrl->aScratchRings = NULL;
        return IX_SUCCESS;
    }

    arg_sa_ctrl->aScratchRings = 
        (ix_hw_ring_handle*)ix_ossl_malloc(sizeof(ix_hw_ring_handle)*sr_count);

    if(arg_sa_ctrl->aScratchRings == NULL){
        return IX_ERROR_PANIC(IX_CC_ERROR_OOM_SYSTEM, 
                              ("Out of memory allocating scratch ring handles."));
    }

    /* enumerate scratch rings and call ix_rm_hw_scratch_ring_create for each */
    ix_rm_cp_property_get_subproperty(sr_prop, idx,
                                      &sr_subprop);
    while(sr_subprop != 0){
        /* on the off chance that the number counted is smaller than
           the number we are trying to create, return an error */
        if(idx == sr_count){
            return IX_ERROR_LOCAL(IX_SA_ERROR_SR_COUNT_MISMATCH,
                                  ("The actual number of scratch rings (%d) does not "
                                   "match the number counted (%d)", idx, sr_count));
        }

        IX_ERROR_CR(_ix_sa_prop_get_uint_value(sr_subprop,
                                              "ELEMENT_SIZE", &element_size));
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(sr_subprop,
                                              "CHAN", &ring_chan));
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(sr_subprop,
                                              "ID", &ring_id));

        /*ix_scratch_ring_size enumeration contains the possible element sizes,
         the sysapp is going to rely on the rm to generate and error if the
         element size is not correct.*/
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
        ix_ossl_message_log("++!!!SCRATCH: %ld S: %ld\n", ring_id, element_size);

#else
        printf("++!!!SCRATCH: %ld S: %ld\n", ring_id, element_size);
#endif

        IX_ERROR_CR(ix_rm_hw_scratch_ring_create(ring_chan, element_size,
                                                 ring_id, 
                                                 &(arg_sa_ctrl->aScratchRings[idx])));
/*
        if(ring_id == 2)
            g_hExceptionRing0 = arg_sa_ctrl->aScratchRings[idx];
        if(ring_id == 3)
            g_hExceptionRing1 = arg_sa_ctrl->aScratchRings[idx];
*/
        if(ring_id == IX_CORE_TO_UBLOCK_PACKET_CTRL_RING_ID)
        {
            /* Register a packet handler callback into RM */
            IX_ERROR_CR(ix_rm_packet_handler_register(IX_CC_QM_CORE_TO_MICROBLOCK_ID, 
                                                      _ix_sa_core_to_ue_ring_pkt_hdlr_cb,
                                                      (void *)arg_sa_ctrl->aScratchRings[idx]));
        }

        idx++;
        ix_rm_cp_property_get_subproperty(sr_prop, idx,
                                          &sr_subprop);
    }


    arg_sa_ctrl->numScratchRings = idx;

    return IX_SUCCESS;
}
/*  _ix_sa_delete_scratch_rings */
/*  ---------------------------------------- */
/**
 * Delete scratch rings.
 *
 * @param arg_sa_ctrl System App control structure  
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_delete_scratch_rings(ix_sa_ctrl *arg_sa_ctrl)
{
    ix_uint32 idx = 0;
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);

    if(arg_sa_ctrl->aScratchRings == NULL){
        arg_sa_ctrl->numScratchRings = 0;
        return IX_SUCCESS;
    }

    /* delete all scratch rings */
    for(idx = 0; idx < arg_sa_ctrl->numScratchRings; idx++){
        IX_ERROR_CR(ix_rm_hw_ring_delete(arg_sa_ctrl->aScratchRings[idx]));
    }
    arg_sa_ctrl->numScratchRings = 0;
    return IX_SUCCESS;
}

/*  _ix_sa_count_scratch_rings */
/*  ---------------------------------------- */
/**
 * Count scratch rings.
 *
 * Count's the number of configured scratch rings in the registry
 *
 * @param arg_count Output variable for the count
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_count_scratch_rings(ix_uint32 *arg_count)
{
    ix_configuration_property_handle sr_prop;
    ix_configuration_property_handle sr_subprop;
    ix_uint32 idx = 0;
    char sr_path[64];

    /* Validate Parameters */
    IX_ERROR_CHECK_ARG_NULL(1, arg_count);

    /* open base scratch ring property */
    sprintf(sr_path, "/SystemApp/SCRATCH_RINGS");
    IX_ERROR_CR(ix_cc_reg_get_prop_from_path(sr_path, 
                                             &sr_prop));


    ix_rm_cp_property_get_subproperty(sr_prop, idx,
	                              &sr_subprop);
    while(sr_subprop != 0){
        *arg_count = *arg_count + 1;
        idx++;
        ix_rm_cp_property_get_subproperty(sr_prop, idx,
	                                  &sr_subprop);
    }
    return IX_SUCCESS;
}


/*  _ix_sa_core_to_ue_ring_pkt_hdlr_cb */
/*  ---------------------------------------- */
/**
 * Packet handler routine for core-to-microblock ring.
 *
 * DESCRIPTION: This function will be the the processing function for the uBlock
 *  packet end point. This function will store the buffer handle and user data onto the
 *  associated end point ring which will passed through arg_pContext argument.
 * 
 * @Param:  - IN arg_hBuffer - the handle of the buffer to be processed
 * @Param:  - IN arg_UserData - the user data associated with the buffer to process
 * @Param:  - IN arg_pContext - context that has been passed to the comm ID at the 
 *          default handler registration time.
 *
 * @Return: IX_SUCCESS if successful or a valid ix_error token for failure.
**/
ix_error _ix_sa_core_to_ue_ring_pkt_hdlr_cb(
    ix_buffer_handle arg_hBuffer, 
    ix_uint32 arg_UserData,
    void* arg_pContext)
{
    ix_hw_ring_handle hRing = (ix_hw_ring_handle)arg_pContext;
    ix_buffer_handle hEopBuffer;
    ix_buffer_handle hBuffer;
    ix_uint32 eopFlag = 0;
    ix_uint32 sopFlag = 0;
    ix_error err = IX_SUCCESS;
    ix_uint32 data[3]; 
    ix_uint32 size;
    ix_buffer_type bufType;
    ix_hw_buffer_meta* pMeta;

    if (arg_hBuffer == IX_NULL_BUFFER_HANDLE)
    {
        return IX_SUCCESS;
    }

    err = ix_rm_buffer_get_type(arg_hBuffer, &bufType);
    if (IX_SUCCESS != err)
    {
        ix_rm_buffer_free_chain(arg_hBuffer);
        return IX_ERROR_WARNING(err,("Fail gettting buffer type"));
    }

    if (IX_BUFFER_TYPE_HARDWARE != bufType)
    {
        /* for software buffer we just free the buffer chain */
        return ix_rm_buffer_free_chain(arg_hBuffer);
    }

    /* Find EOP handle */
    hBuffer = arg_hBuffer;
    IX_ERROR_CG(ix_rm_buffer_is_eop(hBuffer,&eopFlag),
                err, label_1);
		
    while (0 == eopFlag) 
    {
       IX_ERROR_CG(ix_rm_buffer_get_next(hBuffer, &hEopBuffer),
                   err, label_1);
       IX_ERROR_CG(ix_rm_buffer_is_eop(hEopBuffer,&eopFlag),
                   err, label_1);
       hBuffer = hEopBuffer;
    }

    IX_ERROR_CG(ix_rm_buffer_get_meta(hBuffer, (void**)&pMeta),
                err, label_1);
    pMeta->m_HwNext = 0xFF;

    IX_ERROR_CG(ix_rm_buffer_is_sop(hBuffer, &sopFlag),
                err, label_1);

    if(eopFlag && sopFlag) /* eop and sop flag in the same buffer */
    {
        hEopBuffer = 0xFF;
    }
    else
    {
        hEopBuffer = hBuffer;
    }

    size = 3;
    data[0] = (ix_uint32)arg_hBuffer;
    data[1] = (ix_uint32)hEopBuffer;
    data[2] = arg_UserData;

    IX_ERROR_CG(ix_rm_hw_ring_put(hRing, &size, data),
                err, 
                label_1);

    return(IX_SUCCESS);

label_1:

    ix_rm_buffer_free_chain(arg_hBuffer);
    return(IX_SUCCESS);
}

