/*===========================================================================
 * = COPYRIGHT INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 * ===========================================================================
 */
/* =========================================================================*/
/**
 * @file   properties.c
 *
 * Load initial properties.
 *
 * This file contains code to read the initial property configurations from
 * the registry and store them in a linked list to be passed to core components.
 **/
/*=========================================================================*/
#define IX_ERROR_FILE_IDENT "$Id: properties.c,v 1.21 2003/12/12 00:44:55 ktseng Exp $"

#include "ix_cc_error.h"
#include "ix_rm.h"
#include "ix_ossl.h"
#include "sa/internal/internal_sa.h"
#include "sa/internal/internal_registry.h"
#include "cc/ix_cc_reg_util.h"
#include "ix_cc.h"

extern int ix_inet_pton(int, const char *, void *);

static unsigned int ix_inet_addr(char *str)
{
    unsigned	a, b, c, d;
    ix_uint8	arr[4];
    
    if (sscanf(str, "%u.%u.%u.%u", &a, &b, &c, &d) < 4)
    {
        return 0;
    }
    arr[0] = (ix_uint8)a; arr[1] = (ix_uint8)b;
    arr[2] = (ix_uint8)c; arr[3] = (ix_uint8)d;
    return *(unsigned int*)arr;
}


ix_error 
_ix_sa_load_ip_props(ix_configuration_property_handle arg_parent,
                     ix_sa_ctrl *arg_sa_ctrl, ix_cc_properties *arg_props);

ix_uint8
_ix_sa_hex_byte_to_int(char *arg_str);

/*  _ix_sa_load_props */
/*  ---------------------------------------- */
/**
 * Loads the initial properties from the registry.
 *
 * 
 * @param arg_sa_ctrl System Application control struct
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_load_props(ix_sa_ctrl *arg_sa_ctrl, ix_uint32 arg_bladeID)
{
    ix_configuration_property_handle sp_prop = 0;
    ix_configuration_property_handle port_prop = 0;
    ix_uint32 idx = 0;
    ix_cp_property_info prop_info;
    char sp_path[IX_SA_REG_MAX_PATH_SIZE];
    char *valuebuf = NULL;
    int i;
    char *ma = NULL;
    ix_uint32 temp_mtu;
    ix_cc_properties_node *last_node = NULL;
    ix_cc_properties_node *cur_node = NULL;
    ix_error err = IX_SUCCESS;
    ix_uint32 blade_id = 0;
    ix_uint32 cur_port_id = 0;

    /* validate args */
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);
    /* open base property */
    sprintf(sp_path, "/System_Properties");
    IX_ERROR_CR(ix_cc_reg_get_prop_from_path(sp_path, 
                                             &sp_prop));
    /* get the blade id. If the user passed 0xffffffff then get from registry*/
    if(arg_bladeID == 0xFFFFFFFF){
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(sp_prop, "BLADE_ID", &blade_id));
    }else{
        blade_id = arg_bladeID;
    }       
    /* enumerate ports */
    while(!ix_rm_cp_property_get_subproperty(sp_prop, idx,
                                             &port_prop)){
        if(port_prop == 0)
            break;

        IX_ERROR_CG(ix_rm_cp_property_get_info(port_prop, &prop_info),
                    err, post_prop);

        /* make sure it starts with PORT*/


        if(strncmp(prop_info.m_aName, "PORT", 4)){
            /* not a port, continue */
            idx++;
            continue;
        }

        /* allocate new properties node and it's corrisponding
           properties structure*/
        cur_node = (ix_cc_properties_node*)
            ix_ossl_malloc(sizeof(ix_cc_properties_node));
        if(cur_node == NULL){
            if(sp_prop != 0){
                IX_ERROR_CR(ix_rm_cp_property_close(sp_prop));
            }
            if(port_prop != 0){
                IX_ERROR_CR(ix_rm_cp_property_close(port_prop));
            }
            return IX_ERROR_PANIC(IX_CC_ERROR_OOM_SYSTEM, 
                                  ("OOM allocating memory for a "
                                   "ix_cc_properties_node"));
        }
        cur_node->pNextPropNode = NULL;

        cur_node->pPropertyInfo = (ix_cc_properties*)
            ix_ossl_malloc(sizeof(ix_cc_properties));
        if(cur_node->pPropertyInfo == NULL){
            if(cur_node != NULL){
                ix_ossl_free(cur_node);
                cur_node = NULL;
            }
            if(sp_prop != 0){
                IX_ERROR_CR(ix_rm_cp_property_close(sp_prop));
            }
            if(port_prop != 0){
                IX_ERROR_CR(ix_rm_cp_property_close(port_prop));
            }
            return IX_ERROR_PANIC(IX_CC_ERROR_OOM_SYSTEM, 
                                  ("OOM allocating memory for a "
                                   "ix_cc_properties"));
        }
        ix_ossl_memset(cur_node->pPropertyInfo, 0, sizeof(ix_cc_properties));
        cur_node->pPropertyInfo->blade_id = blade_id;

        IX_ERROR_CG(ix_rm_cp_property_get_value_uint32(port_prop,
                                                       &cur_port_id),
                    err, post_alloc_error);
        cur_node->pPropertyInfo->port_id = cur_port_id;
        
        IX_ERROR_CG(_ix_sa_load_ip_props(port_prop, arg_sa_ctrl, 
                                         cur_node->pPropertyInfo),
                    err, post_alloc_error);


        /*--------------------------------------------------------*/
        /* get the mtu */
        IX_ERROR_CG(_ix_sa_prop_get_uint_value(port_prop, "MTU",
                                               &temp_mtu),
                    err, post_alloc_error);
        temp_mtu &= 0xFFFF;
        cur_node->pPropertyInfo->mtu = temp_mtu;

        /*--------------------------------------------------------*/
        /* get the link speed */
        IX_ERROR_CG(_ix_sa_prop_get_uint_value(port_prop, "LINK_SPEED",
                                         &cur_node->pPropertyInfo->link_speed),
                    err, post_alloc_error);

        /*--------------------------------------------------------*/
        /* get mac addr */
        IX_ERROR_CG(_ix_sa_prop_get_string_value(port_prop, "MAC_ADDR",
                                                 &valuebuf),
                    err, post_alloc_error);

        /* convert the hex string into the bytes of the mac addr*/
        ma = valuebuf;
        for(i = 0; i < IX_CC_MAC_ADDR_LEN; i++){
            /* if there are not two characters available to convert
               then the string was too short*/
            if((*ma == 0) || (*(ma+1) == 0)){
                err = IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                     ("Mac address '%s' too short.",
                                      valuebuf));
                goto post_alloc_error;
            }
            /* Convert the next two chars into a byte */
            cur_node->pPropertyInfo->mac_addr[i] = _ix_sa_hex_byte_to_int(ma);
            /* Jump ahead */
            ma += 2;
        }
        /* make sure the mac addr string wasn't too long*/
        if(*ma != 0){
            err = IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                 ("Mac address '%s' too long.",
                                  valuebuf));
            goto post_alloc_error;
        }
        /* free valuebuf string */
        if(valuebuf != NULL){
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }

        /*--------------------------------------------------------*/
        /* get the media type */
        IX_ERROR_CG(_ix_sa_prop_get_string_value(port_prop,
                                                 "MEDIA_TYPE",
                                                 &valuebuf),
                    err, post_alloc_error);
        /* convert the string to one of the choices in the enum*/
        if(!strcmp(valuebuf, "FAST_ETHERNET"))
            cur_node->pPropertyInfo->media_type = IX_MEDIA_TYPE_FAST_ETHERNET;
        else
            if(!strcmp(valuebuf, "1GB_ETHERNET"))
                cur_node->pPropertyInfo->media_type = IX_MEDIA_TYPE_1GB_ETHERNET;
            else        
                if(!strcmp(valuebuf, "ATM"))
                    cur_node->pPropertyInfo->media_type = IX_MEDIA_TYPE_ATM;
                else    
                    if(!strcmp(valuebuf, "POS"))
                        cur_node->pPropertyInfo->media_type = IX_MEDIA_TYPE_POS;
                    else{
                        /* The string doesn't match one of the choices */
                        err = IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                             ("Media type '%s' is not valid.",
                                              valuebuf));

                        goto post_alloc_error;
                    }
        /* free valuebuf string */
        if(valuebuf != NULL){
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }                        
        /*--------------------------------------------------------*/
        /* set the interface state to be up */
        cur_node->pPropertyInfo->physical_if_state = IX_CC_PHYSICAL_IF_STATUS_UP;

        /* Set the link state to be up. */
        cur_node->pPropertyInfo->link_state = IX_CC_LINK_STATUS_UP;


        /* if this is the first port, add it as the head */
        if(last_node == NULL){
            arg_sa_ctrl->init_state.pPropertyList = cur_node;
        }else{
            last_node->pNextPropNode = cur_node;
        }
        last_node = cur_node;
        idx++;
    }
    return IX_SUCCESS;

 post_alloc_error:
    /* the error occured after the allocation of memory
       so free it before returning the error */
    /* free valuebuf string */
    if(valuebuf != NULL)
        ix_ossl_free(valuebuf);

    if(cur_node != NULL){
        if(cur_node->pPropertyInfo != NULL){
            ix_ossl_free(cur_node->pPropertyInfo);
        }
        ix_ossl_free(cur_node);
        cur_node = NULL;
    }

 post_prop:
    if(sp_prop != 0){
        IX_ERROR_CR(ix_rm_cp_property_close(sp_prop));
    }
    if(port_prop != 0){
        IX_ERROR_CR(ix_rm_cp_property_close(port_prop));
    }
    return err;
}

/*  _ix_sa_delete_props */
/*  ---------------------------------------- */
/**
 * Delete properties.
 *
 * This function enumerates the properties linked list and deletes
 * all elements.
 *
 * @param arg_sa_ctrl System Application control struct.
 * @return IX_SUCCESS
 * @retval NONE
**/
ix_error
_ix_sa_delete_props(ix_sa_ctrl *arg_sa_ctrl)
{
    ix_cc_properties_node *cur_node;
    ix_cc_properties_node *hold_node;

    /* Validate arg */
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);
   
    /* get first node*/
    cur_node = arg_sa_ctrl->init_state.pPropertyList;
    
    /* delete nodes and data until none are left in the list*/
    while(cur_node != NULL){
        /* delete property struct */
        if(cur_node->pPropertyInfo != NULL){
            ix_ossl_free(cur_node->pPropertyInfo);
        }

        /* Store pointer to next list entry.*/
        hold_node = cur_node->pNextPropNode;
        /* delete current entry */
        ix_ossl_free(cur_node);
        cur_node = hold_node;
    }
    /* set head pointer to null */
    arg_sa_ctrl->init_state.pPropertyList = NULL;
    return IX_SUCCESS;
}
/*  _ix_sa_hex_byte_to_int */
/*  ---------------------------------------- */
/**
 * Converts two hex characters into a ix_uint8
 *
 * 
 * @param arg_str Pointer to the first of the two characters in the hex byte.
 * @return The ix_uint8 value of the two character hex string
 * @retval byte read from string
**/
ix_uint8
_ix_sa_hex_byte_to_int(char *arg_str)
{
    char bytetext[5];
    unsigned int byte = 0;
    ix_uint8 newbyte = 0;
        
    bytetext[0] = *arg_str++;
    bytetext[1] = *arg_str;
    bytetext[2] = 0;
    
    sscanf(bytetext, "%x", &byte);
    newbyte = (ix_uint8)byte;
    return newbyte;
}
/*  _ix_sa_load_ip_props */
/*  ---------------------------------------- */
/**
 * Load the ip portion of a ports properties
 *
 * The ip portion of the configuration properties are located one
 * level below the rest of the port's properties.
 * 
 * @param arg_parent The parent property.  This must be a 'PORT*' prop.
 * @param arg_sa_ctrl System application control struct.
 * @param arg_props The properties structure to store the data in.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
 *
**/
ix_error
_ix_sa_load_ip_props(ix_configuration_property_handle arg_parent,
                     ix_sa_ctrl *arg_sa_ctrl, ix_cc_properties *arg_props)
{
    ix_configuration_property_handle ipver_prop;
    char *valuebuf;
    ix_uint32 prefixlen;
    char str[IX_CC_MAX_LEN_PROPERTY_NAME]; 
    unsigned int loop; 
    ix_uint32 numVirtualIf;
    ix_uint32 virtual_if_count;
    ix_error err = IX_SUCCESS;
    /* validate args */
    IX_ERROR_CHECK_ARG_NULL(1, arg_parent);
    IX_ERROR_CHECK_ARG_NULL(2, arg_sa_ctrl);
    IX_ERROR_CHECK_ARG_NULL(3, arg_props);

    arg_props->no_virtual_if = 0;
    virtual_if_count = 0;

    /* Get the ip version */
    IX_ERROR_CR(_ix_sa_prop_get_string_value(arg_parent,
                                             "IP_VER",
                                             &valuebuf));

    if ((strcmp(valuebuf, "IPV4V6") == 0))
    {
        /* free valuebuf string */
        if(valuebuf != NULL)
        {
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }

        /* open the ip_ver property */
        IX_ERROR_CR(ix_rm_cp_property_open(arg_parent, "IP_VER", &ipver_prop));

        /* Get the number of IPv4 virtual interfaces */
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(ipver_prop,
                                               "NUM_VIRTUAL_IF_IPV4",
                                               &numVirtualIf));
        if(numVirtualIf > PROP_MAX_IPV4_ADDRESSES)
        {
            ix_rm_cp_property_close(ipver_prop);
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                       ("The num of IPv4 virtual ifs '%d' is not valid.",
                        numVirtualIf));
        }
        arg_props->no_virtual_if = numVirtualIf;

        for (loop = 0;
             loop < numVirtualIf;
             loop++, virtual_if_count++)
        {
            arg_props->ip_prop[virtual_if_count].ip_version =
                IX_CC_IP_VERSION_IPV4;

            /* get the ip address and check for error */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
            snprintf(str, IX_CC_MAX_LEN_PROPERTY_NAME, 
                "%s%u", "IPV4_ADDR_", loop);
#else
            sprintf(str, "%s%u", "IPV4_ADDR_", loop);
#endif
            IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                                     str,
                                                     &valuebuf),
                        err, e_prop);
            arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_address
                = ix_inet_addr(valuebuf);
            if(arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_address == 0)
            {
                /* free valuebuf string */
                if(valuebuf != NULL)
                {
                    ix_ossl_free(valuebuf);
                    valuebuf = NULL;
                }
                ix_rm_cp_property_close(ipver_prop); 
                return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                     ("The ip address '%s' is not valid.",
                                     valuebuf));
            }
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }

            /* get the subnet mask and check for error */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
            snprintf(str, IX_CC_MAX_LEN_PROPERTY_NAME, 
                "%s%u", "IPV4_MASK_", loop);
#else
            sprintf(str, "%s%u", "IPV4_MASK_", loop);
#endif
            IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                                     str,
                                                     &valuebuf),
                        err, e_prop);
            arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_subnet_mask
                = ix_inet_addr(valuebuf);
            if(arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_subnet_mask
                == 0)
            {
                /* free valuebuf string */
                if(valuebuf != NULL)
                {
                    ix_ossl_free(valuebuf);
                    valuebuf = NULL;
                }       
                ix_rm_cp_property_close(ipver_prop); 
                return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                   ("The subnet mask '%s' is not valid.",
                                    valuebuf));
            }
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }

            /* get the broadcast address and check for error */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
            snprintf(str, IX_CC_MAX_LEN_PROPERTY_NAME, 
                "%s%u", "IPV4_BCAST_", loop);
#else
            sprintf(str, "%s%u", "IPV4_BCAST_", loop);
#endif
            IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                                     str,
                                                     &valuebuf),
                        err, e_prop);
            arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_broadcast =
                ix_inet_addr(valuebuf);
            if(arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_broadcast == 0)
            {
                /* free valuebuf string */
                if(valuebuf != NULL)
                {
                    ix_ossl_free(valuebuf);
                    valuebuf = NULL;
                }
                ix_rm_cp_property_close(ipver_prop); 
                return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                   ("The broadcast address '%s' is not valid.",
                                    valuebuf));
            }
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }

            /* get the gateway address and check for error */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
            snprintf(str, IX_CC_MAX_LEN_PROPERTY_NAME, 
                "%s%u", "IPV4_GATE_", loop);
#else
            sprintf(str, "%s%u", "IPV4_GATE_", loop);
#endif
            IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                                     str,
                                                     &valuebuf),
                        err, e_prop);
            arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_gateway =
                ix_inet_addr(valuebuf);
            if(arg_props->ip_prop[virtual_if_count].protocol.ipv4_prop.ipv4_gateway == 0)
            {
                /* free valuebuf string */
                if(valuebuf != NULL)
                {
                    ix_ossl_free(valuebuf);
                    valuebuf = NULL;
                }
                ix_rm_cp_property_close(ipver_prop); 
                return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                   ("The gateway address '%s' is not valid.",
                                    valuebuf));
            }
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }
        }


        /* Get the number of IPv6 virtual interfaces */
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(ipver_prop,
                                               "NUM_VIRTUAL_IF_IPV6",
                                               &numVirtualIf));
        if(numVirtualIf > PROP_MAX_IPV6_ADDRESSES)
        {
            ix_rm_cp_property_close(ipver_prop);
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                       ("The num of IPv6 virtual ifs '%d' is not valid.",
                        numVirtualIf));
        }
        arg_props->no_virtual_if += numVirtualIf;

        for (loop = 0;
             loop < numVirtualIf;
             loop++, virtual_if_count++)
        {
            arg_props->ip_prop[virtual_if_count].ip_version =
                IX_CC_IP_VERSION_IPV6;
 
            /* get the ip address and check for error */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
            snprintf(str, IX_CC_MAX_LEN_PROPERTY_NAME, 
                "%s%u", "IPV6_ADDR_", loop);
#else
            sprintf(str, "%s%u", "IPV6_ADDR_", loop);
#endif
            IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                                     str,
                                                     &valuebuf),
                        err, e_prop);
            if (ix_inet_pton(IX_AF_INET6, valuebuf, 
                   &(arg_props->ip_prop[virtual_if_count].protocol.ipv6_prop.ipv6_address.addr)) != 1)
            {
                if(valuebuf != NULL)
                {
                    ix_ossl_free(valuebuf);
                    valuebuf = NULL;
                }
                ix_rm_cp_property_close(ipver_prop); 
                return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                               ("The ipv6 address is not valid.",
                                valuebuf));
            }
            /* Free the valuebuf memory */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }

            /* get the prefix length */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
            snprintf(str, IX_CC_MAX_LEN_PROPERTY_NAME, 
                "%s%u", "IPV6_PREFIX_LEN_", loop);
#else
            sprintf(str, "%s%u", "IPV6_PREFIX_LEN_", loop);
#endif
            IX_ERROR_CG(_ix_sa_prop_get_uint_value(ipver_prop,
                                                   str,
                                                   &prefixlen),
                        err, e_prop);
            if (prefixlen > 128)
            {
                ix_rm_cp_property_close(ipver_prop); 
                return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                               ("The ipv6 address prefix length is not valid.",
                                valuebuf));
            }

            arg_props->ip_prop[virtual_if_count].protocol.ipv6_prop.ipv6_address.prefixlen = prefixlen;

            /* Get the IPv6 Gateway */
            sprintf(str, "%s%u", "IPV6_GATE_", loop);
            IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                                     str,
                                                     &valuebuf),
                        err, e_prop);
            if (ix_inet_pton(IX_AF_INET6, valuebuf,
                             &(arg_props->ip_prop[virtual_if_count].protocol.ipv6_prop.ipv6_gateway)) != 1)
            {
                if(valuebuf != NULL)
                {
                    ix_ossl_free(valuebuf);
                    valuebuf = NULL;
                }
                ix_rm_cp_property_close(ipver_prop); 
                return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                               ("The ipv6 address is not valid.",
                                valuebuf));
            }
            /* Free the valuebuf memory */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }
        }
        /* Close the property */
        IX_ERROR_CR(ix_rm_cp_property_close(ipver_prop));
    }
    else if (strcmp(valuebuf, "IPV4") == 0)
    {
        /* free valuebuf string */
        if(valuebuf != NULL)
        {
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }
        /* NOTE: that there is only one IPv4 address is being
         *       read. The rest of the code in the system
         *       supports port having multiple IPv4 addresses.
         */ 
        arg_props->no_virtual_if = 1;
        arg_props->ip_prop[0].ip_version = IX_CC_IP_VERSION_IPV4;
        /* open the ip_ver property */
        IX_ERROR_CR(ix_rm_cp_property_open(arg_parent, "IP_VER", &ipver_prop));
 
        /* get the ip address and check for error */
        IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                                 "IP_ADDR",
                                                 &valuebuf),
                    err, e_prop);
 
        arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_address =
            ix_inet_addr(valuebuf);
        if(arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_address == 0)
        {
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }
            ix_rm_cp_property_close(ipver_prop); 
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                                 ("The ip address '%s' is not valid.",
                                 valuebuf));
        }
        /* free valuebuf string */
        if(valuebuf != NULL)
        {
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }
        /* get the subnet mask and check for error */
        IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                              "IP_MASK",
                                              &valuebuf),
                    err, e_prop);

        arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_subnet_mask =
            ix_inet_addr(valuebuf);
        if(arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_subnet_mask == 0)
        {
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }       
            ix_rm_cp_property_close(ipver_prop); 
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                               ("The subnet mask '%s' is not valid.",
                                valuebuf));
        }
        /* free valuebuf string */
        if(valuebuf != NULL)
        {
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }
        /* get the broadcast address and check for error */
        IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                              "IP_BCAST",
                                              &valuebuf),
                    err, e_prop);
 
        arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_broadcast =
            ix_inet_addr(valuebuf);
        if(arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_broadcast == 0)
        {
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }
            ix_rm_cp_property_close(ipver_prop); 
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                               ("The broadcast address '%s' is not valid.",
                                valuebuf));
        }
        /* free valuebuf string */
        if(valuebuf != NULL)
        {
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }
        /* get the gateway address and check for error */
        IX_ERROR_CG(_ix_sa_prop_get_string_value(ipver_prop,
                                           "IP_GATE",
                                           &valuebuf),
                    err, e_prop);

        arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_gateway =
            ix_inet_addr(valuebuf);
        if(arg_props->ip_prop[0].protocol.ipv4_prop.ipv4_gateway == 0)
        {
            /* free valuebuf string */
            if(valuebuf != NULL)
            {
                ix_ossl_free(valuebuf);
                valuebuf = NULL;
            }
            ix_rm_cp_property_close(ipver_prop); 
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA, 
                               ("The gateway address '%s' is not valid.",
                                valuebuf));
        }
        /* free valuebuf string */
        if(valuebuf != NULL)
        {
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }

        IX_ERROR_CR(ix_rm_cp_property_close(ipver_prop));
    }
    else
    {
        /* free valuebuf string */
        if(valuebuf != NULL)
        {
            ix_ossl_free(valuebuf);
            valuebuf = NULL;
        }
        return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA,
                         ("IP_VER value is '%s' but only 'IPV4' and 'IPV4V6'"
			  "are supported at this time.", valuebuf));
    }

    return IX_SUCCESS;

 e_prop:
    /* free valuebuf string */
    if(valuebuf != NULL)
    {
        ix_ossl_free(valuebuf);
        valuebuf = NULL;
    }

    IX_ERROR_CP(ix_rm_cp_property_close(ipver_prop));
    return err;
}
