/*===========================================================================
 * = COPYRIGHT INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 * ===========================================================================
 */

/* =========================================================================*/
/**
 * @file   freelists.c
 *
 * Allocates the freelists.
 *
 * Allocate the freelists based on the freelist configuration within the 
 * registry.
 **/
/*=========================================================================*/
#define IX_ERROR_FILE_IDENT "$Id: freelists.c,v 1.9 2003/09/15 17:21:43 ktseng Exp $"


#include "ix_cc_error.h"
#include "ix_rm.h"
#include "ix_ossl.h"
#include "sa/internal/internal_sa.h"
#include "cc/ix_cc_reg_util.h"

int _ix_sa_prop_is_present(char* arg_parent_path, char* arg_prop);


/*  _ix_sa_alloc_freelists */
/*  ---------------------------------------- */
/**
 * Allocate freelists.
 *
 * Enumerates the freelist entries in the registry and calls 
 * _ix_sa_create_freelist
 *
 * @param arg_sa_ctrl System application control structure
 * @return IX_SUCCESS upon success or an ix_error with one of the following 
 * values.
 * @retval NONE
**/
ix_error
_ix_sa_alloc_freelists(ix_sa_ctrl *arg_sa_ctrl)
{
    ix_configuration_property_handle freelist_subprop;
    ix_uint32 fl_count = 0;
    ix_uint8 counter = 0;
    ix_uint32 element_count;
    ix_uint32 sram_size;
    ix_uint32 sram_chan;
    ix_uint32 dram_size;
    ix_uint32 dram_chan;
    ix_uint32 meta_size;
    ix_uint32 data_size;
    char freelist_path[64];

    /* ensure sa_ctrl is not null */
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);

    /* check on how many channels freelists should be allocated */
    sprintf(freelist_path, "/SystemApp/FREELISTS");

    if(_ix_sa_prop_is_present("/SystemApp/FREELISTS","FL_COUNT")) {

        IX_ERROR_CR(ix_cc_reg_get_prop_from_path(freelist_path, 
                                             &freelist_subprop));

        IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                           "FL_COUNT", &fl_count));

    } else {

        /* If there is no FL_COUNT property then assume single free list */
        fl_count = 1;

    }


    if (fl_count > IX_SA_MAX_FREELISTS)
    {
        return IX_ERROR_GLOBAL(IX_CC_ERROR_FULL,
               ("Number of freelists defined exceeds IX_SA_MAX_FREELISTS.\n"));
    }

    
    arg_sa_ctrl->freelistChannelCount = fl_count;

    for (counter = 1; counter <= fl_count; counter ++)
    {
        /* Get the freelists data from the configuration  */

        if (counter < 9)
        {
            sprintf(freelist_path, "/SystemApp/FREELISTS/FL_0%u", counter);
        }
        else
        {
            sprintf(freelist_path, "/SystemApp/FREELISTS/FL_%u", counter);
        }
        IX_ERROR_CR(ix_cc_reg_get_prop_from_path(freelist_path, 
                                                 &freelist_subprop));
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                               "ELEMENT_COUNT", &element_count));
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                               "SRAM_SIZE", &sram_size));
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                               "SRAM_CHAN", &sram_chan));
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                               "DRAM_SIZE", &dram_size));
        IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                               "DRAM_CHAN", &dram_chan));

       
        /* Create hardware freelist */
        IX_ERROR_CR(ix_rm_hw_buffer_free_list_create(element_count,
                                                     sram_size, dram_size,
                                                     sram_chan, dram_chan,
                                                     &(arg_sa_ctrl->freelist[counter])));
    }

    /* Register the h/w free list into the RM PCI driver */
    IX_ERROR_CR(ix_rm_register_pci_communication_hw_free_list(arg_sa_ctrl->freelist[1]));

    /* Get the message freelist from the configuration. */
    sprintf(freelist_path, "/SystemApp/FREELISTS/FL_MSG");
    IX_ERROR_CR(ix_cc_reg_get_prop_from_path(freelist_path, 
                                             &freelist_subprop));
    IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                           "ELEMENT_COUNT", &element_count));
    IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                           "META_SIZE", &meta_size));
    IX_ERROR_CR(_ix_sa_prop_get_uint_value(freelist_subprop,
                                           "DATA_SIZE", &data_size));

    /* create freelist for messages */
    IX_ERROR_CR(ix_rm_sw_buffer_free_list_create(meta_size, 
                                                 IX_MEMORY_TYPE_DRAM,
                                                 data_size,
                                                 IX_MEMORY_TYPE_DRAM,
                                                 element_count,
                                                 &(arg_sa_ctrl->hMsg_Freelist)));
    arg_sa_ctrl->msg_Buff_Size = data_size;

    /* set the freelist entry in the init_state structure
    *  so all core components will recieve it.*/
    arg_sa_ctrl->init_state.hFreeList = arg_sa_ctrl->freelist[1];
    return IX_SUCCESS;
}

/*  _ix_sa_delete_freelists */
/*  ---------------------------------------- */
/**
 * Delete freelists
 *
 * @param arg_sa_ctrl System application control structure
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_delete_freelists(ix_sa_ctrl *arg_sa_ctrl)
{
    ix_uint16 i = 1;
                        
    ix_buffer_free_list_handle freelist; 

    /* ensure sa_ctrl is not null */
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);

    
    /* Un-register the h/w free list from the RM PCI driver */
    IX_ERROR_CR(ix_rm_unregister_pci_communication_hw_free_list(&freelist));

    /* delete the message freelist */
    IX_ERROR_CP(ix_rm_buffer_free_list_delete(arg_sa_ctrl->hMsg_Freelist));

    /* delete other freelists */
    for (i = 1; i <= arg_sa_ctrl->freelistChannelCount; i++)
    {
        IX_ERROR_CR(ix_rm_buffer_free_list_delete(arg_sa_ctrl->freelist[i]));
    }
    
    return IX_SUCCESS;
}

/*  _ix_sa_prop_get_uint_value */
/*  ---------------------------------------- */
/**
 * Get the value of a specific subproperty of the freelist.
 *
 * Get's the value of a specific subproperty of the freelist.
 * The type is validated and an error is passed if it is not
 * uint32.
 *
 * @param arg_parent Handle to the parent property.
 * @param arg_name   Name of the sub-property
 * @param arg_value  Output parameter for the value.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval IX_SA_ERROR_INVALID_PROP_TYPE The property is not the correct type (uint32)
**/
ix_error
_ix_sa_prop_get_uint_value(ix_configuration_property_handle arg_parent,
                          char *arg_name,
                          ix_uint32 *arg_value)
{
    ix_configuration_property_handle prop;
    ix_cp_property_info prop_info;

    /* Validate arguments */
    IX_ERROR_CHECK_ARG_NULL(1, arg_parent);
    IX_ERROR_CHECK_ARG_NULL(2, arg_name);
    IX_ERROR_CHECK_ARG_NULL(3, arg_value);

    IX_ERROR_CR(ix_rm_cp_property_open(arg_parent, 
                                       arg_name,
                                       &prop));
    IX_ERROR_CR(ix_rm_cp_property_get_info(prop, &prop_info));
        
    /* if the value is the correct type, uint32, then get the value
       otherwise return an error */
    if(prop_info.m_Options & IX_CP_OPTIONS_FLAGS_DATA_UINT32){
        IX_ERROR_CR(ix_rm_cp_property_get_value_uint32(prop,
                                                       arg_value));
        
    }else{
        return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_TYPE,
                              ("Invalid freelist registry property"
                               " type.  All freelist subproperties "
                               "must be uint32."));
    }   
    IX_ERROR_CR(ix_rm_cp_property_close(prop));
    return IX_SUCCESS;
}

/*  _ix_sa_prop_get_string_value */
/*  ---------------------------------------- */
/**
 * Get the value of a specific subproperty of the freelist.
 *
 * Get's the value of a specific subproperty of the freelist.
 * The type is validated and an error is passed if it is not
 * string.
 * Memory is allocated for the string and the recipient is responsible
 * for freeing it.
 *
 * @param arg_parent Handle to the parent property.
 * @param arg_name   Name of the sub-property
 * @param arg_value  Output parameter for the value.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval IX_SA_ERROR_INVALID_PROP_TYPE The property is not the correct type (uint32)
 * @retval IX_SA_ERROR_INVALID_PROP_DATA_SIZE The size of the data returned was invalid.
**/
ix_error
_ix_sa_prop_get_string_value(ix_configuration_property_handle arg_parent,
                          char *arg_name,
                          char **arg_value)
{
    ix_configuration_property_handle prop;
    ix_cp_property_info prop_info;
    char *thestring = NULL;
    ix_uint32 datasize = 0;
    ix_error err;

    /* Validate arguments */
    IX_ERROR_CHECK_ARG_NULL(1, arg_parent);
    IX_ERROR_CHECK_ARG_NULL(2, arg_name);
    IX_ERROR_CHECK_ARG_NULL(3, arg_value);

    IX_ERROR_CR(ix_rm_cp_property_open(arg_parent, 
                                       arg_name,
                                       &prop));
    IX_ERROR_CR(ix_rm_cp_property_get_info(prop, &prop_info));
    datasize = prop_info.m_DataSize;
        
    /* allocate space for the string */
    thestring = (char*)ix_ossl_malloc(datasize);
    if(thestring == NULL){
        return IX_ERROR_PANIC(IX_CC_ERROR_OOM, 
                              ("OOM allocating string for property."));
    }
    
    IX_ERROR_CG(ix_rm_cp_property_get_value(prop, &datasize,
                                            thestring), 
                err, post_alloc_error);
    
    /* make sure we got all of the data */
    if(datasize != prop_info.m_DataSize){
        err = IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_PROP_DATA_SIZE,
                             ("Getting string property, data size"
                              " retrieved was %d and should have "
                              "been %d.", datasize,
                              prop_info.m_DataSize));
        goto post_alloc_error;
    }
    *arg_value = thestring;     

    IX_ERROR_CR(ix_rm_cp_property_close(prop));
    return IX_SUCCESS;

 post_alloc_error:
    if(thestring != NULL){
        ix_ossl_free(thestring);
    }
    return err;
}

int _ix_sa_prop_is_present(char* arg_parent_path, char* arg_prop)
{
  ix_configuration_property_handle prop,subprop;
  ix_uint32 idx = 0;
  ix_cp_property_info prop_info;
  
  if(ix_cc_reg_get_prop_from_path(arg_parent_path, &prop) != IX_SUCCESS)
    {
      return 0;
    }
  
  ix_rm_cp_property_get_subproperty(prop, idx,
                                      &subprop);
  while(subprop != 0)
    {
      if(ix_rm_cp_property_get_info(subprop, &prop_info) != IX_SUCCESS)
	{
	  return 0;
	}
        
      if (strcmp(prop_info.m_aName, arg_prop) == 0)
	{
	  return 1;
        }

      idx++;
      ix_rm_cp_property_get_subproperty(prop, idx, &subprop);
    }
  return 0;
} 
