/*===========================================================================
 * = COPYRIGHT INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 * ===========================================================================
 */
/* =========================================================================*/
/**
 * @file   execution_engines.c
 *
 * SysApp: Launch execution engines and create core components.
 *
 * This file controls the life cycle of the execution engines 
 * configured in the config file.  As each execution engine is 
 * started, it's core components are created.
 **/
/*=========================================================================*/
#define IX_ERROR_FILE_IDENT "$Id: execution_engines.c,v 1.14 2003/09/09 22:51:47 ktseng Exp $"
#include "ix_cc_error.h"
#include "ix_ossl.h"
#include "ix_rm.h"
#include "ix_cci.h"
#include "cc/ix_cc_reg_util.h"
#include "sa/ix_sa_cc_list.h"
#include "sa/internal/internal_sa.h"
#include "sa/internal/internal_registry.h"
#include "cc/ix_cc_msup.h"

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
#include <linux/ctype.h>
int atoi(const char *nptr);
#endif

ix_error _ix_sa_reg_verify_engines(ix_configuration_property_handle arg_hengines);
ix_uint8 _ix_sa_check_engine_prop(ix_uint8 *arg_ee_id_used, ix_uint8 arg_eenum, const char *arg_name);
int      _ix_sa_enum_cc_from_list(char *arg_list, char **arg_listout);
ix_error _ix_sa_exe_init(ix_exe_handle arg_exech, void **ppContext);
ix_error _ix_sa_exe_fini(ix_exe_handle arg_exech, void *pContext);
ix_error _ix_sa_default_exe_init(ix_exe_handle arg_exech, void **ppContext);
ix_error _ix_sa_default_exe_fini(ix_exe_handle arg_exech, void *pContext);

ix_sa_ctrl *g_ix_sa_ctrl = NULL;
/*  _ix_sa_create_execution_engines */
/*  ---------------------------------------- */
/** 
 * Create execution engines.
 * 
 * Create execution engines based on the configuration stored 
 * in the registry.
 *
 * @param arg_sa_ctrl System app control struct.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error 
_ix_sa_create_execution_engines(ix_sa_ctrl *arg_sa_ctrl)
{
    ix_uint8  idx = 0;
    ix_uint32 value;
    ix_error err;
    ix_configuration_property_handle subprop = 0;
    ix_configuration_property_handle hsa_exec_engines = 0;

    /* Validate Arguments*/
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);
    g_ix_sa_ctrl = arg_sa_ctrl;

    /*Get the property that contains the execution engine entries*/
    IX_ERROR_CR(ix_cc_reg_get_prop_from_path(
                    "/SystemApp/SA_EXEC_ENGINES", &hsa_exec_engines));
    /* Although we could simply verify the validity of the registry entries
     * as we were creating engines, this would make recovery difficult.  So
     * we validate the registry information before we start creating 
     * execution engines */
    IX_ERROR_CG(_ix_sa_reg_verify_engines(hsa_exec_engines),
                err, e_prop);
    /* Start each execution engine. */
    ix_rm_cp_property_get_subproperty(hsa_exec_engines, idx, 
                                      &subprop);
    while(subprop != 0){
        /* get the EE number and start the engine */
        IX_ERROR_CG(ix_rm_cp_property_get_value_uint32(subprop,
                                                       &value),
                    err, e_prop);
        IX_ERROR_CG(ix_cci_exe_run(NULL, _ix_sa_exe_init,
                                   _ix_sa_exe_fini,"Exe1", 
                                   &(arg_sa_ctrl->ee_handles[value])),
                    err, e_prop);
        idx++;
        IX_ERROR_CR(ix_rm_cp_property_close(subprop));
        subprop = 0;
        ix_rm_cp_property_get_subproperty(hsa_exec_engines, idx, 
                                          &subprop);
    }
    return IX_SUCCESS;
 e_prop:
    if(subprop != 0){
        IX_ERROR_CP(ix_rm_cp_property_close(subprop));
    }
    IX_ERROR_CP(ix_rm_cp_property_close(hsa_exec_engines));
    return err;
}

/*  _ix_sa_delete_execution_engines */
/*  ---------------------------------------- */
/**
 * Delete execution engines
 *
 * @param arg_sa_ctrl System app control struct.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_delete_execution_engines(ix_sa_ctrl *arg_sa_ctrl)
{
    ix_uint32 idx = 0;

    /* Validate Arguments*/
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);

    for(idx = 0; idx < IX_SA_MAX_EE; idx++){
        if(arg_sa_ctrl->ee_handles[idx] != 0){
            IX_ERROR_CR(ix_cci_exe_shutdown(arg_sa_ctrl->ee_handles[idx]));
            arg_sa_ctrl->ee_handles[idx] = 0;
        }
    }
    return IX_SUCCESS;
}

/*  _ix_sa_exe_init */
/*  ---------------------------------------- */
/**
 * 'init' entry point for each execution engine.
 *
 * This function is called from the execution engine thread of 
 * each execution engine that is created.  It's primary responsibilty
 * is to initialize the message support library and create core-
 * components per the registry configuration.
 *
 * @param arg_exech The execution engine handle
 * @param ppContext This is where the context is to be stored.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
 * 
 * @todo replace temporary message handler with message support lib
 * @todo pass in the main data structure to CCs when created. This requires
 *      exe to pass in a context.
**/
ix_error 
_ix_sa_exe_init(ix_exe_handle arg_exech, void **ppContext)
{
    ix_uint32 ee_num;
    ix_exe_handle ee_handle;
    void *ee_context;
    ix_cp_property_info cclist_info;
    ix_configuration_property_handle cclist_prop;
    char cclist_path[IX_SA_REG_MAX_PATH_SIZE];
    char *cclist_data = NULL;
    char *cclist_cur;
    int cc_id;
    int cc_id_inst;
    ix_sa_ccfunc ccfunc_list[IX_CC_LIST_MAX_SIZE];
    ix_error err = IX_SUCCESS;

    /* Validate Args */
    IX_ERROR_CHECK_ARG_NULL(2, ppContext);
    
    /* Get information about the current execution engine. */
    IX_ERROR_CR(ix_cci_exe_get_info(&ee_handle, &ee_num, &ee_context));
        
    ix_ossl_memset(&ccfunc_list, 0, sizeof(ccfunc_list));
    IX_ERROR_CR(_ix_sa_cc_list(&ccfunc_list[0]));

    /* initialize message support library */
    IX_ERROR_CR(ix_cc_msup_init(g_ix_sa_ctrl->hMsg_Freelist,
                                g_ix_sa_ctrl->msg_Buff_Size,
                                ppContext));
    
    if(ee_num < 10){
        sprintf(cclist_path, "/SystemApp/SA_EXEC_ENGINES/SA_EE_0%ld/CC_LIST", 
                ee_num);
    }else{
        sprintf(cclist_path, "/SystemApp/SA_EXEC_ENGINES/SA_EE_%ld/CC_LIST", 
                ee_num);
    }
    IX_ERROR_CR(ix_cc_reg_get_prop_from_path(cclist_path, &cclist_prop));
    
    IX_ERROR_CR(ix_rm_cp_property_get_info(cclist_prop, &cclist_info));
/*    
    if(cclist_info.m_DataSize == 1){
        return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_CC_LIST, 
                              ("DataSize == 0: %s", cclist_path));
    }
*/
        
    /* allocate space for and fetch the CCLIST*/
    cclist_data = (char *)ix_ossl_malloc(cclist_info.m_DataSize);
    if(cclist_data == NULL){
        return IX_ERROR_LOCAL(IX_CC_ERROR_OOM, 
                              ("Unable to allocate %d bytes for cclist_data.",
                               cclist_info.m_DataSize));
    }
    IX_ERROR_CG(ix_rm_cp_property_get_value(cclist_prop, 
                                            &cclist_info.m_DataSize, 
                                            cclist_data),
                err, e_post_alloc);
    
    /* enumerate each entry in the CCLIST string and create the CCs */
    cc_id_inst = _ix_sa_enum_cc_from_list(cclist_data, &cclist_cur);
    while(cc_id_inst >= 0){
        /* retrieve instance number from extended CC ID */
        g_ix_sa_ctrl->init_state.instanceCC = IX_EXTENDED_ID_TO_INSTANCE(cc_id_inst);
        cc_id = IX_EXTENDED_ID_TO_ID(cc_id_inst);

        if((ccfunc_list[cc_id].init == NULL) ||
           (ccfunc_list[cc_id].fini == NULL)){
            err = IX_ERROR_LOCAL(IX_CC_ERROR_NULL, 
                                 ("init or fini function NULL for cc #%d cc_inst #%d instance #%d", 
                                   cc_id, cc_id_inst, 
                                   g_ix_sa_ctrl->init_state.instanceCC));
            goto e_post_alloc;
        }
        IX_ERROR_CG(ix_cci_cc_create(ee_handle,
                                     ccfunc_list[cc_id].init,
                                     ccfunc_list[cc_id].fini,
                                     (void*)&(g_ix_sa_ctrl->init_state),
                                     &(ccfunc_list[cc_id].cc_handle[g_ix_sa_ctrl->init_state.instanceCC])),
                    err, e_post_alloc);
        cc_id_inst = _ix_sa_enum_cc_from_list(cclist_cur, &cclist_cur);        
    }
                              
    if(cclist_data != NULL)
        ix_ossl_free(cclist_data);

    return IX_SUCCESS;
 e_post_alloc:
    if(cclist_data != NULL){
        ix_ossl_free(cclist_data);
    }
    return err;
}

/*  _ix_sa_enum_cc_from_list */
/*  ---------------------------------------- */
/**
 * Enumerates CC_LIST
 *
 * Supports _ix_sa_exe_init by enumerating each of the CC id numbers
 * stored in CC_LIST.  Supports any non-digit separator.  Each time
 * this function is called, it returns the next number in the list or 
 * (-1) when done.  arg_listout is updated with the new location in 
 * the original list string and should be passed as the first parameter
 * to each call except the first.  On the first call, the original string
 * is passed into arg_list.
 *
 * @param arg_list pointer to current location in list.
 * @param arg_listout new pointer to current location, used in next call
 * @return int the next number in the list or (-1) when complete.
 * @retval -1 enumeration complete
**/
int _ix_sa_enum_cc_from_list(char *arg_list, char **arg_listout)
{
	int ret;

        /* Validate Args */
        if((arg_list == NULL) || (arg_listout == NULL)){
            return -1;
        }

        /* Handle any leading spaces tabs etc.. */
	while((*arg_list != 0) && !isdigit((int)*arg_list)){
		arg_list++;
	}

        /* No more to find */
	if(*arg_list == 0)
		return -1;

        /* Get the value */
	ret = atoi(arg_list);
        /* Zip past the current number */
	while((*arg_list != 0) && isdigit((int)*arg_list)){
		arg_list++;
	}

        /* set arg_listout to the current place in the string */
	*arg_listout = arg_list;
	return ret;
}


/*  _ix_sa_exe_fini */
/*  ---------------------------------------- */
/**
 * 'fini' entry point for each execution engine.
 *
 * This function is called from the execution engine thread of 
 * each execution engine that is created.  It's primary responsibilty
 * is to shutdown the message support library.
 *
 * @param arg_exech execution engine handle for the current ee.
 * @param pContext  pointer to the context returned by _ix_sa_exe_init
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
 *
 * @todo Fini message support lib
**/
ix_error 
_ix_sa_exe_fini(ix_exe_handle arg_exech, void *pContext)
{
    /* Validate Args */

    IX_ERROR_CR(ix_cc_msup_fini(pContext));

    return IX_SUCCESS;
}


/*  _ix_sa_reg_verify_engines */
/*  ---------------------------------------- */
/**
 * Verify the validity of the EE configuration in the registry.
 *
 * Look through each execution engine entry in the registry
 * to verify that the data is correct, valid, and that there
 * are no duplicated.
 * This verification is done ahead of time to provide for easier
 * recovery.
 * arg_hengines should be an open handle to the property that
 * contains all of the ee properties to verify.  Each of these
 * are named 'SA_EE_nn' where nn is the index of the associated
 * execution engine.
 * @param arg_hengines Handle to the engines poperty in the reg.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval IX_SA_ERROR_INVALID_EE One of the exeuction engine entries
 * in the registry is invalid.
**/
ix_error
_ix_sa_reg_verify_engines(
    ix_configuration_property_handle arg_hengines)
{
    ix_uint8 idx = 0;
    ix_uint8 ee_id_used[IX_SA_MAX_EE];
    ix_uint32 propnum = 0;
    ix_configuration_property_handle subprop;
    ix_configuration_property_handle cclist_prop;
    ix_cp_property_info prop_info;
    ix_cp_property_info cclist_info;




    /* Set the initial values of ee_id_used to 0. indicating all id's are free */
    ix_ossl_memset(&ee_id_used[0], 0, sizeof(ee_id_used));

    /* Get each sub property. */
    ix_rm_cp_property_get_subproperty(arg_hengines, idx, 
                                      &subprop);
    while(subprop != 0){
        
        IX_ERROR_CR(ix_rm_cp_property_get_info(subprop, &prop_info));
        
        /* if the value is the correct type, uint32, then get the value
           otherwise return an error */
        if(prop_info.m_Options & IX_CP_OPTIONS_FLAGS_DATA_UINT32){
            IX_ERROR_CR(ix_rm_cp_property_get_value_uint32(subprop,
                                                           &propnum));
            
        }else{
            IX_ERROR_CP(ix_rm_cp_property_close(subprop));
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_EE,
                          ("Invalid Execution Engine registry property"
                           " type.  All execution engine properties "
                           "must be ix_uint32."));
        }
        
        /* make sure the ee number is withing range */
        if(propnum >= IX_SA_MAX_EE){
            IX_ERROR_CP(ix_rm_cp_property_close(subprop));
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_EE,
                                  ("Execution engine id is %ld but must "
                                   "be between 1 and %d", 
                                   propnum, IX_SA_MAX_EE));
        }

        /* verify the engine property has the correct format */
        if(_ix_sa_check_engine_prop(&ee_id_used[0], propnum, 
                                    prop_info.m_aName) == 0){
            IX_ERROR_CP(ix_rm_cp_property_close(subprop));  
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_EE,
                          ("Invalid Execution Engine registry property:"
                           " (NAME: %s  VALUE: %ld) is not valid.",
                           prop_info.m_aName, propnum));
        } 

        /* Verify that this engine has a subprop named CC_LIST */
        IX_ERROR_CR(ix_rm_cp_property_open(subprop, "CC_LIST", &cclist_prop));
        IX_ERROR_CR(ix_rm_cp_property_get_info(cclist_prop, &cclist_info));
        IX_ERROR_CR(ix_rm_cp_property_close(cclist_prop));
        /* If the property doesn't have a data or the data is uint then fail */
        if(cclist_info.m_Options & IX_CP_OPTIONS_FLAGS_DATA_UINT32){
            IX_ERROR_CP(ix_rm_cp_property_close(subprop));
            return IX_ERROR_LOCAL(IX_SA_ERROR_INVALID_CC_LIST,
                          ("Invalid CC_LIST registry property:"
                           " (NAME: %s) value is not valid.\n"
                           " Parent = %s",
                           cclist_info.m_aName, prop_info.m_aName));
        }

        /* next engine */
        idx++;
        ix_rm_cp_property_get_subproperty(arg_hengines, idx, 
                                          &subprop);
    }

    return IX_SUCCESS;
}


/*  _ix_sa_check_engine_prop */
/*  ---------------------------------------- */
/**
 * Validate an individual exeuction engine entry in the registry.
 *
 * This function is used by _ix_sa_reg_verify_engines to do
 * the actual verification of each individual EE entry.  This
 * checks for correct name, correct value, and detects duplicate
 * engine entries.  
 *
 * @param arg_ee_id_used array used to flag used EE id's
 * @param arg_eenum the value (id) of the prop being evaluated.
 * @param arg_name the name of the prop being evaluated.
 *
 * @return ix_uint8 indicating a valid or invalid ee prop.
 * @retval 0 The execution engine propery is invalid.
 * @retval 1 The execution engine propery is valid.
**/
ix_uint8
_ix_sa_check_engine_prop(ix_uint8 *arg_ee_id_used,
                         ix_uint8 arg_eenum, 
                         const char *arg_name)
{
    char *cp;
    char tempname[IX_CP_MAX_PROPERTY_NAME_LENGTH];

    /* Verify that arg_name is valid */
    if((arg_name == NULL) || (*arg_name == 0)){
        return 0;
    }

    /* Verify arg_ee_id_used is valid */
    if(arg_ee_id_used == NULL){
        return 0;
    }
    
    strcpy(tempname, arg_name);
    /* Search from the end of the string for '_' */
    cp = &tempname[strlen(tempname) - 1];
    while((*cp != '_') && (cp != tempname)){
        cp--;
        
    }
    
    /* If '_' was found, null it and increment. otherwise return false
       The effect of this is that arg_name = string up to the last '_'
       and cp = string after last '_'.  This is what we will test against.*/
    if(*cp == '_'){
        *cp = 0;
        cp++;
    }else{
        return 0;
    }
    
    /* Verify the property is named correctly */
    if(strcmp("SA_EE", tempname)){
        return 0;
    }
    
    /* Verify that the ending number matches the value */
    if(arg_eenum != atoi(cp)){
        return 0;
    }

    /* Verify that the id has not already been used. */
    if(arg_ee_id_used[arg_eenum] == 0){
        arg_ee_id_used[arg_eenum] = 1;
    }else{
        return 0;
    }

    return 1;
}

/*  _ix_sa_create_default_engine */
/*  ---------------------------------------- */
/**
 * Creates the default execution engine.
 *
 * @param arg_sa_ctrl System app control struct.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
 * 
**/
ix_error
_ix_sa_create_default_engine(ix_sa_ctrl *arg_sa_ctrl)
{

    /* Validate Args */
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);
    g_ix_sa_ctrl = arg_sa_ctrl;
    IX_ERROR_CR(ix_cci_exe_run(NULL, _ix_sa_default_exe_init,
                               _ix_sa_default_exe_fini,"Exe0", 
                               &(arg_sa_ctrl->def_ee_handle)));
    return IX_SUCCESS;
}

/*  _ix_sa_delete_default_engine */
/*  ---------------------------------------- */
/**
 * Delete default execution engine
 *
 * @param arg_sa_ctrl System app control struct.
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_delete_default_engine(ix_sa_ctrl *arg_sa_ctrl)
{
    /* Validate Args */
    IX_ERROR_CHECK_ARG_NULL(1, arg_sa_ctrl);
    
    /* Delete default execution engine */
    if(arg_sa_ctrl->def_ee_handle != 0){
        IX_ERROR_CR(ix_cci_exe_shutdown(arg_sa_ctrl->def_ee_handle));
        arg_sa_ctrl->def_ee_handle = 0;
    }

    return IX_SUCCESS;
}

/*  _ix_sa_default_exe_init */
/*  ---------------------------------------- */
/**
 * Init function for the default execution engine.
 *
 *
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_default_exe_init(ix_exe_handle arg_exech, void **ppContext)
{
    /* Validate Args */
    IX_ERROR_CHECK_ARG_NULL(2, ppContext);

    IX_ERROR_CR(ix_cc_msup_init(g_ix_sa_ctrl->hMsg_Freelist,
                                g_ix_sa_ctrl->msg_Buff_Size,
                                ppContext));

    return IX_SUCCESS;
}




/*  _ix_sa_default_exe_fini */
/*  ---------------------------------------- */
/**
 * Fini function for the default engine.
 *
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_default_exe_fini(ix_exe_handle arg_exech, void *pContext)
{


    IX_ERROR_CR(ix_cc_msup_fini(pContext));

    return IX_SUCCESS;
}


#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)

#else
/*  _ix_sa_set_cc */
/*  ---------------------------------------- */
/**
 * This function is used to set a init and fini function for a CC
 *
 * @return IX_SUCCESS upon success or an ix_error with one of the following values.
 * @retval NONE
**/
ix_error
_ix_sa_set_cc(ix_sa_ccfunc *arg_pCC_List, ix_uint32 arg_idx,
              ix_cc_init arg_Init, ix_cc_fini arg_Fini)
{
    ix_uint32 i;

    IX_ERROR_CHECK_ARG_NULL(1, arg_pCC_List);
    arg_pCC_List[arg_idx].init = arg_Init;
    arg_pCC_List[arg_idx].fini = arg_Fini;
    for (i = 0; i < IX_CC_INSTANCE_MAX; i++)
    {
        arg_pCC_List[arg_idx].cc_handle[i] = 0;
    }
    return IX_SUCCESS;
}
#endif


