/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a licence agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2003 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 *
 * = FILENAME
 *      vallejo.c
 *
 * = DESCRIPTION
 *      This file contains scripts to initialize the vallejo quad gbeth PHY
 *      on an IXPD2401 platform.
 *
 * = AUTHOR
 *      Tomasz Madajczak
 *      tomasz.madajczak@intel.com
 *
 * = CHANGE HISTORY
 *      01/22/2003      Created by Tomasz Madajczak
 * ============================================================================
 */

/* per port index */
#define INDEX_PORT0 			0x00
#define INDEX_PORT1 			0x80
#define INDEX_PORT2 			0x100
#define INDEX_PORT3 			0x180


/* defines for driver modes (bit 2:0) */
#define IX_CC_ETH_TX_DRIVER_MODE_FIBER               0x0
#define IX_CC_ETH_TX_DRIVER_MODE_GIGA_HALF_DUPLEX    0x1
#define IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX    0x2
#define IX_CC_ETH_TX_DRIVER_MODE_100_HALF_DUPLEX     0x3
#define IX_CC_ETH_TX_DRIVER_MODE_100_FULL_DUPLEX     0x4
#define IX_CC_ETH_TX_DRIVER_MODE_10_HALF_DUPLEX      0x5
#define IX_CC_ETH_TX_DRIVER_MODE_10_FULL_DUPLEX      0x6



/* defines for driver parity (bit 3) */
#define IX_CC_ETH_TX_DRIVER_MODE_PARITY_ODD          0x0
#define IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN         0x8



/* defines for driver block modes (bit 4) */
#define IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY          0x00
#define IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8           0x10


#define IX_CC_ETH_TX_DRIVER_PORT_MASK   0xF
#define IX_CC_ETH_TX_FRAME_SIZE  0x23FC /* 9216 - 4 */

/* Platform specific - in ixdp2401 platform we need to split the port mask
 * onto ports working in fiber/copper mode (front pair) and ports working only
 * in copper mode (backplane pair).
 */
#define IX_CC_ETH_TX_DRIVER_PORT_MASK_01   0x3
#define IX_CC_ETH_TX_DRIVER_PORT_MASK_23   0xc

#ifdef USE_BB_DRV_CONFIG
#define IX_CC_ETH_TX_MAX_NUM_PORTS         12
#else /* USE_BB_DRV_CONFIG */
#define IX_CC_ETH_TX_MAX_NUM_PORTS         4
#endif /* USE_BB_DRV_CONFIG */

/**
 * User defined include files required.
 */

/* ethernet driver files */
#include "ixf1104ce_driver_api.h"
#include "ixf1104ce_ioctl.h"
#include "ixf1104ce_internal.h"

#ifdef USE_BB_DRV_CONFIG
#include "baseboard_media_drv.h"
unsigned int hwConfigEthPortMask;/* ETH Port Mask */
unsigned int hwConfigEthMode[IX_CC_ETH_TX_MAX_NUM_PORTS]; /* ETH Port configuration mode value */
unsigned int hw_port[4];
#endif /* USE_BB_DRV_CONFIG */


/**
 * Pre-processor symbol and macro definitions.
 *
 */
#define hal_delay_us(x) taskDelay(1) /* VxWorks specific */

#define eprintf printf

typedef unsigned int	uint32;

typedef struct {
		uint32 reg_addr;
		int config_val;
		int z;
} configData;

static configData aMacResetConfigTbl[ ] =
{
        {MAC_SOFT_RESET,        0x0000000F, 1},
        {TXFIFO_PORT_RESET,     0x0000000F, 1},
        {RXFIFO_PORT_RESET,     0x0000000F, 1},
        {SPI3_TX_GLOBAL_CONFIG, 0x00E00000, 1},
        {MDIO_RESET,            0x00000001, 1}
};


#define MacResetConfigTblSz \
    (sizeof(aMacResetConfigTbl) /sizeof(aMacResetConfigTbl[0]))

void vCb(int ctx, int port, int statusMask)
{
    printf("\nPort %u, Status Mask %X\n", port, statusMask);
}

#ifdef USE_BB_DRV_CONFIG
int _set_VallejoConfig()
{
	int		i;
	int		bb_handle;
	char	mezzanine_pos_atm = 0;

	hwConfigEthPortMask = 0;
	bb_handle = Bb_Open();

	/* Mezzanine in DB0 */
	switch(Bb_GetDevType(bb_handle, MEDIA_CARD1_ID))
	{
		case BB_FAILURE				:	break;
		case POS_ATM_OC312_ID		:   printf("Mezzanine %s ", POS_ATM_OC312_STR_ID);
										mezzanine_pos_atm |= 0x1;
										break;
		case POS_ATM_OC48_ID		:   printf("Mezzanine %s ", POS_ATM_OC48_STR_ID);
										mezzanine_pos_atm |= 0x1;
										break;
		case GB_ETH_ID				:   printf("Mezzanine %s ", GB_ETH_STR_ID);
										break;
		default						:	printf("ERROR!!! Device in DB0 unknown!!!\n");
										return -1;
	}

	/* MIC1 */
	switch(Bb_GetDevType(bb_handle, MIC1_ID))
	{
		case BB_FAILURE				:	if((mezzanine_pos_atm & 1) != 0)
										{
										 	printf("in DB0\n");
										}
										else
											printf("No device in DB0\n");
										break;
		case MIC_C4xGB_MEZZ_ID 		:   printf("with %s in DB0\n", MIC_C4xGB_MEZZ_STR_ID);
										hwConfigEthPortMask |= 0x30;
										for(i = 4; i < 6; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										break;
		case MIC_F4xGB_MEZZ_ID		:   printf("with %s in DB0\n", MIC_F4xGB_MEZZ_STR_ID);
										hwConfigEthPortMask |= 0x30;
										for(i = 4; i < 6; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										break;
		case MIC_C2xGB_BRD_ID		:   printf("%s on baseboard in slot 1\n", MIC_C2xGB_BRD_STR_ID);
										hwConfigEthPortMask |= 0x3;
										for(i = 0; i < 2; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
										break;
		case MIC_F4xOC12_MEZZ_ID	:   printf("with %s in DB0\n", MIC_F4xOC12_MEZZ_STR_ID);
										break;
		case MIC_F1xOC48_MEZZ_ID	:   printf("with %s in DB0\n", MIC_F1xOC48_MEZZ_STR_ID);
										break;
		case MIC_F2xOC12_MEZZ_ID	:   printf("with %s in DB0\n", MIC_F2xOC12_MEZZ_STR_ID);
										break;
		default						:	printf("ERROR!!! MIC1 unknown!!!\n");
										return -1;
	}


	/* Mezzanine in DB1 */
	switch(Bb_GetDevType(bb_handle, MEDIA_CARD2_ID))
	{
		case BB_FAILURE				:	break;
		case POS_ATM_OC312_ID		:   printf("Mezzanine %s ", POS_ATM_OC312_STR_ID);
										mezzanine_pos_atm |= 0x2;
										break;
		case POS_ATM_OC48_ID		:   printf("Mezzanine %s ", POS_ATM_OC48_STR_ID);
										mezzanine_pos_atm |= 0x2;
										break;
		case GB_ETH_ID				:   printf("Mezzanine %s ", GB_ETH_STR_ID);
										break;
		default						:	printf("ERROR!!! Device in DB1 unknown!!!\n");
										return -1;
	}

	/* MIC2 */
	switch(Bb_GetDevType(bb_handle, MIC2_ID))
	{
		case BB_FAILURE				:	if((mezzanine_pos_atm & 2) != 0)
										{
										 	printf("in DB1\n");
										}
										else
											printf("No device in DB1\n");
										break;
		case MIC_C4xGB_MEZZ_ID 		:   printf("with %s in DB1\n", MIC_C4xGB_MEZZ_STR_ID);
										hwConfigEthPortMask |= 0x300;
										for(i = 8; i < 10; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										break;
		case MIC_F4xGB_MEZZ_ID		:   printf("with %s in DB1\n", MIC_F4xGB_MEZZ_STR_ID);
										hwConfigEthPortMask |= 0x300;
										for(i = 8; i < 10; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_MPHY;
										break;
		case MIC_F2xGB_BRD_ID		:   printf("%s on baseboard in slot 2\n", MIC_F2xGB_BRD_STR_ID);
										hwConfigEthPortMask |= 0x3;
										for(i = 0; i < 2; i++)
										   hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                             									IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             									IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
										break;
		case MIC_F4xOC12_MEZZ_ID	:   printf("with %s in DB1\n", MIC_F4xOC12_MEZZ_STR_ID);
										break;
		case MIC_F1xOC48_MEZZ_ID	:   printf("with %s in DB1\n", MIC_F1xOC48_MEZZ_STR_ID);
										break;
		case MIC_F2xOC12_MEZZ_ID	:   printf("with %s in DB1\n", MIC_F2xOC12_MEZZ_STR_ID);
										break;
		default						:	printf("ERROR!!! MIC2 unknown!!!\n");
										return -1;
	}

	if((hwConfigEthPortMask == 0x3) ||	/* MIC_C2xGB_BRD_ID or MIC_F2xGB_BRD_ID */
	   (hwConfigEthPortMask == 0x30))	/* MIC_C4xGB_MEZZ_ID or MIC_F4xGB_MEZZ_ID */
	{
		hwConfigEthPortMask |= 0xc;	/* Add backplane ports to the port's mask */
		for(i = 2; i < 4; i++)
			hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
        					 	 IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             	 IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
	}
	else /* all ports on baseboard: 2 backplane + 2 front coper (for baseboard spin 1A) */
	{
		hwConfigEthPortMask |= 0xf;	/* Add backplane ports to the port's mask */
		for(i = 0; i < 4; i++)
			hwConfigEthMode[i] = IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
        					 	 IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             	 IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8;
	}

	Bb_Close(bb_handle);
	return 0;
}
#endif /* USE_BB_DRV_CONFIG */


/***
 * reg_write routine now accepts register offset
 * and genetrates the complete address.
 ****/
void _reg_write(volatile uint32 offset, uint32 val)
{
	volatile unsigned int address=0;
	volatile unsigned int val_lo, val_hi;

    val_lo = (val & 0xffff);
    val_hi = (val >> 16) & 0xffff;

	address = 0xC6020000 + (offset << 3); /* 2 * 4 */

	*((volatile unsigned int*)(address)) = val_lo;
    *((volatile unsigned int*)(address + 4)) = val_hi;

}

/***
 * reg_read routine now accepts register offset
 * and genetrates the complete address.
 ****/
uint32 _reg_read(volatile uint32 offset)
{
    volatile unsigned int address;
	volatile unsigned int val_lo, val_hi, val;

	address = 0xC6020000 + (offset << 3); /* 2 * 4 */
	val_lo = *((volatile unsigned int*)(address));
    val_hi = *((volatile unsigned int*)(address + 4));
    val = (val_lo & 0xffff) | ((val_hi << 16) & 0xffff0000);

    return val;
}


#ifdef USE_BB_DRV_CONFIG
uint32 getLogicalPort(uint32 hw_port)
{
	uint32	port = 0;
	switch(hw_port)
	{
		case	0	:				/* MIC_C2xGB_BRD_ID or MIC_F2xGB_BRD_ID or front coper */
		case	4	:               /* MIC_C4xGB_MEZZ_ID or MIC_F4xGB_MEZZ_ID in DB0 */
		case	8	:	port = 0;	/* MIC_C4xGB_MEZZ_ID or MIC_F4xGB_MEZZ_ID in DB1 */
						break;
		case	1	:               /* MIC_C2xGB_BRD_ID or MIC_F2xGB_BRD_ID or front coper */
		case	5	:               /* MIC_C4xGB_MEZZ_ID or MIC_F4xGB_MEZZ_ID in DB0 */
		case	9	:	port = 1;   /* MIC_C4xGB_MEZZ_ID or MIC_F4xGB_MEZZ_ID in DB1 */
						break;
		case	2	:	port = 2;	/* backplane */
						break;
		case	3	:	port = 3;   /* backplane */
						break;
	}
	return port;
}
#endif /* USE_BB_DRV_CONFIG */


uint32 handle1 = 0;
uint32 handle2 = 0;

/*
 * Initializes Vallejo, Alaska and CPLD in the requested mode
 *
 */
void vinit(char mode)
{
#define VALLEJO_BB_BASE_ADDR  0xC6020000

    uint32 err, i, j;
    uint32 vallejoPortMask, vallejoHandle=0;
    gbe_mac_ioctl_ptr macIoctlInfo;
    gbe_mac_ioctl_ptr_64 macIoctlInfo_64;

#ifdef USE_BB_DRV_CONFIG
	int status;
	_set_VallejoConfig();
#endif /* USE_BB_DRV_CONFIG */

#ifdef USE_BB_DRV_CONFIG
    printf("Initializing GbEMAC\n");

    for (i = 0; i < IX_CC_ETH_TX_MAX_NUM_PORTS; i++)
    {
    	if (hwConfigEthPortMask & (1 << i))
        {
        	err = GbEMAC_DeviceStart((1 << i), &handle1, hwConfigEthMode[i]);
            if (err != SUCCESS)
			{
                printf("Media driver start failed!");
				return;
			}
        }
    }

	 /* GbEMAC_Callback(&handle1, vCb, 0); */
	 /* GbEMAC_Callback(&handle2, vCb, 0); */

    /* Set the JUMPO MTU size for all ports */
    for (j = 0; j < IX_CC_ETH_TX_MAX_NUM_PORTS; j++)
	if(hwConfigEthPortMask & (1 << j))
    {
		i = getLogicalPort(j);
		hw_port[i]=j;
        macIoctlInfo_64.portIndex = j;
        macIoctlInfo_64.valueHigh = 0x00000201;
        macIoctlInfo_64.valueLow = 0x02030001 + i;

        macIoctlInfo.portIndex = j;
        macIoctlInfo.value = IX_CC_ETH_TX_FRAME_SIZE;
        err = GbEMAC_Ioctl(&handle1,
                           SET_STN_ADDR,
                           (void *)(&macIoctlInfo_64));
        if (err != SUCCESS) return;

        err = GbEMAC_Ioctl(&handle1,
                           SET_MAX_FRAME_SIZE,
                           (void *)(&macIoctlInfo));
        if (err != SUCCESS) return;
    }
    printf("GbEMAC initialization COMPLETE\n");

#else /* USE_BB_DRV_CONFIG */
    printf("Initializing GbEMAC in mode %c\n", mode);

    switch (mode)
    {
    case 'f':
    case 'F':
        /* setup fiber in Alaska */
        /**(volatile int*)0xc6024018 = 0x00000500;*/
        err = GbEMAC_DeviceStart(IX_CC_ETH_TX_DRIVER_PORT_MASK_01,
                                 &handle1,
                                 IX_CC_ETH_TX_DRIVER_MODE_FIBER |
                                 IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                                 IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8);
        break;
    case 'c':
    case 'C':
        /* Enable Copper in Alaska*/
        /**(volatile int*)(0xc6024018) = 0x000F0003; */
        err = GbEMAC_DeviceStart(IX_CC_ETH_TX_DRIVER_PORT_MASK_01,
                                 &handle1,
                                 IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                                 IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                                 IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8);
        break;
    default:
        return;
    }

    if(err != SUCCESS)
    {
        return;
    }/* end if */
    printf("Enabled fiber ports\n");

    err = GbEMAC_DeviceStart(IX_CC_ETH_TX_DRIVER_PORT_MASK_23,
                             &handle2,
                             IX_CC_ETH_TX_DRIVER_MODE_GIGA_FULL_DUPLEX |
                             IX_CC_ETH_TX_DRIVER_MODE_PARITY_EVEN |
                             IX_CC_ETH_TX_DRIVER_MODE_BLOCK_4_8);

    if(err != SUCCESS)
    {
        return;
    }/* end if */
    printf("Enabled backplane copper ports\n");

	 /* GbEMAC_Callback(&handle1, vCb, 0); */
	 /* GbEMAC_Callback(&handle2, vCb, 0); */

    /* Set the JUMPO MTU size for all ports */
    for (i = 0; i < IX_CC_ETH_TX_MAX_NUM_PORTS; i++)
    {
        macIoctlInfo_64.portIndex = i;
        macIoctlInfo_64.valueHigh = 0x00000201;
        macIoctlInfo_64.valueLow = 0x02030001 + i;

        macIoctlInfo.portIndex = i;
        macIoctlInfo.value = IX_CC_ETH_TX_FRAME_SIZE;
        if (i < 2)
        {
            err = GbEMAC_Ioctl(&handle1,
                               SET_STN_ADDR,
                               (void *)(&macIoctlInfo_64));
            if (err != SUCCESS) return;

            err = GbEMAC_Ioctl(&handle1,
                               SET_MAX_FRAME_SIZE,
                               (void *)(&macIoctlInfo));
        }
        else
        {
            err = GbEMAC_Ioctl(&handle2,
                               SET_STN_ADDR,
                               (void *)(&macIoctlInfo_64));
            if (err != SUCCESS) return;

            err = GbEMAC_Ioctl(&handle2,
                               SET_MAX_FRAME_SIZE,
                               (void *)(&macIoctlInfo));
        }
        if (err != SUCCESS)
        {
            return;
        }/* end if */
    }/* end for */

    printf("GbEMAC initialization COMPLETE\n");
#endif /* USE_BB_DRV_CONFIG */
}


/*
 * Read Vallejo statistics
 *
 */
void vallejo_stats(unsigned int port)
{
    gbe_mac_ioctl_ptr_64 stats;
    gbe_mac_ioctl_ptr param;
    uint32 handle;

#ifdef USE_BB_DRV_CONFIG
    if (port > 3)
		return;
	else
	{
		port = hw_port[port];
		handle = handle1;
	}
#else /* USE_BB_DRV_CONFIG */
    if (port > 3) return;
    if (port < 2)
        handle = handle1;
    else
        handle = handle2;
#endif /* USE_BB_DRV_CONFIG */

    stats.portIndex = port;
    param.portIndex = port;

    printf("---------------------------------------------------------\n");
    printf("Reading Vallejo statistics for port %d\n", port);
    printf("---------------------------------------------------------\n");

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_OCTETS_OK,
                                (void *)(&stats)))
        return;
    printf("GET_RX_OCTETS_OK         = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_OCTETS_BAD,
                                (void *)(&stats)))
        return;
    printf("GET_RX_OCTETS_BAD        = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_UC_PKTS,
                                (void *)(&stats)))
        return;
    printf("GET_RX_UC_PKTS           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_MC_PKTS,
                                (void *)(&stats)))
        return;
    printf("GET_RX_MC_PKTS           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_BC_PKTS,
                                (void *)(&stats)))
        return;
    printf("GET_RX_BC_PKTS           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PKTS_64,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PKTS_64           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PKTS_65_127,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PKTS_65_127       = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PKTS_128_255,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PKTS_128_255      = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PKTS_256_511,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PKTS_256_511      = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PKTS_512_1023,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PKTS_512_1023     = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PKTS_1024_1518,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PKTS_1024_1518    = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PKTS_1519_MAX,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PKTS_1519_MAX     = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_FCS_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_RX_FCS_ERR           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_DATA_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_RX_DATA_ERR          = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_ALLIGN_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_RX_ALLIGN_ERR        = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_LONG_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_RX_LONG_ERR          = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_JABBER_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_RX_JABBER_ERR        = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_PAUSE_MAC_CTL,
                                (void *)(&stats)))
        return;
    printf("GET_RX_PAUSE_MAC_CTL     = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_UNKNOWN_CTL_FRAME,
                                (void *)(&stats)))
        return;
    printf("GET_RX_UNKNOWN_CTL_FRAME = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_VLONG_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_VLONG_ERR            = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RUNT_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_RUNT_ERR             = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_SHORT_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_SHORT_ERR            = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_SEQ_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_SEQ_ERR              = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_SYMBOL_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_SYMBOL_ERR           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    /**
     * TX Statistics Register IOCTL Commands.
     */
    printf("---------------------------------------------------------\n");

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_OCTETS_OK,
                                (void *)(&stats)))
        return;
    printf("GET_TX_OCTETS_OK         = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_OCTETS_BAD,
                                (void *)(&stats)))
        return;
    printf("GET_TX_OCTETS_BAD        = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_UC_PKTS,
                                (void *)(&stats)))
        return;
    printf("GET_TX_UC_PKTS           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_MC_PKTS,
                                (void *)(&stats)))
        return;
    printf("GET_TX_MC_PKTS           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_BC_PKTS,
                                (void *)(&stats)))
        return;
    printf("GET_TX_BC_PKTS           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PKTS_64,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PKTS_64           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PKTS_65_127,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PKTS_65_127       = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PKTS_128_255,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PKTS_128_255      = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PKTS_256_511,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PKTS_256_511      = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PKTS_512_1023,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PKTS_512_1023     = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PKTS_1024_1518,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PKTS_1024_1518    = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PKTS_1519_MAX,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PKTS_1519_MAX     = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_PAUSE_FRAME,
                                (void *)(&stats)))
        return;
    printf("GET_TX_PAUSE_FRAME       = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_DEFERRED_ERR,
                                (void *)(&stats)))
        return;

    printf("GET_TX_DEFERRED_ERR      = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_TOTAL_COLLISION,
                                (void *)(&stats)))
        return;
    printf("GET_TX_TOTAL_COLLISION   = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_SINGLE_COLLISION,
                                (void *)(&stats)))
        return;
    printf("GET_TX_SINGLE_COLLISION  = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_MUL_COLLISION,
                                (void *)(&stats)))
        return;
    printf("GET_TX_MUL_COLLISION     = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_LATE_COLLISION,
                                (void *)(&stats)))
        return;
    printf("GET_LATE_COLLISION       = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_EXCV_COLLISION,
                                (void *)(&stats)))
        return;
    printf("GET_TX_EXCV_COLLISION    = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_EXCV_DEFERRED_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_TX_EXCV_DEFERRED_ERR = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_EXCV_LEN_DROP,
                                (void *)(&stats)))
        return;
    printf("GET_TX_EXCV_LEN_DROP     = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_UNDERRUN,
                                (void *)(&stats)))
        return;
    printf("GET_TX_UNDERRUN          = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_TX_CRC_ERR,
                                (void *)(&stats)))
        return;
    printf("GET_TX_CRC_ERR           = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_FC_COLLISION_SEND,
                                (void *)(&stats)))
        return;
    printf("GET_FC_COLLISION_SEND    = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);


    /**
     * RX FIFO Register IOCTL Commands
     */
    printf("---------------------------------------------------------\n");

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_STN_ADDR,
                                (void *)(&stats)))
        return;
    printf("GET_STN_ADDR             = 0x%08x-%08x\n",
           stats.valueHigh, stats.valueLow);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_FIFO_HIGH_WATERMARK,
                                (void *)(&param)))
        return;
    printf("GET_RX_FIFO_HIGH_WM      = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_FIFO_LOW_WATERMARK,
                                (void *)(&param)))
        return;
    printf("GET_RX_FIFO_LOW_WM       = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_FRAME_REMOVED,
                                (void *)(&param)))
        return;
    printf("GET_RX_FRAME_REMOVED     = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_FIFO_PORT_RESET,
                                (void *)(&param)))
        return;
    printf("GET_RX_FIFO_PORT_RESET   = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_FIFO_FRAME_DROP,
                                (void *)(&param)))
        return;
    printf("GET_RX_FIFO_FRAME_DROP   = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RX_FIFO_OVERFLOW_STT,
                                (void *)(&param)))
        return;
    printf("GET_RX_FIFO_OVERFLOW_STT = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_OUT_SEQUENCE_INFO,
                                (void *)(&param)))
        return;
    printf("GET_OUT_SEQUENCE_INFO    = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_DROPPED_PKTS,
                                (void *)(&param)))
        return;
    printf("GET_DROPPED_PKTS         = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_RW_PTR_RX_FIFO,
                                (void *)(&param)))
        return;
    printf("GET_RW_PTR_RX_FIFO       = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_OCCUPANCY_RX_FIFO,
                                (void *)(&param)))
        return;
    printf("GET_OCCUPANCY_RX_FIFO    = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_JUMBO_PKT_SIZE,
                                (void *)(&param)))
        return;
    printf("GET_JUMBO_PKT_SIZE       = 0x%08x\n",
           param.value);

    if (SUCCESS != GbEMAC_Ioctl(&handle,
                                GET_PKT_DROP_CAP_FIFO,
                                (void *)(&param)))
        return;
    printf("GET_PKT_DROP_CAP_FIFO    = 0x%08x\n",
           param.value);

}

