/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a licence agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2003 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 *
 * = FILENAME
 *      quad_gbeth_memory.h
 *
 * = DESCRIPTION
 *      This file contains the system memory map that can be
 *      customised for a given hardware/application
 *
 * = AUTHOR
 *      Tomasz Madajczak
 *      tomasz.madajczak@intel.com
 *
 * = CHANGE HISTORY
 *      01/22/2003      Created by Tomasz Madajczak
 * ============================================================================
 */
#ifndef __QUAD_GBETH_MEMORY__H__
#define __QUAD_GBETH_MEMORY__H__

#define IPV4_COUNTERS_SCRATCH

/****************************************
 *     SRAM and SDRAM DEFINITIONS
 ***************************************/
/*
 * Amount of SRAM space available for (meta data of) packet buffers. 0,5 MB.
 *  In simulation we use only 10K bytes.
 */

#define     BUF_SRAM_SIZE       (1024 * 10 * 2 * 10)

/*
 *  Amount of SDRAM space available for Packet buffers. 64 MB.
 *  In simulation we use only 640K bytes.
 */
#define     BUF_SDRAM_SIZE      (64 * 1024 * 10 * 2 * 10)

/*
 *  Size of Meta Data (Buffer Descriptors) in SRAM. 32 entries per buffer.
 *  Should be a power of 2.
 */
#define     META_DATA_SIZE      32

/*  Size of packet buffer. */
#define     BUFFER_SIZE         2048

/*  Size of each Queue Descriptor Entry (in Bytes) */
#define     QD_SIZE             16

#ifdef USE_IMPORT_VAR

.import_var BUF_SRAM_BASE
.import_var DL_REL_BASE
.import_var BUF_FREE_LIST0
.import_var SRAM_FLT_TABLE_BASE
.import_var SRAM_PORT_BIT_MAP_BASE
.import_var L2_TABLE_SRAM_BASE

#else   /* not USE_IMPORT_VAR */

/**********************************
 *     SRAM and SDRAM DEFINITIONS
 *********************************/
/*
 *  Base address in SRAM for the packet buffers.
 *  This is for the meta data (buffer descriptors).
 */
#define     BUF_SRAM_BASE       0

/*
 *  Base Address in DRAM for the packet buffers.
 * This is for the actual packet data.
 */
#define     BUF_SDRAM_BASE      0

/*  Base Address for Queue Descriptors. @ 0,5MB in SRAM (channel 0 [31:30]) */
#define     QD_SRAM_BASE        0x80000

/*  Total number of Queue Descriptors */
#define     QD_TOTAL            1025

/*
 *  Following couple of definitions are used as counters by
 *  respective microblocks in SRAM
 *
 *
 * for better DRAM bank utilization, we put the nexthop info in
 * SRAM at this address. We also have a segment in DRAM reserved
 * to keep this info, should we decide to move back to DRAM. This
 * would a compile time option for IPV4 forwarder.
 *
 * Important!!! nexthop table in SRAM should be never used when
 * one SRAM bank available
 */
#define     NEXTHOP_TABLE_SRAM_BASE         (QD_SRAM_BASE + (QD_TOTAL * QD_SIZE))

/*
 * 16MB is enough for 8K buffers. Hence, moved the
 * nexthop base to 16MB.
 */
#define     NEXTHOP_TABLE_SDRAM_BASE        0x1000000

/*  Each Next Hop entry is 16 bytes. So 4K is sufficient */
#define     NEXTHOP_TABLE_SIZE              0x1000


/*  Trie Table is in channel 0 [31:30] at address 2MB in SRAM */
#define     TRIE_TABLE_SRAM_BASE            0x00200000

/*  Trie Table size. 1MB */
#define     TRIE_TABLE_SIZE                 0x100000

#ifdef IPV4_COUNTERS_SCRATCH

    /* Base for IPv4 32bit counters. This counter table is placed in
     * Scratch @ 12KB to improve performance.
     */
    #define     IPV4_STATS_TABLE_BASE       12288

    /* space allocated to maintain these counters */
    #define     IPV4_STATS_TABLE_SIZE       0x800

    /*
     * define the below consts to easy computing of next SRAM locations
     * base for IPv4 32bit counters
     */
    #define     STATS_TABLE_SRAM_BASE       0x00300000

    /* space allocated to maintain these counters */
    #define     STATS_TABLE_SRAM_SIZE       0x0


#else

    /* base for IPv4 32bit counters */
    #define     STATS_TABLE_SRAM_BASE       0x00300000

    /* space allocated to maintain these counters */
    #define     STATS_TABLE_SRAM_SIZE       0x800

#endif  /* IPV4_COUNTERS_SCRATCH */


/*  Base for Packet RX counters */
#define     PACKET_COUNTERS_SRAM_BASE       (STATS_TABLE_SRAM_BASE + STATS_TABLE_SRAM_SIZE)
#define     PACKET_COUNTERS_SRAM_SIZE       0x1000 /*this is probably too much*/

/*  Directed broadcast table base in SRAM */
#define     DBCAST_TABLE_SRAM_BASE          (PACKET_COUNTERS_SRAM_BASE + PACKET_COUNTERS_SRAM_SIZE)
#define     DBCAST_TABLE_SIZE               0x2000

/* control block */
#define     CONTROL_BLOCK_SRAM_BASE         (DBCAST_TABLE_SRAM_BASE + DBCAST_TABLE_SIZE)
#define     CONTROL_BLOCK_SRAM_SIZE         0x4 /* one long word cope with 32
                                                 * ports */


/* Hash table size */
#define     HASH_TABLE_SIZE                 1024*8

/* Ethernet MAC filtering hash table base: should be modified later */
#define     SRAM_FLT_TABLE_BASE             (CONTROL_BLOCK_SRAM_BASE + CONTROL_BLOCK_SRAM_SIZE)
#define     SRAM_FLT_TABLE_SIZE             0x0 /* needs (?) 0x100000
                                                 * if enabled*/

/*
 * Input port bit map base: should be be set and updated by core
 * Each port uses 1 bits to define the port type
 *  bit 0 for port 0, bit 1 for port 1,..., bit 15 for port 15
 *  Bit definition:
 *      0   --- the port is defined as unit cast mode
 *      1  --- The port is defined promisc mode
 */
#define     SRAM_PORT_BIT_MAP_BASE          (SRAM_FLT_TABLE_BASE + SRAM_FLT_TABLE_SIZE)
#define     SRAM_PORT_BIT_MAP_SIZE          0x00000100 /*!check this! if used,
                                                        * 256 bytes should
                                                        * be enough */

/*
 * Packet TX needs 16*3 counters and each counter is 1LW
 * this implies 48*4 bytes. Though Packet TX counter does not require
 * 0x10000 bytes but we start Packet TX counters from 0x310000,
 * to provide memory for any additional counters in future.
 */
#define     PACKET_TX_COUNTER_BASE          (SRAM_PORT_BIT_MAP_BASE + SRAM_PORT_BIT_MAP_SIZE)
#define     PACKET_TX_COUNTER_SIZE          48*4 /*recommended value is
                                                  *0x00010000 (see above)*/

/*
 * L2 Table (each entry has 32 bytes, bit 31 of first long wors is valid bit
 * requires 64K * 32 = 2M, but it is to much -> we have only 1M of available memory so we
 * can hold only 32K MAC addresses
 */
#define     L2_TABLE_SRAM_BASE              0x00100000
#define     L2_TABLE_SRAM_SIZE              (TRIE_TABLE_SRAM_BASE - L2_TABLE_SRAM_BASE)

/*  Area of SRAM which is zero'ed out */

#define     SRAM_ZERO_BLOCK                 (PACKET_TX_COUNTER_BASE + PACKET_TX_COUNTER_SIZE)
#define     SRAM_ZERO_BLOCK_SIZE            16*4 /* 16 longs words needed
                                                  * to initialize scheduler
                                                  * xfers to 0 */

#endif  /* #ifdef USE_IMPORT_VAR */

/*********************************
 *         SCRATCH DEFINITIONS
 ********************************/

/* These are all Scratch ring definitions */

/*  There are two exception ring to send pkts up */
#define         EXCEPTION_RING_OUT_0            0
#define         EXCEPTION_RING_OUT_0_BASE       0
#define         EXCEPTION_RING_OUT_0_SIZE       128

#define         EXCEPTION_RING_OUT_1            1
#define         EXCEPTION_RING_OUT_1_BASE       512
#define         EXCEPTION_RING_OUT_1_SIZE       128

/*  There are two exception ring to send pkts down */
#define         EXCEPTION_RING_IN_0             2
#define         EXCEPTION_RING_IN_0_BASE        1024
#define         EXCEPTION_RING_IN_0_SIZE        128

#define         EXCEPTION_RING_IN_1             3
#define         EXCEPTION_RING_IN_1_BASE        1536
#define         EXCEPTION_RING_IN_1_SIZE        128

/*
 *  Scratch for communicating between ETH RX and IPv4 Blocks
 *  Ring number 4, base 0, and size 1024 Long Words
 */
#define         ETH_RX_TO_IPV4_SCR_RING         4
#define         ETH_RX_TO_IPV4_SCR_RING_BASE    2048
#define         ETH_RX_TO_IPV4_SCR_RING_SIZE    1024

/*
 *  Scratch for communicating between IPV4 and Queue Manager Blocks
 *  Ring number 5, base 512, and size 128 Long Words
 */
#define         IPV4_TO_QM_SCR_RING             5
#define         IPV4_TO_QM_SCR_RING_BASE        6144
#define         IPV4_TO_QM_SCR_RING_SIZE        256

/*
 *  Scratch for communicating between Scheduler and Queue Manager Blocks
 *  Ring number 6, base 1024, and size 256 Long Words
 */
#define         SCHEDULER_TO_QM_SCR_RING        6
#define         SCHEDULER_TO_QM_SCR_RING_BASE   7168
#define         SCHEDULER_TO_QM_SCR_RING_SIZE   256

/*
 *  First scractch ring for communicating between Queue Manager and Packet TX
 *  microblocks
 *  Ring number 6, base 4096, and size 128 long words
 */
#define         QM_TO_PACKET_TX_SCR_RING_0      7
#define         QM_TO_PACKET_TX_SCR_RING_0_BASE 8192
#define         QM_TO_PACKET_TX_SCR_RING_0_SIZE 128

/*
 *  Second scractch ring for communicating between Queue Manager and Packet TX
 *  microblocks
 *  Ring number 7, base 5120, and size 128 long words
 */
#define         QM_TO_PACKET_TX_SCR_RING_1      8
#define         QM_TO_PACKET_TX_SCR_RING_1_BASE 9216
#define         QM_TO_PACKET_TX_SCR_RING_1_SIZE 128

/*
 *  Third scractch ring for communicating between Queue Manager and Packet TX
 *  microblocks
 *  Ring number 8, base 6144, and size 128 long words
 */
#define         QM_TO_PACKET_TX_SCR_RING_2      9
#define         QM_TO_PACKET_TX_SCR_RING_2_BASE 10240
#define         QM_TO_PACKET_TX_SCR_RING_2_SIZE 128

/*
 *  Fourth scractch ring for communicating between Queue Manager and Packet TX
 *  microblocks
 *  Ring number 9, base 7168, and size 128 long words
 */
#define         QM_TO_PACKET_TX_SCR_RING_3      10
#define         QM_TO_PACKET_TX_SCR_RING_3_BASE 11264
#define         QM_TO_PACKET_TX_SCR_RING_3_SIZE 128

/*
 * Scratch address required by Packet TX for updating available
 * Transmit Buffers.
 */
#define         PACKET_TX_AVAIL_TBUF_ELE_SCR_ADDR   14336/*ok*/


#endif      /* __QUAD_GBETH_MEMORY__H__ */




