////////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: tx_helper_util.uc
//
//      Purpose: Utility macros for Tx helper
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __TX_HELPER_UTIL_UC__
#define __TX_HELPER_UTIL_UC__

/////////////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_set_signal()
//
// Description:
// 	
//		Set a signal in a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1 instruction
// 
//
////////////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_set_signal(io_sigmask, in_sig)

.begin
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT in_sig
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, OR, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _tx_helper_set_signal()


/////////////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_set_push_signal()
//
// Description:
// 	
//		Add push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
//
////////////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_set_push_signal(io_sigmask, in_sig)

.begin

	.reg	tmp

#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<indirect]

#endif

.end

#endm // end of #macro _tx_helper_set_push_signal()


/////////////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_clear_signal()
//
// Description:
//		Clear a signal from a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be cleared with in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//
//		None
//
// Size: 
//
//		1 instructions
// 
/////////////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_clear_signal(io_sigmask, in_sig)

.begin

#if ( isnum(in_sig) )
	
	#define_eval SIG_SHIFT sig
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT ]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, AND~, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _tx_helper_clear_signal()


/////////////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_clear_push_signal()
//
// Description:
// 	
//		clear push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
///////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_clear_push_signal(io_sigmask, in_sig)

.begin

	.reg	tmp

#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<indirect]

#endif

.end

#endm // end of #macro _tx_helper_clear_push_signal()


///////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_read_tx_request()
//
// Description:
//		check and get tx request from NN ring 
//
// Outputs:
//		tx_request:			valid tx request  
//
// Inputs:  
//		None
//
// Constants
//		None
//
// Labels:
//		NO_TX_REQUEST_LABEL:	symbolic address to go when there is no tx 
//                              request
//
// Size: 
//		2 instructions
// 
///////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_read_tx_request(out_output_port, out_tx_request, NO_TX_REQUEST_LABEL)

.begin

	br_inp_state[NN_EMPTY, NO_TX_REQUEST_LABEL]	

	// read tx requestenqueue and dequeue messages (3)
	alu[out_output_port, --, B, *n$index++]	
	alu[out_tx_request, --, B, *n$index++]						

#ifdef _DEBUG_COUNTERS_
	alu[@tx_helper_requests_in, @tx_helper_requests_in, +, 1]
#endif

.end

#endm	// _tx_helper_read_tx_request()


///////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_passdown_tx_request()
//
// Description:
//		passs tx request to downstream microblocks either through NN ring 
//		(SPHY_1_32 mode) or scratch ring (SPI_4_16PORTS or SPI_4_10PORTS mode)
//
// Outputs:
//		None
//
// Inputs:  
//		in_tx_request:		tx request  
//		in_outport:			output_port for this tx request   
//
// Constants
//		None
//
// Size: 
//		2 instructions for SPHY_1_32 transmission
//		7 instructions for SPI_4_16PORTS or SPI_4_10PORTS transmission
// 
///////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_passdown_tx_request(in_tx_request, in_outport)

.begin

#if(TX_PHY_MODE == SPHY_1_32)

	wait_till_nn_not_full#:
	 	br_inp_state[nn_full, wait_till_nn_not_full#]
		alu[*n$index++, --, B, in_tx_request]

		#ifdef _DEBUG_COUNTERS_
			alu[@tx_helper_requests_out, @tx_helper_requests_out, +, 1]
		#endif

#endif	// #if(TX_PHY_MODE == SPHY_1_32)


#if((TX_PHY_MODE == SPI_4_16PORTS) || (TX_PHY_MODE == SPI_4_10PORTS))

	#ifndef MULTIPORT_TEST

		.reg	tmp

		alu[tmp, in_tx_request, AND~, 0x0F, <<24]
		alu[$tx_req, tmp, OR, in_outport, <<24]
		_tx_helper_send_tx_req_to_scratch_ring($tx_req, in_outport)	; (5)

	#else // MULTIPORT_TEST
		#ifdef DRR_SCHEDULER
			.reg	tmp

			alu[tmp, in_tx_request, AND~, 0x0F, <<24]
			alu[$tx_req, tmp, OR, in_outport, <<24]
			_tx_helper_send_tx_req_to_scratch_ring($tx_req, in_outport)	; (5)

		#else // round-robin scheduler

			#if (TX_PHY_MODE == SPI_4_16PORTS)
				.reg	port
				.reg	tmp

				alu[port, @mport, AND, 0x0F]
				alu[tmp, in_tx_request, AND~, 0x0F, <<24]
				alu[$tx_req, tmp, OR, port, <<24]
				_tx_helper_send_tx_req_to_scratch_ring($tx_req, port)	; (5)
				alu[@mport, @mport, +, 1]

			#else	// TX_PHY_MODE == SPI_4_10PORTS
				.reg	port
				.reg	tmp

				alu[port, --, B, @mport]
				alu[tmp, in_tx_request, AND~, 0x0F, <<24]
				alu[$tx_req, tmp, OR, port, <<24]
				_tx_helper_send_tx_req_to_scratch_ring($tx_req, port)	; (5)
				alu[@mport, @mport, +, 1]
				alu[--, @mport, -, 9]
				ble[done#]
				alu[@mport, --, B, 0]
			done#:	 

			#endif	// #if (TX_PHY_MODE == SPI_4_16PORTS)

		#endif // #ifdef DRR_SCHEDULER

	#endif //#ifndef MULTIPORT_TEST

#endif	// 	#if(TX_PHY_MODE == SPI_4_16PORTS or SPI_4_10PORTS)

.end

#endm	// _tx_helper_passdown_tx_request()


///////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_send_tx_req_to_scratch_ring()
//
// Description:
//		Send tx request to different scratch ring based on port number  
//
// Outputs: 
//		None 
//
// Inputs:  
//		in_tx_req:			tx request
//		in_port:			port number
//
// Constants
//		None
//
// Size: 
//
//		5 instruction
// 
///////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_send_tx_req_to_scratch_ring(in_tx_req, in_port)

.begin

	.reg	tmp

#if (TX_PHY_MODE == SPI_4_16PORTS) 
	alu[--, in_port, -, 8]
	bge[send_to_tx_ring_1#]
#else	// (TX_PHY_MODE == SPI_4_10PORTS))
	alu[--, in_port, -, 5]
	bge[send_to_tx_ring_1#]
#endif	// #if (TX_PHY_MODE == SPI_4_16PORTS) 

tx_ring_0_full#:	
	br_inp_state[RING_0_FULL, tx_ring_0_full#]

	scratch[put, in_tx_req, packet_tx_ring_0, 0, 1], \
											sig_done[sig_scratch_write]
	br[scartch_put_done#]	

send_to_tx_ring_1#:
tx_ring_1_full#:	
	br_inp_state[RING_1_FULL, tx_ring_1_full#]

	scratch[put, in_tx_req, packet_tx_ring_1, 0, 1], \
											sig_done[sig_scratch_write]	
scartch_put_done#:

	#ifdef _DEBUG_COUNTERS_
		alu[@tx_helper_requests_out, @tx_helper_requests_out, +, 1]
	#endif

.end

#endm	 // _tx_helper_send_tx_req_to_scratch_ring()


///////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_signal_next()
//
// Description: Wakeup next thread in the same ME
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_sig_gpr:		value to be wriiten to SAME_ME_SIGNAL csr to wake up
//						next thread 
//
// Constants:
//		none   
//
// Size: 
//		1 instruction
//
///////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_signal_next(in_sig_gpr)

.begin

	local_csr_wr[SAME_ME_SIGNAL, in_sig_gpr]
.end

#endm // end of #macro _tx_helper_signal_next()


#endif 		// __tx_helper_UTIL_UC__

///////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_calc_queue_struct_addr(out_queue_struct_addr, in_output_port, \
									in_class_id, in_all_structures_base)

.begin

	.reg	port_structures_base

	;get to the base of all class (virtual queue) structures of the particular port
	alu_shf[port_structures_base, in_all_structures_base, OR, 
			in_output_port, <<QUEUE_STRUCTURES_PER_PORT_SHIFT]
	
	;get to the offset of the structure of each class (virtual queue)
	alu_shf[out_queue_struct_addr, port_structures_base, or, in_class_id, \
			<<QUEUE_STRUCTURE_SHIFT]

.end

#endm


///////////////////////////////////////////////////////////////////////////////
#macro _tx_helper_read_deq_cntr(in_queue_struct_addr, rd_xfer_reg, rd_done_sig, \
			next_loc)

.begin
	.reg cam_result
	.reg cam_entry

	cam_lookup[cam_result, in_queue_struct_addr], lm_addr0[0]

	br_bset[cam_result, 7, _tx_helper_read_deq_cntr_cam_hit#]

_tx_helper_read_deq_cntr_cam_miss#:

	;dequeue counter is at offset (DEQ_COUNTER_INDEX*4) from the beginning
	;of the queue structure
	sram[read, rd_xfer_reg, in_queue_struct_addr, (DEQ_COUNTER_INDEX * 4), 1], \
			sig_done[rd_done_sig]

	;get cam entry
	alu_shf[cam_entry, 0xf, AND, cam_result, >>3]

	;no need to write back information because dequeue counter value
	;is always written to SRAM after each increment

	ctx_arb[--], defer[2]
	local_csr_wr[active_ctx_wakeup_events, sig_mask_2]
	cam_write[cam_entry, in_queue_struct_addr, STATUS_VALID]

	.io_completed sig_next_context sig_rd_deq_cntr_done
	
	;move data into local memory
	alu[*l$index0[/**/DEQ_COUNTER_INDEX], --, B, rd_xfer_reg]

	br[next_loc]

_tx_helper_read_deq_cntr_cam_hit#:

	;remove the signal for SRAM read
	_tx_helper_clear_signal[sig_mask_2, sig_rd_deq_cntr_done]

	.io_completed sig_next_context

	ctx_arb[--], defer[1], br[next_loc]
	local_csr_wr[active_ctx_wakeup_events, sig_mask_2]

.end

#endm

///////////////////////////////////////////////////////////////////////////////
#macro _tx_helper_incr_and_write_deq_cntr(in_queue_struct_addr, rd_xfer_reg, \
											wr_xfer_reg, wr_done_sig)

.begin
	;increment data in local memory
	alu[*l$index0[/**/DEQ_COUNTER_INDEX], *l$index0[/**/DEQ_COUNTER_INDEX], +, 1]

	;get data from local memory
	alu[wr_xfer_reg, --, B, *l$index0[/**/DEQ_COUNTER_INDEX]]

	;dequeue counter is in long word DEQ_COUNTER_INDEX from the beginning
	;of the class (virtual queue) structure
	sram[write, wr_xfer_reg, in_queue_struct_addr, (DEQ_COUNTER_INDEX * 4), 1], \
			sig_done[wr_done_sig]

.end

#endm
