////////////////////////////////////////////////////////////////////////////////
//                                                                      
//                   I N T E L   P R O P R I E T A R Y                   
//                                                                       
//      COPYRIGHT (c)  2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//      RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//      BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//      RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//      LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//      MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//      THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                       
//                         INTEL  CORPORATION                            
//                                                                      
//                      2200 MISSION COLLEGE BLVD                        
//                                                                       
//                SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                       
//
////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: tx_helper_init.uc
//
//      Purpose: Initialization code for Tx helper microblock
//
////////////////////////////////////////////////////////////////////////////////

#ifndef __TX_HELPER_INIT_UC__
#define __TX_HELPER_INIT_UC__

#include 	"scheduler_packet.h"
#include 	"stdmac.uc"

///////////////////////////////////////////////////////////////////////////////
//
// _tx_helper_thread_specific_init()
//
// Description: initialize context relative GPRs for specific thread to save 
//              instruction cycles in POS TX processing
//
// Outputs: 
//      out_next_ctx_sig1:	value used to write to SAME_ME_SIGNAL local csr 
//                          to wakeup next thread in phase 1
//
//      out_next_ctx_sig2:	value used to write to SAME_ME_SIGNAL local csr 
//                          to wakeup next thread in phase 2
//
//      out_next_ctx_sig3:	value used to write to SAME_ME_SIGNAL local csr 
//                          to wakeup next thread in phase 3
//
// Inputs:  
//		in_next_thread:		next thread number of this thread 
//
// Constants:
//		none
//
///////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_thread_specific_init(out_next_ctx_sig1, out_next_ctx_sig2, \
									   out_next_ctx_sig3, in_next_ctx)

.begin 
	
	alu_shf[out_next_ctx_sig1, in_next_ctx, or, &sig_next_context, <<3]
	alu_shf[out_next_ctx_sig2, in_next_ctx, or, &sig_next_context, <<3]
	alu_shf[out_next_ctx_sig3, in_next_ctx, or, &sig_next_context, <<3]

.end  // end for local variable next_context

#endm //end of macro _tx_helper_thread_specific_init()


///////////////////////////////////////////////////////////////////////////////
//
// 	_tx_helper_nn_ring_init()
//
// Description:
//		Initialize next neighbor ring, NN ring will be empty only when there
//      is no valid entry
//
// Outputs: 
//		None
//
// Inputs:  
//		None
//		
// Constants
//		None
//
///////////////////////////////////////////////////////////////////////////////

#macro 	_tx_helper_nn_ring_init()

.begin

	.reg	ctx_enables_reg

	.reg set_bits

	; Set bits [15:8] to FF to enable all 8 contexts 
	; Set bits [19:18] to 10 to specify that the message on NN-ring is 2 longwords
	local_csr_rd[ctx_enables]
	immed[ctx_enables_reg, 0]

	immed32[set_bits, 0x0004FF00]
	alu[ctx_enables_reg, ctx_enables_reg, OR, set_bits]
	local_csr_wr[ctx_enables, ctx_enables_reg]

	local_csr_wr[nn_get, 0]
	local_csr_wr[nn_put, 0]	

.end	

#endm	// _tx_helper_nn_ring_init() 


////////////////////////////////////////////////////////////////////////////////
// 
// tx_helper_init()
//
// Description:
// 	
//		Initialization of the microblock
//
// Outputs: 
//		none
//
// Inputs:  
//		non
//		
// Constants
//		none
//
////////////////////////////////////////////////////////////////////////////////

#macro tx_helper_init()

	.reg	 next_context
	
	.set next_context

	move(all_queue_structs_base, SCHED_QUEUE_STRUCTURES_BASE)
	
	.if (ctx() == 0)

		_tx_helper_thread_0_init()

	.else

		.if(ctx() == 1)
				immed[next_context, 2]
		.elif (ctx() == 2)
				immed[next_context, 3]
		.elif (ctx() == 3)
				immed[next_context, 4]		
		.elif (ctx() == 4)
				immed[next_context, 5]
		.elif (ctx() == 5)
				immed[next_context, 6]					
		.elif (ctx() == 6)
				immed[next_context, 7]
		.elif (ctx() == 7)
				immed[next_context, 0]
		.endif				
	
							
		_tx_helper_thread_specific_init(sig1_next_context_gpr, sig2_next_context_gpr,\
												sig3_next_context_gpr, next_context)
	.endif

#endm  // tx_helper_init[]

//////////////////////////////////////////////////////////////////////////////
// 
// tx_helper_thread_0_init()
//
// Description:
// 	
//		Initialization done by thread 0 only. For things that need to be
//		initialized only once per ME.
//
// Outputs: 
// Inputs:
////////////////////////////////////////////////////////////////////////////////

#macro _tx_helper_thread_0_init()
	
	.begin

		.reg sig

		#ifdef _DEBUG_COUNTERS_
			immed[@tx_helper_requests_in, 0]
			immed[@tx_helper_requests_out, 0]
		#endif
	
		_tx_helper_nn_ring_init()

		#if( (TX_PHY_MODE == SPI_4_16PORTS) || (TX_PHY_MODE == SPI_4_10PORTS) )

			#ifdef MULTIPORT_TEST
				alu[@mport, --, B, 0]
			#endif	// #ifdef MULTIPORT_TEST
		#endif	// #if(TX_PHY_MODE == SPI_4_16PORTS) or SPI_4_10PORTS)


		// For kick start thread 0 will send itself one signal for phase 1, 2, and 3


	    alu[sig, --, B,  &sig_next_context]
		alu_shf[sig, 0x0, OR, sig, <<3]         ; value to signal thread 0
		local_csr_wr[SAME_ME_SIGNAL, sig]   	; signal thread 0

		immed[next_context, 1]

		_tx_helper_thread_specific_init(sig1_next_context_gpr, sig2_next_context_gpr,\
												sig3_next_context_gpr, next_context)

		//virtual queue 0 is valid, but the base of queue structures
		//is in SRAM channel 2. Therefore, queue structure address never
		//equals 0. This mean there won't be a look up of 0x0, so 
		//just do a cam_clear, no need to initialize CAM with non-zero data
		cam_clear
		
		ctx_arb[system_init_sig]
	.end

#endm
///////////////////////////////////////////////////////////////////////////////
// 
// _tx_helper_init_sigmasks()
//
// Description:
// 	
//		Initialize the common globals in context relative gprs to save 
//      instruction cycles
//
// Outputs: 
//		out_default_sigmask1:	default sig_mask for phase 1	
//		out_default_sigmask2:	default sig_mask for phase 2
//      out_tx_ring_1:			scratch ring for first packet tx ME
//      out_tx_ring_2:			scratch ring for secnd packet tx ME
//
// Inputs:  
//		None
//		
// Constants
//      None
//
//
////////////////////////////////////////////////////////////////////////////////

#macro	_tx_helper_init_sigmasks(out_default_sigmask1, out_default_sigmask2, \
					out_default_sigmask3)

.begin

	// default phase 1 signal mask at start of each iteration of loop 
	alu[out_default_sigmask1, --, B, 0]			; initialize to 0
	; add sig_next_context
	_tx_helper_set_signal[out_default_sigmask1, sig_next_context]
	; add sig_sram_read for packet meta data read
	_tx_helper_set_signal[out_default_sigmask1, sig_sram_read]


	// default phase 2 signal mask at start of each iteration of loop 
	alu[out_default_sigmask2, --, B, 0]			; initialize to 0
	; add sig_next_context
	_tx_helper_set_signal[out_default_sigmask2, sig_next_context]	
	; add sig_rd_deq_cntr_done for dequeue counter read
	_tx_helper_set_signal[out_default_sigmask2, sig_rd_deq_cntr_done] 
	#if( (TX_PHY_MODE == SPI_4_16PORTS) || (TX_PHY_MODE == SPI_4_10PORTS) )
		_tx_helper_set_signal[out_default_sigmask2, sig_scratch_write]	
	#endif	// #if(TX_PHY_MODE == SPI_4_16PORTS or SPI_4_10PORTS)

	// default phase 3 signal mask at start of each iteration of loop 
	alu[out_default_sigmask3, --, B, 0]			; initialize to 0
	; add sig_next_context
	_tx_helper_set_signal[out_default_sigmask3, sig_next_context]	
	; add sig_wr_deq_cntr_done for dequeue counter write
	_tx_helper_set_signal[out_default_sigmask3, sig_wr_deq_cntr_done] 
	; add sig_scratch_write	to default signals in phase 3	

.end

#endm  // end of #macro _tx_helper_init_sigmasks()

////////////////////////////////////////////////////////////////////////////////

#endif 		//  __tx_helper_INIT_UC__

////////////////////////////////////////////////////////////////////////////////
