////////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ppp_encap_util.uc
//
//      Purpose: Utility macros for L2_ENCAP
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __PPP_ENCAP_UTIL_UC__
#define __PPP_ENCAP_UTIL_UC__

///////////////////////////////////////////////////////////////////////////////
// 
// _ppp_encap_init_gprs()
//
// Description:
// 	
//		Initialize the common globals in context relative gprs to save 
//      instruction cycles
//
// Outputs: 
//      out_src_ring:			scratch ring number used to get ppp_encap source
//      out_sink_ring:			scratch ring number used to send ppp_encap sink
//		out_default_sigmask1:	default sig_mask for phase 1	
//		out_default_sigmask2:	default sig_mask for phase 2
//		out_default_sigmask3:	default sig_mask for phase 3	
//		out_default_sigmask4:	default sig_mask for phase 4		
//		out_id_header_exist:	next hop id value identifies L2 header already exist
//      out_ppp_hader:			PPP header
//
// Inputs:  
//		None
//		
// Constants
//      None
//
//
////////////////////////////////////////////////////////////////////////////////
#macro	_ppp_encap_init_gprs(out_src_ring, out_sink_ring, out_default_sigmask1, \
			out_default_sigmask2, out_default_sigmask3, out_default_sigmask4, \
			out_id_header_exist, out_ppp_header)

.begin

 	alu_shf[out_src_ring, --, b, L2_ENCAP_SOURCE_SCR_RING, <<2]
	alu_shf[out_sink_ring, --, b, L2_ENCAP_SINK_SCR_RING, <<2]

	// default phase 1 signal mask at start of each iteration of loop 
	alu[out_default_sigmask1, --, B, 0]			; initialize to 0
	_ppp_encap_set_signal[out_default_sigmask1, sig2_next_context]	; add sig2_next_context
	_ppp_encap_set_signal[out_default_sigmask1, sig_sram_read]		; add sig_sram_read

	// default phase 2 signal mask at start of each iteration of loop 
	alu[out_default_sigmask2, --, B, 0]			; initialize to 0
	_ppp_encap_set_signal[out_default_sigmask2, sig3_next_context]	; add sig3_next_context
	_ppp_encap_set_signal[out_default_sigmask2, sig_dram_read]		; add sig_dram_read
	_ppp_encap_set_push_signal[out_default_sigmask2, sig_dram_read]	; add push signal

	// default phase 3 signal mask at start of each iteration of loop 
	alu[out_default_sigmask3, --, B, 0]			; initialize to 0
	_ppp_encap_set_signal[out_default_sigmask3, sig4_next_context]	; add sig4_next_context
	_ppp_encap_set_signal[out_default_sigmask3, sig_sram_write]		; add sig_sram_write
	_ppp_encap_set_signal[out_default_sigmask3, sig_dram_write]		; add sig_dram_write
	_ppp_encap_set_push_signal[out_default_sigmask3, sig_dram_write]	; add push signal

	// default phase 4 signal mask at start of each iteration of loop 
	alu[out_default_sigmask4, --, B, 0]			; initialize to 0
	_ppp_encap_set_signal[out_default_sigmask4, sig1_next_context]	; add sig1_next_context
	_ppp_encap_set_signal[out_default_sigmask4, sig_scratch_read]	; add sig_scratch_read 
	_ppp_encap_set_signal[out_default_sigmask4, sig_scratch_write]	; add sig_scratch_write 

	move(out_id_header_exist, _L2_HEADER_ALREADY_EXIST_ID)

// start: MPLS changes 

//	move(out_ppp_header, _PPP_HEADER)
	move(out_ppp_header, _PPP_HEADER_IP)

// end: MPLS changes 

.end

#endm  // end of #macro _ppp_encap_init_gprs()


/////////////////////////////////////////////////////////////////////////////////////
// _ppp_encap_set_signal()
//
// Description:
// 	
//		Set a signal in a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1 instruction
// 
//
////////////////////////////////////////////////////////////////////////////////////

#macro _ppp_encap_set_signal(io_sigmask, in_sig)

.begin
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT in_sig
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, OR, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _ppp_encap_set_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ppp_encap_set_push_signal()
//
// Description:
// 	
//		Add push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
//
////////////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_set_push_signal(io_sigmask, in_sig)
.begin
.reg tmp
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<indirect]

#endif
.end

#endm // end of #macro _ppp_encap_set_push_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ppp_encap_clear_signal()
//
// Description:
//		Clear a signal from a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be cleared with in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//
//		None
//
// Size: 
//
//		1 instructions
// 
//
/////////////////////////////////////////////////////////////////////////////////////

#macro _ppp_encap_clear_signal(io_sigmask, in_sig)

.begin

#if ( isnum(in_sig) )
	
	#define_eval SIG_SHIFT sig
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT ]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, AND~, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _ppp_encap_clear_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ppp_encap_clear_push_signal()
//
// Description:
// 	
//		clear push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
//
////////////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_clear_push_signal(io_sigmask, in_sig)
.begin
.reg tmp
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<indirect]

#endif
.end

#endm // end of #macro _ppp_encap_clear_push_signal()




///////////////////////////////////////////////////////////////////////////////
// _ppp_encap_source()
//
// Description: readfrom the source scratch ring
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_src_ring:		source scratch ring 
//		in_src_sig:			signal for scratch ring read
//
// Constants:
//		none   
//
// Size: 
//		1 instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_source(in_src_ring, in_src_sig)
.begin

	scratch[get, $source_lw0, in_src_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
		sig_done[in_src_sig]

.end
#endm

///////////////////////////////////////////////////////////////////////////////
// _ppp_encap_sink()
//
// Description: Write to the sink scratch ring
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_sink_ring:		sink scratch ring 
//		in_sink_sig:		signal for scratch ring write
//
// Constants:
//		none   
//
// Size: 
//		2 to ?? instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_sink(in_sink_ring, in_sink_sig)
.begin

check_ring_full#:
	br_inp_state[SCR_RING/**/L2_ENCAP_SINK_SCR_RING/**/_STATUS, check_ring_full#]
	scratch[put, $sink_lw0, in_sink_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
		sig_done[in_sink_sig]

//check_ring_full#:

//	; Check if ring is full.
 //	br_inp_state[L2_ENCAP_RING_OUT_0_FULL, check_ring_full#]

	; If not full, write to scratch ring.
//	scratch[put, $sink_lw0, in_sink_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
//	scratch[put, $source_lw0, in_sink_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
//		sig_done[in_sink_sig]

.end
#endm // end of #macro _ppp_encap_sink()

///////////////////////////////////////////////////////////////////////////////
// _ppp_encap_save_to_sink()
//
// Description: move source to sink before got new source
//
// Outputs:							
//		none
//
// Inputs:  
// 		none
//
// Constants:
//		none   
//
// Size: 
//		3 instructions
//
///////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_save_to_sink()
	// save $source to $sink before reading another source
	alu[$sink_lw0, --, B, $source_lw0] 
	alu[$sink_lw1, --, B, $source_lw1] 
	alu[$sink_lw2, --, B, $source_lw2] 
#endm // end of #macro _ppp_encap_save_to_sink()


///////////////////////////////////////////////////////////////////////////////
// _ppp_encap_signal_next()
//
// Description: Wakeup next thread in the same ME
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_sig_gpr:		value to be wriiten to SAME_ME_SIGNAL csr to wake up
//						next thread 
//
// Constants:
//		none   
//
// Size: 
//		1 instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_signal_next(in_sig_gpr)
.begin
	local_csr_wr[SAME_ME_SIGNAL, in_sig_gpr]
.end
#endm // end of #macro _ppp_encap_signal_next()


/////////////////////////////////////////////////////////////////////////////
// _ppp_encap_update_meta_data() 
//
// Description: update buffer size, offset, packet size fields in meta data
//
// Outputs:							
//		none
//
// Inputs:
//      in_meta_handle:		mta handle 
//      in_offset: 			the offset of the payload 
// 		in_bsize:			buffer size 
// 		in_psize:			packet size 
//		in_sig_sram:		signal number for sram write instruction
//
// Constants:
//							none
//
// Size:
//		8 instructions
//
///////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_update_meta_data(in_meta_handle, in_offset, in_bsize, \
						in_psize, in_sig_sram)
.begin 
.reg meta_addr tmpx

	ld_field_w_clr[tmpx, 0111, in_meta_handle]
	alu_shf[meta_addr, --, B, tmpx, <<2]
	alu_shf[tmpx, in_offset, OR, in_bsize, <<16]
	alu[$dl_meta[1], --, B, tmpx]
	ld_field_w_clr[tmpx, 0011, $dl_meta[2]]
	alu_shf[tmpx, tmpx, OR, in_psize, <<16]
	alu[$dl_meta[2], --, B, tmpx]
    sram[write, $dl_meta[1], meta_addr, 4, 2], sig_done[in_sig_sram]

.end 
#endm // end of _ppp_encap_update_meta_data()


/////////////////////////////////////////////////////////////////////////////
// _ppp_encap_write_ppp_header() 
//
// Description: realign and write PPP l2 header back to dram
//
// Outputs:							
//		none
//
// Inputs: 
//		io_buf_size:		buffer size to be updated
//		io_pkt_size:		packet size to be updated
//		in_dramaddr:		dram address of the payload
//      in_offset: 			the offset of the payload 
//		in_sig_dram:		signal number for dram write instruction
//
// Constants:
//							none
//
// Size:
//		19 instructions (worst case)
//
///////////////////////////////////////////////////////////////////////////////
#macro _ppp_encap_write_ppp_header(io_buf_size, io_pkt_size, in_dramaddr, \
				in_offset, in_sig_dram)
.begin
.reg offset_mod_8 new_offset tmp

	alu[offset_mod_8, in_offset, AND, _CONST_07]

	// jump to different port
	jump[offset_mod_8, jmp_tbl#], defer[3], targets[oset_0#, oset_1#, \
		oset_2#, oset_3#, oset_4#, oset_5#, oset_6#, oset_7#] 
		// reduce offset by two longwords to get new_offset, because two 
		// longwords (ppp header occupies lower 2 bytes) is actually added 
		// before payload in the case payload offset is multiple of 8
		alu[new_offset, in_offset, -, 8]
		alu[io_buf_size, io_buf_size, +, _PPP_L2_HEADER_SIZE]
		alu[io_pkt_size, io_pkt_size, +, _PPP_L2_HEADER_SIZE]


//	Start of jump table. 
jmp_tbl#:

oset_0#:
		br[offset_mod_8_eq_0#]
			
oset_1#:
		br[offset_mod_8_eq_1#]

oset_2#:
		br[offset_mod_8_eq_2#]

oset_3#:
		br[offset_mod_8_eq_3#]

oset_4#:
		br[offset_mod_8_eq_4#]

oset_5#:
		br[offset_mod_8_eq_5#]

oset_6#:
		br[offset_mod_8_eq_6#]

oset_7#:
		br[offset_mod_8_eq_7#]

offset_mod_8_eq_0#:					; [3]
		// don't care $$dram_lw0 write transfer register, it would not be 
		// transmitted out as part of packet, put 2 bytes PPP L2 header just 
		// before payload
		alu[$$dram_lw1, --, B, ppp_header]
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 1], sig_done[in_sig_dram]	 
		br[write_dram_done#]

offset_mod_8_eq_2#:			; [6]
		// put 2 bytes PPP L2 header just before payload
		ld_field_w_clr[tmp, 1100, ppp_header, <<16]
		ld_field[tmp, 0011, $$dram_lw0]
		alu[$$dram_lw0, --, B, tmp]
		alu[$$dram_lw1, --, B, $$dram_lw1]
		dram[write, $$dram_lw0, in_dramaddr, in_offset, 1], sig_done[in_sig_dram]	 
		br[write_dram_done#]

offset_mod_8_eq_3#:		; [6]
		// put 2 bytes PPP L2 header just before payload
		ld_field_w_clr[tmp, 1110, ppp_header, <<8]
		ld_field[tmp, 0001, $$dram_lw0]
		alu[$$dram_lw0, --, B, tmp]
		alu[$$dram_lw1, --, B, $$dram_lw1]
		dram[write, $$dram_lw0, in_dramaddr, in_offset, 1], sig_done[in_sig_dram]	 
		br[write_dram_done#]


offset_mod_8_eq_4#:	; [4]
		// put 2 bytes PPP L2 header just before payload
		alu[$$dram_lw0, --, B, ppp_header]
		alu[$$dram_lw1, --, B, $$dram_lw1]
		dram[write, $$dram_lw0, in_dramaddr, in_offset, 1], sig_done[in_sig_dram]		 
		br[write_dram_done#]

offset_mod_8_eq_5#:		; [7]
		// put 2 bytes PPP L2 header just before payload
		ld_field_w_clr[tmp, 0111, ppp_header, >>8]
		alu[$$dram_lw0, --, B, tmp]
		ld_field_w_clr[tmp, 1000, ppp_header, <<24]
		ld_field[tmp, 0111, $$dram_lw1]
		alu[$$dram_lw1, --, B, tmp]
		dram[write, $$dram_lw0, in_dramaddr, in_offset, 1], sig_done[in_sig_dram]		 
		br[write_dram_done#]

offset_mod_8_eq_6#:		; [5]
		// don't care $$dram_lw0 write transfer register, it would not be 
		// transmitted out as part of packet, put 2 bytes PPP L2 header just 
		// before payload
		ld_field_w_clr[tmp, 1100, ppp_header, <<16]
		ld_field[tmp, 0011, $$dram_lw1]
		alu[$$dram_lw1, --, B, tmp]
		dram[write, $$dram_lw0, in_dramaddr, in_offset, 1], sig_done[in_sig_dram]		 
		br[write_dram_done#]


offset_mod_8_eq_7#:		; [5]
		// don't care $$dram_lw0 write transfer register, it would not be 
		// transmitted out as part of packet, put 2 bytes PPP L2 header just 
		// before payload
		ld_field_w_clr[tmp, 1110, ppp_header, <<8]
		ld_field[tmp, 0001, $$dram_lw1]
		alu[$$dram_lw1, --, B, tmp]
		dram[write, $$dram_lw0, in_dramaddr, in_offset, 1], sig_done[in_sig_dram]		 
		br[write_dram_done#]

offset_mod_8_eq_1#: 	; [7]
		// don't care $$dram_lw0 write transfer register, it would not be 
		// transmitted out as part of packet, put 2 bytes PPP L2 header just 
		// before payload
		ld_field_w_clr[tmp, 0111, ppp_header, >>8]
		alu[$$dram_lw1, --, B, tmp]
		ld_field_w_clr[tmp, 1000, ppp_header, <<24]
		ld_field[tmp, 0111, $$dram_lw0]
		alu[$$dram_lw2, --, B, tmp]
		alu[$$dram_lw3, --, B, $$dram_lw1]
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 2], sig_done[in_sig_dram]	 

write_dram_done#:

.end
 #endm // end of macro _ppp_encap_write_ppp_header()

#endif 		// __PPP_ENCAP_UTIL_UC__

///////////////////////////////////////////////////////////////////////////////

