///////////////////////////////////////////////////////////////////////
//                                                                   
//                  I N T E L   P R O P R I E T A R Y                
//                                                                   
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS  
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY   
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A 
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER 
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL, 
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT 
//     THE PRIOR WRITTEN PERMISSION OF :                             
//                                                                   
//                        INTEL  CORPORATION                         
//                                                                   
//                     2200 MISSION COLLEGE BLVD                     
//                                                                   
//               SANTA  CLARA,  CALIFORNIA  95052-8119               
//                                                                   
///////////////////////////////////////////////////////////////////////
//                                                                   
//                                                                   
//      File Name: llc_encap_util.uc                                         
//                                                                   
//      Description: This file contains definitions and utility macros
//                   that are used by the main file.
//                                                                   
///////////////////////////////////////////////////////////////////////

#ifndef	_LLC_ENCAP_UTIL_UC_
#define	_LLC_ENCAP_UTIL_UC_

#include <xbuf.uc>
#include <stdmac.uc>
#include <dl_system.h>
#include <dispatch_loop.uc>


//------------------------------------------------------------------
// _llc_encap_init()
//
//    Description: Initializes gpr's used and waits for system init.
//
//    Parameters:
//        None.
//
//------------------------------------------------------------------
#macro _llc_encap_init()

	// LLC/SNAP header size in bytes.
	#define LLC_SNAP_SIZE				8

	.sig dram_rd_sig
	.sig dram_wr_sig

	.sig eop_meta_wr_sig

    ; Initialize inter-thread signals that we will use.
	; Set signal_next_thread to contain information to signal the 
	; next thread.
	.reg next_ctx
	move(next_ctx, 0x80)
    alu[next_thread, next_ctx, OR, &prev_thread_done_sig, <<3]

	; Initialize signal mask gpr's.

	immed[default_sigmask_a, 0]
	alu[default_sigmask_a, default_sigmask_a, OR, 1, <<(&prev_thread_done_sig)]

	alu[sigmask_a, --, B, default_sigmask_a]

	immed[default_sigmask_b, 0]
	alu[default_sigmask_b, default_sigmask_b, OR, 1, <<(&prev_thread_done_sig)]

	alu[sigmask_b, --, B, default_sigmask_b]

#endm


//------------------------------------------------------------------
// _llc_encap_get_first_qw_of_payload()
//
//    Description: Fetches the first quadword of payload from dram.
//
//    Parameters:
//        Outputs:
//        out_payload :
//            d-xfer-rd registers into which the payload is read.
//        Inputs:
//		  in_payload_addr :
//			  Absolute dram address of start of buffer.
//        in_payload_offset:
//            Offset of start of payload within buffer.
//		  in_dram_rd_sig:
//			  Signal generated on I/O completion.
//
//------------------------------------------------------------------
#macro _llc_encap_get_first_qw_of_payload(out_payload, \
			in_payload_addr, in_payload_offset, in_dram_rd_sig)
.begin
		dram[read, out_payload[0], in_payload_addr, \
			in_payload_offset, 1], sig_done[in_dram_rd_sig]
.end
#endm


//------------------------------------------------------------------
// _llc_encap_read_next_hop_entry()
//
//    Description: Reads the next hop entry of the packet.
//
//    Parameters:
//        Outputs:
//        out_next_hop_entry :
//            s-xfer-rd registers into which the next hop entry
//            is read.
//        Inputs:
//		  in_next_hop_id :
//			  Next hop id of the packet.
//		  in_next_hop_entry_rd_sig:
//			  Signal generated on I/O completion.
//
//------------------------------------------------------------------
#macro _llc_encap_read_next_hop_entry(out_next_hop_entry, \
			in_next_hop_id, in_next_hop_entry_rd_sig)
.begin

	; Convert next_hop_id to the SRAM offset containing the 
	; next hop entry. Each next hop entry is NEXT_HOP_ENTRY_SIZE bytes.
	alu[in_next_hop_id, --, B, in_next_hop_id, <<PO2_NEXT_HOP_ENTRY_SIZE]

	sram[read, out_next_hop_entry[0], NEXT_HOP_TABLE_BASE_SRAM_GPR, \
		in_next_hop_id, 4], sig_done[in_next_hop_entry_rd_sig]

.end
#endm


//------------------------------------------------------------------
// _llc_encap_compute_packet_cell_count()
//
//    Description: Computes number of 48-byte cells in the packet.
//
//    Parameters:
//        Outputs:
//        out_packet_cell_count :
//            Number of 48-byte cells in the packet.
//        Inputs:
//        in_packet_size:
//            Packet size in bytes.
//        IN_ARG :
//            When set to 'ADDN_CELL' the macro computes cell count
//            taking into consideration the need for an additional
//            cell for the packet if the payload of the last cell
//            of the packet is >40. The additional cell is needed
//            in this case to store the 8-byte CPCS trailer.
//
//     Example :
//                packet size  IN_ARG      packet cell count
//                (bytes)
//                 40           --          1
//                 47           --          1
//                 48           --          1
//                 49           --          2
//                 40           ADDN_CELL   1
//                 47           ADDN_CELL   2
//                 48           ADDN_CELL   2
//                 49           ADDN_CELL   2
//
//------------------------------------------------------------------
#macro _llc_encap_compute_packet_cell_count(out_packet_cell_count, \
			in_packet_size, IN_ARG)
.begin

	//    Shown below is the C implementation of the algorithm.
	//
	//    int compute_packet_cell_count(int in_packet_size)
	//    {
	//        int p, c;
	//
	//        p = in_packet_size + (IN_ARG == 'ADDN_CELL')?56:48;
	//
	//        c = ((((p << 2) + p) << 2) + p);
	//
	//        c = (c + (c << 6));
	//
	//        out_packet_cell_count = \
	//            ((c >> 16) + ((((c & 0xffff) << 12) + c) >> 28));
	//
	//        return out_packet_cell_count;
	//    }

	.reg p, c

	#if (streq('IN_ARG', 'ADDN_CELL'))
	alu[p, in_packet_size, +, 56]
	#else
	alu[p, in_packet_size, +, 48]
	#endif

	.reg tmp

	alu[tmp, --, B, p, <<2]
	alu[tmp, tmp, +, p]
	alu[tmp, --, B, tmp, <<2]
	alu[c, tmp, +, p]

	alu[tmp, --, B, c, <<6]
	alu[c, c, +, tmp]

	alu[tmp, c, AND, OxFFFF]
	alu[tmp, --, B, tmp, <<12]
	alu[tmp, tmp, +, c]
	alu[tmp, --, B, tmp, >>28]
	alu[c, --, B, c, >>16]

	alu[out_packet_cell_count, c, +, tmp]

.end
#endm


//------------------------------------------------------------------
// _llc_encap_add_llc_to_payload()
//
//    Description: Reads the next hop entry of the packet.
//
//    Parameters:
//        I/O:
//        io_payload :
//            Input: d-xfer-rd registers contain the first qw of
//                   payload.
//            Output: d-xfer-wr registers are assigned the llc/snap
//                    header aligned with the first qw of payload.
//            s-xfer-rd registers into which the next hop entry
//            is read.
//        Inputs:
//		  in_payload_addr :
//			  Absolute dram address of start of buffer.
//        in_payload_offset :
//            Offset of start of payload within buffer.
//        in_payload_offset_0_7 :
//            Non 8-byte aligned payload offset.
//		  in_dram_wr_sig:
//			  Signal generated on I/O completion.
//
//------------------------------------------------------------------
#macro _llc_encap_add_llc_to_payload(io_payload, \
			in_payload_addr, in_payload_offset, in_payload_offset_0_7, \
			in_next_hop_entry, in_dram_wr_sig)
.begin

	jump[in_payload_offset_0_7, jmp_base#],
		targets[br_jmp_0#, br_jmp_1#, br_jmp_2#,  br_jmp_3#, \
			    br_jmp_4#, br_jmp_5#, br_jmp_6#,  br_jmp_7#]

jmp_base#:

#define_eval i 0		
#while (i <= 7)
br_jmp_/**/i/**/#:		
	br[jmp_/**/i/**/#]	
	#define_eval i (i+1)	
#endloop
#undef i

jmp_0#:
.begin

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Copy the llc/snap header to start of payload.
	alu[io_payload[0], --, B, in_next_hop_entry[2]]
	alu[io_payload[1], --, B, in_next_hop_entry[3]]

	; Write the llc/snap header to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 1], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_1#:
.begin

	.reg payload0 payload1 payload2 payload3

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Align the llc/snap header to start of payload.
	ld_field_w_clr[payload0, 0111, in_next_hop_entry[2], >>8]
	ld_field_w_clr[payload1, 1000, in_next_hop_entry[2], <<24]
	ld_field[payload1, 0111, in_next_hop_entry[3], >>8]
	ld_field_w_clr[payload2, 1000, in_next_hop_entry[3], <<24]
	ld_field[payload2, 0111, io_payload[0]]
	ld_field_w_clr[payload3, 1111, io_payload[1]]

	; Copy the combined payload into d-xfer-wr registers.
	alu[io_payload[0], --, B, payload0]
	alu[io_payload[1], --, B, payload1]
	alu[io_payload[2], --, B, payload2]
	alu[io_payload[3], --, B, payload3]

	; Write the combined payload to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 2], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_2#:
.begin

	.reg payload0 payload1 payload2 payload3

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Align the llc/snap header to start of payload.
	ld_field_w_clr[payload0, 0011, in_next_hop_entry[2], >>16]
	ld_field_w_clr[payload1, 1100, in_next_hop_entry[2], <<16]
	ld_field[payload1, 0011, in_next_hop_entry[3], >>16]
	ld_field_w_clr[payload2, 1100, in_next_hop_entry[3], <<16]
	ld_field[payload2, 0011, io_payload[0]]
	ld_field_w_clr[payload3, 1111, io_payload[1]]

	; Copy the combined payload into d-xfer-wr registers.
	alu[io_payload[0], --, B, payload0]
	alu[io_payload[1], --, B, payload1]
	alu[io_payload[2], --, B, payload2]
	alu[io_payload[3], --, B, payload3]

	; Write the combined payload to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 2], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_3#:
.begin

	.reg payload0 payload1 payload2 payload3

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Align the llc/snap header to start of payload.
	ld_field_w_clr[payload0, 0001, in_next_hop_entry[2], >>24]
	ld_field_w_clr[payload1, 1110, in_next_hop_entry[2], <<8]
	ld_field[payload1, 0001, in_next_hop_entry[3], >>24]
	ld_field_w_clr[payload2, 1110, in_next_hop_entry[3], <<8]
	ld_field[payload2, 0001, io_payload[0]]
	ld_field_w_clr[payload3, 1111, io_payload[1]]

	; Copy the combined payload into d-xfer-wr registers.
	alu[io_payload[0], --, B, payload0]
	alu[io_payload[1], --, B, payload1]
	alu[io_payload[2], --, B, payload2]
	alu[io_payload[3], --, B, payload3]

	; Write the combined payload to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 2], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_4#:
.begin

	.reg payload1 payload2 payload3

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Align the llc/snap header to start of payload.
	ld_field_w_clr[payload1, 1111, in_next_hop_entry[2]]
	ld_field_w_clr[payload2, 1111, in_next_hop_entry[3]]
	ld_field_w_clr[payload3, 1111, io_payload[1]]

	; Copy the combined payload into d-xfer-wr registers.
	.set io_payload[0]
	alu[io_payload[1], --, B, payload1]
	alu[io_payload[2], --, B, payload2]
	alu[io_payload[3], --, B, payload3]

	; Write the combined payload to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 2], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_5#:
.begin

	.reg payload1 payload2 payload3

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Align the llc/snap header to start of payload.
	ld_field_w_clr[payload1, 0111, in_next_hop_entry[2], >>8]
	ld_field_w_clr[payload2, 1000, in_next_hop_entry[2], <<24]
	ld_field[payload2, 0111, in_next_hop_entry[3], >>8]
	ld_field_w_clr[payload3, 1000, in_next_hop_entry[3], <<24]
	ld_field[payload3, 0111, io_payload[1]]

	; Copy the combined payload into d-xfer-wr registers.
	.set io_payload[0]
	alu[io_payload[1], --, B, payload1]
	alu[io_payload[2], --, B, payload2]
	alu[io_payload[3], --, B, payload3]

	; Write the combined payload to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 2], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_6#:
.begin

	.reg payload1 payload2 payload3

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Align the llc/snap header to start of payload.
	ld_field_w_clr[payload1, 0011, in_next_hop_entry[2], >>16]
	ld_field_w_clr[payload2, 1100, in_next_hop_entry[2], <<16]
	ld_field[payload2, 0011, in_next_hop_entry[3], >>16]
	ld_field_w_clr[payload3, 1100, in_next_hop_entry[3], <<16]
	ld_field[payload3, 0011, io_payload[1]]

	; Copy the combined payload into d-xfer-wr registers.
	.set io_payload[0]
	alu[io_payload[1], --, B, payload1]
	alu[io_payload[2], --, B, payload2]
	alu[io_payload[3], --, B, payload3]

	; Write the combined payload to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 2], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_7#:
.begin

	.reg payload1 payload2 payload3

	// in_next_hop_entry[2] contains LLC_SNAP_0
	// in_next_hop_entry[3] contains LLC_SNAP_1

	; Align the llc/snap header to start of payload.
	ld_field_w_clr[payload1, 0001, in_next_hop_entry[2], >>24]
	ld_field_w_clr[payload2, 1110, in_next_hop_entry[2], <<8]
	ld_field[payload2, 0001, in_next_hop_entry[3], >>24]
	ld_field_w_clr[payload3, 1110, in_next_hop_entry[3], <<8]
	ld_field[payload3, 0001, io_payload[1]]

	; Copy the combined payload into d-xfer-wr registers.
	.set io_payload[0]
	alu[io_payload[1], --, B, payload1]
	alu[io_payload[2], --, B, payload2]
	alu[io_payload[3], --, B, payload3]

	; Write the combined payload to dram.
	dram[write, io_payload[0], in_payload_addr, in_payload_offset, 2], \
		sig_done[in_dram_wr_sig]

	br[jmp_end#]

.end

jmp_end#:

.end
#endm


//------------------------------------------------------------------
// _llc_encap_update_vcq()
//
//    Description: Update vcq number for the packet based on
//    next hop entry information.
//
//    Parameters:
//        Outputs :
//        out_src_sink_data2 :
//            Scratch ring cache that holds the vcq number.
//        Inputs:
//		  in_next_hop_entry1 :
//			  Next hop entry that holds the new vcq number.
//
//------------------------------------------------------------------
#macro _llc_encap_update_vcq(out_src_sink_data2, in_next_hop_entry1)
.begin

	; Update vcq number using the next hop entry information, if 
	; the information is valid.
	.if(in_next_hop_entry1 != 0xFFFFFFFF)

		.reg qnum

		alu[qnum, 0, +16, in_next_hop_entry1]

		alu[out_src_sink_data2, out_src_sink_data2, AND, OxFFFF, <<16]
		alu[out_src_sink_data2, out_src_sink_data2, OR, qnum]

	.endif

.end
#endm


//------------------------------------------------------------------
// _llc_encap_store_cell_count()
//
//    Description: Update vcq number for the packet based on
//    next hop entry information.
//
//    Parameters:
//        Outputs :
//        out_src_sink_data0 :
//            Scratch ring cache that holds the sop buffer cell count.
//        out_src_sink_data2 :
//            Scratch ring cache that holds the packet cell count.
//        Inputs:
//		  in_original_packet_cell_count :
//            Cell count of the packet when it entered this microblock.
//            This cell count does not include the bytes added by this
//            microblock to the packet. It also does not take into
//            account the additional cell that may be needed for this
//            packet if the last cell is > 40 bytes.
//        in_correct_packet_cell_count :
//            Cell count of the packet when it leaves this microblock.
//            This cell count includes the bytes added by this
//            microblock to the packet. It also takes into account the
//            additional cell that may be needed for this  packet if
//            the last cell is > 40 bytes.
//
//------------------------------------------------------------------
#macro _llc_encap_store_cell_count(out_src_sink_data0, out_src_sink_data2, \
			in_original_packet_cell_count, in_correct_packet_cell_count)
.begin

	.reg diff_in_packet_cell_count

	; Store packet cell count on scratch ring.
	alu[out_src_sink_data2, out_src_sink_data2, AND, OxE003FFFF]

	alu[out_src_sink_data2, out_src_sink_data2, OR, \
		in_correct_packet_cell_count, <<18]

	; The difference between in_correct_packet_cell_count and 
	; in_original_packet_cell_count is the number of cells that need
	; to be added to the sop buffer cell count so that the sum of cell
	; counts in all buffers of this packet equals the packet cell count.
	alu[diff_in_packet_cell_count, \
		in_correct_packet_cell_count, -, in_original_packet_cell_count]

	.if (diff_in_packet_cell_count > 0)

		; Update the sop buffer cell count.
		alu[diff_in_packet_cell_count, --, B, diff_in_packet_cell_count, <<24]
		alu[out_src_sink_data0, out_src_sink_data0, +, diff_in_packet_cell_count]

	.endif

.end
#endm


#macro _llc_encap()
.begin

		; If source scratch ring is empty (dl_next_block == IX_NULL)
		; or if queue belongs to a drop queue (dl_next_block == BID_QM),
		; execute null path, else execute the llc encap path.
		alu[--, dl_next_block, -, BID_LLC_ENCAP]
		bne[null_path#]

		;-------------------------------------------------------
		; Step 1 - Read meta-data for the packet.
		;-------------------------------------------------------

		; Get packet size.
		.reg packet_size
		dl_meta_get_packet_size(packet_size)

		; Compute original packet cell count.
		.reg original_packet_cell_count
		_llc_encap_compute_packet_cell_count(
			original_packet_cell_count, packet_size, --)

		.reg next_hop_id

		dl_meta_get_nexthop_id(next_hop_id)

		; Allocate 4 lw's to read the next hop entry for the packet.
		xbuf_alloc($next_hop_entry, 4, read)

		.sig next_hop_entry_rd_sig

		_llc_encap_read_next_hop_entry($next_hop_entry,
			next_hop_id, next_hop_entry_rd_sig)

		ctx_arb[prev_thread_done_sig, next_hop_entry_rd_sig]

		signal_ctx(next_thread)

		// If LLC/SNAP needs to added to payload.
		#ifdef ADD_L2_ENCAP

		.if ($next_hop_entry[2] != 0xFFFFFFFF)

		;-------------------------------------------------------
		; Step 2 - Read first qw of payload if necessary.
		;-------------------------------------------------------

		; Compute absolute payload address in dram using sop
		; buffer handle.
		.reg payload_addr
		dl_buf_get_data(payload_addr, dl_buf_handle)

		.set $$payload[0] $$payload[1]

		.reg payload_offset \
			 payload_offset_0_7

		dl_meta_get_offset(payload_offset)

		; Compute non 8-byte aligned payload offset.
		alu[payload_offset_0_7, payload_offset, AND, 0x7]

		; Read first qw of payload if it starts at a non 8-byte
		; aligned address.
		beq[end_payload_rd#]

		alu[sigmask_a, default_sigmask_a, OR, 3, <<(&dram_rd_sig)]

		_llc_encap_get_first_qw_of_payload($$payload,
			payload_addr, payload_offset, dram_rd_sig)

end_payload_rd#:

		;-------------------------------------------------------
		; Step 3 - Update sop buffer meta-data.
		;-------------------------------------------------------

		; Update sop buffer size.
		.reg buf_size
		dl_meta_get_buffer_size(buf_size)
		alu[buf_size, buf_size, +, LLC_SNAP_SIZE]
		dl_meta_set_buffer_size(buf_size)

		; Update packet size.
		alu[packet_size, packet_size, +, LLC_SNAP_SIZE]
		dl_meta_set_packet_size(packet_size)

		; Update payload offset.
		alu[payload_offset, payload_offset, -, LLC_SNAP_SIZE]
		dl_meta_set_offset(payload_offset)

		;-------------------------------------------------------
		; Step 4 - Add LLC/SNAP header to start of payload.
		;-------------------------------------------------------

		; Wait on prev_thread_done_sig and optionally dram_rd_sig.
		ctx_arb[--], defer[2]

		; ctx_arb on sigmask_a.
		local_csr_wr[active_ctx_wakeup_events, sigmask_a]

		; Reset sigmask_a to its default value.
		alu[sigmask_a, --, B, default_sigmask_a]

		.io_completed prev_thread_done_sig
		.io_completed dram_rd_sig

		signal_ctx(next_thread)

		; Add LLC/SNAP header to start of payload.
		alu[sigmask_b, sigmask_b, OR, 3, <<(&dram_wr_sig)]

		_llc_encap_add_llc_to_payload($$payload, payload_addr,
			payload_offset, payload_offset_0_7, $next_hop_entry,
			dram_wr_sig)


		.else // (i.e. if $next_hop_entry[2] == 0xFFFFFFFF)

		ctx_arb[prev_thread_done_sig]
		signal_ctx(next_thread)

		.endif // ($next_hop_entry[2] != 0xFFFFFFFF)

		#endif // ADD_L2_ENCAP

		;-------------------------------------------------------
		; Step 5 - Compute vcq#, packet cell count and update
		;          sop buffer cell count if necessary.
		;-------------------------------------------------------

		.reg no_addn_cell_packet_cell_count

		; Compute packet cell count without taking ADDN_CELL into account.
		_llc_encap_compute_packet_cell_count(
			no_addn_cell_packet_cell_count, packet_size, --)

		.reg correct_packet_cell_count

		; Compute correct packet cell count.
		_llc_encap_compute_packet_cell_count(
			correct_packet_cell_count, packet_size, ADDN_CELL)

		; If (correct_packet_cell_count - no_addn_cell_packet_cell_count) == 1
		; increment eop buffer size by 1. This implies an additional cell needs
		; to be scheduled.
		; We increment eop buffer size so that the AAL5 transmit block does not
		; free this buffer until the additional cell dequeue request comes in.
		alu[--, correct_packet_cell_count, -, no_addn_cell_packet_cell_count]
		beq[compute_vcq#]

		alu[--, src_sink_data[1], -, IX_NULL]
		beq[sep#]

		; Stores eop buffer handle address in bytes.
		.reg eop_buf_hdl
		ld_field_w_clr[eop_buf_hdl, 0111, src_sink_data[1]]
		alu[eop_buf_hdl, --, B, eop_buf_hdl, <<2]

		alu[sigmask_b, sigmask_b, OR, 1, <<(&eop_meta_wr_sig)]

		alu[$eop_buf_size_incr, --, B, 1, <<16]
		sram[add, $eop_buf_size_incr, eop_buf_hdl, 4], sig_done[eop_meta_wr_sig]

		br[compute_vcq#]

sep#:
		; Update sop buffer size.
		.reg sep_buf_size
		dl_meta_get_buffer_size(sep_buf_size)
		alu[sep_buf_size, sep_buf_size, +, 1]
		dl_meta_set_buffer_size(sep_buf_size)

compute_vcq#:

		; Compute vcq and store it on the sink scratch ring.
		_llc_encap_update_vcq(src_sink_data[2], $next_hop_entry[1])

		; Store packet cell count on the sink scratch ring.
		; Also if needed, update sop buffer cell count on the sink
		; scratch ring.
		_llc_encap_store_cell_count(src_sink_data[0], src_sink_data[2],
				original_packet_cell_count, correct_packet_cell_count)

		; Set dl_next_block
		alu[dl_next_block, --, B, BID_LLC_ENCAP_NEXT1]

		br[llc_encap_end#]

		xbuf_free($next_hop_entry)

null_path#:

	; Null dequeue and drop queue path.
	_llc_encap_null_path()

llc_encap_end#:

.end
#endm


//------------------------------------------------------------------
// _llc_encap_null_path()
//
//    Description: Path taken by a thread that receives no data on
//    the source scratch ring.
//
//    Parameters: None.
//
//------------------------------------------------------------------
#macro _llc_encap_null_path()
.begin

	;-------------------------------------------------------
	; Step 1 - Read meta-data for the packet
	;-------------------------------------------------------

	; Delay here to simulate actual code. We do this to 
	; avoid flooding scratch[read] requests.
	#repeat 4
	alu[--, --, b, 0]
	#endloop

	ctx_arb[prev_thread_done_sig]

	signal_ctx(next_thread)

	;-------------------------------------------------------
	; Step 2 - Read first qw of payload if necessary.
	;-------------------------------------------------------

	; Delay here to simulate actual code. We do this to 
	; avoid flooding scratch[read] requests.
	#repeat 2
	alu[--, --, b, 0]
	#endloop

	;-------------------------------------------------------
	; Step 3 - Update sop buffer meta-data.
	;-------------------------------------------------------

	; Delay here to simulate actual code. We do this to 
	; avoid flooding scratch[read] requests.
	#repeat 5
	alu[--, --, b, 0]
	#endloop

	;-------------------------------------------------------
	; Step 4 - Add LLC/SNAP header to start of payload.
	;-------------------------------------------------------

	#ifdef ADD_L2_ENCAP
	ctx_arb[prev_thread_done_sig]
	signal_ctx(next_thread)
	#endif

	; Delay here to simulate actual code. We do this to 
	; avoid flooding scratch[read] requests.
	#repeat 4
	alu[--, --, b, 0]
	#endloop

	;-------------------------------------------------------
	; Step 5 - Compute vcq#, packet cell count and update
	;          sop buffer cell count if necessary.
	;-------------------------------------------------------

	; Delay here to simulate actual code. We do this to 
	; avoid flooding scratch[read] requests.
	#repeat 3
	alu[--, --, b, 0]
	#endloop

.end
#endm

#endif // _LLC_ENCAP_UTIL_UC_