////////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ethernet_encap_util.uc
//
//      Purpose: Utility macros for L2_ENCAP
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ETHERNET_ENCAP_UTIL_UC__
#define __ETHERNET_ENCAP_UTIL_UC__

///////////////////////////////////////////////////////////////////////////////
// 
// _ethernet_encap_init_gprs()
//
// Description:
// 	
//		Initialize the common globals in context relative gprs to save 
//      instruction cycles
//
// Outputs: 
//      out_src_ring:			scratch ring number used to get ethernet_encap source
//      out_sink_ring:			scratch ring number used to send ethernet_encap sink
//		out_default_sigmask1:	default sig_mask for phase 1	
//		out_default_sigmask2:	default sig_mask for phase 2
//		out_default_sigmask3:	default sig_mask for phase 3	
//		out_default_sigmask4:	default sig_mask for phase 4		
//		out_id_header_exist:	next hop id value identifies L2 header already exist
//      out_l2_table_base:      L2 table base
// Inputs:  
//		None
//		
// Constants
//      None
//
//
////////////////////////////////////////////////////////////////////////////////
#macro	_ethernet_encap_init_gprs(out_src_ring, out_sink_ring, out_default_sigmask1, \
			out_default_sigmask2, out_default_sigmask3, out_default_sigmask4, \
			out_id_header_exist, out_l2_table_base)

.begin

 	alu_shf[out_src_ring, --, b, L2_ENCAP_SOURCE_SCR_RING, <<2]
	alu_shf[out_sink_ring, --, b, L2_ENCAP_SINK_SCR_RING, <<2]

	// default phase 1 signal mask at start of each iteration of loop 
	alu[out_default_sigmask1, --, B, 0]			; initialize to 0
	_ethernet_encap_set_signal[out_default_sigmask1, sig2_next_context]	; add sig2_next_context
	_ethernet_encap_set_signal[out_default_sigmask1, sig_sram_read]		; add sig_sram_read

	// default phase 2 signal mask at start of each iteration of loop 
	alu[out_default_sigmask2, --, B, 0]			; initialize to 0
	_ethernet_encap_set_signal[out_default_sigmask2, sig3_next_context]	; add sig3_next_context
	_ethernet_encap_set_signal[out_default_sigmask2, sig_sram_read]		; add sig_sram_read
	_ethernet_encap_set_signal[out_default_sigmask2, sig_dram_read]		; add sig_dram_read
	_ethernet_encap_set_push_signal[out_default_sigmask2, sig_dram_read]	; add push signal

	// default phase 3 signal mask at start of each iteration of loop 
	alu[out_default_sigmask3, --, B, 0]			; initialize to 0
	_ethernet_encap_set_signal[out_default_sigmask3, sig4_next_context]	; add sig4_next_context
	_ethernet_encap_set_signal[out_default_sigmask3, sig_sram_write]		; add sig_sram_write
	_ethernet_encap_set_signal[out_default_sigmask3, sig_dram_write]		; add sig_dram_write
	_ethernet_encap_set_push_signal[out_default_sigmask3, sig_dram_write]	; add push signal

	// default phase 4 signal mask at start of each iteration of loop 
	alu[out_default_sigmask4, --, B, 0]			; initialize to 0
	_ethernet_encap_set_signal[out_default_sigmask4, sig1_next_context]	; add sig1_next_context
	_ethernet_encap_set_signal[out_default_sigmask4, sig_scratch_read]	; add sig_scratch_read 
	_ethernet_encap_set_signal[out_default_sigmask4, sig_scratch_write]	; add sig_scratch_write 

	move(out_id_header_exist, _L2_HEADER_ALREADY_EXIST_ID)
//	move(out_l2_table_base, _L2_TABLE_BASE)
	move(out_l2_table_base, L2_TABLE_SRAM_BASE)

.end

#endm  // end of #macro _ethernet_encap_init_gprs()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_set_signal()
//
// Description:
// 	
//		Set a signal in a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1 instruction
// 
//
////////////////////////////////////////////////////////////////////////////////////

#macro _ethernet_encap_set_signal(io_sigmask, in_sig)

.begin
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT in_sig
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, OR, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _ethernet_encap_set_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_set_push_signal()
//
// Description:
// 	
//		Add push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
//
////////////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_set_push_signal(io_sigmask, in_sig)
.begin
.reg tmp
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<indirect]

#endif
.end

#endm // end of #macro _ethernet_encap_set_push_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_clear_signal()
//
// Description:
//		Clear a signal from a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be cleared with in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//
//		None
//
// Size: 
//
//		1 instructions
// 
//
/////////////////////////////////////////////////////////////////////////////////////

#macro _ethernet_encap_clear_signal(io_sigmask, in_sig)

.begin

#if ( isnum(in_sig) )
	
	#define_eval SIG_SHIFT sig
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT ]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, AND~, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _ethernet_encap_clear_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_clear_push_signal()
//
// Description:
// 	
//		clear push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
//
////////////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_clear_push_signal(io_sigmask, in_sig)
.begin
.reg tmp
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<indirect]

#endif
.end

#endm // end of #macro _ethernet_encap_clear_push_signal()




///////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_source()
//
// Description: readfrom the source scratch ring
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_src_ring:		source scratch ring 
//		in_src_sig:			signal for scratch ring read
//
// Constants:
//		none   
//
// Size: 
//		1 instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_source(in_src_ring, in_src_sig)
.begin

	scratch[get, $source_lw0, in_src_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
		sig_done[in_src_sig]

.end
#endm

///////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_sink()
//
// Description: Write to the sink scratch ring
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_sink_ring:		sink scratch ring 
//		in_sink_sig:		signal for scratch ring write
//
// Constants:
//		none   
//
// Size: 
//		2 to ?? instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_sink(in_sink_ring, in_sink_sig)
.begin

check_ring_full#:
	br_inp_state[SCR_RING/**/L2_ENCAP_SINK_SCR_RING/**/_FULL, check_ring_full#]
	scratch[put, $sink_lw0, in_sink_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
		sig_done[in_sink_sig]

//check_ring_full#:

//	; Check if ring is full.
 //	br_inp_state[L2_ENCAP_RING_OUT_0_FULL, check_ring_full#]

	; If not full, write to scratch ring.
//	scratch[put, $sink_lw0, in_sink_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
//	scratch[put, $source_lw0, in_sink_ring, 0, _NUM_WORDS_L2_ENCAP_MSG], \
//		sig_done[in_sink_sig]

.end
#endm // end of #macro _ethernet_encap_sink()

///////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_save_to_sink()
//
// Description: move source to sink before got new source
//
// Outputs:							
//		none
//
// Inputs:  
// 		none
//
// Constants:
//		none   
//
// Size: 
//		3 instructions
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_save_to_sink()
	// save $source to $sink before reading another source
	alu[$sink_lw0, --, B, $source_lw0] 
	alu[$sink_lw1, --, B, $source_lw1] 
	alu[$sink_lw2, --, B, $source_lw2] 
#endm // end of #macro _ethernet_encap_save_to_sink()


///////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_signal_next()
//
// Description: Wakeup next thread in the same ME
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_sig_gpr:		value to be wriiten to SAME_ME_SIGNAL csr to wake up
//						next thread 
//
// Constants:
//		none   
//
// Size: 
//		1 instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_signal_next(in_sig_gpr)
.begin
	local_csr_wr[SAME_ME_SIGNAL, in_sig_gpr]
.end
#endm // end of #macro _ethernet_encap _signal_next()




/////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_read_dram() 
//
// Description: read in_num_lw from dram
//
// Outputs:							
// 		out_dxfer:			dram xfer registers to hold vlue read 
//							instruction
//		out_dramaddr:		dram address of the payload
//
// Inputs:  
//		in_handle:			meta data handle 
//      in_offset: 			the offset of the payload 
//		in_num_lw:			number of doublewords to read
//		in_sig_dram:		signal number for dram read instruction
//
// Constants:
//							none
//
// Size:
//		4 instructions
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_read_dram(out_dxfer, out_dramaddr, in_handle, in_offset, \
		in_num_dw, in_sig_dram)							 						
.begin
.reg tmpx 
	ld_field_w_clr[tmpx, 0111, in_handle]
	alu_shf[tmpx, --, b, tmpx, <<2]
    // use following dl_buf_get_data_from_meta macro from dispatch_loop.uc to
	// get dram address at the beginning of the buffer from true_dl_buf_hdl
	// now it takes one instruction
    dl_buf_get_data_from_meta[out_dramaddr, tmpx]

    dram[read, out_dxfer, out_dramaddr, in_offset, in_num_dw], \
		sig_done[in_sig_dram]
.end // tmpx
#endm // end of macro _ethernet_encap_read_dram()



/////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_update_meta_data() 
//
// Description: update buffer size, offset, packet size fields in meta data
//
// Outputs:							
//		none
//
// Inputs:
//      in_meta_handle:		mta handle 
//      in_offset: 			the offset of the payload 
// 		in_bsize:			buffer size 
// 		in_psize:			packet size 
//		in_sig_sram:		signal number for sram write instruction
//
// Constants:
//							none
//
// Size:
//		8 instructions
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_update_meta_data(in_meta_handle, in_offset, in_bsize, \
						in_psize, in_sig_sram)
.begin 
.reg meta_addr tmpx
	// following dispatch loop macros ???
	//dl_meta_set_offset(offset)
	//dl_meta_set_buffer_size(buf_size)
	//dl_meta_set_packet_size(pkt_size)
	// update meta data by dispatch macros (3 to 12)
	//dl_meta_flush_cache(dl_meta, $source_lw0, sig_sram_write, SIG_NONE, 1, 2)
	ld_field_w_clr[tmpx, 0111, in_meta_handle]
	alu_shf[meta_addr, --, B, tmpx, <<2]
	alu_shf[tmpx, in_offset, OR, in_bsize, <<16]
	alu[$dl_meta1, --, B, tmpx]
	ld_field_w_clr[tmpx, 0011, $dl_meta2]
	alu_shf[tmpx, tmpx, OR, in_psize, <<16]
	alu[$dl_meta2, --, B, tmpx]
    sram[write, $dl_meta1, meta_addr, 4, 2], sig_done[in_sig_sram]

.end 
#endm // end of _ethernet_encap_update_meta_data()



/////////////////////////////////////////////////////////////////////////////
// _ethernet_encap_write_ethernet_header() 
//
// Description: realign and write Ethernet l2 header back to dram
//
// Outputs:							
//		none
//
// Inputs: 
//		in_dramaddr:		dram address of the payload
//      in_offset: 			the offset of the payload 
//		in_sig_dram:		signal number for dram write instruction
//
// Constants:
//							none
//
// Size:
//		26 instructions (worst case)
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_encap_write_ethernet_header(in_dramaddr, in_offset, in_sig_dram)
.begin
.reg offset_mod_8 new_offset tmp

	alu[offset_mod_8, in_offset, AND, _CONST_07]

	// jump to different port
	jump[offset_mod_8, jmp_tbl#], defer[3], targets[oset_0#, oset_1#, oset_2#, \
		oset_3#, oset_4#, oset_5#, oset_6#, oset_7#] 
		// reduce offset by two longwords, because two longword (ethernet header
		// occupies lower 14 bytes) is actually added before payload
		alu[new_offset, in_offset, -, 16]
		nop
		nop

//	Start of jump table. 
jmp_tbl#:

oset_0#:
		br[offset_mod_8_eq_0#]
			
oset_1#:
		br[offset_mod_8_eq_1#]

oset_2#:
		br[offset_mod_8_eq_2#]

oset_3#:
		br[offset_mod_8_eq_3#]

oset_4#:
		br[offset_mod_8_eq_4#]

oset_5#:
		br[offset_mod_8_eq_5#]

oset_6#:
		br[offset_mod_8_eq_6#]

oset_7#:
		br[offset_mod_8_eq_7#]

offset_mod_8_eq_0#:					; (10)
		local_csr_wr[BYTE_INDEX, 3]	; need 3 cycles for local_csr_wr to settle
			// shift out last one bytes in first 4 long word L2 header
			ld_field_w_clr[tmp, 0011, $l2_entry_lw0, >>8]
			alu[$$dram_lw0, --, B, tmp]
			nop
		byte_align_be[--, $l2_entry_lw0]
		byte_align_be[$$dram_lw1, $l2_entry_lw1]
		byte_align_be[$$dram_lw2, $l2_entry_lw2]
		byte_align_be[$$dram_lw3, $l2_entry_lw3]
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 2], sig_done[in_sig_dram]	 
		br[write_dram_done#]

offset_mod_8_eq_1#: 	; (13)
		local_csr_wr[BYTE_INDEX, 2]	; need 3 cycles for local_csr_wr to settle
			// now $$dram_lw0 should have one byte Ethernet header
			ld_field_w_clr[tmp, 0001, $l2_entry_lw0, >>16]
			alu[$$dram_lw0, --, B, tmp]
			// now shift last byte of Ether header into original offset area (1 byte)  
			ld_field_w_clr[tmp, 1000, $l2_entry_lw3, <<16]
		ld_field[tmp, 0111, $$dram_lw0]
		alu[$$dram_lw4, --, B, tmp]
		alu[$$dram_lw5, --, B, $$dram_lw1]

		byte_align_be[--, $l2_entry_lw0]
		byte_align_be[$$dram_lw1, $l2_entry_lw1]
		byte_align_be[$$dram_lw2, $l2_entry_lw2]
		byte_align_be[$$dram_lw3, $l2_entry_lw3]
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 3], sig_done[in_sig_dram]	 
		br[write_dram_done#]

offset_mod_8_eq_2#:			; (12)
		local_csr_wr[BYTE_INDEX, 1]	; need 3 cycles for local_csr_wr to settle
			// now $$dram_lw0 should have no Ethernet header, reset to 0
			alu[$$dram_lw0, --, B, 0]
			// now shift last two byte of Ether header into original offset area (2 byte)  
			ld_field_w_clr[tmp, 1100, $l2_entry_lw3, <<8]
			ld_field[tmp, 0011, $$dram_lw0]
		alu[$$dram_lw4, --, B, tmp]
		alu[$$dram_lw5, --, B, $$dram_lw1]

		byte_align_be[--, $l2_entry_lw0]
		byte_align_be[$$dram_lw1, $l2_entry_lw1]
		byte_align_be[$$dram_lw2, $l2_entry_lw2]
		byte_align_be[$$dram_lw3, $l2_entry_lw3]
//		dram[write, $$dram_lw0, in_dramaddr, new_offset, 3], sig_done[in_sig_dram]	 
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 2], sig_done[in_sig_dram] /* test */	 
		br[write_dram_done#]

offset_mod_8_eq_3#:		; (11)
		// now $$dram_lw0 should have no Ethernet header, reset to 0
		alu[$$dram_lw0, --, B, 0]
		// now shift last three bytes of Ether header into original offset 
		// area (3 bytes)  
		ld_field_w_clr[tmp, 1110, $l2_entry_lw3]
		ld_field[tmp, 0001, $$dram_lw0]
		alu[$$dram_lw4, --, B, tmp]
		alu[$$dram_lw5, --, B, $$dram_lw1]

		ld_field_w_clr[tmp, 0111, $l2_entry_lw0]
		alu[$$dram_lw1, --, B, tmp]
		alu[$$dram_lw2, --, B, $l2_entry_lw1]
		alu[$$dram_lw3, --, B, $l2_entry_lw2]
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 3], sig_done[in_sig_dram]
		br[write_dram_done#]

offset_mod_8_eq_4#:	; (11)
		local_csr_wr[BYTE_INDEX, 3]	; need 3 cycles for local_csr_wr to settle
			// now $$dram_lw0 should have no Ethernet header, reset to 0
			alu[$$dram_lw0, --, B, 0]
			ld_field_w_clr[tmp, 0011, $l2_entry_lw0, >>8]
			alu[$$dram_lw1, --, B, tmp]
		
		alu[$$dram_lw5, --, B, $$dram_lw1]
		byte_align_be[--, $l2_entry_lw0]
		byte_align_be[$$dram_lw2, $l2_entry_lw1]
		byte_align_be[$$dram_lw3, $l2_entry_lw2]
		byte_align_be[$$dram_lw4, $l2_entry_lw3]
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 3], sig_done[in_sig_dram]	 
		br[write_dram_done#]

offset_mod_8_eq_5#:		; (13)
		local_csr_wr[BYTE_INDEX, 2]	; need 3 cycles for local_csr_wr to settle
			// now $$dram_lw0 should have no Ethernet header, reset to 0
			alu[$$dram_lw0, --, B, 0]
			// now $$dram_lw5 should have one byte Ethernet header
			ld_field_w_clr[tmp, 1000, $l2_entry_lw3, <<16]
			ld_field[tmp, 0111, $$dram_lw1]
		alu[$$dram_lw5, --, B, tmp]
		// MSB Etherbyte in $$dram_lw1
		ld_field_w_clr[tmp, 0001, $l2_entry_lw0, >>16]
		alu[$$dram_lw1, --, B, tmp]
		byte_align_be[--, $l2_entry_lw0]
		byte_align_be[$$dram_lw2, $l2_entry_lw1]
		byte_align_be[$$dram_lw3, $l2_entry_lw2]
		byte_align_be[$$dram_lw4, $l2_entry_lw3]
		dram[write, $$dram_lw0, in_dramaddr, new_offset, 3], sig_done[in_sig_dram]	 
		br[write_dram_done#]

offset_mod_8_eq_6#:		; (11)
		local_csr_wr[BYTE_INDEX, 1]	; need 3 cycles for local_csr_wr to settle
			// now both $$dram_lw0 and $$dram_lw1 should have no Ethernet header, 
			// reset to 0
			//alu[$$dram_lw0, --, B, 0]
			//alu[$$dram_lw1, --, B, 0]

			// now shift last two byte of Ether header into original offset area (2 byte)  
			ld_field_w_clr[tmp, 1100, $l2_entry_lw3, <<8]
			ld_field[tmp, 0011, $$dram_lw1]
			alu[$$dram_lw5, --, B, tmp]
		byte_align_be[--, $l2_entry_lw0]
		byte_align_be[$$dram_lw2, $l2_entry_lw1]
		byte_align_be[$$dram_lw3, $l2_entry_lw2]
		byte_align_be[$$dram_lw4, $l2_entry_lw3]
		alu[new_offset, new_offset, +, 8]	; no need to write $$dram_lw0 and $$dram_lw1											
		dram[write, $$dram_lw2, in_dramaddr, new_offset, 2], sig_done[in_sig_dram]	
		br[write_dram_done#]


offset_mod_8_eq_7#:		; (11)
		// now both $$dram_lw0 and $$dram_lw1 should have no Ethernet header, 
		// reset to 0
		alu[$$dram_lw0, --, B, 0]
		alu[$$dram_lw1, --, B, 0]
		// now shift last three bytes of Ether header into original offset 
		// area (3 bytes)  
		ld_field_w_clr[tmp, 1110, $l2_entry_lw3]
		ld_field[tmp, 0001, $$dram_lw1]
		alu[$$dram_lw5, --, B, tmp]

		ld_field_w_clr[tmp, 0111, $l2_entry_lw0]
		alu[$$dram_lw2, --, B, tmp]
		alu[$$dram_lw3, --, B, $l2_entry_lw1]
		alu[$$dram_lw4, --, B, $l2_entry_lw2]
		alu[new_offset, new_offset, +, 8]	; no need to write $$dram_lw0 and $$dram_lw1											
		dram[write, $$dram_lw2, in_dramaddr, new_offset, 2], sig_done[in_sig_dram]	


write_dram_done#:

.end
 #endm // end of macro _ethernet_encap_write_ethernet_header()


#endif 		// __ETHERNET_ENCAP_UTIL_UC__

///////////////////////////////////////////////////////////////////////////////

