////////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ethernet_encap_util.uc
//
//      Purpose: Utility macros for L2_ENCAP
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ETHERNET_ARP_UTIL_UC__
#define __ETHERNET_ARP_UTIL_UC__

///////////////////////////////////////////////////////////////////////////////
// 
// _ethernet_arp_init_gprs()
//
// Description:
// 	
//		Initialize the common globals in context relative gprs to save 
//      instruction cycles
//
// Outputs: 
//      out_src_ring:			scratch ring number used to get ethernet_encap source
//      out_sink_ring:			scratch ring number used to send ethernet_encap sink
//		out_default_sigmask1:	default sig_mask for phase 1	
//		out_default_sigmask2:	default sig_mask for phase 2
//		out_default_sigmask3:	default sig_mask for phase 3	
//		out_id_header_exist:	next hop id value identifies L2 header already exist
//      out_l2_table_base:      L2 table base
// Inputs:  
//		None
//		
// Constants
//      None
//
//
////////////////////////////////////////////////////////////////////////////////
#macro	_ethernet_arp_init_gprs(out_src_ring, out_sink_ring, out_default_sigmask1, \
			out_default_sigmask2, out_default_sigmask3, \
			out_id_header_exist, out_l2_table_base)

.begin

#if IS_IXPTYPE(__IXP24XX)
	alu_shf[out_src_ring, --, b, ETHERNET_ARP_SOURCE_SCR_RING, <<2]
	alu_shf[out_sink_ring, --, b, ETHERNET_ARP_SINK_SCR_RING, <<2]
#endif	// #if IS_IXPTYPE(__IXP24XX)

#if IS_IXPTYPE(__IXP28XX)
	alu_shf[out_src_ring, --, b, ETHERNET_ARP_RING_IN, <<2]
	alu_shf[out_sink_ring, --, b, ETHERNET_ARP_RING_OUT, <<2]
#endif	// #if IS_IXPTYPE(__IXP28XX)

	// default phase 1 signal mask at start of each iteration of loop 
	alu[out_default_sigmask1, --, B, 0]			; initialize to 0
	_ethernet_arp_set_signal[out_default_sigmask1, sig2_next_context]	; add sig2_next_context
	_ethernet_arp_set_signal[out_default_sigmask1, sig_sram_read]		; add sig_sram_read

	// default phase 2 signal mask at start of each iteration of loop 
	alu[out_default_sigmask2, --, B, 0]			; initialize to 0
	_ethernet_arp_set_signal[out_default_sigmask2, sig3_next_context]	; add sig3_next_context
	_ethernet_arp_set_signal[out_default_sigmask2, sig_sram_read]		; add sig_sram_read

	// default phase 3 signal mask at start of each iteration of loop 
	alu[out_default_sigmask3, --, B, 0]			; initialize to 0
	_ethernet_arp_set_signal[out_default_sigmask3, sig1_next_context]	; add sig1_next_context
	_ethernet_arp_set_signal[out_default_sigmask3, sig_scratch_read]	; add sig_scratch_read 
	_ethernet_arp_set_signal[out_default_sigmask3, sig_scratch_write]	; add sig_scratch_write 

	move(out_id_header_exist, _L2_HEADER_ALREADY_EXIST_ID)
//	move(out_l2_table_base, _L2_TABLE_BASE)
	immed32(out_l2_table_base, L2_TABLE_SRAM_BASE)

.end

#endm  // end of #macro _ethernet_arp_init_gprs()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_set_signal()
//
// Description:
// 	
//		Set a signal in a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1 instruction
// 
//
////////////////////////////////////////////////////////////////////////////////////

#macro _ethernet_arp_set_signal(io_sigmask, in_sig)

.begin
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT in_sig
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, OR, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _ethernet_arp_set_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_set_push_signal()
//
// Description:
// 	
//		Add push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to set
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
//
////////////////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_set_push_signal(io_sigmask, in_sig)
.begin
.reg tmp
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, OR, 1, <<indirect]

#endif
.end

#endm // end of #macro _ethernet_arp_set_push_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_clear_signal()
//
// Description:
//		Clear a signal from a signal mask 
//
// Outputs: 
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be cleared with in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//
//		None
//
// Size: 
//
//		1 instructions
// 
//
/////////////////////////////////////////////////////////////////////////////////////

#macro _ethernet_arp_clear_signal(io_sigmask, in_sig)

.begin

#if ( isnum(in_sig) )
	
	#define_eval SIG_SHIFT sig
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT ]
	#undef SIG_SHIFT

#else

	alu[io_sigmask, io_sigmask, AND~, 1, <<(&in_sig)]

#endif

.end

#endm // end of #macro _ethernet_arp_clear_signal()


/////////////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_clear_push_signal()
//
// Description:
// 	
//		clear push signal (i.e. dram access has puu and push signal) in a signal mask 
//
// Outputs:
//		None 
//
// Inputs/Outputs:  
//		io_sigmask:			signal mask to be added with push signal of in_sig
//
// Inputs:  
//		in_sig:				signal number to clear
//
// Constants
//		None
//
// Size: 
//
//		1-4 instructions
// 
//
////////////////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_clear_push_signal(io_sigmask, in_sig)
.begin
.reg tmp
#if ( isnum(in_sig) )

	#define_eval SIG_SHIFT (in_sig +1)
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<SIG_SHIFT]
	#undef SIG_SHIFT

#else
	alu[tmp, --, B, &in_sig]
	alu[tmp, tmp, +, 1]
	alu[--, tmp, OR, 0]										
	alu_shf[io_sigmask, io_sigmask, AND~, 1, <<indirect]

#endif
.end

#endm // end of #macro _ethernet_arp_clear_push_signal()




///////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_source()
//
// Description: readfrom the source scratch ring
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_src_ring:		source scratch ring 
//		in_src_sig:			signal for scratch ring read
//
// Constants:
//		none   
//
// Size: 
//		1 instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_source(in_src_ring, in_src_sig)
.begin

	scratch[get, $source_lw0, in_src_ring, 0, _NUM_WORDS_ETHERNET_ARP_MSG], \
		sig_done[in_src_sig]

.end
#endm

///////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_sink()
//
// Description: Write to the sink scratch ring
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_sink_ring:		sink scratch ring 
//		in_sink_sig:		signal for scratch ring write
//
// Constants:
//		none   
//
// Size: 
//		2 to ?? instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_sink(in_sink_ring, in_sink_sig)
.begin

check_ring_full#:
	br_inp_state[ETHER_ARP_SINK_RING_FULL, check_ring_full#]

	scratch[put, $sink_lw0, in_sink_ring, 0, _NUM_WORDS_ETHERNET_ARP_MSG], \
		sig_done[in_sink_sig]

.end
#endm // end of #macro _ethernet_arp_sink()

///////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_save_to_sink()
//
// Description: move source to sink before got new source
//
// Outputs:							
//		none
//
// Inputs:  
// 		none
//
// Constants:
//		none   
//
// Size: 
//		3 instructions
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_save_to_sink()
	// save $source to $sink before reading another source
	alu[$sink_lw0, --, B, $source_lw0] 
	alu[$sink_lw1, --, B, $source_lw1] 
	alu[$sink_lw2, --, B, $source_lw2] 
#endm // end of #macro _ethernet_arp_save_to_sink()


///////////////////////////////////////////////////////////////////////////////
// _ethernet_arp_signal_next()
//
// Description: Wakeup next thread in the same ME
//
// Outputs:							
//		none
//
// Inputs:  
// 		in_sig_gpr:		value to be wriiten to SAME_ME_SIGNAL csr to wake up
//						next thread 
//
// Constants:
//		none   
//
// Size: 
//		1 instruction
//
///////////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_signal_next(in_sig_gpr)
.begin
	local_csr_wr[SAME_ME_SIGNAL, in_sig_gpr]
.end
#endm // end of #macro _ethernet_encap _signal_next()


////////////////////////////////////////////////////////////////////////
// Macro Name  : _ethernet_arp_core_exception_send
// Description : Macro to write the two longwords of exception
//				 and trigger interrupt to activate RM ISR for 
//				 exception packets
// Output:
//		none
//
// Input:
//		in_buf_handle:			input sop buffer handle 
//      in_exception_code:		input exception code
//		in_eop_buf_handle:		input eop buffer handle
//		in_sigmask:				input signal msak 
// 		in_scratch_write_sig:	input signal for putting exception 
//								request into exception scratch ring
//                             
//	Labels:
//		IN_PHASE1_START_LABEL:	symbolic address for start of inifinite 
//								Ethernet ARP processing loop
//
// Constant:	
//		none
//
// Size:
//		18 instructions
//
////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_core_exception_send(in_buf_handle, \
			in_eop_buf_handle, in_sigmask, in_scratch_write_sig, \
			IN_PHASE1_START_LABEL)

.begin
    .reg    priority

	/* check for priority */
    dl_exception_get_priority[priority]
	br_bclr[priority, 0, low_priority#]

	_ethernet_arp_core_exception_write_ring(in_buf_handle, \
		dl_exception_reg, EXCEPTION_RING_OUT_1, XSCALE_INT_B, \
		in_eop_buf_handle, in_sigmask, in_scratch_write_sig, \
		IN_PHASE1_START_LABEL)

low_priority#:

	_ethernet_arp_core_exception_write_ring(in_buf_handle, \
		dl_exception_reg, EXCEPTION_RING_OUT_0, XSCALE_INT_A, \
		in_eop_buf_handle, in_sigmask, in_scratch_write_sig, \
		IN_PHASE1_START_LABEL)
.end

end#:

#endm /* end of _ethernet_arp_core_exception_send() */

////////////////////////////////////////////////////////////////////////
// Macro Name  : _ethernet_arp_core_exception_write_ring
// Description : Macro to write the two longwords of exception
//				 and trigger interrupt to activate RM ISR for 
//				 exception packets
//
// Output:
//		none
//
// Input:
//		in_buffer_handle:		input sop buffer handle 
//      in_exception_code:		input exception code
//      IN_RING_NUM:			input exception scratch ring number 
//      IN_INTERRUPT:			input interrupt number 
//		in_eop_buffer_handle:	input eop buffer handle
//		in_sigmask:				input signal msak 
// 		in_scratch_write_sig:	input signal for putting exception 
//								request into exception scratch ring
//                             
//	Labels:
//		IN_PHASE1_START_LABEL:	symbolic address for start of inifinite 
//								Ethernet ARP processing loop
//
// Constant:	
//		none
//
// Size:
//		16 instructions
// Branches    : 
//
////////////////////////////////////////////////////////////////////////
#macro _ethernet_arp_core_exception_write_ring(in_buffer_handle, \
		in_exception_code, IN_RING_NUM, IN_INTERRUPT, \
		in_eop_buffer_handle, in_sigmask, in_scratch_write_sig, \
		IN_PHASE1_START_LABEL)

.begin
   .reg $excep_lw0, $excep_lw1, ring
   .xfer_order $excep_lw0, $excep_lw1
   .sig scratch_wr_done


   // set the ring number

   alu[ring, --,b, IN_RING_NUM, <<2]

   // set the exception longwords

   alu[$excep_lw0, --,b, in_buffer_handle]
   alu[$excep_lw1, --,b, in_exception_code]

	// get source for next run  [1]
	_ethernet_arp_source(src_ring, sig_scratch_read) 	

   // check if transmit ring is full

   #define_eval RING_NUM IN_RING_NUM

   scr_write#:
	
   br_inp_state[SCR_RING/**/RING_NUM/**/_FULL, scratch_ring_full#]

   // write the two longwords to the exception ring
	_ethernet_arp_set_signal[in_sigmask, in_scratch_write_sig]	; add signal 

   scratch[put, $excep_lw0, ring, 0,2], sig_done[in_scratch_write_sig]
   local_csr_wr[active_ctx_wakeup_events, in_sigmask]
   ctx_arb[--] 
   .io_completed sig1_next_context sig_scratch_read sig_scratch_write
   nop
   nop

   // trigger the interrupt

   cap[fast_wr, 0, IN_INTERRUPT]

   // take three  cycles for the CSR to be updated
   // using two defer cycles
   nop
 
   br[IN_PHASE1_START_LABEL], defer[2]
   nop
   nop

scratch_ring_full#:
	// 	Drop the packet. Two cases to handle. Single buffer or a bufffer chain.
	//	It is determined by looking at the dl_eop_buf_handle

	alu[--, in_eop_buffer_handle, -, IX_NULL]			; if dl_eop_buf_handle == IX_NULL
	beq[drop_single_buffer_1#]								; then it is single buffer.

	dl_drop_buffer_chain[in_buffer_handle, in_eop_buffer_handle]	; otherwise, buffer chain.
	br[end#]

drop_single_buffer_1#:
	dl_drop_buffer[in_buffer_handle]

end#:
   local_csr_wr[active_ctx_wakeup_events, in_sigmask]
   .io_completed sig1_next_context sig_scratch_read sig_scratch_write
	ctx_arb[--] , br[IN_PHASE1_START_LABEL]

.end

#undef RING_NUM

#endm /* end of _ethernet_arp_core_exception_write_ring() */


#endif 		// __ETHERNET_ARP_UTIL_UC__

///////////////////////////////////////////////////////////////////////////////

