/*****************************************************************************
 *                            Intel Proprietary
 *
 * Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.
 * No part of this program or publication may be reproduced, transmitted,
 * transcribed, stored in a retrieval system, or translated into any language
 * or computer language in any form or by any means, electronic, mechanical,
 * magnetic, optical, chemical, manual, or otherwise, without the prior
 * written permission of:
 *                        Intel Corporation
 *                        2200 Mission College Blvd.
 *                        Santa Clara, CA  95052-8119
 *****************************************************************************/

/**
 *****************************************************************************
 * @ingroup SarUt
 * File:  ix_cc_llc_snap_config.c
 * @description
 *      The file defines for core component infrestructure  interface and
 *      internal functions for initialization and shutdown.
 *****************************************************************************/

/*****************************************************************************
 *
 * Contents:
 *        Definitions of following exported data:
 *            none
 *        Definitions of following exported functions
 *        Definitions of following local functions:
 *
 *****************************************************************************/


/*****************************************************************************
 *                   System defined include files
 *****************************************************************************/
#include "ix_cc_error.h"
#include "ix_rm.h"
#include "ix_ossl.h"
#include "ix_cc.h"
#include "ix_cc_macros.h"
#include "ix_cc_properties.h"
#include "ix_config.h"
#include "bindings.h"


#include "cc/ix_cc_msup.h"


/*****************************************************************************
 *                    User defined include files
 *****************************************************************************/
#include "ix_cc_llc_snap_config_private.h"
#include <stdio.h>

ix_error llc_snap_encap_cb_add_next_hop_callback(ix_error arg_Result,
                                                 void* arg_pContext)
{
   ix_ossl_sem_t *pSem = (ix_ossl_sem_t*) arg_pContext;

   if (arg_Result != IX_SUCCESS)
    {
        /*
         * Print out the error for the user's benefit.
         */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
        ix_error_dump(0, arg_Result);
#else
        ix_error_dump(stdout, arg_Result);
#endif
    }

   
   /* print the received results */
   ix_ossl_message_log("callback for add_next_hop - result : %ld\n",
                       (ix_uint32)arg_Result);

   
   if (pSem != NULL)
   {
       /*
        * If we're not able to give the semaphore, print the error.  Always
        * return success.
        */
       IX_ERROR_CP(ix_ossl_sem_give(*pSem));
   }
   
   return IX_SUCCESS;
}

ix_error llc_snap_encap_cb_del_next_hop_callback(ix_error arg_Result,
                                                 void* arg_pContext)
{
   ix_ossl_sem_t *pSem = (ix_ossl_sem_t*) arg_pContext;

   if (arg_Result != IX_SUCCESS)
    {
        /*
         * Print out the error for the user's benefit.
         */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
        ix_error_dump(0, arg_Result);
#else
        ix_error_dump(stdout, arg_Result);
#endif
    }

   
   /* print the received results */
   ix_ossl_message_log("callback for del_next_hop - result : %ld\n",
                       (ix_uint32)arg_Result);

   
   if (pSem != NULL)
   {
       /*
        * If we're not able to give the semaphore, print the error.  Always
        * return success.
        */
       IX_ERROR_CP(ix_ossl_sem_give(*pSem));
   }
   
   return IX_SUCCESS;
}


int addNextHop(const char *argv)
{
    ix_uint32 nextHop;
    ix_uint32 vpi;
    ix_uint32 vci;
    ix_uint32 vcq;
    ix_uint32 encapBit;
    ix_uint32 dSap;
    ix_uint32 sSap;
    ix_uint32 ctrl;
    ix_uint32 oui;
    ix_uint32 pid;
    ix_error err;
    ix_ossl_sem_t sem;

    if (argv == NULL)
    {
        ix_ossl_message_log("addNextHop: parameters missing.");
        return 1;
    }
    
    /*
     * Parse the command line and extract all required parameters.
     */
    {
        
        if (sscanf(argv, "%li %li %li %li %li %li %li %li %li %li",
                   &nextHop, &vpi, &vci, &vcq, &encapBit, &dSap, &sSap, &ctrl, &oui, &pid ) != 10)
        {
            ix_ossl_message_log("addNextHop: Command line parse error.");
            return 1;
        } 
        
        ix_ossl_message_log("Adding next hop:\n");
        ix_ossl_message_log("nextHop:%ld vpi:%ld vci:%ld vcq:%ld "
                            "encapBit:%ld dSap:%ld sSap:%ld "
                            "ctrl:%ld oui:%ld pid:%ld\n",
                            nextHop, vpi, vci, vcq,
                            encapBit, dSap, sSap, ctrl,
                            oui, pid);
    }

    
    /* Allocate the semaphore - it is released by calback functions */
    IX_ERROR_CR(ix_ossl_sem_init(IX_OSSL_SEM_UNAVAILABLE, &sem));

    /* send the vc remove async message */
    IX_ERROR_CG(ix_cc_llc_snap_encap_async_add_next_hop(nextHop,
                                                        encapBit,
                                                        vpi,
                                                        vci,
                                                        vcq,
                                                        dSap,
                                                        sSap,
                                                        ctrl,
                                                        oui,
                                                        pid,
                                                        llc_snap_encap_cb_add_next_hop_callback,
                                                        &sem),
                err,clean_sem);
    IX_ERROR_CG(ix_ossl_sem_take(sem, LLCSNAP_CONFIG_TIMEOUT), err, clean_sem);
    IX_ERROR_CR(ix_ossl_sem_fini(sem));
    
    return 0;

clean_sem:
    (void) ix_ossl_sem_fini(sem);

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
    ix_error_dump(0, err);
#else
    ix_error_dump(stdout, err);
#endif
    
    return -1;
}



int delNextHop(const char *argv)
{
    ix_uint32 nextHop;
    ix_error err;
    ix_ossl_sem_t sem;

    
    if (argv == NULL)
    {
        ix_ossl_message_log("delNextHop: parameters missing.");
        return 1;
    }
    
    /*
     * Parse the command line and extract all required parameters.
     */
    {
        
        if (sscanf(argv, "%li",
                   &nextHop) != 1)
        {
            ix_ossl_message_log("delNextHop: Command line parse error.");
            return 1;
        } 
        
        ix_ossl_message_log("Removing next hop: nextHop:%ld\n",
                            nextHop);
    }

    
    /* Allocate the semaphore - it is released by calback functions */
    IX_ERROR_CR(ix_ossl_sem_init(IX_OSSL_SEM_UNAVAILABLE, &sem));


    /* actually issue the command */
    IX_ERROR_CG(ix_cc_llc_snap_encap_async_delete_next_hop(nextHop,
                                                           llc_snap_encap_cb_del_next_hop_callback,
                                                           &sem),
                err,clean_sem);

    IX_ERROR_CG(ix_ossl_sem_take(sem, LLCSNAP_CONFIG_TIMEOUT), err, clean_sem);
    IX_ERROR_CR(ix_ossl_sem_fini(sem));
    
    return 0;

clean_sem:
    (void) ix_ossl_sem_fini(sem);

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
    ix_error_dump(0, err);
#else
    ix_error_dump(stdout, err);
#endif
    
    return -1;
}


