/*****************************************************************************
 *                            Intel Proprietary
 *
 * Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.  
 * No part of this program or publication may be reproduced, transmitted,
 * transcribed, stored in a retrieval system, or translated into any language
 * or computer language in any form or by any means, electronic, mechanical,
 * magnetic, optical, chemical, manual, or otherwise, without the prior
 * written permission of:
 *                        Intel Corporation
 *                        2200 Mission College Blvd.
 *                        Santa Clara, CA  95052-8119
 *****************************************************************************/

/**
 *****************************************************************************
 * @ingroup LlcSnapEncap
 *
 * File:  ix_cc_llc_snap_encap_msg_handler.c
 *
 * @description
 *      This file defines the message handling functions for the LLC Snap
 *      Encapsulation Core Component.
 *
 *****************************************************************************/

/*****************************************************************************
 *
 * Contents:
 *        Definitions of following exported data:
 *
 *        Definitions of following exported functions:
 *            ix_cc_llc_snap_encap_msg_handler
 *
 *        Definitions of following local data:
 *
 *        Definitions of following local functions:
 *
 *****************************************************************************/


/*
 * User defined include files
 */
#include "ix_cc_llc_snap_encap.h"
#include "ix_cc_llc_snap_encap_private.h"

#include "cc/ix_cc_msup.h"


/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * LLC Snap CC message handler
 * @description
 *    This primitive is the message handler function for LLC Snap Encap core
 *    component. The core component receives messages from other core
 *    components through this message handler function and internally calls
 *    appropriate library function to process the message. This message
 *    handler should be used to update dynamic properties.
 *
 * @param hDataToken - IN: buffer handle embedding information for
 *                     the message passed in userData.
 * @param userData - IN: message type
 * @param pContext - IN: pointer to the context data
 *
 * @retval IX_SUCCESS - message processed successfully
 * @retval IX_CC_ERROR_NULL - null argument
 * @retval IX_CC_ERROR_UNDEFINED_MSG - unsupported message
 * @retval IX_CC_LLC_SANP_ENCAP_ERROR_MSG_LIBRARY - error from message
 *                                                  support library
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_BUFFER_FREE - error from RM, failed
 *                                                  freeing the buffer
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_msg_handler(ix_buffer_handle hDataToken,
                                 ix_uint32 userData,
                                 void* pContext)
{
    ix_error err = IX_SUCCESS;
    void *pLocalContext = NULL;
    void *pMsg = NULL;
    void *pRetMsg = NULL;
    ix_uint32 msgRetLength = 0;
    ix_uint8 noResponse = FALSE;

    
    /* Validate input arguments */
    if (pContext == NULL)
    {
        return IX_ERROR_WARNING(
        IX_CC_ERROR_NULL,
        ("LLC Snap Encap Msg Handler: pContext is NULL"));
    }

    if ((userData < IX_CC_COMMON_MSG_ID_FIRST) ||
        (userData >= IX_CC_LLC_SNAP_ENCAP_MSG_LAST))
    {
    
        return IX_ERROR_WARNING(
        IX_CC_ERROR_UNDEFINED_MSG,
        ("LLC Snap Encap Msg Handler: Invalid message type"));
    }

    /* get data from the message */
    IX_MSUP_EXTRACT_MSG(hDataToken,
                        &pMsg,
                        &pLocalContext);

    if ((userData != IX_CC_LLC_SNAP_ENCAP_MSG_ADD_NEXT_HOP) &&
        (userData != IX_CC_LLC_SNAP_ENCAP_MSG_DELETE_NEXT_HOP))
    {
        return IX_ERROR_WARNING(
        IX_CC_ERROR_UNDEFINED_MSG,
        ("LLC Snap Encap Msg Handler: Invalid message type"));
    }

    if (pMsg == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_ERROR_NULL,
                                ("LLC Snap Encap Msg Handler: Invalid Buffer"));
    }
 
    
    /* invoke proper function according to the message type */
    switch (userData)
    {
    case IX_CC_LLC_SNAP_ENCAP_MSG_ADD_NEXT_HOP:
    {
        /* get data from the message */
        ix_cc_llc_snap_encap_add_next_hop_data_t* pData = \
            (ix_cc_llc_snap_encap_add_next_hop_data_t*)pMsg;
        
        /* invoke proper library function; pass received parameters */
        err = ix_cc_llc_snap_encap_add_next_hop(pData->nextHop,
                                                pData->encapType,
                                                pData->vpi,
                                                pData->vci,
                                                pData->vcq,
                                                pData->dSap,
                                                pData->sSap,
                                                pData->ctrl,
                                                pData->oui,
                                                pData->pid);
        break;
    }
    
    case IX_CC_LLC_SNAP_ENCAP_MSG_DELETE_NEXT_HOP:
    {
        /* get data from the message */
        ix_cc_llc_snap_encap_delete_next_hop_data_t* pData = \
            (ix_cc_llc_snap_encap_delete_next_hop_data_t*)pMsg;

        /* invoke proper library function; pass received parameters */
        err = ix_cc_llc_snap_encap_delete_next_hop(pData->nextHop);
    
        break;
    }
    
    default:
        IX_ERROR_CRT(ix_rm_buffer_free_chain(hDataToken),
                     IX_CC_LLC_SNAP_ENCAP_ERROR_BUFFER_FREE,
                     0);
        
        return IX_ERROR_WARNING(
        IX_CC_ERROR_UNDEFINED_MSG,
        ("LLC Snap Encap Msg Handler: Invalid Message Type"));
        
        break;

    } /* switch */

    /* do not send reply if not needed */
    if(noResponse == TRUE)
    {
        IX_ERROR_CRT(ix_rm_buffer_free_chain(hDataToken),
                     IX_CC_LLC_SNAP_ENCAP_ERROR_BUFFER_FREE,
                     0);
        
        return IX_SUCCESS;
    }

    /* send return message */
    err = ix_cc_msup_send_reply_msg(pLocalContext,
                                    pRetMsg,
                                    msgRetLength,
                                    err);
    if (err != IX_SUCCESS)
    {
        err = IX_ERROR_WARNING(
        IX_CC_LLC_SNAP_ENCAP_ERROR_MSG_LIBRARY,
        ("LLC Snap Encap Msg Handler: Send reply message failed"));
    }

    /* free message buffer */
    IX_ERROR_CRT(ix_rm_buffer_free_chain(hDataToken),
                 IX_CC_LLC_SNAP_ENCAP_ERROR_BUFFER_FREE,
                 0);

    return err;

} /* ix_cc_llc_snap_encap_msg_handler */

