/*****************************************************************************
 *                            Intel Proprietary
 *
 * Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.  
 * No part of this program or publication may be reproduced, transmitted,
 * transcribed, stored in a retrieval system, or translated into any language
 * or computer language in any form or by any means, electronic, mechanical,
 * magnetic, optical, chemical, manual, or otherwise, without the prior
 * written permission of:
 *                        Intel Corporation
 *                        2200 Mission College Blvd.
 *                        Santa Clara, CA  95052-8119
 *****************************************************************************/

/**
 *****************************************************************************
 * @ingroup LlcSnapEncap
 *
 * File:  ix_cc_llc_snap_encap_fini.c
 *
 * @description
 *      This file defines all shutdown functions for the LLC Snap
 *      Encapsulation Core Component.
 *
 *****************************************************************************/

/*****************************************************************************
 *
 * Contents:
 *        Definitions of following exported data:
 *
 *        Definitions of following exported functions:
 *            ix_cc_llc_snap_encap_fini
 *
 *        Definitions of following local data:
 *
 *        Definitions of following local functions:
 *            ix_cc_llc_snap_encap_free_symbols
 *            ix_cc_llc_snap_encap_unregister_message_handler
 *            ix_cc_llc_snap_encap_unregister_packet_handlers
 *
 *****************************************************************************/


/*
 * User defined include files
 */
#include "ix_cc_llc_snap_encap.h"
#include "ix_cc_llc_snap_encap_private.h"



/** 
 * Function definitions.
 */

/*****************************************************************************
 * Abstract:
 *    This primitive will be used to free memory previously allocated for
 *    patching.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - memory allocated successfully
 *    IX_CC_ERROR_OOM - unable to free memory
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_free_symbols()
{
    /* Free memory allocated for microblock symbols */
    IX_ERROR_CRT(ix_rm_mem_free(pLlcSnapEncapContext->pNextHopTableSramBase),
                 IX_CC_ERROR_OOM,
                 IX_ERROR_LEVEL_LOCAL);

    /* clear the relevant pointers in the context */
    pLlcSnapEncapContext->pNextHopTableSramBase = NULL;

    return IX_SUCCESS;

} /* ix_cc_llc_snap_encap_free_symbols */


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to deregister the message handler used
 *    by the LLC Snap Encapsulation Core Component.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - message handler unregistered successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - error from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_unregister_message_handler()
{
    /* Call CCI if CCI is enabled */
#if defined(IX_EXCLUDE_CCI)
    
    IX_ERROR_CRT(ix_rm_message_handler_unregister(
                 IX_CC_LLC_SNAP_ENCAP_MSG_INPUT),
                 IX_CC_LLC_SNAP_ENCAP_ERROR_CCI,
                 IX_ERROR_LEVEL_WARNING);
#else
    
    IX_ERROR_CRT(ix_cci_cc_remove_message_handler(
                 IX_CC_LLC_SNAP_ENCAP_MSG_INPUT),
                 IX_CC_LLC_SNAP_ENCAP_ERROR_CCI,
                 IX_ERROR_LEVEL_WARNING);
#endif

    return IX_SUCCESS;

} /* ix_cc_llc_snap_encap_unregister_message_handler */


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to deregister the packet handlers used
 *    by the LLC Snap Encapsulation Core Component.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - packet handlers unregistered successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - error from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_unregister_packet_handlers()
{
    /* Call CCI if CCI is enabled */
#if defined(IX_EXCLUDE_CCI)
    
    IX_ERROR_CRT(ix_rm_message_handler_unregister(IX_CC_LLC_SNAP_ENCAP_PKT_INPUT),
                 IX_CC_LLC_SNAP_ENCAP_ERROR_CCI,
                 IX_ERROR_LEVEL_WARNING);
#else
    
    IX_ERROR_CRT(ix_cci_cc_remove_message_handler(IX_CC_LLC_SNAP_ENCAP_PKT_INPUT),
                 IX_CC_LLC_SNAP_ENCAP_ERROR_CCI,
                 IX_ERROR_LEVEL_WARNING);
#endif

    return IX_SUCCESS;

} /* ix_cc_llc_snap_encap_unregister_packet_handlers */


/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * LLC Snap CC main shutdown function
 * @description
 *    This routine is the main shutdown function for the LLC Snap
 *    Encapsulation Core Component. It performs memory deallocation,
 *    handlers deregistration and all other activities necessary to gracefully
 *    stop the Core Component's regular activity.
 *
 * @param hCcHandle - IN: handle to the LLC Snap Encap core component
 * @param pContext - IN: pointer to the control block memory allocated 
 *                   earlier in the_init function. 
 *
 * @retval IX_SUCCESS - shutdown successful
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM - invalid input
 *                                                          parameter(s)
 * @retval IX_CC_ERROR_OOM - failure of memory deallocation
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - failure from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_fini(ix_cc_handle hCcHandle,
                          void* pContext)
{
    /* validate input parameters */
    if(pContext == NULL)
    {
        return IX_ERROR_WARNING(IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM,
                                ("LLC Snap Encap fini: pContext is null"));
    }


    if((hCcHandle != pLlcSnapEncapContext->hLlcSnapEncapHandle) ||
       (ix_cc_llc_snap_encap_context_t*)pContext != pLlcSnapEncapContext)
    {
        return IX_ERROR_WARNING(
        IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM,
        ("LLC Snap Encap fini: Invalid pContext parameter"));
    }
  
    /* Free memory allocated for symbols */
    IX_ERROR_CR(ix_cc_llc_snap_encap_free_symbols());   

    /* Unregister the Message Handler */
    IX_ERROR_CR(ix_cc_llc_snap_encap_unregister_message_handler());
  
    /* Unregister the Packet Handler */
    IX_ERROR_CR(ix_cc_llc_snap_encap_unregister_packet_handlers());
    
    /* Free the Context */
    ix_ossl_free(pLlcSnapEncapContext);
        
    return IX_SUCCESS;

} /* ix_cc_llc_snap_encap_fini */

