#ifndef IX_CC_LLC_SNAP_ENCAP_PRIVATE_H
#define IX_CC_LLC_SNAP_ENCAP_PRIVATE_H
/*****************************************************************************
 *                            Intel Proprietary
 *
 * Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.  
 * No part of this program or publication may be reproduced, transmitted,
 * transcribed, stored in a retrieval system, or translated into any language
 * or computer language in any form or by any means, electronic, mechanical,
 * magnetic, optical, chemical, manual, or otherwise, without the prior
 * written permission of:
 *                        Intel Corporation
 *                        2200 Mission College Blvd.
 *                        Santa Clara, CA  95052-8119
 *****************************************************************************/

/**
 *****************************************************************************
 * @ingroup LlcSnapEncap
 *
 * File:  ix_cc_llc_snap_encap_private.h
 *
 * @description
 *      This file defines all private data types and function prototypes
 *      required internally by the LLC Snap Encapsulation Core Component.
 *
 *****************************************************************************/

/*****************************************************************************
 *
 * Contents:
 *        Definitions of following exported data:
 *
 *        Definitions of following exported functions:
 *
 *        Definitions of following local data:
 *            ix_cc_llc_snap_encap_messages_t
 *            ix_cc_llc_snap_encap_error_code_t
 *            ix_cc_llc_snap_encap_context_t
 *            ix_cc_llc_snap_encap_registry_data_t
 *            ix_cc_llc_snap_encap_add_next_hop_callback_data_t
 *            ix_cc_llc_snap_encap_add_next_hop_data_t
 *            ix_cc_llc_snap_encap_delete_next_hop_callback_data_t
 *            ix_cc_llc_snap_encap_delete_next_hop_data_t
 *            ix_cc_llc_snap_encap_pkt_lookup_result_t
 *
 *        Definitions of following local functions:
 *            ix_cc_llc_snap_encap_retrieve_registry_data
 *            ix_cc_llc_snap_encap_registry
 *            ix_cc_llc_snap_encap_alloc_symbols
 *            ix_cc_llc_snap_encap_patch_symbols
 *            ix_cc_llc_snap_encap_register_message_handler
 *            ix_cc_llc_snap_encap_register_packet_handlers
 *            ix_cc_llc_snap_encap_free_symbols
 *            ix_cc_llc_snap_encap_unregister_message_handler
 *            ix_cc_llc_snap_encap_unregister_packet_handlers
 *            _ix_cc_llc_snap_encap_cb_add_next_hop
 *            _ix_cc_llc_snap_encap_cb_delete_next_hop
 *
 *****************************************************************************/

#if defined(__cplusplus)
extern "C" {
#endif /* end defined(__cplusplus) */

#include "ix_cc_llc_snap_encap.h"

/*
 * patched symbol names
 */
#define LLC_SNAP_ENCAP_PATCHNAME_NEXT_HOP_TABLE "NEXT_HOP_TABLE_SRAM_BASE"

/* number of symbols to be patched */
#define IX_CC_LLC_SNAP_ENCAP_NUMBER_OF_SYMBOLS 1


/*
 * names of the symbols taken from the registry
 */
#define LLC_SNAP_ENCAP_REGNAME_NUMBER_OF_NEXT_HOP_ENTRIES \
"LLC_SNAP_ENCAP/NUMBER_OF_NEXT_HOP_ENTRIES"
#define LLC_SNAP_ENCAP_REGNAME_MEV2_NUM_MASK "LLC_SNAP_ENCAP/MEV2_NUM_MASK"
#define LLC_SNAP_ENCAP_REGNAME_SRAM_CHANNEL "LLC_SNAP_ENCAP/SRAM_CHANNEL"


/* LLC Snap static configuration data - used when the registry is not present */
/* TBD: all values in this section are dummy! */

/* number of Next Hop Table Entries */
#define NUMBER_OF_NEXT_HOP_ENTRIES 10

/* IDs of Microengines used by the LLC Snap Encap */
#define IX_CC_LLC_SNAP_ENCAP_MEV2_NUM_MASK 0x1

/* SRAM channels used by LLC Snap Encap */
#define IX_CC_LLC_SNAP_ENCAP_SRAM_CHANNEL 0

/* end of static configuration data */


/* packet handler related definitions (TBD - dummy values) */
#define LLC_SNAP_ENCAP_NEXT1 0x01
#define LLC_SNAP_ENCAP_NEXT2 0x02
#define LLC_SNAP_ENCAP_NEXT3 0x03

#define LLC_SNAP_ENCAP_DEFAULT_NEXT_BLOCK   LLC_SNAP_ENCAP_NEXT1

#define IX_CC_LLC_SNAP_ENCAP_DEFAULT_OUTPUT 0x01
#define IX_CC_LLC_SNAP_ENCAP_FWD_OUTPUT     0x02


/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation message codes
 *
 * Purpose/Intended Use:
 *    This enum contains all message codes used within the LLC Snap Encap
 *    Core Component.
 *
 *****************************************************************************/
typedef enum ix_cc_llc_snap_encap_messages
{
    IX_CC_LLC_SNAP_ENCAP_MSG_FIRST = IX_CC_COMMON_MSG_ID_LAST,

    IX_CC_LLC_SNAP_ENCAP_MSG_ADD_NEXT_HOP,
    IX_CC_LLC_SNAP_ENCAP_MSG_DELETE_NEXT_HOP,

    /* Add new messages before this */
    IX_CC_LLC_SNAP_ENCAP_MSG_LAST
    
} ix_cc_llc_snap_encap_messages_t;


/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation error codes
 *
 * Purpose/Intended Use:
 *    This enum contains all private error codes used within the LLC Snap Encap
 *    Core Component.
 *
 *****************************************************************************/
typedef enum ix_cc_llc_snap_encap_error_code
{
    IX_CC_LLC_SNAP_ENCAP_ERROR_FIRST   = IX_ERROR_MAKE_GROUP(CC_LLC_SNAP_ENCAP),
    IX_CC_LLC_SNAP_ENCAP_ERROR_SUCCESS = IX_CC_LLC_SNAP_ENCAP_ERROR_FIRST,
    
    IX_CC_LLC_SNAP_ENCAP_ERROR_FAILED_PATCHING,
    IX_CC_LLC_SNAP_ENCAP_ERROR_REGISTRY,
    IX_CC_LLC_SNAP_ENCAP_ERROR_CCI,
    IX_CC_LLC_SNAP_ENCAP_ERROR_BUFFER_FREE,
    IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM,
    IX_CC_LLC_SNAP_ENCAP_ERROR_MSG_LIBRARY,
    IX_CC_LLC_SNAP_ENCAP_ERROR_DUPLICATED,
    IX_CC_LLC_SNAP_ENCAP_ERROR_NOT_FOUND,
    IX_CC_LLC_SNAP_ENCAP_ERROR_CALLBACK,
    IX_CC_LLC_SNAP_ENCAP_ERROR_NOT_INITIALIZED,

    /* unknown error code */
    IX_CC_LLC_SNAP_ENCAP_ERROR_UNKNOWN,
    IX_CC_LLC_SNAP_ENCAP_ERROR_LAST
    
} ix_cc_llc_snap_encap_error_code_t;


/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation CC context structure
 *
 * Purpose/Intended Use:
 *    This structure contains all private data important for the LLC Snap
 *    CC, not exported for the rest of the system. These are "global" only
 *    in context of the LLC Snap Core Component.
 *
 *****************************************************************************/
typedef struct ix_cc_llc_snap_encap_context
{
    ix_cc_handle hLlcSnapEncapHandle; /* Handle of the LLC Snap core comp. */
    ix_uint32* pNextHopTableSramBase; /* Base address of the Next Hop table
                                         in SRAM */
    ix_uint8 initialized;             /* LLC Snap CC initialization state */
    
} ix_cc_llc_snap_encap_context_t;


/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation CC registry data structure
 *
 * Purpose/Intended Use:
 *    This structure contains all private data for the LLC Snap
 *    CC read from the System Registry.
 *
 *****************************************************************************/
typedef struct ix_cc_llc_snap_encap_registry_data
{
    ix_uint32 numberOfNextHopEntries; /* number of entries in the Next Hop
                                         Table */
    ix_uint32 mev2NumMask;            /* relevant uEngines IDs */
    ix_uint32 sramChannel;            /* SRAM channel used for the table
                                         allocation*/    
} ix_cc_llc_snap_encap_registry_data_t;



/*
 * Variables global for the whole LLC Snap Encapsulation Core Component
 */
extern ix_cc_llc_snap_encap_context_t*      pLlcSnapEncapContext;
extern ix_cc_llc_snap_encap_registry_data_t llcSnapEncapRegistryData;



/*****************************************************************************
 * Abstract:
 *    This primitive will be used to retrieve data from the registry 
 *    for a given property type. If there is no registry, then the data
 *    structure will be filled with the defaults from the header file.
 *
 * Parameters:
 *    pPropertyName  - IN:  pointer to the property name string
 *    pPropertyValue - OUT: pointer to the property value
 *
 * Return values:
 *    IX_SUCCESS - data retrieved successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_REGISTRY - failed to access the registry
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_retrieve_registry_data(const char* pPropertyName,
                                            ix_uint32* pPropertyValue);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to retrieve data from the registry for
 *    all LLC Snap Encapsulation static data stored in an
 *    ix_cc_llc_snap_encap_registry_data_t type data structure.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - data retrieved successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_REGISTRY - failed to access the registry
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_registry(void);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to allocate SRAM memory for the table
 *    which base address is to be patched into microcode.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - memory allocated successfully
 *    IX_CC_ERROR_OOM - unable to allocate memory
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_alloc_symbols(void);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to actually patch the symbols into
 *    the LLC Snap Encap microcode.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - symbol patching ended successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_FAILED_PATCHING - patching failed
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_patch_symbols(void);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to register the message handler for the
 *    LLC Snap Encapsulation Core Component.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - message handler registered successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - error from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_register_message_handler(void);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to register the packet handlers for the
 *    LLC Snap Encapsulation Core Component.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - packet handlers registered successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - error from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_register_packet_handlers(void);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to free memory previously allocated for
 *    patching.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - memory allocated successfully
 *    IX_CC_ERROR_OOM - unable to free memory
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_free_symbols(void);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to deregister the message handler used
 *    by the LLC Snap Encapsulation Core Component.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - message handler unregistered successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - error from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_unregister_message_handler(void);


/*****************************************************************************
 * Abstract:
 *    This primitive will be used to deregister the packet handlers used
 *    by the LLC Snap Encapsulation Core Component.
 *
 * Parameters:
 *    None
 *
 * Return values:
 *    IX_SUCCESS - packet handlers unregistered successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - error from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_unregister_packet_handlers(void);


/*****************************************************************************
 * Abstract:
 *    This is an "Add Next Hop" internal callback function. Message support
 *    library calls this function to return the status of the requested
 *    asynchronous operation to the original "async" function caller.
 *
 * Parameters:
 *    result  - IN: result of the asynchronous operation
 *    pContext - IN: context provided by caller
 *    pMsg - IN: return message
 *    msgLength - IN: length of the message
 *
 * Return values:
 *    IX_SUCCESS - all operations ended successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CALLBACK - user callback returns error message
 *
 *****************************************************************************/
ix_error
_ix_cc_llc_snap_encap_cb_add_next_hop(ix_error result,
                                      void* pContext,
                                      void* pMsg,
                                      ix_uint32 msgLength);

    
/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation "Add Next Hop" callback data
 *
 * Purpose/Intended Use:
 *    This is the data structure used in LLC Snap Encap "Add Next Hop"
 *    message/callback operations.
 *
 * Fields:
 *    pUserCallback - pointer to the user callback function
 *    pUserContext  - pointer to the user provided context data (opaque for
 *                    the LLC Snap Encap CC).
 *
 *****************************************************************************/
typedef struct ix_cc_llc_snap_encap_add_next_hop_callback_data
{
    ix_cc_llc_snap_encap_cb_add_next_hop pUserCallback;
    void* pUserContext;
} ix_cc_llc_snap_encap_add_next_hop_callback_data_t;


/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation "Add Next Hop" callback data
 *
 * Purpose/Intended Use:
 *    This is the data structure used in LLC Snap Encap "Add Next Hop"
 *    message/callback operations. It contains all user-provided data
 *    necessary to fulfill the "Add Next Hop" request.
 *
 *****************************************************************************/
typedef struct ix_cc_llc_snap_encap_add_next_hop_data
{
    ix_uint16 nextHop;  /* Next Hop ID                 */
    ix_uint8 encapType; /* encapsulation type bit      */
    ix_uint16 vpi;      /* outgoing VPI for the packet */
    ix_uint16 vci;      /* outgoing VCI for the packet */
    ix_uint16 vcq;      /* VCQ number for the packet   */
    ix_uint8 dSap;      /* LLC Header: DSAP field      */
    ix_uint8 sSap;      /* LLC Header: SSAP field      */
    ix_uint8 ctrl;      /* LLC Header: Ctrl field      */
    ix_uint32 oui;      /* SNAP Header: OUI field      */
    ix_uint16 pid;      /* SNAP Header: PID field      */
} ix_cc_llc_snap_encap_add_next_hop_data_t;


/*****************************************************************************
 * Abstract:
 *    This is a "Delete Next Hop" internal callback function. Message support
 *    library calls this function to return the status of the requested
 *    asynchronous operation to the original "async" function caller.
 *
 * Parameters:
 *    result  - IN: result of the asynchronous operation
 *    pContext - IN: context provided by caller
 *    pMsg - IN: return message
 *    msgLength - IN: length of the message
 *
 * Return values:
 *    IX_SUCCESS - all operations ended successfully
 *    IX_CC_LLC_SNAP_ENCAP_ERROR_CALLBACK - user callback returns error message
 *
 *****************************************************************************/
ix_error
_ix_cc_llc_snap_encap_cb_delete_next_hop(ix_error result,
                                         void* pContext,
                                         void* pMsg,
                                         ix_uint32 msgLength);

    
/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation "Delete Next Hop" callback data
 *
 * Purpose/Intended Use:
 *    This is the data structure used in LLC Snap Encap "Delete Next Hop"
 *    message/callback operations.
 *
 * Fields:
 *    pUserCallback - pointer to the user callback function
 *    pUserContext  - pointer to the user provided context data (opaque for
 *                    the LLC Snap Encap CC).
 *
 *****************************************************************************/
typedef struct ix_cc_llc_snap_encap_delete_next_hop_callback_data
{
    ix_cc_llc_snap_encap_cb_delete_next_hop pUserCallback;
    void* pUserContext;
} ix_cc_llc_snap_encap_delete_next_hop_callback_data_t;


/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation "Delete Next Hop" callback data
 *
 * Purpose/Intended Use:
 *    This is the data structure used in LLC Snap Encap "Delete Next Hop"
 *    message/callback operations. It contains all user-provided data
 *    necessary to fulfill the "Delete Next Hop" request.
 *
 *****************************************************************************/
typedef struct ix_cc_llc_snap_encap_delete_next_hop_data
{
    ix_uint16 nextHop;  /* Next Hop ID */
} ix_cc_llc_snap_encap_delete_next_hop_data_t;


/*****************************************************************************
 * Abstract 
 *    LLC Snap Encapsulation packet lookup result
 *
 * Purpose/Intended Use:
 *    This structure contains lookup results for the LLC Snap
 *    Encapsulation Core Component packet handler.
 *
 *****************************************************************************/
typedef struct ix_cc_llc_snap_encap_pkt_lookup_result 
{
    ix_error result;    
    ix_uint16 nextBlockId;
} ix_cc_llc_snap_encap_pkt_lookup_result_t;


#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

/*****************************************************************************
 *                           TRUE/FALSE defines
 *****************************************************************************/
#ifndef TRUE
#define TRUE  0x1
#endif
#ifndef FALSE
#define FALSE 0x0
#endif

#endif /* IX_CC_LLC_SNAP_ENCAP_PRIVATE_H */

