#ifndef IX_CC_LLC_SNAP_ENCAP_H
#define IX_CC_LLC_SNAP_ENCAP_H
/*****************************************************************************
 *                            Intel Proprietary
 *
 * Copyright (c) 1998-2002 By Intel Corporation.  All rights reserved.  
 * No part of this program or publication may be reproduced, transmitted,
 * transcribed, stored in a retrieval system, or translated into any language
 * or computer language in any form or by any means, electronic, mechanical,
 * magnetic, optical, chemical, manual, or otherwise, without the prior
 * written permission of:
 *                        Intel Corporation
 *                        2200 Mission College Blvd.
 *                        Santa Clara, CA  95052-8119
 *****************************************************************************/

/**
 *****************************************************************************
 * @ingroup LlcSnapEncap
 *
 * File:  ix_cc_llc_snap_encap.h
 *
 * @description
 *      This file defines all exported data types and function prototypes
 *      for the LLC Snap Encapsulation Core Component.
 *
 *****************************************************************************/

/*****************************************************************************
 *
 * Contents:
 *        Definitions of following exported data:
 *            llc_snap_encap_next_hop_entry_t
 *            ix_cc_llc_snap_encap_cb_add_next_hop
 *            ix_cc_llc_snap_encap_cb_delete_next_hop
 *
 *        Definitions of following exported functions:
 *            ix_cc_llc_snap_encap_async_add_next_hop
 *            ix_cc_llc_snap_encap_async_delete_next_hop
 *            ix_cc_llc_snap_encap_add_next_hop
 *            ix_cc_llc_snap_encap_delete_next_hop
 *            ix_cc_llc_snap_encap_init
 *            ix_cc_llc_snap_encap_fini
 *            ix_cc_llc_snap_encap_msg_handler
 *            ix_cc_llc_snap_encap_pkt_handler
 *
 *        Definitions of following local data:
 *
 *        Definitions of following local functions:
 *
 *****************************************************************************/

#include "ix_types.h"
#include "ix_error.h"
#include "ix_ossl.h"
#include "ix_cc.h"
#include "ix_cc_error.h"

#include "bindings.h"


#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */


/**
 *****************************************************************************
 * @defgroup LlcSnapEncap
 *   LLC Snap Encapsulation Core Component
 * @purpose
 *   This group contains all definitions of API and data structures exported
 *   by the LLC Snap Encapsulation Core Component.
 * 
 *****************************************************************************/

/* Encapsulation types */
#define ENCAP_LLC   0
#define ENCAP_VCMUX 1

/* default table filling for VCMUX encapsulation */
#define VCMUX_FILL 0xFFFFFFFF

/**
 *****************************************************************************
 * @ingroup LlcSnapEncap
 *    Next Hop table entry structure
 * @description
 *    These structures hold a single entry in the Next Hop table used by
 *    the LLC Snap Encapsulation microblock.
 *
 *****************************************************************************/
typedef struct llc_snap_llc_header
{
    /**< LLC/Snap Encapsulation header */
    ix_uint8 dSap;        /**< LW0 [ 0: 7] LLC header: DSAP field        */
    ix_uint8 sSap;        /**< LW0 [ 8:15] LLC header: SSAP field        */
    ix_uint32 ctrl   :8;  /**< LW0 [16:23] LLC header: Ctrl field        */
    ix_uint32 oui   :24;  /**< LW0 [24:31] 
                             LW1 [ 0:15] SNAP header: Organizationally
                             Unique Identifier (OUI) field             */
    ix_uint16 pid;        /**< LW1 [16:31] SNAP header: Protocol
                             Identifier (PID) field                    */    
} __attribute__ ((packed)) llc_snap_llc_header_t;

typedef struct llc_snap_mux_header
{
    ix_uint32 lw0; /**< LW0 [ 0:31] VCMUX 0xFFFFFFFF default filling */ 
    ix_uint32 lw1; /**< LW1 [ 0:31] VCMUX 0xFFFFFFFF default filling */ 
} __attribute__ ((packed)) llc_snap_mux_header_t;

typedef struct llc_snap_encap_next_hop_entry
{
    ix_uint32 reserved1  : 8; /**< [24:31] reserved == 0               */
    ix_uint32 vci        :12; /**< [12:23] outgoing VCI for the packet */
    ix_uint32 vpi        :12; /**< [ 0:11] outgoing VPI for the packet */

    ix_uint16 reserved2;      /**< reserved == 0                       */
    ix_uint16 vcq;            /**< VCQ number for the packet           */

    union
    {
        llc_snap_llc_header_t llc_snap;
        llc_snap_mux_header_t vcmux;        
    } encap_header;
    
} __attribute__ ((packed)) llc_snap_encap_next_hop_entry_t;



/******************************************************************************
 *
 * TYPE DEFINITIONS FOR CALLBACK FUNCTIONS
 *
 ******************************************************************************/
typedef ix_error (*ix_cc_llc_snap_encap_cb_add_next_hop) (ix_error arg_Result,
                                                          void* arg_pContext);

typedef ix_error (*ix_cc_llc_snap_encap_cb_delete_next_hop) (ix_error arg_Result,
                                                             void* arg_pContext);



/******************************************************************************
 *
 * LLC SNAP ENCAPSULATION CORE COMPONENT MESSAGE HELPER API
 *
 ******************************************************************************/

/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * Add a new Next Hop entry to the table
 * @description
 *    This message helper function builds and sends an "Add Next Hop"
 *    message for the LLC Snap Encapsulation core component. All the user
 *    provided data are sent along to fill the entry in the Next Hop table.
 *
 * @param nextHop - IN: Next Hop ID
 * @param vpi - IN: outgoing VPI for the packet
 * @param vci - IN: outgoing VCI for the packet
 * @param vcq - IN: VCQ number for the packet
 * @param dSap - IN: LLC header: DSAP field
 * @param sSap - IN: LLC header: SSAP field
 * @param ctrl - IN: LLC header: Ctrl field
 * @param oui - IN: SNAP header: Organizationally Unique Identifier (OUI) field
 * @param pid - IN: SNAP header: Protocol Identifier (PID) field   
 * @param pUserCallback - IN: pointer to the user callback function
 * @param pUserContext - IN: pointer to the user provided context data (opaque
 *                       for the LLC Snap Encap CC).
 *
 * @retval IX_SUCCESS - message was successfully built and sent 
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM - invalid pointer
 *                                                          in input parameters
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_MSG_LIBRARY - failed sending a message 
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_NOT_INITIALIZED - Core Component not
 *                                                      initialized yet
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_async_add_next_hop(
    ix_uint16 nextHop,
    ix_uint8 encapType,
    ix_uint16 vpi,
    ix_uint16 vci,
    ix_uint16 vcq,
    ix_uint8 dSap,
    ix_uint8 sSap,
    ix_uint8 ctrl,
    ix_uint32 oui,
    ix_uint16 pid,
    ix_cc_llc_snap_encap_cb_add_next_hop pCallback,
    void *pUserContext);


/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * Delete an entry in the Next Hop table
 * @description
 *    This message helper function builds and sends a "Delete Next Hop"
 *    message for the LLC Snap Encapsulation core component. All the user
 *    provided data are sent along to find the entry in the Next Hop table.
 *
 * @param nextHop - IN: Next Hop ID
 * @param pUserCallback - IN: pointer to the user callback function
 * @param pUserContext - IN: pointer to the user provided context data (opaque
 *                       for the LLC Snap Encap CC).
 *
 * @retval IX_SUCCESS - message was successfully built and sent 
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM - invalid pointer
 *                                                          in input parameters
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_MSG_LIBRARY - failed sending a message 
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_NOT_INITIALIZED - Core Component not
 *                                                      initialized yet
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_async_delete_next_hop(
    ix_uint16 nextHop,
    ix_cc_llc_snap_encap_cb_delete_next_hop pCallback,
    void *pUserContext);


/******************************************************************************
 *
 * LLC SNAP ENCAPSULATION CORE COMPONENT LIBRARY API
 *
 ******************************************************************************/

/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * Add a new Next Hop entry to the table
 * @description
 *    This library function actually fills the entry in the Next Hop table.
 *    Table index is the nextHop function argument. The CC checks if the
 *    desired entry is empty; if it is - the entry is filled with received
 *    data; if it is not - proper error code is returned.
 *
 * @param nextHop - IN: Next Hop ID
 * @param vpi - IN: outgoing VPI for the packet
 * @param vci - IN: outgoing VCI for the packet
 * @param vcq - IN: VCQ number for the packet
 * @param dSap - IN: LLC header: DSAP field
 * @param sSap - IN: LLC header: SSAP field
 * @param ctrl - IN: LLC header: Ctrl field
 * @param oui - IN: SNAP header: Organizationally Unique Identifier (OUI) field
 * @param pid - IN: SNAP header: Protocol Identifier (PID) field   
 *
 * @retval IX_SUCCESS - entry successfully added
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM - invalid input
 *                                                          parameter(s)
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_DUPLICATED - entry already exists
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_NOT_INITIALIZED - Core Component not
 *                                                      initialized yet
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_add_next_hop(ix_uint16 nextHop,
                                  ix_uint8 encapType,
                                  ix_uint16 vpi,
                                  ix_uint16 vci,
                                  ix_uint16 vcq,
                                  ix_uint8 dSap,
                                  ix_uint8 sSap,
                                  ix_uint8 ctrl,
                                  ix_uint32 oui,
                                  ix_uint16 pid);


/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * Delete an entry in the Next Hop table
 * @description
 *    This library function actually deletes the entry in the Next Hop table.
 *    Table index is the nextHop function argument. The CC checks if the
 *    desired entry exists; if it does - the entry is cleared (thus removed);
 *    if it does not - proper error code is returned.
 *
 * @param nextHop - IN: Next Hop ID
 *
 * @retval IX_SUCCESS - message was successfully built and sent 
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM - invalid input
 *                                                          parameter
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_NOT_FOUND - such entry does not exist 
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_NOT_INITIALIZED - Core Component not
 *                                                      initialized yet
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_delete_next_hop(ix_uint16 nextHop);


/******************************************************************************
 *
 * LLC SNAP ENCAPSULATION CORE COMPONENT SYSTEM FUNCTIONS
 *
 ******************************************************************************/

/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * LLC Snap CC main initializaton function
 * @description
 *    This routine is the main initialization function for the LLC Snap
 *    Encapsulation Core Component. It performs memory allocations,
 *    handlers registration and all other activities necessary to start
 *    the Core Component's regular activity.
 *
 * @param hCcHandle - IN: handle to the core component, created by the core
 *                    component infrastructure; this should be used later to
 *                    get other services from the core component infrastructure.
 * @param ppContext - OUT: location where the pointer to the control block
 *                    allocated by the core component will be stored. The
 *                    control block is internal to the CC and contains
 *                    information about LLC Snap Encap internal data structures,
 *                    allocated memory and other relevant information. Later
 *                    this pointer is to be passed into the
 *                    ix_cc_llc_snap_encap_fini function for freeing memory and
 *                    other housekeeping operations when the core component
 *                    is being destroyed.  
 *
 * @retval IX_SUCCESS - initialization successful
 * @retval IX_CC_ERROR_NULL - input parameter arg_ppContext is null
 * @retval IX_CC_ERROR_OOM - memory allocation failure      
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_REGISTRY - registry access failure
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - failure from CCI 
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_FAILED_PATCHING - patching failure
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_init(ix_cc_handle hCcHandle,
                          void** ppContext);


/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * LLC Snap CC main shutdown function
 * @description
 *    This routine is the main shutdown function for the LLC Snap
 *    Encapsulation Core Component. It performs memory deallocation,
 *    handlers deregistration and all other activities necessary to gracefully
 *    stop the Core Component's regular activity.
 *
 * @param hCcHandle - IN: handle to the LLC Snap Encap core component
 * @param pContext - IN: pointer to the control block memory allocated 
 *                   earlier in the_init function. 
 *
 * @retval IX_SUCCESS - shutdown successful
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM - invalid input
 *                                                          parameter(s)
 * @retval IX_CC_ERROR_OOM - failure of memory deallocation
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_CCI - failure from CCI
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_fini(ix_cc_handle hCcHandle,
                          void* pContext);


/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * LLC Snap CC message handler
 * @description
 *    This primitive is the message handler function for LLC Snap Encap core
 *    component. The core component receives messages from other core
 *    components through this message handler function and internally calls
 *    appropriate library function to process the message. This message
 *    handler should be used to update dynamic properties.
 *
 * @param hDataToken - IN: buffer handle embedding information for
 *                     the message passed in userData.
 * @param userData - IN: message type
 * @param pContext - IN: pointer to the context data
 *
 * @retval IX_SUCCESS - message processed successfully
 * @retval IX_CC_ERROR_NULL - null argument
 * @retval IX_CC_ERROR_UNDEFINED_MSG - unsupported message
 * @retval IX_CC_LLC_SANP_ENCAP_ERROR_MSG_LIBRARY - error from message
 *                                                  support library
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_BUFFER_FREE - error from RM, failed
 *                                                  freeing the buffer
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_msg_handler(ix_buffer_handle hDataToken,
                                 ix_uint32 userData,
                                 void* pContext);


/**
 ****************************************************************************
 * @ingroup LlcSnapEncap
 * LLC Snap CC packet handler
 * @description
 *    This is the only registered function to receive exception
 *    packets from the LLC Snap Encap microblock.
 *
 * @param hDataToken - IN: handle to the buffer which contains
 *                     exception packets from the microblock
 * @param exceptionCode - IN: ignored in this case
 * @param pContext - IN: pointer to the context data
 *
 * @retval IX_SUCCESS - packet processed successfully
 * @retval IX_CC_ERROR_INTERNAL - error from CCI
 * @retval IX_CC_ERROR_SEND_FAIL - error sending the packet forth
 * @retval IX_CC_LLC_SNAP_ENCAP_ERROR_INVALID_INPUT_PARAM - invalid input
 *                                                          parameter(s)
 *
 *****************************************************************************/
ix_error
ix_cc_llc_snap_encap_pkt_handler(ix_buffer_handle hDataToken,  
                                 ix_uint32 exceptionCode,
                                 void* arg_pContext);



#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end ifndef IX_CC_LLC_SNAP_ENCAP_H */

