/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2002 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK for the IXP2X00 Network Processor
 *
 * = FILENAME
 *      ix_cc_eth_tx_int.h
 *
 * = DESCRIPTION
 *      This file contains parameters used internally by Ethernet Tx core
 *      component.
 *
 * = CHANGE HISTORY
 *      11/20/2002 - Created.
 *
 * ============================================================================
 * $Id: ix_cc_eth_tx_int.h,v 1.38 2003/10/29 01:52:25 ktseng Exp $
 */

#if !defined(__IX_CC_ETH_TX_INT_H__)
#define __IX_CC_ETH_TX_INT_H__



/**
 * System defined include files required.
 */

#include "ix_types.h"
#include "cc/ix_cc_eth_tx.h"

#include "ix_ossl.h" /* for malloc and free */
#include "cc/ix_cc_l2tm.h"

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)

#if (_IX_BOARD_TYPE_ != _IX_IXDP2401_) && (_IX_BOARD_TYPE_ != _IX_IXDP2801_) && !defined(IX_PLATFORM_2800)
#include "cc/internal/ix_cc_eth_tx_drv_wrapper.h"
#endif

#ifndef BOOL
#define BOOL ix_uint32
#endif

#endif /* _IX_OS_TYPE_ */

#if defined(IX_PLATFORM_2800)
#include "common_types.h"
#include "common_def.h"
#include "bld1110.h"
#include "ixf_api_ether_d.h"
#include "ixf_api_d.h" /* BroadBand Utility Definitions and Function headers */
#include "ixf_api.h"
#include "hal.h"
#endif /* defined(IX_PLATFORM_2800) */

/**
 * User defined include files required.
 */


#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */

#define IX_CC_ETH_TX_L2TM_NAME "DEFAULT"

/* various size definitions for 32bit and 64-bit counter support */
#define IX_CC_ETH_TX_NUM_COUNTERS_PER_PORT    4
#define IX_CC_ETH_TX_TOTAL_NUM_COUNTERS       (IX_CC_ETH_TX_MAX_NUM_PORTS * IX_CC_ETH_TX_NUM_COUNTERS_PER_PORT)
#define IX_CC_ETH_TX_SIZE_32BIT_COUNTER_BUF   (IX_CC_ETH_TX_TOTAL_NUM_COUNTERS * sizeof(ix_uint32))
#define IX_CC_ETH_TX_SIZE_64BIT_COUNTER_HDL   (IX_CC_ETH_TX_TOTAL_NUM_COUNTERS * sizeof(ix_counter_64bit_handle))

/* size hint of l2 table */
#define IX_CC_NUMBER_OF_NODES_PER_PORT        253
#define IX_CC_ETH_TX_L2_TABLE_SIZE_HINT       (IX_CC_NUMBER_OF_NODES_PER_PORT * IX_CC_ETH_TX_MAX_NUM_PORTS)

/* Offset into the IP header for destination IP address */
#define IX_CC_ETH_TX_DEST_IP_OFFSET           16
#define IX_CC_ETH_TX_DEST_IP_OFFSET_IPV6      24
#define IX_CC_ETH_TX_DEST_IP_OFFSET_LAST_32BIT_IPV6 (IX_CC_ETH_TX_DEST_IP_OFFSET_IPV6 + 12)
#define IX_CC_ETH_TX_IPV6_NBR_SOL_MCAST_ADDR  0xFF02 /* IPv6 Nbr Sol MCast Pkt */
#define IX_CC_ETH_TX_MAC_PROT_IPV6            0x86DD /* IPv6 protocol */

/* IP version mask and number */
#define IX_CC_ETH_TX_IP_VER_MASK              0xF0 /* high nibble */
#define IX_CC_ETH_TX_IP_VER_IPV6              6
#define IX_CC_ETH_TX_IP_VER_SFT               4    /* 4-bit shift */

/* size of MAC header */
#define IX_CC_ETH_TX_SIZE_MAC_ADDR            6
#define IX_CC_ETH_TX_SIZE_MAC_PROT            2
#define IX_CC_ETH_TX_SIZE_MAC_HDR             14

/**
 * IP address  table structure
 */
typedef struct ix_s_cc_eth_tx_ip_addr_table
{
    ix_uint32 ipAddr;                                /* IP Address */
    struct ix_s_cc_eth_tx_ip_addr_table *pNextEntry; /* next IP address node */
} ix_cc_eth_tx_ip_addr_table;


/**
 * local interface table structure
 */
typedef struct ix_s_cc_eth_tx_local_interface_table
{
    ix_uint32 portId;                         /* Port Id */
    ix_ether_addr etherAddr;                  /* ethernet MAC address */
    ix_cc_eth_tx_ip_addr_table *pIpAddrTable; /* IP address list */
} ix_cc_eth_tx_local_interface_table;


/**
 * Control block structure
 */
typedef struct ix_s_cc_eth_tx_ctl_blk
{

    /**
     * Handle of the ethernet tx core component, provided by
     * CCI during init function
     */
    ix_cc_handle hEthTxHandle;

    /**
     * Handle of ethernet driver, this gets from driver
     * during initialization and will be used for accessing
     * driver services (callback registeration, IOCTL calls ..).
     */
    unsigned int hEthDriver;

    /* Handle of L2TM */
    ix_cc_l2tm hL2TM;

    /* L2 table manager's symbols; this will be used for patching */
    ix_cc_l2tm_symbols l2TMSymbol;

    /**
     * Local blade id and port id of each ethernet channel.
     * This will be derived from ix_cc_init_context structure.
     */
    ix_uint32 portId[IX_CC_ETH_TX_MAX_NUM_PORTS];
    ix_uint32 localBladeId;

    /**
     * number of channel will be derived from property node
     */
     ix_uint32 numberOfChannels;

    /**
     * MEv2 numbers for Ethernet Tx microblock.
     */
    ix_uint32 mev2NumMask; /*  Tx microblock MEv2 number */


    /**
     * 64-bit counter handler Array.
     */
    ix_counter_64bit_handle ahCounterHandlers64Bit[IX_CC_ETH_TX_TOTAL_NUM_COUNTERS];

    /**
     * base addresses of 32-bit counter memories.
     */
    ix_uint32 *pCounterBase32Bit;

    /**
     * Mutex id used used by the component in critical sections.
     */
    ix_ossl_mutex_t mid;

    /**
     * Local Interface table, this will be used to store port id,MAC address
     * and ip address of a port
     */
    ix_cc_eth_tx_local_interface_table *pInterfaceTable;


    /**
     * User callback for link status notification.
     */
    ix_cc_eth_tx_cb_notify_data userCbData;

    /**
     * Free list handle
     */
    ix_cc_handle hBufferFreeList;

    /**
     * Component initialization flag.
     */
    BOOL initialized;

#if defined(IX_PLATFORM_2401) || defined(IX_PLATFORM_2801)

/* Platform specific - in an ixdp2801 platform we have up to 12 Ethernet ports,
 * but not all ports are always assigned to Ethernet. A number of them could be
 * POS or ATM, so provide the portMask variable to store only Ethernet ports.
 */
#if (_IX_BOARD_TYPE_ == _IX_IXDP2401_) || (_IX_BOARD_TYPE_ == _IX_IXDP2801_)

    ix_uint32 portMask;

#endif
#endif

#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
#if (_IX_BOARD_TYPE_ != _IX_IXDP2401_) && (_IX_BOARD_TYPE_ != _IX_IXDP2801_) && !defined(IX_PLATFORM_2800)
    /**
     * GBE MAC user callback function.
     */
    _ix_cc_eth_tx_drv_cb pGbeMacCallback;
#endif
#endif /* (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_) */

#if defined(IX_PLATFORM_2800)

    /**
     * Pointer to IXFAPI media driver specific handle structures.
     */
    bb_ChipData_t *pChipData;

    /**
     * Pointer to IXF1110 configuration register table.
     */
    InitRegTable_t *ixf1110RegInitTable;

#endif /* (defined IX_PLATFORM_2800) */

} ix_cc_eth_tx_ctl_blk;



/**
 * Callback context entry structure
 */
typedef struct s_ix_cc_eth_tx_cb_entry
{
    void *pUserCtx;     /* user context */
    void *pUserCb;      /* user callback */
} ix_cc_eth_tx_cb_entry;



/**
 * MACRO NAME: IX_CC_ETH_TX_MAC_ADDR_N2H
 *
 * DESCRIPTION: This macro converts the input network-byte order mac
 *              address into host-byte order and place them into
 *              two ix_uint32 output variables.
 *
 * @Param: arg_pMacAddr - IN: pointer to the 6-byte array containing
 *                            the network-byte order MAC address.
 * @Param: arg_LowWord - OUT: 32-bit variable to store bit [31:0] of the
 *                            MAC address in host-byte order.
 * @Param: arg_HighWord - OUT: 32-bit variable to store bit [47:32] of the
 *                            MAC address in host-byte order.
 *
 * @Return:  - None
 */
#define IX_CC_ETH_TX_MAC_ADDR_N2H(                             \
    arg_pMacAddr,                                              \
    arg_LowWord,                                               \
    arg_HighWord)                                              \
{                                                              \
    arg_LowWord = (ix_uint32)(arg_pMacAddr[5]) |               \
                  ((ix_uint32)(arg_pMacAddr[4]) << 8) |        \
                  ((ix_uint32)(arg_pMacAddr[3]) << 16) |       \
                  ((ix_uint32)(arg_pMacAddr[2]) << 24);        \
                                                               \
    arg_HighWord = (ix_uint32)(arg_pMacAddr[1]) |              \
                   ((ix_uint32)(arg_pMacAddr[0]) << 8);        \
}



/**
 * Prototypes for internal interface functions.
 */
void _ix_cc_eth_tx_arp_send_packet(
    ix_buffer_handle);

ix_error _ix_cc_eth_tx_send_packet(
    ix_buffer_handle,
    ix_uint32);

ix_error _ix_cc_eth_tx_unregister_msg_pkt_handlers(void);

ix_error _ix_cc_eth_tx_free_symbols(
    ix_cc_eth_tx_ctl_blk *arg_pCtlBlk);

ix_error _ix_cc_eth_tx_free_local_interface_table(
    ix_cc_eth_tx_ctl_blk *arg_pCtlBlk);

ix_error _ix_cc_eth_tx_delete_64bit_counters(
    ix_cc_eth_tx_ctl_blk *arg_pCtlBlk);

/* Local interface table routines */
ix_error _ix_cc_eth_tx_add_port_ip_address(
    void *arg_pCtlBlk,
    ix_uint32 arg_PortId,
    ix_uint32 arg_IpAddr);

ix_error _ix_cc_eth_tx_del_port_ip_address(
    void *arg_pCtlBlk,
    ix_uint32 arg_PortId,
    ix_uint32 arg_IpAddr);

ix_error _ix_cc_eth_tx_get_port_data(
    void *arg_pCtlBlk,
    ix_uint32 arg_PortId,
    ix_ether_addr *arg_pL2Addr,
    ix_uint32 *arg_pIpAddr);

ix_error _ix_cc_eth_tx_get_port_mac_address(
    void *arg_pCtlBlk,
    ix_uint32 arg_IpAddr,
    ix_ether_addr *arg_pL2Addr);

ix_error _ix_cc_eth_tx_update_port_mac_address(
    void *arg_pCtlBlk,
    ix_uint32 arg_PortId,
    ix_ether_addr *arg_pL2Addr);

ix_error _ix_cc_eth_tx_disp_port_data(
    void *arg_pCtlBlk);

/* Message helper callback functions */
ix_error _ix_cc_eth_tx_cb_get_statistics_info(
    ix_error arg_Result,
    void *arg_pContext,
    void *arg_Msg,
    ix_uint32 arg_MsgLen);

ix_error _ix_cc_eth_tx_cb_get_interface_state(
    ix_error arg_Result,
    void *arg_pContext,
    void *arg_Msg,
    ix_uint32 arg_MsgLen);

ix_error _ix_cc_eth_tx_cb_arp(
    ix_error arg_Result,
    void *arg_pContext,
    void *arg_Msg,
    ix_uint32 arg_MsgLen);

ix_error _ix_cc_eth_get_channel_number(
    ix_cc_eth_tx_ctl_blk *,
    ix_uint32,
    ix_uint32 *);

/* driver callback function */
void _ix_cc_eth_tx_driver_callback(void *arg_pCtlBlk,
                                   ix_uint32 arg_PortId,
                                   ix_uint32 arg_PortStatus);



#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__IX_CC_ETH_TX_INT_H__) */
