///////////////////////////////////////////////////////////////////////
//
//
//                  I N T E L   P R O P R I E T A R Y
//
//     COPYRIGHT [c]  2002 BY  INTEL  CORPORATION.  ALL RIGHTS
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT
//     THE PRIOR WRITTEN PERMISSION OF :
//
//                        INTEL  CORPORATION
//
//                     2200 MISSION COLLEGE BLVD
//
//               SANTA  CLARA,  CALIFORNIA  95052-8119
//
///////////////////////////////////////////////////////////////////////
//
//
//      File Name: statistics_util.uc
//
//      Purpose: Contains macros used by stats.uc.
//
//
///////////////////////////////////////////////////////////////////////

#ifndef	_STATISTICS_UTIL_UC_
#define	_STATISTICS_UTIL_UC_

#include <dl_system.h>
#include <dispatch_loop.uc>

///////////////////////////////////////////////////////////////////////
//
// _stats_init
//
//	 	Description: Sets up next thread signaling, input scratch
//                   ring number and waits for system initialization
//                   signal.
//
//	 	Outputs:
//
//			out_next_thread	:	gpr that contains information to signal
//                              next thread.
//
//			out_scr_get_ring:	Stores incoming scratch ring number
//                              in a gpr.
//
//		Inputs: None.
//
///////////////////////////////////////////////////////////////////////
#macro _stats_init(out_next_thread, out_scr_get_ring, port_mask)
.begin

	; Register used to signal the next thread.

    immed[out_next_thread, (0x80 | (INTER_THD_SIGNAL <<3))]

	; Store input scratch ring number in a register.

	alu_shf[out_scr_get_ring, --, B, STATS_RING_IN, <<2]

	move(drop_queue_number, QM_DROP_QUEUE)
	move(OxFFF, 0xFFF)

	; Wait for system initialization signal.

	.if(ctx()==0)

	#ifdef _DEBUG_COUNTERS_
	; Initialize debug counters
	immed[@stats_num_requests_in, 0]
	immed[@stats_num_requests_out, 0]
	immed[@stats_num_out_nn_ring_full, 0]
	#endif

	.sig volatile system_init_sig
	.addr system_init_sig ME_INIT_SIGNAL
	ctx_arb[system_init_sig]

	.else

	ctx_arb[sig_prev]

	.endif

.end
#endm


///////////////////////////////////////////////////////////////////////
//
// dl_stats_source
//
//	 	Description: Read incoming request from scratch ring.
//
//	 	Outputs:
//			out_scr_get		:	SRAM read xfer registers used to read
//                              the incoming request.
//
//		Inputs:
//			in_scr_get_ring	:	Incoming scratch ring number.
//
///////////////////////////////////////////////////////////////////////
#macro dl_stats_source(out_scr_get, in_scr_get_ring, in_scr_get_sig)
.begin

	; Fetch incoming request from scratch ring.

	scratch[get, out_scr_get[0], 0, in_scr_get_ring, 3], \
		sig_done[in_scr_get_sig]

.end
#endm


///////////////////////////////////////////////////////////////////////
//
// dl_stats_sink
//
//	 	Description: Write outgoing request to nn ring.
//
//	 	Outputs: None
//
//		Inputs:
//			in_data0, 1, 2:	GPR's that contain outgoing request
//                          to write to the nn ring.
//
///////////////////////////////////////////////////////////////////////
#macro dl_stats_sink(in_data0, in_data1, in_data2)
.begin
	
	br_inp_state[NN_FULL, nn_ring_full#]

nn_ring_not_full#:

	; Write outgoing request to nn ring.

	#ifdef _DEBUG_COUNTERS_
	alu[@stats_num_requests_out, @stats_num_requests_out, +, 1]
	#endif

	#ifndef INGRESS_A
	alu[*n$index++, --, b, in_data0]
	#endif

	alu[*n$index++, --, b, in_data1]

	#ifndef INGRESS_A
	alu[*n$index++, --, b, in_data2]
	#endif

	br[end#]

nn_ring_full#:

	#ifdef _DEBUG_COUNTERS_
	alu[@stats_num_out_nn_ring_full, @stats_num_out_nn_ring_full, +, 1]
	#endif

nn_ring_full_loop#:
	br_inp_state[NN_FULL, nn_ring_full_loop#]

	br[nn_ring_not_full#]

end#:

.end
#endm


///////////////////////////////////////////////////////////////////////
//
// _stats_compute_packet_cell_count
//
//	 	Description: Compute cell count for the entire packet.
//
//	 	I/O:
//			io_data0		:	GPR that contains information needed
//                              to compute packet cell count. The same
//                              gpr is then used to store the computed
//                              packet cell count.
//
//		Inputs:
//			in_data1, 2		:	GPR's that contain sop and eop
//                              buffer handle (and cell counts).
//
///////////////////////////////////////////////////////////////////////
#macro _stats_compute_packet_cell_count(io_data0, in_data1, in_data2)
.begin

	.reg packet_cell_count
		 
	; Compute sop buffer cell count.

	.reg sop_cell_count
	alu[sop_cell_count, 0x3F, and, in_data1, >>24]
	alu[packet_cell_count, sop_cell_count, +, 1]

	; Compute mop and eop buffer cell counts, if the buffers exist.

	.if (in_data2 != DL_BUF_NULL)

		; mop_eop_buf_size = sum of all mop buffers + eop buffer size.
		; It is obtained by packet_size - sop_buffer_size

		.reg mop_eop_buf_size
		ld_field_w_clr[mop_eop_buf_size, 0011, io_data0, >>16]

		; Compute number of mop buffers.
		; num_mop_buffers = floor( (mop_eop_buf_size - 1)/BUFFER_SIZE )

		.reg num_mop_buffers
		alu[mop_eop_buf_size, mop_eop_buf_size, -, 1]
		alu[num_mop_buffers, --, b, mop_eop_buf_size, >>PO2_BUFFER_SIZE]

		; Compute total mop buffer cell count.
		; mop_cell_count = num_mop_buffers * NUM_CELLS_IN_MOP_BUFFER
		; NUM_CELLS_IN_MOP_BUFFER is a compile time constant.

		.reg mop_cell_count
		multiply(mop_cell_count, num_mop_buffers, NUM_CELLS_IN_MOP_BUFFER)
		alu[packet_cell_count, packet_cell_count, +, mop_cell_count]

		; Compute eop buffer cell count.

		.reg eop_cell_count
		alu[eop_cell_count, 0x3F, and, in_data2, >>24]
		alu[eop_cell_count, eop_cell_count, +, 1]
		alu[packet_cell_count, packet_cell_count, +, eop_cell_count]

	.endif

	; Copy packet cell count to nn ring gpr.

	ld_field[io_data0, 1100, packet_cell_count, <<16]

.end
#endm


///////////////////////////////////////////////////////////////////////
//
// _stats_enq_buffer_to_freelist
//
//	 	Description: If a buffer cell count is 0, enqueue it 
//					 to the buffer freelist.
//
//		Inputs:
//			in_buf_handle	:	Buffer handle to enqueue to the
//								buffer freelist.
//
///////////////////////////////////////////////////////////////////////
#macro _stats_enq_buffer_to_freelist(in_sop_buffer_handle, in_eop_buffer_handle)
.begin

	.reg buffer_handle eop_buffer_handle

	xbuf_alloc($meta, 1, read)

	.sig meta_sig


	; Mask off the cell count and sop/eop bits from sop and eop buffer handles.

	alu[eop_buffer_handle, in_eop_buffer_handle, AND~, 0xFF, <<24]

	alu[buffer_handle, in_sop_buffer_handle, AND~, 0xFF, <<24]


	; Free all buffers belonging to the packet in a while loop.

	.while (buffer_handle != eop_buffer_handle)

	; Read the next buffer handle to free into $meta[0].

	dl_meta_load_cache(buffer_handle, $meta, meta_sig, 0, 1)

	; Free a buffer.

	#ifdef FREELIST_MANAGER

	; Put the buffer handle on the drop ring.

	alu[$drop, --, B, buffer_handle]
	alu[drop_ring_num, --, b, DROP_RING, <<2]
	scratch[put, $drop, 0, drop_ring_num, 1], sig_done[drop_sig]

	ctx_arb[meta_sig, drop_sig]

	#else

	; Wait for the meta read to complete before
	; issuing an sram enqueue i.e., dl_buf_free
	; Because sram read and sram enqueue are not ordered

	ctx_arb[meta_sig]

	; Enqueue the buffer handle to the buffer freelist.

	dl_buf_free(buffer_handle, BUF_FREE_LIST0)


	#endif

	; Copy $meta[0] into 'buffer_handle'.

	alu[buffer_handle, $meta[0], AND~, 0xFF, <<24]

	.endw

	; Free the eop buffer.

	#ifdef FREELIST_MANAGER

	; Put the buffer handle on the drop ring.

	alu[$drop, --, B, buffer_handle]
	alu[drop_ring_num, --, b, DROP_RING, <<2]
	scratch[put, $drop, 0, drop_ring_num, 1], sig_done[drop_sig]

	ctx_arb[drop_sig]

	#else

	; Enqueue the buffer handle to the buffer freelist.

	dl_buf_free(buffer_handle, BUF_FREE_LIST0)

	#endif


	xbuf_free($meta)

.end
#endm

#endif // _STATISTICS_UTIL_UC_