///////////////////////////////////////////////////////////////////////////////
//
//
//                  I N T E L   P R O P R I E T A R Y
//
//     COPYRIGHT [c]  2002 BY  INTEL  CORPORATION.  ALL RIGHTS
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT
//     THE PRIOR WRITTEN PERMISSION OF :
//
//                        INTEL  CORPORATION
//
//                     2200 MISSION COLLEGE BLVD
//
//               SANTA  CLARA,  CALIFORNIA  95052-8119
//
///////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: statistics.uc
//
//      Purpose: This microblock computes pipeline statistics (future work).
//				 It also computes packet cell count for the scheduler microcblock.
//               It also frees buffers belonging to dropped packets.
//
//      History:
//
//      Date            Comment                         By
//      ---------------------------------------------------------------------
//
//		11/11/2002		Created							asv
//
/////////////////////////////////////////////////////////////////////////////

#ifndef	_STATISTICS_UC_
#define	_STATISTICS_UC_

#include <dl_system.h>
#include <dispatch_loop.uc>
#include <statistics_util.uc>

// CSIX cell payload size in bytes.
#define			CELL_SIZE					120

// Compute number of CSIX cells each mop buffer would contain.
// NUM_CELLS_IN_MOP_BUFFER = ceiling(BUFFER_SIZE/CELL_SIZE).

#if ((BUFFER_SIZE % CELL_SIZE) == 0)
#define_eval 	NUM_CELLS_IN_MOP_BUFFER		(BUFFER_SIZE / CELL_SIZE)
#else
#define_eval 	NUM_CELLS_IN_MOP_BUFFER		(BUFFER_SIZE / CELL_SIZE) + 1
#endif

// Compute BUFFER_SIZE as a power of 2.
#define_eval	PO2_BUFFER_SIZE				LOG2(BUFFER_SIZE)

//---------------------------------------------------------------------
// Register and signal definitions.
//---------------------------------------------------------------------

	#ifdef _DEBUG_COUNTERS_
	.reg @stats_num_requests_in			; Number of incoming requests
	.reg @stats_num_requests_out		; Number of outgoing requests
	.reg @stats_num_out_nn_ring_full	; Number of outgoing nn ring full hits
	#endif

	; Used for next thread signaling.
	.reg next_thread

	; Used to store the incoming scratch ring number.
	.reg scr_get_ring

	; Used to read incoming request from previous microblock,
	xbuf_alloc($scr_get, 3, read)

	; Used to compute and write outgoing request to next microblock.
	.reg data0 data1 data2

	; Used to signal incoming scratch request on scratch ring.
	.sig scr_get_sig

	; Used to store drop queue number.
	.reg drop_queue_number

	; Used to store queu number.
	.reg queue_number

	#ifdef PACKET_MODE
	; Used to store port number.
	.reg port_number
	#endif

	.reg OxFFF
	
	#ifdef FREELIST_MANAGER
	.reg drop_ring_num
	.reg write $drop
	.sig drop_sig
	#endif


//---------------------------------------------------------------------
// Initialization phase.
//---------------------------------------------------------------------

	; Setup gpr's and wait for system initialization.
	; Note: Q-array drop queue initialization is done by the queue 
	; manager ME during its initialization phase.
	
	_stats_init(next_thread, scr_get_ring, port_mask)

	; Read incoming request from the previous microblock.

	dl_stats_source($scr_get, scr_get_ring, scr_get_sig)

//---------------------------------------------------------------------
// Recurring phase.
//---------------------------------------------------------------------

loop#:

	; Signal next thread.

	signal_ctx(next_thread)

	; Wait for the incoming request and next thread signals.

	ctx_arb[scr_get_sig, sig_prev]

	; Copy incoming request into gpr's.

	alu[data0, --, B, $scr_get[0]]	; mop + eop buffer size and queue number.
	alu[data1, --, B, $scr_get[1]]	; sop buffer handle.
	alu[data2, --, B, $scr_get[2]]	; eop buffer handle.

	; Read incoming request from the previous microblock for next iteration.

	dl_stats_source($scr_get, scr_get_ring, scr_get_sig)

	; Check if we got a valid incoming request.

	alu[--, data0, -, 0]
	beq[loop#]

	; Valid incoming request.

	#ifdef _DEBUG_COUNTERS_
	alu[@stats_num_requests_in, @stats_num_requests_in, +, 1]
	#endif

	#ifdef PACKET_MODE

	; Extract port number and queue number.

	alu[queue_number, OxFFF, AND, data0]
	alu[port_number, 0xF, AND, data0, >>12]

	#else

	; Extract queue number.

	alu[queue_number, 0, +16, data0]

	#endif

	; Check if the packet belongs to the drop queue.

	alu[--, queue_number, -, drop_queue_number]
	beq[drop_queue_packet#]
		
	; Normal packet.

	#ifndef PACKET_MODE

	; Compute packet cell count.

	_stats_compute_packet_cell_count(data0, data1, data2)

	#else

	; LW0 is packet len [15:0]

	alu[data0, --, B, data0, >>16]
			
	; LW1 (data1) is still the SOP handle.

	; LW2 is port number [31:16] and queue number [15:0]

	alu[data2, queue_number, OR, port_number, <<16]

	#endif

	; Write outgoing request to the next microblock.

	dl_stats_sink(data0, data1, data2)

	br[loop#]


drop_queue_packet#:

	; This packet has been assigned to the drop queue.		
	; We need to drop this packet by freeing all its buffers.

	; Enqueue all packet buffers to the buffer freelist.

	_stats_enq_buffer_to_freelist(data1, data2)

	br[loop#]

#endif // _STATISTICS_UC_