/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2001 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2000 Network Processor, Release 6
 *
 * = FILENAME
 *      ix_cc_stkdrv_common.h
 *
 * = DESCRIPTION
 *      Header file for common data structures for the Stack Driver CC Module
 *      and its communication handlers (VIDD, TM, etc.).
 *
 * = AUTHOR
 *      Aaron Luk
 *      aaron.luk@intel.com
 *
 * = CHANGE HISTORY
 *       6/19/2002 - initial revision
 *
 * ============================================================================
 * $Id: ix_cc_stkdrv_common.h,v 1.49 2003/11/01 02:47:43 rranjeet Exp $
 */
#if !defined(__IX_CC_STKDRV_COMMON_H__)
#define __IX_CC_STKDRV_COMMON_H__

#if defined(IX_PLATFORM_2401) || defined(IX_PLATFORM_2801)
/* Improvement
 * Cleanup in headers - don't include user space stdio.h, use <> when including
 * headers from global include paths.
 */
#include <ix_ossl.h>
#include <ix_rm.h>

#include <ix_cc_error.h>
#include <ix_cc.h>
#include <ix_cc_properties.h>

#include <cc/ix_cc_msup.h>

#else /* IX_PLATFORM_2x01 */

#include "ix_cc_error.h"
#include <stdio.h>
#include "ix_ossl.h"
#include "ix_rm.h"
#include "ix_cc.h"
#include "ix_cc_properties.h"

#include "cc/ix_cc_msup.h"

#endif /* IX_PLATFORM_2x01 */

#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */

/**
 * Pre-processor symbol and macro definitions.
 */

/* Compile in MUX stubs for VxSim-Lite only. */
#if (defined(IX_TARGET_SIMNTgnu_Debug) || defined(IX_TARGET_SIMNTgnu))
    #define MUX_STUB
#endif /* (defined(IX_TARGET_SIMNTgnu_Debug) || defined(IX_TARGET_SIMNTgnu)) */

/* Comment out this line in order to disable packet classification. */
/*#define IX_CC_STKDRV_PKT_CLASS*/

/* Comment out this line in order to disable transport module support. */
#define IX_CC_STKDRV_TRANSPORT_MODULE

/* Comment out this line in order to disable packet dumps for debugging. */
/*#define IX_CC_STKDRV_DUMP_PACKET*/

/**************************************************************************** 
 * Initialization items
 * IX_CC_STKDRV_DEV_NAME
 *                      - The string name of this driver.  This is the name
 *                        the driver will use to register with the kernel
 * IX_CC_STKDRV_DEV_NAME_LEN
 *                      - The length of IX_CC_STKDRV_DEV_NAME
 */
#define IX_CC_STKDRV_DEV_NAME     "ixdev"
#define IX_CC_STKDRV_DEV_NAME_LEN 8

/* Number of characters in an IP address string, i.e. "XXX.XXX.XXX.XXX" */
#define IX_CC_STKDRV_IP_ADDR_STR_LEN 16

/* Minimum buffer size for stack driver. */
#define IX_CC_STKDRV_MIN_BUFFER_SIZE 2048

/* Offset in longwords for destination IP address in IP header. */
#define IX_CC_STKDRV_IPV4_DST_IP_OFFSET 4

/**
 * MACRO NAME: _IX_CC_STKDRV_DEBUG_ERROR_CG
 *
 * DESCRIPTION: Wrapper for IX_ERROR_CG to only be used in debug mode.
 *
 * @Param:  - IN __expr - expression to be evaluated.
 * @Param:  - OUT __err - the return error code.
 * @Param:  - IN __label - the label to go to on error.
 *
 * @Return: 0 if successful or a valid ix_error token on failure. 
 */
#if defined(IX_DEBUG)
    #define _IX_CC_STKDRV_DEBUG_ERROR_CG(__expr, __err, __label) \
        IX_ERROR_CG(__expr, __err, __label)
#else
    #define _IX_CC_STKDRV_DEBUG_ERROR_CG(__expr, __err, __label) \
        __expr
#endif /* defined(IX_DEBUG) */

/**
 * MACRO NAME: _IX_CC_STKDRV_DEBUG_ERROR_CGT
 *
 * DESCRIPTION: Wrapper for IX_ERROR_CGT to only be used in debug mode.
 *
 * @Param:  - IN __expr - expression to be evaluated.
 * @Param:  - OUT __err - the return error code.
 * @Param:  - IN __code - the translated error code to return.
 * @Param:  - IN __level - the translated error level to return.
 * @Param:  - IN __label - the label to go to on error.
 *
 * @Return: 0 if successful or a valid ix_error token on failure. 
 */
#if defined(IX_DEBUG)
    #define _IX_CC_STKDRV_DEBUG_ERROR_CGT(__expr, __err, __code, __level, __label) \
        IX_ERROR_CGT(__expr, __err, __code, __level, __label)
#else
    #define _IX_CC_STKDRV_DEBUG_ERROR_CGT(__expr, __err, __code, __level, __label) \
        __expr
#endif /* defined(IX_DEBUG) */

/**
 * MACRO NAME: _IX_CC_STKDRV_DEBUG_ERROR_CRT
 *
 * DESCRIPTION: Wrapper for IX_ERROR_CGT to only be used in debug mode.
 *
 * @Param:  - IN __expr - expression to be evaluated.
 * @Param:  - IN __code - the translated error code to return.
 * @Param:  - IN __level - the translated error level to return.
 *
 * @Return: 0 if successful or a valid ix_error token on failure. 
 */
#if defined(IX_DEBUG)
    #define _IX_CC_STKDRV_DEBUG_ERROR_CRT(__expr, __code, __level) \
        IX_ERROR_CRT(__expr, __code, __level)
#else
    #define _IX_CC_STKDRV_DEBUG_ERROR_CRT(__expr, __code, __level) \
        __expr
#endif /* defined(IX_DEBUG) */

/**
 * Type definitions.
 */

/* Stack Driver-specific error codes. */
enum ix_cc_stkdrv_error
{
    IX_CC_STKDRV_ERROR_FIRST = IX_ERROR_MAKE_GROUP(CC_STKDRV),
    IX_CC_STKDRV_VIDD_ERROR_MUX = IX_CC_STKDRV_ERROR_FIRST,
    IX_CC_STKDRV_VIDD_ERROR_INVALID_PROPERTY,
    IX_CC_STKDRV_VIDD_ERROR_INVALID_MSG_ID,
    IX_CC_STKDRV_ERROR_CANNOT_CLASSIFY_PKT,
    IX_CC_STKDRV_ERROR_HANDLER_RECV,
    IX_CC_STKDRV_ERROR_CANNOT_CLASSIFY_OUTPUT_ID,
    IX_CC_STKDRV_ERROR_INVALID_BUF_TYPE,
    IX_CC_STKDRV_ERROR_HANDLER_INIT,
    IX_CC_STKDRV_ERROR_HANDLER_FINI,
    IX_CC_STKDRV_ERROR_REPOSITORY_OPEN,
    IX_CC_STKDRV_ERROR_REPOSITORY_READ,
    IX_CC_STKDRV_ERROR_REPOSITORY_CLOSE,
    IX_CC_STKDRV_ERROR_CCI_ADD_HANDLER,
    IX_CC_STKDRV_ERROR_RM_ADD_HANDLER,
    IX_CC_STKDRV_ERROR_CCI_REMOVE_HANDLER,
    IX_CC_STKDRV_ERROR_RM_REMOVE_HANDLER,
    IX_CC_STKDRV_ERROR_INVALID_BLADE_ID,

    /* Error codes specific to packet classification. */
    #if defined(IX_CC_STKDRV_PKT_CLASS)
        IX_CC_STKDRV_ERROR_FILTER_INIT,
        IX_CC_STKDRV_ERROR_FILTER_FINI,
        IX_CC_STKDRV_ERROR_OUT_OF_FILTERS,
        IX_CC_STKDRV_ERROR_INVALID_FILTER,
    #endif /* defined(IX_CC_STKDRV_PKT_CLASS) */
    IX_CC_STKDRV_VIDD_ERROR_KERNEL,
    IX_CC_STKDRV_ERROR_LAST
};

/**
 * TYPENAME: ix_cc_stkdrv_packet_type
 * 
 * DESCRIPTION: Used to label a packet's destination - either to the local
 * legacy stacks, local NPF stacks, remote legacy stacks, or remote NPF
 * stacks. This data type is used by the CC Module, the communication
 * handlers, and the packet classifier.
 *
 */
typedef enum ix_e_cc_stkdrv_packet_type
{
    IX_CC_STKDRV_PACKET_TYPE_FIRST = 0,
    IX_CC_STKDRV_PACKET_TYPE_LOCAL_LEGACY = IX_CC_STKDRV_PACKET_TYPE_FIRST,
    IX_CC_STKDRV_PACKET_TYPE_LOCAL_NPF,
    IX_CC_STKDRV_PACKET_TYPE_REMOTE_LEGACY,
    IX_CC_STKDRV_PACKET_TYPE_REMOTE_NPF,
    IX_CC_STKDRV_PACKET_TYPE_UNKNOWN,
    IX_CC_STKDRV_PACKET_TYPE_LAST
} ix_cc_stkdrv_packet_type;

/**
 * TYPENAME: ix_cc_stkdrv_handler_id
 * 
 * DESCRIPTION: Identification for OS dependent packet handler.
 *
 */
typedef enum ix_e_cc_stkdrv_handler_id {
    IX_CC_STKDRV_HANDLER_ID_FIRST = 0,
    IX_CC_STKDRV_HANDLER_ID_LOCAL_DRIVER = IX_CC_STKDRV_HANDLER_ID_FIRST,
    IX_CC_STKDRV_HANDLER_ID_TRANSP_MODULE,
    IX_CC_STKDRV_HANDLER_ID_UNDEFINED_HANDLER,
    IX_CC_STKDRV_HANDLER_ID_LAST
} ix_cc_stkdrv_handler_id;


/**
 * TYPENAME: ix_cc_stkdrv_virtual_if
 * 
 * DESCRIPTION: This structure is the internal representation of virtual
 * interfaces existing on the forwarding plane.  This structure
 * represents a node in the linked list of available virtual interfaces.
 *
 */
typedef struct ix_s_cc_stkdrv_virtual_if ix_cc_stkdrv_virtual_if;
struct ix_s_cc_stkdrv_virtual_if
{
    ix_uint8    name[IX_CC_STKDRV_DEV_NAME_LEN]; /* virtual interface name */
    ix_uint8    macAddr[IX_CC_MAC_ADDR_LEN];     /* MAC address of the interface */
    /* Contains IPv4/v6 address, subnet mask, gateway, and broadcast.
     * Also contains the virtual IF ID, IP version.
     */
    ix_cc_ip_properties ipProp;

    /* next interface structure in the list of the structures. */
    ix_cc_stkdrv_virtual_if *pNextVirtualIf;
};

/**
 * TYPENAME: ix_cc_stkdrv_physical_if_info
 * 
 * DESCRIPTION: Structure to contain physical interface properties.
 *
 */
typedef struct ix_s_cc_stkdrv_physical_if_info ix_cc_stkdrv_physical_if_info;
struct ix_s_cc_stkdrv_physical_if_info
{
    /* ID of forwarding plane */
    ix_uint32 fpId;

    /* number of the interface */
    ix_uint32 portId;

    /** 
     * Status of the port - up/down, etc.
     * Modified when an application brings the
     * port up or down using the property API.
     */
    ix_cc_physical_if_status physical_if_status;

    /* Link state. */
    ix_cc_link_status link_status;
    
    /* Media type of the interface */
    ix_media_type mediaType;

    /* MTU for this interface */
    ix_uint16 MTU;

    /* link speed in megabits per second */
    ix_uint32 linkSpeed;

    /* Number of virtual interfaces having IPv4 address currently present */
    ix_uint32 no_virtual_if_with_ipv4_addr;

    /* Number of virtual interfaces having IPv6 address currently present */
    ix_uint32 no_virtual_if_with_ipv6_addr;

    /* list of virtual interfaces associated with this physical interface */
    ix_cc_stkdrv_virtual_if* pVirtualIfs;
};

/**
 * TYPENAME: ix_cc_stkdrv_physical_if_node
 * 
 * DESCRIPTION: This structure represents a node in the linked list of available
 * physical interfaces. 
 *
 */
typedef struct ix_s_cc_stkdrv_physical_if_node ix_cc_stkdrv_physical_if_node;
struct ix_s_cc_stkdrv_physical_if_node 
{
    ix_cc_stkdrv_physical_if_info* pPhysicalIfInfo;
    ix_cc_stkdrv_physical_if_node* pNextPhysicalIf;
};

/**
 * TYPENAME: ix_cc_stkdrv_fp_node
 * 
 * DESCRIPTION: This is a linked list of forwarding plane information structures.
 *
 */
typedef struct ix_s_cc_stkdrv_fp_node ix_cc_stkdrv_fp_node;
struct ix_s_cc_stkdrv_fp_node
{
    ix_uint32 fpId; /*id of the forwarding plane */
    ix_cc_stkdrv_physical_if_node* pPhysicalIfs;    /* list of interfaces for this forwarding plane */
    ix_uint32 numPorts; /* number of ports on this FP. */
    ix_cc_stkdrv_fp_node * pNextFP; /* next ix_cc_stkdrv_fp_info in the list */     
};

/**
 * TYPENAME: ix_cc_stkdrv_cc_module_msg_id
 * 
 * DESCRIPTION: This is an enumerated type identifying message types coming
 * from the VIDD to the CC Module.  It can be expanded as necessary - in
 * this release no messages need be sent from the VIDD to the CC Module,
 * so there are no msg IDs.
 *
 */
typedef enum ix_e_cc_stkdrv_cc_module_msg_id
{
    IX_CC_STKDRV_CC_MODULE_MSG_ID_FIRST = 0,
    IX_CC_STKDRV_CC_MODULE_MSG_ID_LAST
} ix_cc_stkdrv_cc_module_msg_id;

/**
 * TYPENAME: ix_cc_stkdrv_vidd_msg_id
 * 
 * DESCRIPTION: This is an enumerated type identifying message types coming
 * from the CC Module to the VIDD.  It can be expanded as necessary.
 *
 */
typedef enum ix_e_cc_stkdrv_vidd_msg_id
{
    IX_CC_STKDRV_VIDD_MSG_ID_FIRST = 0,
    IX_CC_STKDRV_VIDD_MSG_ID_PROPERTY_UPDATE = IX_CC_STKDRV_VIDD_MSG_ID_FIRST,
    IX_CC_STKDRV_VIDD_MSG_ID_LAST
} ix_cc_stkdrv_vidd_msg_id;

/**
 * TYPENAME: ix_cc_stkdrv_tm_msg_id
 * 
 * DESCRIPTION: Identification for messages sent to the
 * Transport Module from the CC Module.  Currently no
 * messages are supported.
 *
 */
typedef enum ix_e_cc_stkdrv_tm_msg_id {
    IX_CC_STKDRV_TM_MSG_ID_FIRST = 0,
    IX_CC_STKDRV_TM_MSG_ID_LAST
} ix_cc_stkdrv_tm_msg_id;

/**
 * TYPENAME: ix_cc_stkdrv_packet_cb
 * 
 * DESCRIPTION: Callback function is provided to the core component
 * Module to call in order to send packet data to the driver. Each
 * handler module must implement a packet processing function and provide
 * address of the function to the core component Module. 
 *
 * @Param:  - IN ix_buffer_handle arg_hBuffer - handle to the IXA SDK
 *          representations of network packets.
 * @Param:  - IN void* arg_pContext - pointer to the context - depending
 *          on the operation, we can use either the handler context stored
 *          in the handler module structure, or the port-specific context
 *          stored in the physical interface structure.
 * @Param:  - IN ix_cc_stkdrv_packet_type arg_packetType - type of the packet.
 *
 * @Return: IX_SUCCESS, or an appropriate ix_error.
 */
typedef ix_error (*ix_cc_stkdrv_packet_cb)(
                                           ix_buffer_handle arg_hBuffer,
                                           void* arg_pContext,
                                           ix_cc_stkdrv_packet_type arg_packetType
                                           );


/**
 * TYPENAME: ix_cc_stkdrv_msg_str_cb
 * 
 * DESCRIPTION: Callback functions are provided to the core component module
 * in order to send configuration and control messages to the driver. Two
 * types of messages are defined - byte strings and integer values. Each driver
 * module must implement such functions and provide addresses of the functions
 * to the core component Module.
 *
 * @Param:  - IN ix_uint32 arg_msgId - identification of the message.
 * @Param:  - IN const char* arg_pMsg - string of bytes in the message.
 * @Param:  - IN ix_uint32 arg_msgSize - length of the message - represent actual
 *          number of bytes, no addition for the possible null terminator is made.
 * @Param:  - IN void* arg_pContext - pointer to the context stored
 *          in the handler module structure.
 * @Param:  - OUT void** arg_ppReplyMsg - location of pointer to reply message,
 *          if any.
 * @Param:  - OUT ix_uint32* arg_pReplyMsgSize - pointer to size of reply message,
 *          if any.
 *
 * @Return: IX_SUCCESS, or an appropriate ix_error.
 */
typedef ix_error (*ix_cc_stkdrv_msg_str_cb)(
                                            ix_uint32 arg_msgId,
                                            const char* arg_pMsg,
                                            ix_uint32 arg_msgSize,
                                            void* arg_pContext,
                                            void** arg_ppReplyMsg,
                                            ix_uint32* arg_pReplyMsgSize
                                            );

/**
 * TYPENAME: ix_cc_stkdrv_msg_int_cb
 * 
 * DESCRIPTION: Callback functions are provided to the core component module
 * in order to send configuration and control messages to the driver. Two
 * types of messages are defined - byte strings and integer values. Each driver
 * module must implement such functions and provide addresses of the functions
 * to the core component Module.
 *
 * @Param:  - IN ix_uint32 arg_msgId - identification of the message.
 * @Param:  - IN ix_uint32 arg_value - message value.
 * @Param:  - IN void* arg_pContext - pointer to the context context stored
 *          in the handler module structure.
 * @Param:  - OUT void** arg_ppReplyMsg - location of pointer to reply message,
 *          if any.
 * @Param:  - OUT ix_uint32* arg_pReplyMsgSize - pointer to size of reply message,
 *          if any.
 *
 * @Return: IX_SUCCESS, or an appropriate ix_error.
 */
typedef ix_error (*ix_cc_stkdrv_msg_int_cb)(
                                            ix_uint32 arg_msgId,
                                            ix_uint32 arg_value,
                                            void* arg_pContext,
                                            void** arg_ppReplyMsg,
                                            ix_uint32* arg_pReplyMsgSize
                                            );



/**
 * TYPENAME: ix_cc_stkdrv_handler_module_fini_cb
 * 
 * DESCRIPTION: Callback function to finalize a communication handler.
 *
 * @Param:  - IN void* arg_pContext - pointer to the handler module-
 *          specific context.
 *
 * @Return: IX_SUCCESS, or an appropriate ix_error.
 */
typedef ix_error (*ix_cc_stkdrv_handler_module_fini_cb)(
                                                        void *arg_pContext
                                                        );

/**
 * TYPENAME: ix_cc_stkdrv_handler_func
 * 
 * DESCRIPTION: During initialization, each handler object populates the
 * ix_cc_stkdrv_handler_func data structure with function addresses and
 * corresponding contexts.
 *
 */
typedef struct ix_s_cc_stkdrv_handler_func
{
    ix_cc_stkdrv_packet_cb receive_pkt; /*callback - receiving packet */
    ix_cc_stkdrv_msg_str_cb receive_msg_str; /* callback - receiving string message*/
    ix_cc_stkdrv_msg_int_cb receive_msg_int; /* callback - receiving integer msg */
    ix_cc_stkdrv_handler_module_fini_cb fini; /* callback - finalizes handler module. */
    void* pPktContext;  /* pointer to packet context */
    void* pMsgStrContext;   /* pointer to string message context */
    void* pMsgIntContext;   /* pointer to integer message context */
    void* pFiniContext; /* context passed in to fini function of handler */
} ix_cc_stkdrv_handler_func;


/**
 * TYPENAME: ix_cc_stkdrv_handler_module
 * 
 * DESCRIPTION: This data structure describes a registered communication handler
 * and keeps the necessary information in order to identify that handler. In
 * addition, it points to the list of physical interfaces associated with this module.
 *
 */
typedef struct ix_s_cc_stkdrv_handler_module ix_cc_stkdrv_handler_module;
struct ix_s_cc_stkdrv_handler_module
{
    ix_cc_stkdrv_handler_id id; /* id of the handler */
    ix_cc_stkdrv_handler_func* pHandler_func; /* holder of packet and message callbacks */
    ix_cc_stkdrv_handler_module* pNextHandler;  /*next struct in the list of handler modules*/
};

/**
 * TYPENAME: ix_cc_stkdrv_handler_module_init_cb
 * 
 * DESCRIPTION: Callback function to initialize a communication handler.
 *
 * @Param:  - IN ix_buffer_free_list_handle arg_hFreeList - handle to
 *          the freelist from which ix_buffers will be allocated to contain
 *          packet data from the handler module.
 * @Param:  - IN ix_cc_stkdrv_fp_node* arg_pFP - pointer to list of FP data,
 *          to get information for handler-specific initialization.
 * @Param:  - IN void* arg_pInitData - pointer to any additional initialization
 *          data that may be required by a specific handler module.
 * @Param:  - OUT ix_cc_stkdrv_handler_module* arg_pHandlerModule - pointer
 *          to handler module structure to be filled in by this callback.
 *
 * @Return: IX_SUCCESS, or an appropriate ix_error.
 */
typedef ix_error (*ix_cc_stkdrv_handler_module_init_cb)(
                                                        ix_buffer_free_list_handle arg_hFreeList,
                                                        ix_cc_stkdrv_fp_node* arg_pFP,
                                                        void* arg_pInitData,
                                                        ix_cc_stkdrv_handler_module* arg_pHandlerModule
                                                        );

/**
 * TYPENAME: ix_cc_stkdrv_vidd_property_update_msg
 * 
 * DESCRIPTION: Structure to contain property update information between
 *              the stack driver core component and the local VIDD.
 *
 */
typedef struct ix_s_cc_stkdrv_vidd_property_update_msg
{
    /* ID of the properties being updated. */
    ix_uint32 propID;

    /* Pointer to the property info containing updates.. */
    ix_cc_properties* pProperty;

} ix_cc_stkdrv_vidd_property_update_msg;


#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
/**
 * TYPENAME: ix_cc_stkdrv_og_pkt_type
 * 
 * DESCRIPTION: 
 * This is used to label the outgoing pkt type 
 */
typedef enum ix_e_cc_stkdrv_og_pkt_type
{
    IX_CC_STKDRV_OG_PKT_TYPE_FIRST = 0,
    IX_CC_STKDRV_OG_PKT_TYPE_IPV4,
    IX_CC_STKDRV_OG_PKT_TYPE_IPV6,
    IX_CC_STKDRV_OG_PKT_TYPE_UNAVAILABLE,
    IX_CC_STKDRV_OG_PKT_TYPE_UNDEFINED,
    IX_CC_STKDRV_OG_PKT_LAST
} ix_cc_stkdrv_og_pkt_type;
#endif


/**
 * Extern function prototypes.
 */

/**
 * NAME: ix_cc_stkdrv_vidd_init
 *
 * DESCRIPTION: This function is called by the CC Module to initialize the
 * local VIDD and register the VIDD's entry points with the CC Module.
 * 
 * @Param:  - IN ix_buffer_free_list_handle arg_hFreeList - handle to
 *          freelist used to allocated buffers for passing packets down
 *          from the stack.
 * @Param:  - IN ix_cc_stkdrv_fp_node* arg_pFP - pointer to the FP structure
 *          used to get data for all ports.
 * @Param:  - IN void* arg_pInitData - pointer to any additional initialization
 *          data that may be required by a specific handler module - unused
 *          in this case - unused in this case.
 * @Param:  - INOUT ix_cc_stkdrv_handler_module* arg_pHandlerModule -
 *          the function will fill in this structure with its entry points
 *          and contexts.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 *          IX_CC_STKDRV_VIDD_ERROR_MUX
 */
extern ix_error ix_cc_stkdrv_vidd_init(
                                      ix_buffer_free_list_handle arg_hFreeList,
                                      ix_cc_stkdrv_fp_node* arg_pFP,
                                      void* arg_pInitData,
                                      ix_cc_stkdrv_handler_module* arg_pHandlerModule
                                      );


/**
 * NAME: ix_cc_stkdrv_tm_init
 *
 * DESCRIPTION: This function is called by the CC Module to initialize the
 * Transport Module and register the Transport Module's entry points with
 * the CC Module.
 * 
 * @Param:  - IN ix_buffer_free_list_handle arg_hFreeList - handle to
 *          freelist used to allocated buffers for passing packets down
 *          from the FP Module.
 * @Param:  - IN ix_cc_stkdrv_fp_node* arg_pFP - pointer to the FP structure
 *          used to get data for all ports.
 * @Param:  - IN void* arg_pInitData - pointer to any additional initialization
 *          data that may be required by a specific handler module - unused
 *          in this case - unused in this case.
 * @Param:  - INOUT ix_cc_stkdrv_handler_module* arg_pHandlerModule -
 *          the function will fill in this structure with its entry
 *          points and contexts.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 */
extern ix_error ix_cc_stkdrv_tm_init(
                                    ix_buffer_free_list_handle arg_hFreeList,
                                    ix_cc_stkdrv_fp_node* arg_pFP,
                                    void* arg_pInitData,
                                    ix_cc_stkdrv_handler_module* arg_pHandlerModule
                                    );
                                    
/**
 * Prototypes for interface functions.
 */

/**
 * NAME: ix_cc_stkdrv_send_packet
 *
 * DESCRIPTION: This function is called by the handler module to the core
 * component module to queue the packet for transmission. In a future
 * implementation it may perform a downstream classification to determine
 * which output core component to send the packet to (e.g. IPv4 or IPv6 core
 * component). For now this classification is stubbed and all output packets go
 * to the IPv4 core component.
 * 
 * @Param:  - IN ix_buffer_handle arg_hBuffer - handle to the packet to be transmitted.
 * @Param:  - IN void* arg_pContext - pointer to the context.  In this case it
 *          will be a pointer to the CC port structure, ix_cc_stkdrv_physical_if_node.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_SEND_FAIL
 */
#if (_IX_OS_TYPE_ == _IX_OS_LINUX_KERNEL_)
ix_error ix_cc_stkdrv_send_packet(
                                  ix_buffer_handle arg_hBuffer,
                                  void* arg_pContext,
                                  ix_cc_stkdrv_og_pkt_type arg_pktType
                                  );
#else
ix_error ix_cc_stkdrv_send_packet(
                                  ix_buffer_handle arg_hBuffer,
                                  void* arg_pContext
                                  );
#endif


/**
 * NAME: ix_cc_stkdrv_send_msg_str
 *
 * DESCRIPTION: These message functions are used for sending messages
 * from a handler module to the core component module.  In this
 * release there is no need to send any messages from a handler
 * module to the CC Module, so these functions are stubs.
 * 
 * @Param:  - IN ix_cc_stkdrv_cc_module_msg_id arg_msgID - ID of the msg.
 * @Param:  - IN const char* arg_pMsg - pointer to the msg data.
 * @Param:  - IN ix_uint32 arg_MsgSize - size of the msg in bytes.
 * @Param:  - IN void* arg_pContext - pointer to the context.
 *
 * @Return: IX_SUCCESS
 */
ix_error ix_cc_stkdrv_send_msg_str(
                                   ix_cc_stkdrv_cc_module_msg_id arg_msgID,
                                   const char* arg_pMsg,
                                   ix_uint32 arg_MsgSize,
                                   void* arg_pContext
                                   );

/**
 * NAME: ix_cc_stkdrv_send_msg_int
 *
 * DESCRIPTION: These message functions are used for sending messages
 * from a handler module to the core component module.  In this
 * release there is no need to send any messages from a handler
 * module to the CC Module, so these functions are stubs.
 * 
 * @Param:  - IN ix_cc_stkdrv_cc_module_msg_id arg_msgID - ID of the msg.
 * @Param:  - IN ix_uint32 arg_MsgValue - the integer msg.
 * @Param:  - IN void* arg_pContext - pointer to the context.
 *
 * @Return: IX_SUCCESS
 */
ix_error ix_cc_stkdrv_send_msg_int(
                                   ix_cc_stkdrv_cc_module_msg_id arg_msgID,
                                   ix_uint32 arg_MsgValue,
                                   void* arg_pContext
                                   );

/**
 * NAME: ix_cc_stkdrv_async_send_handler_msg
 *
 * DESCRIPTION: Function used to send a message to a communication handler
 *              asynchronously.  This caller must be in the same address
 *              space as the communication handler to which it is trying
 *              to send a message.
 *
 * @Param:  - IN ix_cc_msghlp_callback arg_pCallback -
 *          pointer to the client's callback function to be invoked once
 *          the message returns.
 * @Param:  - IN void* arg_pContext - pointer to client context.
 * @Param:  - IN ix_cc_stkdrv_handler_id arg_handlerID - ID of the communication
 *          handler to which the message is to be delivered.
 * @Param:  - IN ix_uint32 arg_msgID - handler-specific message ID.
 * @Param:  - IN void* arg_pMsg - pointer to the message data.
 * @Param:  - IN ix_uint32 arg_msgSize - size of the message in bytes.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_OOM_SYSTEM
 */
ix_error ix_cc_stkdrv_async_send_handler_msg(
                                             ix_cc_msghlp_callback arg_pCallback,
                                             void* arg_pContext,
                                             ix_cc_stkdrv_handler_id arg_handlerID,
                                             ix_uint32 arg_msgID,
                                             void* arg_pMsg,
                                             ix_uint32 arg_msgSize
                                             );
                                             
/**
 * NAME: _ix_cc_strkdrv_dump_packet
 *
 * DESCRIPTION: Dumps data to the screen.
 * 
 * @Param:  - IN ix_uint8* arg_pData - pointer to start of data to be dumped.
 * @Param:  - IN ix_uint32 arg_len - length of data to be dumped in bytes.
 * @Param:  - IN ix_uint8* arg_pOutputString - string to be output before dump.
 *
 * @Return: None.
 */
void _ix_cc_stkdrv_dump_packet(
                              ix_uint8* arg_pData,
                              ix_uint32 arg_len,
                              ix_uint8* arg_pOutputString
                              );


/**
 * Exported variables.
 */

#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__IX_CC_STKDRV_COMMON_H__) */
