/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2002 Intel Corporation.All rights reserved.
 *
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2000 Network Processor, Release 6
 *
 * = FILENAME
 *      internal_stkdrv_pktclass.h
 *
 * = DESCRIPTION
 *      Internal types and functions for the packet classifier that are
 *      shared between the stack driver core component source and the
 *      packet classifier source.
 *
 * = AUTHOR
 *       Aaron Luk
 *       aaron.luk@intel.com
 *
 * = CHANGE HISTORY
 *      11/20/2002 3:48:33 PM - creation time
 *
 * ============================================================================
 * $Id: internal_stkdrv_pktclass.h,v 1.5 2002/12/17 20:20:15 asluk Exp $
 * $History: $
 */

#if !defined(__INTERNAL_STKDRV_PKTCLASS_H__)
#define __INTERNAL_STKDRV_PKTCLASS_H__


#include "cc/ix_cc_stkdrv_pktclass.h"

#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */

/**
 * TYPENAME: _ix_cc_stkdrv_filter_ops_context
 * 
 * DESCRIPTION: Context for asynchronous filter operations.
 *
 */
typedef struct _ix_s_cc_stkdrv_filter_ops_context
{
    ix_cc_stkdrv_cb_filter_ops pUserCallback;
    void* pUserContext;
} _ix_cc_stkdrv_filter_ops_context;

/**
 * TYPENAME: _ix_cc_stkdrv_add_filter_msg
 * 
 * DESCRIPTION: Structure to hold filter add messages.
 *
 */
typedef struct _ix_s_cc_stkdrv_add_filter_msg
{
    ix_cc_stkdrv_filter filter;
    ix_cc_stkdrv_filter_priority priority;
} _ix_cc_stkdrv_add_filter_msg;

/**
 * TYPENAME: _ix_cc_stkdrv_remove_filter_msg
 * 
 * DESCRIPTION: Structure to hold filter remove messages.
 *
 */
typedef struct _ix_s_cc_stkdrv_remove_filter_msg
{
    ix_cc_stkdrv_filter_handle filterHandle;
} _ix_cc_stkdrv_remove_filter_msg;


/**
 * TYPENAME: _ix_cc_stkdrv_modify_filter_msg
 * 
 * DESCRIPTION: Structure to hold filter modify messages.
 *
 */
typedef struct _ix_s_cc_stkdrv_modify_filter_msg
{
    ix_cc_stkdrv_filter filter;
    ix_cc_stkdrv_filter_priority priority;
    ix_cc_stkdrv_filter_handle filterHandle;
} _ix_cc_stkdrv_modify_filter_msg;


/**
 * TYPENAME: _ix_cc_stkdrv_register_filter_type_msg
 * 
 * DESCRIPTION: Structure to hold filter register messages.
 *
 */
typedef struct _ix_s_cc_stkdrv_register_filter_type_msg
{
    ix_cc_stkdrv_filter_type filterType;
    ix_cc_stkdrv_cb_check_filter checkFilterCB;
} _ix_cc_stkdrv_register_filter_type_msg;


/**
 * TYPENAME: _ix_cc_stkdrv_unregister_filter_type_msg
 * 
 * DESCRIPTION: Structure to hold filter unregister messages.
 *
 */
typedef struct _ix_s_cc_stkdrv_unregister_filter_type_msg
{
    ix_cc_stkdrv_filter_type filterType;
} _ix_cc_stkdrv_unregister_filter_type_msg;


/**
 * NAME: _ix_cc_stkdrv_add_filter
 *
 * DESCRIPTION: This function adds a filter to the array of filters and also
 * places the filter index into the appropriate priority list.
 * 
 * @Param:  - IN ix_cc_stkdrv_filter* arg_pFilter - pointer to the filter to be added.
 * @Param:  - IN ix_cc_stkdrv_filter_priority arg_priority - determines
 *          whether this filter is of high, medium, or low priority. During
 *          classification higher priority filters will be checked first.
 * @Param:  - IN ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl - pointer to the
 *          filter control structure.
 * @Param:  - OUT ix_cc_stkdrv_filter_handle* arg_pFilterHandle - pointer to
 *          the handle of the added filter. The handle will be passed in to
 *          ix_cc_stkdrv_remove_filter to remove the filter.
 * @Return: IX_SUCCESS
 *          IX_CC_STKDRV_ERROR_OUT_OF_FILTERS
 *          IX_CC_ERROR_RANGE
 *          IX_CC_ERROR_INTERNAL
 */
ix_error _ix_cc_stkdrv_add_filter(
                                 ix_cc_stkdrv_filter* arg_pFilter,
                                 ix_cc_stkdrv_filter_priority arg_priority,
                                 ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl,
                                 ix_cc_stkdrv_filter_handle* arg_pFilterHandle
                                 );

/**
 * NAME: _ix_cc_stkdrv_remove_filter
 *
 * DESCRIPTION: This function removes a filter from the array of filters and
 * the appropriate priority list.
 * 
 * @Param:  - IN ix_cc_stkdrv_filter_handle arg_filterHandle - opaque handle
 *          to the filter to be removed.
 * @Param:  - IN ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl - pointer to the
 *          filter control structure.
 * @Return: IX_SUCCESS
 *          IX_CC_STKDRV_ERROR_INVALID_FILTER
 *          IX_CC_ERROR_INTERNAL
 *
 */
ix_error _ix_cc_stkdrv_remove_filter(
                                    ix_cc_stkdrv_filter_handle arg_filterHandle,
                                    ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl
                                    );

/**
 * NAME: _ix_cc_stkdrv_remove_all_filters
 *
 * DESCRIPTION: This function removes all filters.
 * 
 * @Param:  - INOUT ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl - pointer to the
 * filter control structure.
 * @Return: IX_SUCCESS
 *
 */
ix_error _ix_cc_stkdrv_remove_all_filters(
                                         ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl
                                         );

/**
 * NAME: _ix_cc_stkdrv_modify_filter
 *
 * DESCRIPTION: This function modifies a filter with new input filter
 * criteria and data.
 * 
 * @Param:  - IN ix_cc_stkdrv_filter* arg_pFilter - pointer to new filter data
 *          to be used for the modified filter.
 * @Param:  - IN ix_cc_stkdrv_filter_priority arg_priority - determines whether
 *          this filter is of high, medium, or low priority. During
 *          classification higher priority filters will be checked first.
 * @Param:  - IN ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl - pointer to the
 *          filter control structure.
 * @Param:  - INOUT ix_cc_stkdrv_filter_handle* arg_pFilterHandle - pointer to
 *          the handle of the filter to be modified.
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_RANGE
 *          IX_CC_ERROR_INTERNAL
 *          IX_CC_STKDRV_ERROR_INVALID_FILTER
 *          IX_CC_STKDRV_ERROR_OUT_OF_FILTERS
 *
 */
ix_error _ix_cc_stkdrv_modify_filter(
                                    ix_cc_stkdrv_filter* arg_pFilter,
                                    ix_cc_stkdrv_filter_priority arg_priority,
                                    ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl, 
                                    ix_cc_stkdrv_filter_handle* arg_pFilterHandle
                                    );


/**
 * NAME: _ix_cc_stkdrv_register_filter_type
 *
 * DESCRIPTION: This function registers a callback for a specific filter type.
 * 
 * @Param:  - IN ix_cc_stkdrv_filter_type arg_filterType - type of the filter
 *          being registered.
 * @Param:  - IN ix_cc_stkdrv_cb_check_filter arg_checkFilterCB - callback
 *          function to check a filter of a specific type.
 * @Param:  - IN ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl - pointer to the
 *          filter control structure.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_DUPLICATE_ENTRY
 *
 */
ix_error _ix_cc_stkdrv_register_filter_type(
                                            ix_cc_stkdrv_filter_type arg_filterType,
                                            ix_cc_stkdrv_cb_check_filter arg_checkFilterCB,
                                            ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl
                                            );
/**
 * NAME: _ix_cc_stkdrv_unregister_filter_type
 *
 * DESCRIPTION: This function unregisters a callback for a specific filter type.
 * 
 * @Param:  - IN ix_cc_stkdrv_filter_type arg_filterType - type of the filter
 *          being unregistered.
 * @Param:  - IN ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl - pointer to the
 *          filter control structure.
 *
 * @Return: IX_SUCCESS
 *          IX_CC_ERROR_NOT_FOUND
 *
 */
ix_error _ix_cc_stkdrv_unregister_filter_type(
                                              ix_cc_stkdrv_filter_type arg_filterType,
                                              ix_cc_stkdrv_filter_ctrl* arg_pFilterCtrl
                                              );


#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__INTERNAL_STKDRV_PKTCLASS_H__) */
