/////////////////////////////////////////////////////////////////////////////////////
//
//
//                  I N T E L   P R O P R I E T A R Y
//
//     COPYRIGHT [c]  2002 BY  INTEL  CORPORATION.  ALL RIGHTS
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT
//     THE PRIOR WRITTEN PERMISSION OF :
//
//                        INTEL  CORPORATION
//
//                     2200 MISSION COLLEGE BLVD
//
//               SANTA  CLARA,  CALIFORNIA  95052-8119
//
//
/////////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: pkt_copier_util.uc
//
//      Purpose: Packet replication microblock utility functions.  
//
/////////////////////////////////////////////////////////////////////////////////////

#ifndef	_PKT_COPIER_UTIL_UC_
#define	_PKT_COPIER_UTIL_UC_

/////////////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_ctx_init()
//
// 	Description:
// 	
//		Initialize the packet copier context in Local memory.
//
//	Output
//
//		None.
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_ctx_init()

.begin
	.reg	cnt

	/*
	 * initiaze the context first time
	 */
	 ; set local memory
	local_csr_wr[active_lm_addr_0, 0]
	nop
	nop
	nop

	immed[cnt, 640]

	.while (cnt > 0)

		alu[*l$index0++, --, B, IX_NULL]

		alu[cnt, cnt, -, 1]

	.endw
.end

#endm

/////////////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_2me_ctx_init()
//
// 	Description:
// 	
//		Initialize the packet copier context in Local memory. (in 2 ME configuration)
//
//	Output
//
//		None.
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_2me_ctx_init()

.begin

	.reg	base

	move[base, PKT_COPIER_CTX_BASE]

	local_csr_wr[active_lm_addr_0, base]

	nop
	nop
	nop

	alu[_PORT_MASK, --, B, 0x0]

.end

#endm


/////////////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_queue_desc_init()
//
// 	Description:
// 	
//		Initializes the context queue descriptor
//
//	Output
//
//		None.
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_queue_desc_init()
	/*
	 * Initialize the Queue descriptor
	 */
	immed[_QD_CNT, 0x0]
	immed[_QD_HEAD, PKT_COPIER_Q_BASE]
	immed[_QD_TAIL, PKT_COPIER_Q_BASE]
	immed[_QD_TSHLD, PKT_COPIER_Q_TSHLD]
	immed32[_QD_SIZE_MASK, PKT_COPIER_WRAP_MASK]
#endm


/////////////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_get_request()
//
// 	Description:
// 	
//		Get next copy request
//
//	Output
//
//		None.
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_get_request()
	scratch[get, $request[0], 0, scr_ring_request, PKT_COPIER_REQUEST_SIZE], \
		sig_done[sig_scr_req]
#endm


/////////////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_local_dequeue_next_request()
//
// 	Description:
// 	
//		Dequeue next copy request (context) from local queue.
//
//	Output
//
//		None.
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_local_dequeue_next_request()

	alu[_QD_CNT, _QD_CNT, -, 1]
	
	// update Q head to next location 

	alu[_QD_HEAD, _QD_HEAD, +, PKT_COPIER_Q_ENTRY_SIZE]

	// round the address if needed

	alu[_QD_HEAD, _QD_HEAD, AND, _QD_SIZE_MASK]

#endm


/////////////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_local_enqueue()
//
// 	Description:
// 	
//		Enqueue copy request (context) into Local queue.
//
//	Output
//
//		None.
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_local_enqueue()
.begin
	
	.reg	parent_buf_id

	// set the LM index to queue tail (4)

	local_csr_wr[active_lm_addr_0, _QD_TAIL]

	// increment q cnt
	
	alu[_QD_CNT, _QD_CNT, +, 1]

	// update Q tail to next location 

	alu[_QD_TAIL, _QD_TAIL, +, PKT_COPIER_Q_ENTRY_SIZE]

	// round the address if needed 

	alu[_QD_TAIL, _QD_TAIL, AND, _QD_SIZE_MASK]

	// get parent buf ID (3)

	_pkt_copier_get_parent_buffer_id(parent_buf_id, $request[1])

	// also store the meta data. No need to store LW 0 (6)

	alu[_PARENT_META_1, --, B, $parent_meta[1]]

	alu[_PARENT_META_2, --, B, $parent_meta[2]]

	// clear the outport
	alu[_PARENT_META_3, $parent_meta[3], AND~, mask_ffff]

	alu[_PARENT_META_4, --, B, $parent_meta[4]]

	alu[_PARENT_META_5, --, B, $parent_meta[5]]


	alu_shf[_PARENT_META_6, $parent_meta[6], OR, parent_buf_id]

	// now store the request (4)

	// first time, we need to set the ref. cnt bit

	alu_shf[_REF_CNT, --, B, 0x1, <<REF_CNT_BIT]

	alu[_PACKET_SIZE_CLASS_ID, --, B, $request[0]]
	alu[_PARENT_SOP_HANDLE, --, B, $request[1]]
	alu[_PARENT_EOP_HANDLE, --, B, $request[2]]
	alu_shf[_CNT_PORT_MASK, --, B, $request[3]]

.end


#endm


/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_is_request_complete()
//
// 	Description:
// 	
//		Check if the current request is processed completely.
//
//	Output
//
//		None.
//
//	Input
//
//		PENDING_LABEL	=	Address to go if request is not complete
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_is_request_complete(PENDING_LABEL)
.begin

	; simply check port mask

	alu[--, 0x0, +16, _PORT_MASK]

	bne[PENDING_LABEL]

.end
#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_get_next_copy_num()
//
// 	Description:
// 	
//		Obtain next copy number from the request being processed.
//
//	Output
//
//		None.
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_get_next_copy_num()

.begin
	.reg temp

	alu[temp, 0x0, +16, _PORT_MASK]

	ffs[copy_for_outport, temp]

	; now clear this bit in port mask

	alu[temp, copy_for_outport, OR, 0x0]

	alu_shf[_PORT_MASK, _PORT_MASK, AND~, 0x1, <<indirect]
.end

#endm


/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_check_and_issue_get_request()
//
// 	Description:
// 	
//		Check and get next copy request if we have space in local queue.
//
//	Output
//
//		None.
//
//	Input
//
//		NO_SPACE_LABEL	=	address to go if there is no space in local 
//							queue.
// 
/////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_check_and_issue_get_request(NO_SPACE_LABEL)

	; check if we can pull in next request 

	alu[--, _QD_CNT, -, _QD_TSHLD]

	; if so, don't issue request

	bgt[NO_SPACE_LABEL]

	; otherwise issue a request 

	_pkt_copier_get_request()

#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_get_parent_buffer_id()
//
// 	Description:
// 	
//		Obtain parent buffer ID from handle
//
//	Output
//
//		out_buf_id		= Buffer ID
//
//	Input
//
//		in_buf_handle	= Parent buffer handle to convert.
// 
/////////////////////////////////////////////////////////////////////////////


#macro	_pkt_copier_get_parent_buffer_id(out_buf_id, in_buf_handle)

	// get the descriptor LW address

	ld_field_w_clr[out_buf_id, 0111, in_buf_handle]

	// which descriptor

	alu[out_buf_id, out_buf_id, -, parent_meta_base_lw]

	// get the ID

	alu_shf[out_buf_id, mask_ffff, AND, out_buf_id, >>3]	

#endm


/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_create_child_meta()
//
// 	Description:
// 	
//		Create child meta data and store into xfer registers
//
//	Output
//
//		None
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////


#macro	_pkt_copier_create_child_meta()
.begin
	.reg	temp	/* for packing data */

	; we will create child meta and load into $child_meta[X]

	; set LW1
	alu[$child_meta[1], --, B, _PARENT_META_1]

	; set LW 2
	alu_shf[$child_meta[2], _PARENT_META_2, OR, rx_stat_constant]	

#ifdef	PKT_COPIER_SPHY

	; set LW3. Note that outport is already cleared.
	alu[$child_meta[3], --, B, _PARENT_META_3]

#else
	; set LW3. Note that outport is already cleared.
	alu[$child_meta[3], _PARENT_META_3, OR, copy_for_outport]

#endif

	; LW4
	alu[$child_meta[4], --, B, _PARENT_META_4]

	; LW5
	alu[$child_meta[5], --, B, _PARENT_META_5]

	; LW6. Note that parent Buffer ID is already set.
	alu[$child_meta[6], --, B, _PARENT_META_6]

.end
#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_write_child_meta()
//
// 	Description:
// 	
//		Write child meta data into SRAM
//
//	Output
//
//		None
//
//	Input
//
//		None.
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_write_child_meta()
.begin
	.reg addr

	// now write the meta data 

	dl_meta_flush($child_meta, cur_child_handle, CHILD_FREELIST_ID, sig_child_meta_wr, SIG_NONE, 0, 7);

.end
#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_set_ref_cnt()
//
// 	Description:
// 	
//		Set reference count in parent buffer
//
//	Output
//
//		None
//
//	Input
//
//		in_count		= Reference count
//		in_buf_handle	= buffer handle of parent
//		in_freelist		= freelist ID of parent
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_set_ref_cnt(in_count, in_buf_handle, in_freelist)
.begin

	.reg	sram_addr

	; set single ref. cnt and count
	alu[$pkt_copier_ref_cnt[0], --, B, in_count] 

	; get SRAM address of meta data

	dl_buf_get_desc(sram_addr, in_buf_handle, in_freelist)


#ifdef	DL_SPLIT_META_DATA

	; clear the channel number

	alu_shf[sram_addr, sram_addr, AND~, 0x3, <<30]

	#if	(BUF_SRAM_SPLIT_CHANNEL != 0)

	; Set the channel number
	alu[sram_addr, sram_addr, OR, BUF_SRAM_SPLIT_CHANNEL, <<30]

	#endif

#endif

	sram[write, $pkt_copier_ref_cnt[0], 28, sram_addr, 1], sig_done[sig_parent_meta_wr]

.end
#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_2me_ctx_ptr_set()
//
// 	Description:
// 	
//		Set context pointer in 2nd ME in 2 ME configuration
//
//	Output
//
//		None
//
//	Input
//
//		None
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_2me_ctx_ptr_set()

.begin

	.reg	base

	move[base, PKT_COPIER_CTX_BASE]

	local_csr_wr[active_lm_addr_0, base]

	nop
	nop
	nop

.end


#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_ctx_ptr_set()
//
// 	Description:
// 	
//		Set context pointer in 1 ME configuration
//
//	Output
//
//		None
//
//	Input
//
//		None
// 
/////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_ctx_ptr_set()


	local_csr_wr[active_lm_addr_0, context_ptr_base]


#endm


/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_check_n_write_response()
//
// 	Description:
// 	
//		Check if scratch ring is full, and if not write the response.
//
//	Output
//
//		None
//
//	Input
//
//		RING_FULL_LABEL		= address to jump if the ring is full
// 
/////////////////////////////////////////////////////////////////////////////
#macro	_pkt_copier_check_n_write_response(RING_FULL_LABEL)


#define_eval	TMP							PKT_COPIER_RING_OUT
#define_eval	PKT_COPIER_RING_OUT_FULL	SCR_Ring/**/TMP/**/_Full

	// check if ring is full 

	br_inp_state[PKT_COPIER_RING_OUT_FULL, RING_FULL_LABEL]

	scratch[put, $response[0], 0, scr_ring_response, PKT_COPIER_RESPONSE_SIZE], \
		sig_done[sig_scr_rsp]

measure_perf_here#:

end#:

#undef	PKT_COPIER_RING_OUT_FULL
#undef	TMP

#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_create_response()
//
// 	Description:
// 	
//		Construct the copy response.
//
//	Output
//
//		None
//
//	Input
//
//		None
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_create_response()

	// by the time this macro is called, the parent meta flush
	// flag is cleared

#ifdef	PKT_COPIER_SPHY
	alu[$response[0], --, B, _PACKET_SIZE_CLASS_ID]
#else
	alu[$response[0], _PACKET_SIZE_CLASS_ID	, OR, copy_for_outport, <<12]
#endif

	alu[$response[1], --, B, cur_child_handle]
	alu[$response[2], --, B, _PARENT_EOP_HANDLE]

#endm

/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_setup_next_thread_chain()
//
// 	Description:
// 	
//		Construct the chain of threads to use running in 1 ME configuration.
//
//	Output
//
//		None
//
//	Input
//
//		None
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_setup_next_thread_chain()

	// sets up next thread chain by setting a GPR which when
	// written to SAME_ME_SIGNAL local CSR, wakes up next
	// thread in the chain.

	// If this block were to run on fewer threads, only changing this
	// macro would be enough to maintain required thread ordering.

	alu_shf[pkt_copier_next_thread_gpr, --, B, PKT_COPIER_NEXT_THD_SIGNAL, <<3]

	.if (ctx() == 0)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_0]
	
	.elif (ctx() == 1)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_1]
	
	.elif (ctx() == 2)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_2]
	
	.elif (ctx() == 3)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_3]
	
	.elif (ctx() == 4)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_4]
	
	.elif (ctx() == 5)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_5]
	
	.elif (ctx() == 6)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_6]
	
	.elif (ctx() == 7)

		alu_shf[pkt_copier_next_thread_gpr, pkt_copier_next_thread_gpr, OR, NEXT_THREAD_OF_7]
	
	.endif

#endm


/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_send_req_to_next_me()
//
// 	Description:
// 	
//		Send request to 2nd ME in 2 ME configuration.
//
//	Output
//
//		None
//
//	Input
//
//		in_scr_req		= 	Xfer registers containing the copy request
//		in_parent_meta	=	Xfer registers containing parent meta data
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_send_req_to_next_me(in_scr_req, in_parent_meta)

.begin

	.reg	parent_buf_id
	.reg	sop_eop_seg_cnt
	.reg	tmp

	// get parent buffer ID (3)

	_pkt_copier_get_parent_buffer_id(parent_buf_id, in_scr_req[1])

	ld_field_w_clr[sop_eop_seg_cnt, 1000, in_scr_req[1]]

1st_write#:

	br_inp_state[nn_full, 1st_write#]

	// port mask and SOP/EOP/SegCnt

	alu[tmp, in_scr_req[3], AND, mask_ffff]

	alu[*n$index++, tmp, OR, sop_eop_seg_cnt]

	// packet size and class ID

	alu[*n$index++, --, B, in_scr_req[0]]

	// EOP handle

	alu[*n$index++, --, B, in_scr_req[2]]

2nd_write#:

	br_inp_state[nn_full, 2nd_write#]

	alu[*n$index++, --, B, in_parent_meta[1]]
	alu[*n$index++, --, B, in_parent_meta[2]]
	alu[*n$index++, in_parent_meta[3], AND~, mask_ffff]

3rd_write#:

	br_inp_state[nn_full, 3rd_write#]

	alu[*n$index++, --, B, in_parent_meta[4]]
	alu[*n$index++, --, B, in_parent_meta[5]]
	alu[*n$index++, in_parent_meta[6], OR, parent_buf_id]

.end

#endm


/////////////////////////////////////////////////////////////////////////////
// 
// _pkt_copier_rcv_req_from_prev_me()
//
// 	Description:
// 	
//		Receive request from 1st ME in 2 ME configuration.
//
//	Output
//
//		None
//
//	Input
//
//		None
// 
/////////////////////////////////////////////////////////////////////////////

#macro	_pkt_copier_rcv_req_from_prev_me()

	// LM context pointer is already set

1st_read#:

	br_inp_state[nn_empty, 1st_read#]
	
	// SOP/EOP/Segcnt

	ld_field_w_clr[_SOP_EOP_SEG_CNT, 1000, *n$index]

	// port mask

	alu[_PORT_MASK, --, B, *n$index++]

	// packet size and class

	alu[_PACKET_SIZE_CLASS_ID, --, B, *n$index++]

	// eop handle

	alu[_PARENT_EOP_HANDLE, --, B, *n$index++]


2nd_read#:

	br_inp_state[nn_empty, 2nd_read#]

	alu[_PARENT_META_1, --, B, *n$index++]
	alu[_PARENT_META_2, --, B, *n$index++]
	alu[_PARENT_META_3, --, B, *n$index++]


3rd_read#:

	br_inp_state[nn_empty, 3rd_read#]

	alu[_PARENT_META_4, --, B, *n$index++]
	alu[_PARENT_META_5, --, B, *n$index++]
	alu[_PARENT_META_6, --, B, *n$index++]

#endm



#endif	//_PKT_COPIER_UTIL_UC_
