/////////////////////////////////////////////////////////////////////////////////////
//                                                                      
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                 
//
// Change History
// --------------
//
// Date			Description											Whom
// ------------------------------------------------------------------------------------
//
// 11/11/01    	Created for IXP2400								Uday Naik
// 10/06/02		Modified for IXP2800							Prashant Chandra
//
/////////////////////////////////////////////////////////////////////////////////////
//
//
// Flow Control Handling Routines
//
//
/////////////////////////////////////////////////////////////////////////////////////

#ifndef __CSIX_SCHEDULER_FC_UC__
#define __CSIX_SCHEDULER_FC_UC__

/////////////////////////////////////////////////////////////////////////////////////

// Offset of the flow control fifo

#define FCIFIFO 0x100

/////////////////////////////////////////////////////////////////////////////////////
//
//
// HandleFlowControl:
//
//	  Handle Flow Control Messages from the Fabric
//
//    This runs in one single thread and handles flow control 
//	  messages coming in from the Fabric
//
//    Each frame has atleast n messages of 1 word each. Additional there is 
//    a payload length (16 bits) and vertical parity (16 bits)
//
//
//  Flow Control Word 1
// 
//  |----------------------------------------------------------------------|
//  |	xxxxxxxx | PayloadLength | xxxx	    | Class   | xxxxxxx | On/Off   |
//  |	(31..24) | (16..23)	     | (15..12)	| (8..11) | (7..1)  | (0)      |
//  |----------------------------------------------------------------------|
//
//
//  Flow Control Word 2
//
//  |------------------------------------------------------------------|
//  |	xxxxxxxx | Port Number   | Next Message (or vertical parity)   |
//  |	(31..22) | (16..21)	     |    (0..15)				     	   |
//  |------------------------------------------------------------------|
//
//
//  Flow Control Word 3..n (last word has vertical parity)
//
//  |------------------------------------------------------------------|
//  |	xxxxxxxx | Port Number   | Next Message (or vertical parity)   |
//  |	(31..22) | (16..21)	     |    (0..15)				     	   |
//  |------------------------------------------------------------------|
//
//  
// 
//
/////////////////////////////////////////////////////////////////////////////////////



#macro _scheduler_handle_flow_control()

.begin

	.reg 			$msg1, $msg2			// xfer registers for flow control
	.xfer_order 	$msg1, $msg2			// declare the order for xfer registers
	.reg			fcfifo					// stores address of fcfifo
	.reg 			message1				// GPR to store first word of fc message 
	.reg			class					// class (0-15) within a port
	.reg			port					// port (0..31)
	.reg			number_of_messages		// number of messages in the FC frame
	.reg			temp					// temporary variable
	.reg			port_number_mask
	.reg			queue_addr
	.reg			prev_queue_addr
	.reg			next_index
	.reg			port_addr
	.sig			FLOW_CONTROL_SIGNAL		// signal for reading msf
	
	// load the start address for the ingress flow control fifo 

	immed[fcfifo, FCIFIFO]
	
	// load bit mask to extract lsb bit from port number. We store it in a 
	// register so we dont have to compute it at run time
	
	immed32[port_number_mask, 0x10000]

READ_IFIFO#:
	
	// Read 2 words from the IFIFO and swap out waiting for it to 
	// get done

 	msf[read, $msg1, fcfifo, 0, 2], ctx_swap [FLOW_CONTROL_SIGNAL]

	// Get xfer register into GPR message1 

	alu[message1, --, B, $msg1]
		
	// Get the number of messages. which is payloadLength/4 

	alu_shf[number_of_messages, 0x3F, AND, message1, >>18]

	// Get the port number (we take 5 bits from the port number field

	alu_shf[port, 0x1f, AND, $msg2, >>16]

	// Get the class number (we want last 3 bits to be zero)

	alu_shf[class, 0x38, AND, message1, >>5]

HANDLE_FLOW_CONTROL_MESSAGE#:

	// set the local memory address for the queue
	alu_shf[queue_addr, class, OR, port, <<(NUM_CLASSES_SHF+3)]
	local_csr_wr[active_lm_addr_0, queue_addr]

	// check the speed bit. If it is 0, then flow control is on
	// else it is off

	alu[--, 0x1, AND, message1] 

	beq[FLOW_CONTROL_ON#]

	// load the prev queue addr to LM ADDR1
	ld_field_w_clr[prev_queue_addr, 0011, active, >>16]
	local_csr_wr[active_lm_addr_1, prev_queue_addr]

	// check if this is the first active queue
	br_bset[active, LIST_INACTIVE_BIT, FIRST_ACTIVE_QUEUE#]
	ld_field[*l$index0, 0001, active, >>3]
	br[FLOW_CONTROL_END#], defer[2]
	ld_field[active, 1100, queue_addr, <<16]
	ld_field[*l$index1, 0001, queue_addr, >>3]

FIRST_ACTIVE_QUEUE#:
	br[FLOW_CONTROL_END#], defer[2]
	ld_field_w_clr[active, 0011, queue_addr]
	ld_field[active, 1100, queue_addr, <<16]


FLOW_CONTROL_ON#:

	// set the flow bit in the queue data structure 

	alu_shf[*l$index0, *l$index0, OR, 1, <<QUEUE_FC_BIT]
	

FLOW_CONTROL_END#:

	alu[number_of_messages, number_of_messages, -, 1]

	beq[FLOW_CONTROL_DONE#] 
		
	alu[message1, --, B, $msg2]

	// Read 1 word from the IFIFO and swap out waiting for it to 
	// get done. Since we already have message1 we will start to use
	// it

	msf[read, $msg2, fcfifo, 0, 1], ctx_swap[FLOW_CONTROL_SIGNAL] , defer[1]

	alu_shf[class, 0x38, AND, message1, >>5]

	br[HANDLE_FLOW_CONTROL_MESSAGE#]

FLOW_CONTROL_DONE#:
.end

#endm
	
/////////////////////////////////////////////////////////////////////////////////////

#endif	// __CSIX_SCHEDULER_FC_UC__

/////////////////////////////////////////////////////////////////////////////////////