//------------------------------------------------------------------------------
//                                                                      
//                   I N T E L   P R O P R I E T A R Y                   
//                                                                       
//      COPYRIGHT (c)  2001 BY  INTEL  CORPORATION.  ALL RIGHTS          
//      RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//      BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//      RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//      LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//      MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//      THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                       
//                         INTEL  CORPORATION                            
//                                                                      
//                      2200 MISSION COLLEGE BLVD                        
//                                                                       
//                SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                       
//------------------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: 	llc_snap_decap.uc
//
//      Purpose: 	This file contains macros that perform LLC/SNAP classify and
//					decap function
//					
//
//      History:
//
//
//      Date            Comment                         By
//      ------------------------------------------------------------------------
//
//		07/9/2002		Created							Minal Patel
//
////////////////////////////////////////////////////////////////////////////////
#ifndef	_LLC_SNAP_DECAP_UC_
#define _LLC_SNAP_DECAP_UC_




;-------------------------------------------------------------------------------
; include files
;-------------------------------------------------------------------------------
#include 	"llc_snap_decap.h"
#include	"xbuf.uc"



////////////////////////////////////////////////////////////////////////////////
//	_llc_snap_get_header_type:
//
//		Description:
//		It gets the protocol id specified in the SNAP header underneath the 
//		LLC header.
//
//		Output:
//			snap_pid: The protocol ID specified in SNAP header.
//	
//		Input: 
//
//			in_header: The header read from DRAM which has IP header along
//			with L2 encap if any.
//
//       	in_header_offset: The offset at which the AAL5 header starts.
//
//			HDR_PTYPE_START: The offset(in bytes) at which the protocol ID starts.
//
//			HDR_PTYPE_SZ: The size in bytes of the protocol ID.
//
//		Size: 1-2 cycles
//		
////////////////////////////////////////////////////////////////////////////////


#macro _llc_snap_get_header_type(snap_pid, in_header, in_header_offset, HDR_PTYPE_START, HDR_PTYPE_SZ)
.begin

	xbuf_extract[snap_pid, in_header, in_header_offset, HDR_PTYPE_START, HDR_PTYPE_SZ]

.end
#endm


////////////////////////////////////////////////////////////////////////////////
//	_llc_snap_decap:
//
//		Description:
//		It updates the offset to the header, the sop buffer size and the
//		packet size according to the L2 header size passed as input parameter.
//
//		Input: 
//
//			HEADER_SIZE: The size of the L2 header( eg LLC_SNAP header size = 8)
//
//		Size: 9 cycles
//		
////////////////////////////////////////////////////////////////////////////////


#macro	_llc_snap_decap(HEADER_SIZE)

.begin
	.reg	offset
	.reg	size


	; Update offset.
	dl_meta_get_offset[offset]
	alu[offset, offset, +, HEADER_SIZE]
	dl_meta_set_offset[offset]

	; Update packet size.
	dl_meta_get_packet_size[size]
	alu[size, size, -, HEADER_SIZE]
	dl_meta_set_packet_size[size]

	; Update SOP buffer size. (SOP buffer cell count is recomputed in dl_qm_sink())
	dl_meta_get_buffer_size[size]
	alu[size, size, -, HEADER_SIZE]
	dl_meta_set_buffer_size[size]

.end 
#endm


////////////////////////////////////////////////////////////////////////////////
//	_llc_snap_decap_classify:
//
//		Description:
//		It classifies the packet according to the header type specified in 
//		meta data.
//
//		Output:
//			ip_header_offset: The offset in bytes pointing to the IP header.
//
//		Input: 
//
//			in_header: The header read from DRAM which has IP header along
//			with L2 encap if any.
//
//       	AAL5_HDR_OFFSET: The offset at which the AAL5 header starts.
//
//		Size: 
//		
////////////////////////////////////////////////////////////////////////////////


#macro _llc_snap_decap_classify(ip_hdr_offset_reg, in_header, AAL5_HDR_OFFSET) 
.begin

	.reg pkt_proto_type
	.reg snap_hdr_pid

	.reg ether_pid_val 


 	; Check for our block id. If doesn't match, exit.
	alu[--, dl_next_block, -, BID_LLC_SNAP_DECAP]
	bne[BY_PASS#]

	; Get PKT_PROTO_TYPE.
	dl_meta_get_header_type(pkt_proto_type)

	alu[--, pkt_proto_type, -, PKT_PROTO_LLC_SNAP]
	bne[NOT_LLC_SNAP#]

	;--------------------------------------------------------------------------
	; Here frame is LLC-SNAP encapsulated.
	;--------------------------------------------------------------------------
	
	; Set the ip_header_offset to 8.
	alu[ip_hdr_offset_reg, --, b, 8]
				
	; Get the header_type underneath the LLC_SNAP header.
	_llc_snap_get_header_type(snap_hdr_pid, in_header, AAL5_HDR_OFFSET, 
				SNAP_HDR_PID_START, SNAP_HDR_PID_SZ)   
			
	; Update offset, sop buffer size, pkt_size.
	_llc_snap_decap(LLC_SNAP_HDR_SIZE)

	; Check if snap_hdr_pid = ETHER_IPV4_PID.
	move(ether_pid_val, ETHER_IPV4_PID)
	alu[--, snap_hdr_pid, -, ether_pid_val]
	bne[NOT_ETHER_IPV4#]

	br[CLASSIFY_DONE#], defer[2]
	dl_meta_set_header_type(PKT_PROTO_IPV4)
	alu[dl_next_block, -- , b, TARGET_LLCSNAP_IPV4]

NOT_ETHER_IPV4#:

	; Check if snap_hdr_pid = ETHER_IPV6_PID.
	move(ether_pid_val, ETHER_IPV6_PID)
	alu[--,snap_hdr_pid, -, ether_pid_val]
	bne[NOT_ETHER_IPV6#]

	br[CLASSIFY_DONE#], defer[2]
	dl_meta_set_header_type(PKT_PROTO_IPV6)
	alu[dl_next_block, -- , b, TARGET_LLCSNAP_IPV6]
		
NOT_ETHER_IPV6#:

	; Check if snap_hdr_pid = ETHER_ARP_PID.
	move(ether_pid_val, ETHER_ARP_PID)
	alu[--,snap_hdr_pid, -, ether_pid_val]
	bne[NOT_ETHER_ARP#]

	br[CLASSIFY_DONE#], defer[2]
	dl_meta_set_header_type(PKT_PROTO_ARP)
	alu[dl_next_block, -- , b, TARGET_LLCSNAP_ARP]

NOT_ETHER_ARP#:
	
	; Check if snap_hdr_pid = ETHER_PPP_DISCOVERY_PID.
	move(ether_pid_val, ETHER_PPP_DISCOVERY_PID)
	alu[--,snap_hdr_pid, -, ether_pid_val]
	bne[NOT_ETHER_PPP_DISC#]

	br[CLASSIFY_DONE#], defer[2]
	dl_meta_set_header_type(PKT_PROTO_PPP_DISCOVERY)
	alu[dl_next_block, -- , b, TARGET_LLCSNAP_PPP_DISCOVERY]			
				
NOT_ETHER_PPP_DISC#:

	; Check if snap_hdr_pid = ETHER_PPP_SESSION_PID.
	move(ether_pid_val, ETHER_PPP_SESSION_PID)
	alu[--,snap_hdr_pid, -, ether_pid_val]
	bne[NOT_ETHER_PPP_SESSION#]

	br[CLASSIFY_DONE#], defer[2]
	dl_meta_set_header_type(PKT_PROTO_PPP_SESSION)
	alu[dl_next_block, -- , b, TARGET_LLCSNAP_PPP_SESSION]	

NOT_ETHER_PPP_SESSION#:

	; Check if snap_hdr_pid = ETHER_MPLS_UNICAST_PID.
	immed32[ether_pid_val, ETHER_MPLS_UNICAST_PID]
	alu[--,snap_hdr_pid, -, ether_pid_val]
	bne[NOT_ETHER_MPLS_UNICAST#]

	br[CLASSIFY_DONE#], defer[2]
	dl_meta_set_header_type(PKT_PROTO_MPLS_UNICAST)
	alu[dl_next_block, -- , b, TARGET_LLCSNAP_MPLS_UNICAST]

NOT_ETHER_MPLS_UNICAST#:

	; It has unsupported snap_hdr_pid.
	br[CLASSIFY_DONE#], defer[1]
	alu[dl_next_block, --, b, BID_DROP]

NOT_LLC_SNAP#:



	;--------------------------------------------------------------------------
	; Here frame is not encapsulated.
	;--------------------------------------------------------------------------

	; Check if pkt_proto_type = PKT_PROTO_IPV4.
	alu[--,pkt_proto_type,-,PKT_PROTO_IPV4]
	bne[NOT_IPV4#]

	alu[ip_hdr_offset_reg, --, b, 0]

	; No need to update offset, sop buffer size, pkt_size.
	br[CLASSIFY_DONE#], defer[1]
	alu[dl_next_block,-- ,b, TARGET_LLCSNAP_IPV4]

NOT_IPV4#:

	; Check if pkt_proto_type = PKT_PROTO_IPV6.
	alu[--,pkt_proto_type,-,PKT_PROTO_IPV6]
	bne[NOT_IPV6#]

	; Later, update ip_hdr_offset_reg, offset, sop buffer size, pkt_size.
	br[CLASSIFY_DONE#], defer[1]
	alu[dl_next_block,-- ,b, TARGET_LLCSNAP_IPV6]

NOT_IPV6#:

	; Check if pkt_proto_type = PKT_PROTO_ETHER.
	alu[--,pkt_proto_type,-,PKT_PROTO_ETHER]
	bne[NOT_ETHER#]

	; Later, update ip_hdr_offset_reg, offset, sop buffer size, pkt_size.
	br[CLASSIFY_DONE#], defer[1]
	alu[dl_next_block,-- ,b, TARGET_LLCSNAP_ETHER]

NOT_ETHER#:

	; Check if pkt_proto_type = PKT_PROTO_ARP.
	alu[--,pkt_proto_type,-,PKT_PROTO_ARP]
	bne[NOT_ARP#]

	; Later, update ip_hdr_offset_reg, offset, sop buffer size, pkt_size.
	br[CLASSIFY_DONE#], defer[1]
	alu[dl_next_block,-- ,b, TARGET_LLCSNAP_ARP]

NOT_ARP#:

	; Check if pkt_proto_type = PKT_PROTO_PPP_DISCOVERY.
	alu[--,pkt_proto_type,-,PKT_PROTO_PPP_DISCOVERY]
	bne[NOT_PPP_DISC#]

	; Later, update ip_hdr_offset_reg, offset, sop buffer size, pkt_size.
	br[CLASSIFY_DONE#], defer[1]
	alu[dl_next_block,-- ,b, TARGET_LLCSNAP_PPP_DISCOVERY]

NOT_PPP_DISC#:

	; Check if pkt_proto_type = PKT_PROTO_PPP_SESSION.
	alu[--,pkt_proto_type,-,PKT_PROTO_PPP_SESSION]
	bne[NOT_PPP_SESSION#]

	; Later, update ip_hdr_offset_reg, offset, sop buffer size, pkt_size.
	br[CLASSIFY_DONE#], defer[1]
	alu[dl_next_block,-- ,b, TARGET_LLCSNAP_PPP_SESSION]
	
NOT_PPP_SESSION#:
				 		
	; It has unsupported pkt_proto_type.
	alu[dl_next_block,-- ,b, BID_DROP]

BY_PASS#:

CLASSIFY_DONE#:


.end
#endm






#endif   //_LLC_SNAP_DECAP_UC_
