////////////////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ether.uc
//
//      Purpose: This file contains the  macro which implements the 
//		ether decapsulation and MAC filering 
//
//      History:
//
//
//      Date            Comment                         			By
//      --------------------------------------------------------------------------------	
//
//      05/29/2002      Created                         			David Meng
//
//
/////////////////////////////////////////////////////////////////////////////////////////

#ifndef	__ETHER_UC__
#define	__ETHER_UC__

/////////////////////////////////////////////////////////////////////////////////////////

//	Include Files

// include xbuf.uc from IXPblocks portable library
#include	"xbuf.uc"
#include	"sig_macros.uc"
#include	"dl_source.uc"
#include	"ether.h"
#include    "ix_cc_microengines_bindings.h"

// include Ether files for initialization, utilities such as


/////////////////////////////////////////////////////////////////////////////////////////

//	Constants 
//	None.

/////////////////////////////////////////////////////////////////////////////////////////

//	Global Registers
#ifdef ETHER_WITHOUT_L2
	.reg	eth_flt_table_base
	.reg	port_bit_base
#endif

//	Use DL variables.

/////////////////////////////////////////////////////////////////////////////////////////

//	Global Signals
//	None.

/////////////////////////////////////////////////////////////////////////////////////////
//
// _ether_init()
//
//	 	Description: 
//			This is the general ethernet processing macro.  This macro only set up 
//			global gprs used by the ethernet processing.  If there is a l2 processing 
//          on the top, this init macro may be not need to be called 
//
//	 	Outputs:
//
//		Inputs:
//			in_port_type: 		The input port type
//			in_flt_table_base:  The MAC filter table base address	
//		Size: 
//	
//
////////////////////////////////////////////////////////////////////////////////////////
#macro	_ether_init(in_flt_table_base, in_port_bit_base)
.begin
	immed32(eth_flt_table_base,in_flt_table_base)		; set up ether filtering table base
	immed32(port_bit_base,in_port_bit_base)			; set up port bit map base
.end 
#endm

/////////////////////////////////////////////////////////////////////////////////////////
//
// _ether_decap()
//
//	 	Description: 
//			Decap the ethernet II frame.
//		
//
//	 	Outputs:
//			meta data offset: offset of the IP packet in the frame.
//			meta data pkt_size: the packet size of the IP packet from the frame.
//
//		Inputs:
//			None
//		Size: 
////////////////////////////////////////////////////////////////////////////////////////
#macro	_ether_decap()

.begin
	.reg	offset
	.reg	size


	// update offset	

	dl_meta_get_offset[offset]
	alu[offset, offset, +, ETHER_HEADER_LENGTH]
	dl_meta_set_offset[offset]

	// update packet size
	dl_meta_get_packet_size[size]
	alu[size, size, -, ETHER_HEADER_LENGTH]
	dl_meta_set_packet_size[size]

	// Update SOP buffer size
	dl_meta_get_buffer_size[size]
	alu[size, size, -, ETHER_HEADER_LENGTH]
	dl_meta_set_buffer_size[size]

.end 
#endm


/////////////////////////////////////////////////////////////////////////////////////////
//
// _ether_strip_fcs()
//
//	 	Description: 
//			Strip the FCS in SOP buffer and adjust the meta data accordingly.
//		
//
//	 	Outputs:
//
//		Inputs:
//			None
//		Size: 
////////////////////////////////////////////////////////////////////////////////////////
#macro	_ether_strip_fcs()

.begin
	.reg	size

	// update packet size
	dl_meta_get_packet_size[size]
	alu[size, size, -, 4]
	dl_meta_set_packet_size[size]

	// Update SOP buffer size
	dl_meta_get_buffer_size[size]
	alu[size, size, -, 4]
	dl_meta_set_buffer_size[size]

.end 
#endm


/////////////////////////////////////////////////////////////////////////////////////////
//
// _ether_classify()
//
//	 	Description: 
//			This classification is currently designed for Ethernet II.  For the 802.2 and 
//			802.3 Ethernet, LLC/SNAP should be considered.
//
//	 	Outputs:
//			dl_next_block
//			
//		
//
//		Inputs:
//			in_header : Ethernet frame header buffer
//			in_header_offset: The offset of the Ethernet frame in the buffer 
//
//		
//		
//
//		
//		
//		Size: 
//
//		
//
////////////////////////////////////////////////////////////////////////////////////////
#macro	_ether_classify(in_header, in_header_offset)
.begin
	.reg	type
		

#ifdef MPLS

	xbuf_extract[type, in_header, in_header_offset,\
				4, 2]											; get frame type


	alu[ether_frame_type, --, B, type]							; Keep the frame type 

#else

	_ether_get_frame_type(type, in_header, in_header_offset); Get Ethernet frame type.

#endif // MPLS

	.if(type == ETHER_IPV4)									; This is an IP packet.
		dl_meta_set_header_type(BID_ETHER_DECAP_NEXT1)
		alu[dl_next_block, --, B, TARGET_ETHER_DECAP_IPV4]	; Next block is IPv4.
	.elif(type == ETHER_IPV6)								; This is an IPV6 packet.
		dl_meta_set_header_type(BID_ETHER_DECAP_NEXT2)
		alu[dl_next_block, --, B, TARGET_ETHER_DECAP_IPV6]	; Next block is IPv6.
	.elif(type == ETHER_ARP)								; This is ARP packet.
		dl_meta_set_header_type(BID_ETHER_DECAP_NEXT3)
		alu[dl_next_block, --, B, TARGET_ETHER_DECAP_ARP]	; Next block is ARP (core).
		dl_set_exception(BID_ETHER, IX_ARP_PACKET)
		dl_set_exception_priority(1)						; ARP_PACKET is high priority

#ifdef MPLS													; MPLS processng
	.elif(type == ETHER_MPLS_UCAST)							; This is an MPLS ucast packet.
		dl_meta_set_header_type(BID_ETHER_DECAP_NEXT4)
		alu[dl_next_block, --, B, TARGET_ETHER_DECAP_MPLS]	; Next block is MPLS ILM.
#endif //MPLS


 	.elif(type == ETHER_PPP_DISCOVERY)						; This is PPP discovery.
		dl_meta_set_header_type(ETHER_PPP_DISCOVERY_TYPE)	; Set PPP discovery type.
		alu[dl_next_block, --, B, TARGET_ETHER_DECAP_PPP_DISCOVERY]	; Next block PPP discovery.
		dl_set_exception(BID_ETHER, IX_NON_IP_PACKET)
	.elif(type == ETHER_PPP_SESSION)						; This is PPP session.
		dl_meta_set_header_type(ETHER_PPP_SESSION_TYPE)		; Set PPP session type.
		alu[dl_next_block, --, B, TARGET_ETHER_DECAP_PPP_SESSION]	; Next block PPP session.
		dl_set_exception(BID_ETHER, IX_NON_IP_PACKET)		
	.else
		dl_meta_set_header_type(UNKNOWN_TYPE)				; set Unknown type
		alu[dl_next_block, --, B, IX_DROP]					; unknown packet to drop
	.endif
.end 
#endm


//////////////////////////////////////////////////////////////////////////////
// _ether_filter_mac
//
// Description:
//    This macro will check the given DMAC address against the given list
//    of MAC addresses (hash table).  In addition, it will also check if the 
//    DMAC is broadcast, multicast, or unicast and if the flags will permit
//    the packet (e.g., PROMISC will allow all packets, broadcast is always 
//	  allowed.  Multi cast is teated as unicast but set flag as multi cast).
//	  
// Parameters:
//    out_rx_flags          - (OUT) The receive flags associated with this MAC
//                          (e.g., IX_RXSTAT_UCAST, IX_RXSTAT_MCAST, 
//                          IX_RXSTAT_BCAST)
//    in_dmac_hi            - (IN) The most significant 32 bits of the DMAC
//    in_dmac_lo            - (IN) The least significant 16 bits of the 
//                          DMAC
//    in_mac_filters_addr   - (IN) The address of the MAC address hash table
//    in_flags              - (IN) The ingress flags
//    in_port               - (IN) The ingress port number
//
// Return:
//    A the zero condition code will be set of the packet is to be discarded
//
//		Size: 
//
////////////////////////////////////////////////////////////////////////////////////////

#macro _ether_filter_mac(out_rx_flags, in_dmac_hi, in_dmac_lo, \
						 in_port)
.begin
	.sig volatile flt_rd									; read filter table signal
	.sig volatile get_bit_map								; get bit map signal
	.reg $port_type											; $xfer for port type
	.reg type												; type gpr
	.reg filter_stat										; filter stat 0,no pass, 1 pass


	
	br_bclr[dmac_hi, ETHER_MCAST_BIT, not_multicast#]


	// Check for broadcast, a special case of multicast

	.if (in_dmac_lo == ETHER_BC_MAC_LO)
		.if (in_dmac_hi == ETHER_BC_MAC_HI)
			move(out_rx_flags, ETHER_RX_BCAST)
			br[pass#]										; if broadcast packet, pass
		.endif
	.endif

	//  So it is simply multicast, not broadcast

	move(out_rx_flags, ETHER_RX_MCAST)						; set multi cast flag
	br[filter_start#]										; go to filtering

not_multicast#:
	move(out_rx_flags, ETHER_RX_UCAST)						; set unicast flag

filter_start#:
	alu[filter_stat,--,B,0x0]								; set stat no pass
	sram[read, $port_type, port_bit_base, 0,1],\
		 sig_done[get_bit_map]								; read the port type bit map
												
	// Search the MAC addr hash table
	.reg index
	.reg index_mask
	.reg cur_port 
	.reg port_mask 
	.reg next_index
	.reg temp

	xbuf_alloc($cur_entry, 2, read)
	_ether_hash_mac(next_index, in_dmac_hi, in_dmac_lo)		; get  Hash Index
	immed[index_mask, 0x3ff] 
	immed[port_mask, 0x3f]
	alu[next_index, next_index, AND, index_mask]			; get valid index bits
	br!=0[loop#]
	// A hash of zero is invalid, set it to 1
	immed[next_index, 1]
loop#:
	alu_shf[index, --, b, next_index, <<3] 					; each entry is 8 bytes wide
	sram[read, $cur_entry[0], eth_flt_table_base, \
		 index, 2], ctx_swap[flt_rd]    					; read the entry
	alu_shf[next_index, $cur_entry[1], AND, index_mask] 	; extract the next index field
	br=0[check_promisc#]									; if next_index == 0 then 
															; there is no hash entry
	alu[temp,--,B,next_index,<<3]							; new index * 8		
	alu[--, temp, -, index] 								; if index == NextIndex we got 
															; the right pointer
	br!=0[next#], defer[2]
	alu_shf[cur_port, port_mask, AND, $cur_entry[1], >>10]	; Compare the port
	alu[--, cur_port, -, in_port]
	br!=0[check_promisc#]									; port does not match

	// No collision so compare the MAC address to see if we have a match
	alu[--, in_dmac_hi, -, $cur_entry[0]]
	br!=0 [check_promisc#]
	alu[temp,--, B, $cur_entry[1], >>16]
	alu[--, in_dmac_lo, -, temp]
	br=0[check_promisc#], defer[1]
	alu[filter_stat,--,B,0x1]								; filer pass
    
	alu[filter_stat,--,B,0x0]
	br[check_promisc#]										; filetr no pass 

next#:
	br!=0[loop#] 											; Port comparison
	alu[--, in_dmac_hi, -, $cur_entry[0]]
	br!=0 [loop#]
	alu[temp,--, B, $cur_entry[1], >>16]
	alu[--, in_dmac_lo, -, temp]
	br=0[check_promisc#], defer[1]
	alu[filter_stat,--,B,0x1]								; filer pass

	br[loop#]

check_promisc#:
	ctx_arb[get_bit_map]									; wait the bit map read done
	br_bset[filter_stat,0,pass#]							; If pass, go to pass.
	alu[--, in_port, or, 0x0]
	alu_shf[--, $port_type, and, 1, <<indirect]				; Check the port type
	br!=0[pass#]											; Port is promisc, pass	
	br[end#],defer[1]										; Port not promisc,not pass
	alu[--, --, B, 0]										; Set result as 0. 
	xbuf_free($cur_entry)									; Free x_buf

pass#:
	// clear the zero condition code since we will keep this packet
	alu[--, --, B, 1]
end#:
.end
#endm

//////////////////////////////////////////////////////////////////////////////
// _ether_hash_mac
//
// Parameters:
//    out_index             - (OUT) The hash of the given mac address
//    in_mac_hi             - (IN) The most significant 32 bits of the MAC
//    in_mac_lo             - (IN) The least significant 16 bits of the 
//                          MAC
//
// Description:
//    L2 Address Soft hash function. Takes a 6 byte L2 address as argument and returns
//    a single byte Index
//
// Returns:
//    The software hash of the given MAC address
//
// Context:
//
// Comments:
//

#macro _ether_hash_mac(out_index, in_mac_hi, in_mac_lo)
.begin
	.reg temp, temp1
	alu[out_index, --, B, in_mac_hi, <<16]
	alu[out_index, --, B, out_index, >>16]
	alu[out_index, in_mac_lo, XOR, out_index]
	alu[temp, --, B, in_mac_hi, >>16]
	alu[out_index, out_index, XOR, temp]
	alu[temp, 0xff, AND, out_index] 
	alu[temp1, 0xff,AND, out_index, >>8] 
	alu[out_index, temp, XOR, temp1] 
.end
#endm

///////////////////////////////////////////////////////////////////////////////
//
// _ether_get_dmac_hi()
//
//	 	Description: 
//				get destination MAC address high 4 bytes from ether header buffer
//			
//
//	 	Outputs:
//				dmac_hi:	destination MAC high 4 bytes 
//		Inputs:
//
//			ether_buf		:	Ethernet header buffer 
//		Size: 
//
//			1 instructions at a minimum. (max depends on the loop)
//
///////////////////////////////////////////////////////////////////////////////

#macro _ether_get_dmac_hi(dmac_hi, in_header, in_header_offset)
.begin
	xbuf_extract[dmac_hi, in_header, in_header_offset,\
				MAC_HI_START, MAC_HI_LEN]						; get DMAC hi

.end
#endm

///////////////////////////////////////////////////////////////////////////////
//
// _ether_get_dmac_lo()
//
//	 	Description: 
//				get destination MAC address low 2 bytes from ether header buffer
//			
//
//	 	Outputs:
//				dmac_lo:	destination MAC low 2 bytes 
//		Inputs:
//
//			in_header	:	Ethernet header buffer 
//			in_header_offset:	Etherney header offset 		
//		Size: 
//
//			 instructions at a minimum. (max depends on the loop)
//
///////////////////////////////////////////////////////////////////////////////

#macro _ether_get_dmac_lo(dmac_lo, in_header, in_header_offset)
.begin
	xbuf_extract[dmac_lo, in_header, in_header_offset,\
				MAC_LO_START, MAC_LO_LEN]						; get DMAC low

.end
#endm

///////////////////////////////////////////////////////////////////////////////
//
// _ether_get_frame_type()
//
//	 	Description: 
//				get Ethernet frame type 
//			
//
//	 	Outputs:
//				type:	Ethernet frame type
//		Inputs:
//
//			ether_buf		:	Ethernet header buffer 
//		Size: 
//
//			1 instructions at a minimum. (max depends on the loop)
//
///////////////////////////////////////////////////////////////////////////////

#macro _ether_get_frame_type(type, in_header, in_header_offset)
.begin

//#define_eval	START	in_header_offset
	xbuf_extract[type, in_header, in_header_offset,\
				FRAME_TYPE_START, FRAME_TYPE_LEN]				; get frame type

.end
#endm

///////////////////////////////////////////////////////////////////////////////
//
// _ether_get_ip_ver()
//
//	 	Description: 
//				get Ip packet version 
//			
//
//	 	Outputs:
//				ip_ver:	IP packet version
//		Inputs:
//
//			ether_buf		:	Ethernet header buffer 
//		Size: 
//
//			2 instructions at a minimum. (max depends on the loop)
//
///////////////////////////////////////////////////////////////////////////////

#macro _ether_get_ip_ver(ip_ver, in_header, in_header_offset)
.begin
	xbuf_extract(ip_ver, in_header, in_header_offset,\
				 ETHER_IP_VER_START, IP_VER_LEN)				; get IP version
	alu[ip_ver, IP_VER_MSK, and, ip_ver,>>4]					; mask out HLEN field
.end
#endm

///////////////////////////////////////////////////////////////////////////////
//
// _ether_decap_classify(ether_buf)
//
//	 	Description: 
//				Ethernet frame destination MAC filtering
//				Decap and classification
//			
//
//	 	Outputs:
//				rx_flag:	The packet filtering type
//				dl_next_block
//		Inputs:
//			
//			ether_buf		:	Ethernet header buffer 
//			flt_table_base	:	MAC filtering hash table base
//		Size: 
//
//			 instructions at a minimum. (max depends on the loop)
//
///////////////////////////////////////////////////////////////////////////////

#macro _ether_decap_classify(in_header, IN_HEADER_OFFSET)

#define_eval	_START	IN_HEADER_OFFSET

.begin
	.reg	dmac_hi										; destination MAC hi
	.reg	dmac_lo										; destination MAC lo
	.reg	in_port										; input port
	.reg	rx_stat

	.if(dl_next_block == BID_ETHER)						; not for me, pass

#ifndef DISABLE_MAC_FILTERING 
		dl_meta_get_input_port(in_port)					; get input port number
		_ether_get_dmac_hi(dmac_hi, in_header,\
						   _START)						; get DMAC hi
		_ether_get_dmac_lo(dmac_lo, in_header,\
						   _START)						; get DMAC lo
		_ether_filter_mac(rx_stat, dmac_hi, dmac_lo,\
					  in_port)							; filtering the MAC address
		br=0[ether_mac_filter_fail#]					
		dl_meta_set_rx_stat(rx_stat)					; set rx state 
#endif
		_ether_classify(in_header, _START)				; do an Ether classification

#ifndef	DISABLE_DECAP
 		.if((dl_next_block != BID_ARP) &&\
		    (dl_next_block != IX_EXCEPTION) )			; ARP and EXCEPTION no decap
		_ether_decap()									; do an Ethernet frame decap

#ifdef	STRIP_4BYTES_CRC
		_ether_strip_fcs()
#endif

#ifdef	INCLUDE_PKT_CKSUM
#ifdef	DISABLE_MAC_FILTERING
#error	"DISABLE_MAC_FILTERING cannot be defined if INCLUDE_PKT_CKSUM is enabled"
#endif
		// since we removed the Ethernet header, we need to update the packet 
		// checksum.
		.begin
		.reg over_all_cksum hdr_cksum
		.reg hdr[4]	

#ifdef		STRIP_4BYTES_CRC
		.reg fcs
			xbuf_extract(fcs, __PKTDATA, 0, fcs_offset, 4)
#endif

			dl_meta_get_cksum(over_all_cksum)

			// extract 14 bytes of the header
			xbuf_extract(hdr[0], in_header, _START, 0, 4);
			xbuf_extract(hdr[1], in_header, _START, 4, 4);
			xbuf_extract(hdr[2], in_header, _START, 8, 4);
			xbuf_extract(hdr[3], in_header, _START, 12, 2);

			// calculate the checksum for the above header
			.begin
			.reg	temp1 temp2
				alu[temp1, hdr[0], +, hdr[1]]
				alu[temp1, hdr[2], +carry, temp1]
				alu[temp1, hdr[3], +carry, temp1]

#ifdef		STRIP_4BYTES_CRC
				alu[temp1, fcs, +carry, temp1]
#endif
				alu[temp1, 0x0, +carry, temp1]	// add final carry, if any
				ld_field_w_clr[temp2, 0011, temp1, >>16]
				alu[temp1, temp2, +16, temp1]
				alu_shf[temp2, --, B, temp1, >>16]
				alu[hdr_cksum, temp2, +16, temp1]
			.end

			// now we need to subtract this from the overall checksum
			alu[hdr_cksum, --, ~B, hdr_cksum]
			alu[hdr_cksum, over_all_cksum, + , hdr_cksum]
			alu[over_all_cksum, 0x0, +carry, hdr_cksum]	// upper 16bits might be set
			dl_meta_set_cksum(over_all_cksum)

		.end
#endif
		.endif
#endif

		br[ether_process_done#]

#ifndef DISABLE_MAC_FILTERING 
ether_mac_filter_fail#:
		alu[dl_next_block, --, B, BID_DROP]				; the packet should be droped
#endif

ether_process_done#:
	.endif	
.end
#endm

#endif	//	__ETHER_UC__
