/////////////////////////////////////////////////////////////////////////////////////
//                                                                      
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
/////////////////////////////////////////////////////////////////////////////////////
//
// 		File Name: qm_scheduler_packet.uc
// 
// 		Purpose:	  DRR scheduler and queue manager 
//
/////////////////////////////////////////////////////////////////////////////////////
//
// 		History:
//
// 		Date			Comment										By
//		---------------------------------------------------------------------------
//
//		03/17/02		Created										Uday Naik
//
/////////////////////////////////////////////////////////////////////////////////////

#ifndef __QM_SCHEDULER_PACKET_UC__
#define __QM_SCHEDULER_PACKET_UC__

/////////////////////////////////////////////////////////////////////////////////////
// 
// This scheduler runs on the egress Sausalito. It is a frame or packet based 
// scheduler. It implements WRR scheduling on the ports and DRR on the queues
// within a port. Currently 16 ports and 6 classes per port are supported
//
// 
// The following are the issues with implementing DRR under the IXP2400 environment
// 
//      1)  The packet size is not available for a queue when it is being scheduled.
//          Once the dequeue is issued, the packet size is received N beats later 
//          where each beat is 88 cycles.
//
//      2)  The bit vector with information on which queues have data may not be
//          current. This could mean that a dequeue is issued on a queue that has
//          no data
//
//      3)  The scheduler schedules a packet every beat (88 cycles). This means that
//          for large packets, the scheduler is running faster than the transmit
//
//      4)  The exact packet size is not available. The packet size is in multiples 
//			of CHUNK_SIZE which is MTU/128. The queue credit increments are also 
//			given in CHUNK_SIZE units.
//
//      5)  During a DRR round, a queue may go empty or come alive. While a queue is
//          empty it should not receive credit. But queues that frequently go empty
//          should not affect the bandwidth allocation.
//
// The above issues are worked around with modifications to DRR as follows
//    
//      1) We use a scheme of negative credits. The criteria for a queue to be 
//		   eligible to send is that it has data, flow control is off on the port and 
//		   the credits for the queue are positive. A packet is transmitted from a 
//		   queue if it meets the above criteria. Once the packet length is received, 
//		   (N beats later), the packet length is decremented from the current credit 
//		   of the queue. When the current credit of the queue goes negative, it can 
//		   no longer transmit. When all the queues on a port go negative, one DRR 
//		   round is over. Each queue gets another round of credit at this point. 
//		   To ensure that all the queues are schedulable with one round of credit, 
//		   we need to keep the minimum quantum for a queue as (N *  MTU)/CHUNK_SIZE.
//
//      2) If a dequeue is issued on a queue that has no data, then the QM returns 
//		   the packet size as 0. This is treated as a special case. The scheduler 
//		   will run slightly faster than the QM to allow it to make up for lost slots.
//
//      3) If the queue between TX and QM gets full due to large packets or because 
//		   the scheduler is running slightly faster, then the QM will not dequeue the
//		   packet and instead return a 0 for the packet size.
//
//      4) The algorithm will round robin among ports first and queues next. i.e. if 
//         queue i of port j is scheduled, the next queue scheduled will be queue k 
//		   of port j + 1. When the scheduler comes back to port j, the next queue 
//		   scheduled in port j will be port i+1. This increases the probability that 
//		   the packet length is back by the time the queue 
//  
//      5) While a queue is empty or flow control is on, it's credit remains 
//		   untouched. If a queue comes alive in the middle of a round, it is allowed 
//		   to participate right away with the available credit. 
//         
//  
/////////////////////////////////////////////////////////////////////////////////////
// 
// Absolute register
// 
// PortEmptyVector	: Port Empty Bit Vector  : 1 word (NUMBER_OF_PORTS bits are used)
//                    If bit is set then port has data
//
/////////////////////////////////////////////////////////////////////////////////////

//.reg @port_empty_vector			; Absolute GPR (Global shared across threads)

////////////////////////////////////////////////////////////////////////////////////

// include stdmac.uc in IXPblocks Portable library

#include <stdmac.uc>

// include localmem.uc in IXPblocks Portable library for read/write local memory

#include <localmem.uc>

// include file system.h for system constants

#include "dl_system.h"

// header file with constants for algorithm

#include "scheduler_packet.h"
#include "qm_packet_include.h"

// include the queue manager message handling code 

#include "scheduler_qm.uc"
#include "scheduler_packet.uc"

#include "qm_scheduler_packet_init.uc"

#include "qm_packet_macro.uc"
#include "qm_packet_message.uc"
#include "qm_packet_code.uc"

// The xfer registers 0..15 will be used as a target for a reflector write by 
// the transmit microengine. They will contain the count of the packets transmitted 
// and will be accessed in absolute indexed mode. 

.reg volatile visible $$txd_p0, $$txd_p1, $$txd_p2, $$txd_p3, $$txd_p4, $$txd_p5, $$txd_p6, \
			          $$txd_p7, $$txd_p8, $$txd_p9, $$txd_p10, $$txd_p11, $$txd_p12, \
			          $$txd_p13, $$txd_p14,$$txd_p15
	
.xfer_order	 $$txd_p0, $$txd_p1, $$txd_p2, $$txd_p3, $$txd_p4, $$txd_p5, $$txd_p6, \
			 $$txd_p7, $$txd_p8, $$txd_p9, $$txd_p10, $$txd_p11, $$txd_p12, \
			 $$txd_p13, $$txd_p14, $$txd_p15

/* This signal is declared in system_init() macro */
.sig visible sys_init_signal

#define INTER_THREAD_SIGNAL 	5

.sig volatile	prev_thread_signal
.addr 			prev_thread_signal		INTER_THREAD_SIGNAL

/////////////////////////////////////////////////////////////////////////////////////
//
// main() : Code execution starts here
//
/////////////////////////////////////////////////////////////////////////////////////


// If thread number is 6, go to the Queue Manager Message Handler routine

br=ctx[ 6, QM_HANDLER_THREAD#]

// If thread number is 7, go to the scheduler routine. 

br=ctx[ 7, SCHEDULER_THREAD#]

// All other threads support Packet Queue Manager

/////////////////////////////////////////////////////////////////////////////////////	

// The xfer registers 0..15 will be used as a target for a reflector write by 
// the transmit microengine. They will contain the count of the packets transmitted 
// and will be accessed in absolute indexed mode. Since xfer registers 0..15 belong 
// to thread 0 in relative mode, this thread will initialize them to 0 by reading in 
// 0's into these registers initially. It will signal the scheduler thread once done

/////////////////////////////////////////////////////////////////////////////////////

QM_PACKET#:

.begin

    qm_packet_init[]

    queue_manager[]

.end


/////////////////////////////////////////////////////////////////////////////////////

QM_HANDLER_THREAD#:

	// Set up the registers for next neighbor and scratch rings

	_qm_scheduler_init_rings()

	// signal the scheduler thread 

	#define_eval INTER_THREAD_SIGNAL_REG_VALUE (0x80 | (INTER_THREAD_SIGNAL << 3))

	local_csr_wr[SAME_ME_SIGNAL, INTER_THREAD_SIGNAL_REG_VALUE]

	nop
	nop

	// Execute in an infinite loop handling QM messages

	_scheduler_qm_message_handler[]

/////////////////////////////////////////////////////////////////////////////////////

SCHEDULER_THREAD#:

.begin
	
	// Scheduler thread

	.reg	port_credit_vector
	.reg	port_init_credit_vector
	.reg 	port_mask
	.reg 	minus_two
	.reg    ring
	.reg	$deq1, $deq2, $deq3, $deq4
	.sig	deq_signal1, deq_signal2, deq_signal3, deq_signal4

	// Wait for a signal from thread 0

	qm_scheduler_init[port_credit_vector, port_init_credit_vector, ring, port_mask, minus_two]

	ctx_arb[prev_thread_signal]

	// Execute in thread 1 in an infinite loop

SCHEDULE#:
	
	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
			 minus_two, $deq1, deq_signal1, voluntary]

	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
			 minus_two, $deq2, deq_signal2, voluntary]

	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
			 minus_two, $deq3, deq_signal3, voluntary]

	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
			 minus_two, $deq4, deq_signal4, deq_signal1]


SCHEDULE_LOOP#:

	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
			 minus_two, $deq1, deq_signal1, deq_signal2]

	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
	 		 minus_two, $deq2, deq_signal2, deq_signal3]

	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
			 minus_two, $deq3, deq_signal3, deq_signal4]

	schedule[port_credit_vector, port_init_credit_vector, ring, port_mask, \
			 minus_two, $deq4, deq_signal4, deq_signal1]

	br [SCHEDULE_LOOP#]

.end

/////////////////////////////////////////////////////////////////////////////////////

#endif   // __SCHEDULER_EGRESS_DRR_UC__

/////////////////////////////////////////////////////////////////////////////////////
/*******************************  FILE HISTORY  *******************************
 * \main\1 Wed Aug 13 14:03:20 2003 adziecia
 * IGK0100025024: ATM-IPV4-ETH: simulation project check-in 
 *  
 *****************************************************************************/
