/////////////////////////////////////////////////////////////////////////////////////////
//                                                                      
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119 
//
/////////////////////////////////////////////////////////////////////////////////////////
// 		
//		Change History
// 		--------------
//
// Date			Description											Whom
// ---------------------------------------------------------------------------------
//
// 12/11/01    	Queue Manager for IXP2400  						Aneet Chopra                 
// 11/01/02		Modified for IXP2800 - Cell Mode				Uday Naik
// 12/10/02		Modified for IXP2800 - Packet Mode				Dennis Tran
//                                                                  
/////////////////////////////////////////////////////////////////////////////////////////

#ifndef __QM_CODE_UC__
#define __QM_CODE_UC__

/////////////////////////////////////////////////////////////////////////////////////////

#include "qm_macro.uc"

/////////////////////////////////////////////////////////////////////////////////////////
//
// Macro Name  : _qm_handle_enqueue
//
// Description : Handle the enqueue request 
//
// Cycles:	20 in miss and 11 in hit path
//
/////////////////////////////////////////////////////////////////////////////////////////

#macro _qm_handle_enqueue[ $out_qdesc_enqueue, in_enqueue_queue_number, \
						   in_sigmask, in_sop_handle, in_eop_handle ]
.begin

	.reg	qarray_entry	// index 0..15 into Q-Array hardware 

	// Check if entry is in CAM. If there is a CAM miss then write back the 
	// current CAM entry and read in the new entry based on the queue number. (14)
		
	_qm_cam_check( in_enqueue_queue_number, qarray_entry, $out_qdesc_enqueue, \
				   in_sigmask, sig_q_array_enq_r_done, 0, ENQ)  

	// Enqueue a packet (6)

	#ifdef PACKET_MODE

	_qm_enqueue_packet_mode(in_sop_handle, qarray_entry) 

	#else

	_qm_enqueue_cell_mode(in_sop_handle, in_eop_handle, qarray_entry) 
		
	#endif

	#ifdef _DEBUG_COUNTERS_
	alu[@qm_num_enq_requests, @qm_num_enq_requests, +, 1]
	#endif

.end
#endm

/////////////////////////////////////////////////////////////////////////////////////////
//
// Macro Name  : _qm_handle_dequeue
//
// Description : Handle the dequeue request  
//
// Cycles:	15 in miss and 6 in hit path
//
/////////////////////////////////////////////////////////////////////////////////////////

#macro _qm_handle_dequeue[ $out_qdesc_dequeue0, $out_dequeued_handle, \
						   in_dequeue_queue_number, in_sigmask ]
.begin

	.reg	qarray_entry	// index 0..15 into Q-Array hardware 

	// Check if entry is in CAM. If there is a CAM miss then write back the 
	// current CAM entry and read in the new entry based on the queue number. (14)
		
	_qm_cam_check( in_dequeue_queue_number, qarray_entry, $out_qdesc_dequeue0, \
				   in_sigmask, sig_q_array_deq_r_done, 0, DEQ)  

	// Dequeue a cell/packet (1)

	_qm_dequeue(qarray_entry, $out_dequeued_handle)

.end
#endm

/////////////////////////////////////////////////////////////////////////////////////////
//
// Macro Name  : _qm_read_enqueue_dequeue_request
//
// Description : Read the enqueue/dequeue request  
//
// Cycles:	4
//
/////////////////////////////////////////////////////////////////////////////////////////

#macro _qm_read_enqueue_dequeue_request[ in_message0, in_sop_handle, in_eop_handle, \
										 no_request_label ]
.begin

	br_inp_state[NN_EMPTY, no_request_label]	

	// Read enqueue and dequeue messages (3)

	alu[in_message0,   --, B, *n$index++]						
	alu[in_sop_handle, --, B, *n$index++]
	alu[in_eop_handle, --, B, *n$index++]

.end
#endm

/////////////////////////////////////////////////////////////////////////////////////////
//
// Macro Name  : queue_manager
//
// Description : This is the main macro where the queue manager algorithm
//				 is implemented. The QM is designed to handle an enqueue 
//	 			 dequeue request per beat. Based on this the QM then needs
//				 to send messages to the Transmit blocks.
//
// Output      : Nil
//
// Input       : message 	: register array for enqueue/dequeue messages
//			
//
// Constant	   : Nil 
//
// Size        : 
//
// Branches    : 
//
/////////////////////////////////////////////////////////////////////////////////////////

#macro queue_manager()	
.begin

	.reg		message0							// message LW0 from NN ring 
	.reg		enqueue_queue_number				// enqueue queue number
	.reg		dequeue_queue_number				// dequeue queue number
	.reg		$qdesc_enqueue						// queue descriptor for enqueue
	.reg		$qdesc_dequeue						// queue descriptor for dequeue
	.reg		sop_buffer_handle					// sop buffer handle
	.reg		eop_buffer_handle					// eop buffer handle 
	.reg		$dequeued_handle					// handle dequeue from qArray
	.reg		sigmask								// signal mask 

	// This is where the loop begins for the queue manager

qm_loop#:

	// set signal mask to default (1)

	alu[sigmask, --, B, SIGMASK_DEFAULT]

	// signal the next ctx well before we go to sleep (1)

	_qm_signal_next_ctx(gl_next_context_sig) 
	
	// Read the enqueue/dequeue request (4)

	_qm_read_enqueue_dequeue_request[ message0, sop_buffer_handle, eop_buffer_handle, \
									  no_request# ]

handle_enqueue_request#:

	// extract enqueue queue number from the message long word 0 (1)

	ld_field_w_clr[enqueue_queue_number, 0011, message0] , load_cc
	
	// check if queue number is invalid (1)

	beq[handle_dequeue_request#]

	// handle the enqueue request (20 for miss, 11 in hit case)

	_qm_handle_enqueue[ $qdesc_enqueue, enqueue_queue_number, 
						sigmask, sop_buffer_handle, eop_buffer_handle ]

handle_dequeue_request#:

	// extract dequeue queue number  (1)

	ld_field_w_clr[dequeue_queue_number, 0011, message0, >>16], load_cc 

	// check if it is invalid (1)

	beq[no_dequeue_request#]

	// handle the dequeue (15 for miss and 6 for hit)

	_qm_handle_dequeue[ $qdesc_dequeue, $dequeued_handle, dequeue_queue_number, \
						sigmask ]

wait_for_all#:

	// Wait for the I/O to finish and prev thread signal (3)
		
	ctx_arb[--] , defer [1]
		
		local_csr_wr[active_ctx_wakeup_events, sigmask]	

	// let the assembler know these signals are done

	.io_completed sig_q_array_deq_r_done sig_q_array_enq_r_done sig_deq_done 

	// check if dequeued handle is valid
	alu[--, $dequeued_handle, -, 0]

	// if invalid, do not transmit
	beq[qm_loop#]

	// send dequeued handle to transmit block and loop back to the top (4)

	_qm_send_transmit_message[$dequeued_handle, dequeue_queue_number, qm_loop#] 
		
	// The previous macro ends by jumping to the top of the loop 

no_dequeue_request#:		

	// Wait for the I/O to finish and prev thread signal (4)
	
	ctx_arb[--] , defer [2] 

		// defer slot 1 - remove the dequeue signal from the signal mask 

		alu[sigmask, sigmask, AND~, 1, <<(&sig_deq_done)]
			
		// load the signal mask into the signal event register 
			
		local_csr_wr[active_ctx_wakeup_events, sigmask]	

	// let the assembler know these signals are done

	.io_completed sig_q_array_deq_r_done sig_q_array_enq_r_done 

	br[qm_loop#]

no_request#:

	// No request for enqueue or dequeue. so just swap out and wait for the signal
	// from the previous thread. Then jump back to the top of the loop 

	ctx_arb[sig_prev_thread] , br[qm_loop#]


.end
#endm

/////////////////////////////////////////////////////////////////////////////////////////
//
// This is where the code begins..................
//
/////////////////////////////////////////////////////////////////////////////////////////

main#:

.begin

	// Initialization of global and absolute registers, Q_array and local memory

	qm_init()

	// Here is where the Queue Manager loop begins
	// Code loops inside queue_manager()

	queue_manager()

error#:

	nop ; should never get here

.end

/////////////////////////////////////////////////////////////////////////////////////////

#endif // __QMCODE_UC__

/////////////////////////////////////////////////////////////////////////////////////////