////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ipv4_mcast_fwder.uc
//
//      Purpose: This file contains the ipv4_mcast_fwder macro which implements the 
//		IPV4 Multicast fwding functionality. Calls in internal macros. 
//
//
//////////////////////////////////////////////////////////////////////////////

#ifndef	__IPV4_MCAST_FWDER_UC__
#define	__IPV4_MCAST_FWDER_UC__

//	Include Files

	// include xbuf.uc from IXPblocks portable library
#include	"xbuf.uc"

	// include ipv4.h in IXPblocks Portable library
#include	"ipv4.h"

	// include ipv4.uc in IXPblocks Portable library to process IP header and
	// LPM macro.
#include	"ipv4.uc"

	// include IPV4 Multicast Forwarder definitions for symbols, excp codes, 
	// and other misc. definitions.  

#include	"ipv4_mcast_fwder_uc.h"
#include	"ipv4_fwder_uc.h"
#include	"ipv4_fwder_util.uc"



/////////////////////////////////////////////////////////////////////////////
//	
//	ipv4_mcast_fwder
//
//	This macro performs address checks specific to Multicasting only. Rest
//	of the RFC1812/RFC2644 checks are expected to be completed by IPV4
//	Unicast Forwarder (a.k.a simply IPV4 Forwarder) block.
//
//	This block processes only multicast packets.
//
//	Currently no statistics are maintained.
//
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//	_ipv4_mcast_fwder_init()
//
//	DESCRIPTION
//		Initialize necessary global variables.
//
//	Output
//
//		None.
//
//	Input
//
//		None.
//
/////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_mcast_fwder_init()

	// declare some registers

.reg	mfib_table_base			; pointer to MFIB
.reg	@ipv4_mcast_stats_base
.reg	table_entry_mask_const

	move[mfib_table_base, IPV4_MFIB_TABLE_BASE]

	move[@ipv4_mcast_stats_base, IPV4_STATS_TABLE_BASE]

	move[table_entry_mask_const, 0xffff]

	alu_shf[table_entry_mask_const, --, B, table_entry_mask_const, <<MFIB_ENTRY_SHF_VAL]

#endm

/////////////////////////////////////////////////////////////////////////////
//
//	_ipv4_mcast_fwder_validate_header()
//
//	Description
//		Validate IPV4 header. Perform multicast related checks. Also perform
//		checks not performed by unicast forwarder microblock
//
//	Output
//
//		out_result	=	Result of this macro IXP_IPV4_SUCCESS or IPV4_FAILURE
//						or exception code.
//
//	Input
//
//		in_ip		=	buffer containing the IPV4 Packet.
//		RD_START	=	Start offset of IPV4 header within this buffer.
//
/////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_mcast_fwder_validate_header(out_result, in_ip, RD_START)

.begin

	.reg	ver_hdr
	.reg	sa
	.reg	da
	.reg	tmp
	.reg	tmp1

	// check if this packet has options. This is not checked by
	// IPV4 Unicast Forwarder. We also don't check for IPV4 Version
	// as this is already checked by IPV4 Unicast Forwarder

	xbuf_extract(ver_hdr, in_ip, RD_START, IP_VERSION_LENGTH)

	alu[--, ver_hdr, - , 0x45]
	bgt[option_exception#]

	xbuf_extract(sa, in_ip, RD_START, IP_SOURCE_ADDRESS)

	xbuf_extract(da, in_ip, RD_START, IP_DESTINATION_ADDRESS)

	// if dst. is not multicast, drop

	alu[tmp, da, AND, 0xf, <<28]

	br!=byte[tmp, 3, 224, invalid#]

	// even within multicast address, currently we don't process
	// all

	// drop 232.0.0.0

	move[tmp, 0xe8000000]

	alu[--, da, -, tmp]

	beq[invalid#]

	// exception for 224.0.0.x
		
	move[tmp, 0xe00000]

	alu_shf[tmp1, --, B, da, >>8]

	alu[--, tmp1, -, tmp]

	beq[special_addr_excp#]

	// drop all other except 232.x.x.x

	br!=byte[da, 3, 0xe8, invalid#]

	// succesful, exit this macro

	br[exit#], defer[1]

	immed[out_result, IXP_IPV4_SUCCESS]

	// error and exception cases below.

invalid#:
	br[exit#], defer[1]

	immed[out_result, IPV4_FAILURE]

option_exception#:

	br[exit#], defer[2]

	dl_set_exception_priority[1]

	immed[out_result, IPV4_MCAST_EXCP_OPTIONS]

special_addr_excp#:

	br[exit#], defer[2]

	dl_set_exception_priority[1]

	immed[result, IPV4_MCAST_EXCP_SPECIAL_ADDR]


exit#:
.end

#endm


/////////////////////////////////////////////////////////////////////////////
//
//	_ipv4_mcast_fwder_lookup()
//
//	Description
//		Perform multicast forwarding information base lookup. This lookup
//		is performed using S, G fields from the packet.
//
//	Output
//
//		out_result	=	Result of this macro IXP_IPV4_SUCCESS or IPV4_FAILURE
//						or exception code.
//
//	Input
//
//		in_index	=	Index of the bucket (primary) to start processing.
//						Index is calculated outside this macro.
//		in_sa		=	Source address of IPV4 packet (S)
//		in_da		=	Destination address of IPV4 packet (G)
//		in_mfib_table_base	=	Base address of Multicast forwarding 
//								Information base.
//
/////////////////////////////////////////////////////////////////////////////

#macro	_ipv4_mcast_fwder_lookup(out_result, in_index, in_sa, in_da, \
			in_mfib_table_base)

.begin

	.reg	entry_offset

	// get the entry offset

	alu_shf[entry_offset, --, B, in_index, <<MFIB_ENTRY_SHF_VAL]
	
lookup_start#:

	sram[read, $entry[0], entry_offset, in_mfib_table_base, IPV4_MFIB_ENTRY_SIZE_LW], \
		ctx_swap[sig_mfib]

	// check if valid bit is set. We need to check for only the primary try.
	
	br_bclr[$entry[MFIB_NEXT_ENTRY], MFIB_ENTRY_VALID_BIT, match_not_found#]

lookup_loop#:

	// compare the entries

	alu[--, $entry[MFIB_GRP_ADDR], -, da]
	bne[lookup_next#]

	alu[--, $entry[MFIB_SRC_ADDR], -, sa]
	bne[lookup_next#]

	// RPF check

	ld_field_w_clr[tmp, 0001, $entry[MFIB_SRC_PORT], >>16]
	ld_field[tmp, 1000, $entry[MFIB_SRC_PORT]]

	alu[--, tmp, -, in_port]
	beq[drop#]

	// match found. Check flags

	br_bset[$entry[MFIB_FLAGS], IPV4_MCAST_LOCAL_BIT, local_exception#]
	br_bset[$entry[MFIB_FLAGS], IPV4_MCAST_DROP_BIT, drop#]
	
	// forward the packet.
	
	ld_field_w_clr[tmp, 0011, $entry[MFIB_MTID]]
	dl_meta_set_nexthop_id[tmp]

	dl_meta_set_output_port[0x0]

	dl_meta_set_fabric_port[MCAST_RESERVED_BLADE_ID]

	dl_meta_set_class_id[0x0]

	br[exit#], defer[1]

	immed[result, IPV4_SUCCESS]

lookup_next#:
	
	// get new hash key
	
	alu_shf[entry_offset, table_entry_mask_const, AND, $entry[MFIB_NEXT_ENTRY], >>4]

	beq[match_not_found#]

	sram[read, $entry[0], entry_offset, in_mfib_table_base, 5], sig_done[sig_mfib]
	
	ctx_arb[sig_mfib], br[lookup_loop#]

match_not_found#:

	br[exception#], defer[1]
	immed[result, IPV4_MCAST_EXCP_LOOKUP_FAILED]
	
drop#:

	br[exit#], defer[1]
	immed[result, IPV4_FAILURE]

local_exception#:
	
	dl_set_exception_priority[1]
	immed[result, IPV4_MCAST_EXCP_LOCAL]


exit#:

.end

#endm

/////////////////////////////////////////////////////////////////////////////
//
//	_ipv4_mcast_fwder()
//
//	Description
//		Implement IPV4 Multicast Forwarding functionality. Performs
//		header checks, hash lookup.
//
//		Implement PIM-SM (draft-ietf-pim-sm-v2-new-08.ps) = Future 
//		Release.
//
//	Output
//
//		out_ip	=	Buffer containing the IPV4 packet copied from 
//					input. The IPV4 Packet in this buffer has TTL
//					and checksum updated.
//
//	Input
//
//		in_ip		=	buffer containing the IPV4 Packet.
//		WR_START	=	Start offset of IPV4 header within output buffer.
//		RD_START	=	Start offset of IPV4 header within input buffer.
//		EXCP_COPY_HDR	=	Specify whether to copy header or not, incase
//					this block encounters an exception. 
//
/////////////////////////////////////////////////////////////////////////////

#macro	_ipv4_mcast_fwder(out_ip, in_ip, WR_START, RD_START, EXCP_COPY_HDR)

.begin

	.reg	sa				; source addr
	.reg	da				; dst addr
	.reg	tmp				; for storing intermediate values
	.reg	hash_key		; to store address of hash table entry
	.reg	in_port			; Blade ID and Input port information
	.reg	result		; store the result of various checks
	.reg	counter_base	; stats counter base

	.sig	sig_mfib		; signal used to read MFIB

	// start processing here

	xbuf_alloc[$entry, 5, read]

start#:
	
	// check if the packet is for this block
	
	br!=byte[dl_next_block, 0, BID_IPV4_MCAST_FWDER, hdr_copy#]

	immed[result, IPV4_SUCCESS]

	// activate header caching

	#ifdef	IP_HDR_CACHE_LM

		.begin
			.reg	thread_id

			dl_get_thread_id(thread_id)

			xbuf_activate(in_ip, 0, thread_id, 0)
			xbuf_activate(out_ip, 1, thread_id, 0)

		.end
	#endif


	// get input blade ID

	dl_meta_get_input_port(in_port)

	// build counter base for this port.

	alu[counter_base, -- , B, @ipv4_mcast_stats_base]

	alu_shf[counter_base, counter_base, OR , in_port, <<STATS_PER_PORT_SHF_VAL]

	alu_shf[in_port, in_port, OR, THIS_BLADE_ID, <<24]

	// reset the remainder in CRC

	local_csr_wr[crc_remainder, 0x0]

	// check if lookup is needed 

	_ipv4_fwder_is_lookup_needed[update_ipv4_hdr#]


	// extract source and destination addresses. Also start CRC
	// operation.
	
	xbuf_extract(sa, in_ip, RD_START, IP_SOURCE_ADDRESS)

	crc_be[crc_32, --, sa]	

	xbuf_extract(da, in_ip, RD_START, IP_DESTINATION_ADDRESS)

	crc_be[crc_32, --, da]

	_ipv4_mcast_fwder_validate_header(result, in_ip, RD_START)

	br!=byte[result, 0, IXP_IPV4_SUCCESS, check_macro_result#]

	// now perform hash lookup. read the CRC remainder. We MUST have
	// atleast 5 instructions before reading remainder.

	local_csr_rd[crc_remainder]

	immed[hash_key, 0x0]

	// take LSB 10 bits

	move[tmp, 0x3ff]

	alu[hash_key, hash_key, AND, tmp]

#ifdef	_UNIT_TEST_

	// for unit testing, we simply do some xor to get hash key. We will overwrite
	// the hash key obtained earlier

	.begin

		.reg	tmp1

		alu[hash_key, sa, XOR, da]

		ld_field_w_clr[tmp1, 0011, hash_key, >>16]

		alu_shf[hash_key, hash_key, XOR, tmp1]

		alu[hash_key, hash_key, AND, tmp]

	.end
	

#endif	// _UNIT_TEST_

	_ipv4_fwder_update_counter(OFFSET_PKTS_FWD)

	// perform the lookup

	_ipv4_mcast_fwder_lookup(result, hash_key, sa, da, mfib_table_base)

	br!=byte[result, 0, IPV4_SUCCESS, check_macro_result#]

	// otherwise succesful

	// now decrement the TTL, update the checksum. 

update_ipv4_hdr#:

	_ipv4_fwder_update_header(result, out_ip, in_ip, WR_START, RD_START)

	br!=byte[result, 0, IPV4_SUCCESS, drop#]

	// done processing, exit this block

	br[exit#], defer[1]

	immed[dl_next_block, BID_IPV4_MCAST_FWDER_NEXT1]


///////////////////////////////   ERROR CASES ///////////////////////////////////

check_macro_result#:
	
	alu[--, result, - , IPV4_FAILURE]
	beq[drop#]
	
exception#:
	// if here, we are in exception path
	_ipv4_fwder_update_counter(OFFSET_PKTS_EXCP)

	dl_set_exception[BID_IPV4, result]
	immed[dl_next_block, BID_IPV4_MCAST_EXCP_NEXT1]

hdr_copy#:

	#if	(EXCP_COPY_HDR == 1)

		// if required we copy header for exception packets.
		xbuf_copy(out_ip, 0, WR_START, in_ip, RD_START, 0, 20, 0]

	#endif

	br[exit#]

	// drop packet if it reaches here

drop#:

	_ipv4_fwder_update_counter(OFFSET_PKTS_DROP)

	immed[dl_next_block, IX_DROP]


	// deactivate header caching

	#ifdef	IP_HDR_CACHE_LM

		xbuf_deactivate(in_ip)
		xbuf_deactivate(out_ip)

	#endif

////////////////////////  exit the macro here ///////////////////////////////
	xbuf_free[$entry]

.end

exit#:

#endm	// end of ipv4_mcast_fwder


#endif	// __IPV4_MCAST_FWDER_UC__



