/////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
/////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ipv4_fwder_util.uc
//
//      Purpose: Implements internal macros for ipv4_fwder. 
//				 Calls in IXP IPV4 macros.
//
/////////////////////////////////////////////////////////////////////////////

#ifndef	__IPV4_FWDER_UTIL_UC__
#define	__IPV4_FWDER_UTIL_UC__


//	Contents
//	--------
//	_ipv4_fwder_validate_header()	:	Macro to validate IPV4 header according
//										to RFC1812/RFC2644.
//	_ipv4_fwder_proc_nexthop_info()	:	Macro to process nexthop information.
//	_ipv4_fwder_update_header()		:	Macro to perform necessary modification
//										to IPV4 header.
//	_ipv4_fwder_is_lookup_needed()	:	Macro to determine if route lookup 
//										needed.
//	_ipv4_fwder_lookup()			:	Macro to perform LPM.
//	_ipv4_fwder_update_counter()	:	Macro to atomically update counter.
//	
//	No public API exported.
//

/////////////////////////////////////////////////////////////////////////////

//	Include Files
//	None.

/////////////////////////////////////////////////////////////////////////////

//	Constants 
//	None.

/////////////////////////////////////////////////////////////////////////////

//	Global Registers
//	None.

/////////////////////////////////////////////////////////////////////////////

//	Global Signals
//	None.

/////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_validate_header[]
//
//	 	Description: 
//
//		Validates IP header according to RFC1812. 
//
//	 	Outputs:
//
//			result	-	result of this macro. Can be IPV4_SUCCESS
//				or IPV4_FAILURE or an exception code.
//
//		Inputs
//
//			in_ip	-	Registers (Transfer) containing the IP
//				packet. 
//
//			IPHDR_READ_START	- This is where the IP hdr starts in buffer
//
//			options_label	- Packet with options will bypass rest of the 
//						checks as this packet will go to the core.
//
//		Global variables
//			Assumes the following variables are set. Simply reads
//			these variables.
//				ver_hdr
//				ip_total_len
//
//		CONSTANTS
//
//			IPHDR_READ_START	- The relavive start byte position 
//				of IP header w.r.t start of the register buffer.
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
//
/////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_validate_header[out_result, in_ip, IPHDR_READ_START, options_label]

// remove spaces from arguments passed
#define_eval	VH_RD_START		IPHDR_READ_START

.begin
	.reg	packet_size
	.reg	temp
	// verify if the packet received is atleast 20 bytes
	dl_meta_get_packet_size[packet_size]	; get packet size
	alu[ -- , packet_size, - , 20] 
	blt[ipv4_fwder_small_pkt#]	; if size<20, then error.

	// verify Hdr version and options
	br!=byte[ver_hdr, 0, 0x45, check_for_options#]

#ifdef	RFC1812_SHOULD
	// check if size reported by L2 is lessthan the total length
	// field.
	alu[ -- , packet_size, - , ip_total_len]
	blt[length_mismatch#]
#endif

	ipv4_verify[out_result, in_ip, VH_RD_START]
	br!=byte[out_result, 0, IPV4_SUCCESS, check_exception#]

#ifdef	RFC2644_CHECKS

	.begin
	.reg	src_addr
		xbuf_extract(src_addr, in_ip, VH_RD_START, IP_SOURCE_ADDRESS)

#ifdef	LM_DBCAST_TABLE
		ipv4_is_dbcast_lm(out_result, src_addr, LM_DBCAST_BASE, //
				LM_DBCAST_BLOCK_COUNT, LM_DBCAST_BLOCK_SIZE, LM_DBCAST_INDEX)
#else
		// directed broadcast checks for source address
		ipv4_is_dbcast(out_result, src_addr, sram_dbcast_base,DBCAST_TABLE_BLOCK_SIZE)
#endif

	.end
#endif

	// if we execute this, we got a good packet. Exit this macro.
	br[ipv4_fwder_validate_header_end#]

	// Below, we perform non-critical path checks for errors and
	// exceptions.


#ifdef	RFC1812_SHOULD
	// length mismatch
length_mismatch#:

	_ipv4_fwder_update_counter(OFFSET_PKTS_BAD_LEN)
	br[ipv4_fwder_validate_header_end#], defer[1]
	immed[out_result, IPV4_EXCP_LENGTH_MISMATCH]
#endif

	// here we check for options
	// check if the version is 0x4. We 

check_for_options#:

	alu[temp, ver_hdr, AND, 0xf0]
	br!=byte[temp, 0, 0x40, invalid#]

	// now check if hdr length is atleast 5 LW
	alu[--, ver_hdr, - , 0x45]
	blt[invalid#]

	// if here, this packet is with options. Jump to options
	// label as we will check for options after
	// filtering packets destined to local stack.

	br[options_label]

check_exception#:
	alu[--, out_result, - , IPV4_FAILURE]
	bne[ipv4_fwder_validate_header_end#]

	// packet with bad header.
invalid#:
	// update counter
	_ipv4_fwder_update_counter(OFFSET_PKTS_BAD_HDR)

	br[ipv4_fwder_validate_header_end#], defer[1]
	immed[out_result, IPV4_FAILURE]

	// The received packet was too small. So increment a counter and 
	// drop the packet.
ipv4_fwder_small_pkt#:
	_ipv4_fwder_update_counter(OFFSET_PKTS_TOO_SMALL)
	br[ipv4_fwder_validate_header_end#], defer[1]
	immed[out_result, IPV4_FAILURE]

ipv4_fwder_validate_header_end#:

// cleanup the namespace.
#undef	VH_RD_START
.end

#endm

//////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_proc_nexthop_info[]
//
//	 	Description: 
//
//		Read and process the Next hop information. Set some
//		DL variables accordingly.
//
//			Note the order of parameters. This order, output, output/input,
//			input, input constants MUST be followed.
//
//	 	Outputs:
//
//			result	-	result of this macro. Can be IPV4_SUCCESS
//				or IPV4_FAILURE or an exception code.
//
//		Inputs
//
//			NONE. 
//
//		Global variables.
//			Uses the following state variables.
//			in_port	
//			ip_total_len
//			nexthop_index 
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_proc_nexthop_info(out_result)

.begin
proc_nexthop_info_begin#:

	// see the description of this compilation switch in ipv4_fwder_init.uc
#ifdef	NEXTHOP_INFO_SRAM

#define	NH_DATA		$nh
.sig	nh_sram_signal

#else

#define	NH_DATA		$$nh
.sig	nh_dram_signal

#endif

	.reg	nexthop_addr

	// allocate buffer to hold Next hop info. We will read 2 quadwords.
	xbuf_alloc[NH_DATA, NEXTHOP_INFO_SIZE_LW, read]

	// We have the index of the next hop info entry. Each entry is of
	// 16bytes. So, to get the relative address of the entry multiply
	// the index by 16.
	alu_shf[nexthop_addr, --, B , nexthop_index, <<NEXTHOP_INFO_SHF_VAL]

#ifdef	NEXTHOP_INFO_SRAM
	// read next hop information from SRAM

#ifdef	PROCESS_CONTROL_BLOCK

	sram[read, NH_DATA[0], nexthop_addr, nexthop_info_base, //	
		NEXTHOP_INFO_SRAM_REF_CNT], sig_done[nh_sram_signal]
	ctx_arb[nh_sram_signal, control_block_sig], AND

#else

	sram[read, NH_DATA[0], nexthop_addr, nexthop_info_base, // 	
		NEXTHOP_INFO_SRAM_REF_CNT], ctx_swap[nh_sram_signal]

#endif

#else
	// read next hop information from DRAM
	dram[read, NH_DATA[0], nexthop_addr, nexthop_info_base, //	
		NEXTHOP_INFO_DRAM_REF_CNT], sig_done[nh_dram_signal]

#ifdef	PROCESS_CONTROL_BLOCK
	ctx_arb[nh_dram_signal, control_block_sig], AND
#else
	ctx_arb[nh_dram_signal]
#endif

#endif

	// Now set the DL variables with the information we find in the
	// next hop entry.

	#ifdef COMPRESSED_NEXTHOP_INFO

	// Compressed next hop id entry is 2 LW's. Used for IXP2800

	.begin
	.reg	tmp
		
		ld_field_w_clr[tmp, 0011, NH_DATA[NH_NHID_LW], >>16]
		// set next hop ID. Only LSB16 bits are used.
		dl_meta_set_nexthop_id[tmp]

		// set the output port on egress. output port is 8 bits
		ld_field_w_clr[tmp, 0001,NH_DATA/**/NH_OUTPORT_LW,0]		
		dl_meta_set_output_port[tmp]

		// set the blade ID of egress
		ld_field_w_clr[tmp, 0001, NH_DATA[NH_BLADE_LW], >>8]	
		dl_meta_set_fabric_port[tmp]

		// set the nexthop ID type. This is 8 bits.
		ld_field_w_clr[tmp,0001,NH_DATA/**/NH_NHID_TYPE_LW,>>8]	
		dl_meta_set_nexthop_id_type[tmp]

		// if the nexthop_id_type != 0, then SKIP IPV4 tests.
		br!=byte[tmp,0,0,proc_nexthop_info_success#]

	.end

	#else	// COMPRESSED_NEXTHOP_INFO

	.begin
	.reg	tmp

		// set next hop ID. Only LSB16 bits are used.
		dl_meta_set_nexthop_id[NH_DATA[NH_NHID_LW]]

		// set the output port on egress. output port is 16 bits
		ld_field_w_clr[tmp, 0011, NH_DATA[NH_OUTPORT_LW], 0]	
		dl_meta_set_output_port[tmp]	

		// set the blade ID of egress
		ld_field_w_clr[tmp, 0001, NH_DATA[NH_BLADE_LW], >>24]	
		dl_meta_set_fabric_port[tmp]

		// set the nexthop ID type. This is 4 bits.
		alu_shf[tmp, 0xf, AND, NH_DATA[NH_BLADE_LW], >>20]
		dl_meta_set_nexthop_id_type[tmp]

		// if the nexthop_id_type != 0, then SKIP IPV4 tests.
		br!=byte[tmp, 0, 0, proc_nexthop_info_success#]
	.end

	#endif	// COMPRESSED_NEXTHOP_INFO

	// Now, we start processing the flags. These flags normally result in
	// exception packets. In general, these flags are cleared. If any of
	// the flags is set, we will process flags in non-critical code path.
	// The flags are in bits 16 thru 31.

	// first check for local packets
	alu[--, NH_DATA[NH_FLAGS_LW], AND , ipv4_flags_mask_for_stack]
	bne[check_flags_for_stack#]

	// now check for options. During header validation we already
	// filtered out packets with version !=4 and hdr length < 5
	alu[--, ver_hdr, - , 0x45]
	bne[options#]

	// At this point, we are ready to forward the packet as we 
	// have filtered all the local packets.
	// the 'nexthop_index' could be zero. This entry may have
	// a default route or may indicate that no route exists. If no route
	// exists, the nexthop ID of this entry has a value of '-1'.

	#ifdef	COMPRESSED_NEXTHOP_INFO
	// Compressed next hop id entry is 2 LW's.

	.begin
	.reg	tmp
		alu[tmp, --, b, NH_DATA[NH_NHID_LW], >>16]
		alu[tmp, 0x1, + , tmp]
		br_bset[tmp, 16, no_route#]; no route for this destination.
	.end

	#else	// COMPRESSED_NEXTHOP_INFO
	
	.begin
	.reg	tmp
		alu[tmp, 0x1, + , NH_DATA[NH_NHID_LW]]
		// here's trick. Now if NH id=-1, adding a 1 sets the reserved field
		// to 1. Else, this byte will be zero.
		br_bset[tmp, 16, no_route#]; no route for this destination.
	.end

	#endif	// COMPRESSED_NEXTHOP_INFO

	// Check remaining flags. We don't come here if any of 
	// ipv4_flags_mask_for_stack were set.
	br!=byte[NH_DATA[NH_FLAGS_LW], 0, 0, check_flags_other#]

#ifdef	PROCESS_CONTROL_BLOCK
	// now, before checking further, see, if fwding is enabled on the
	// inport? If enabled, a bit is set for that inport. At this point
	// inport is actually blade | inport. Currently, we check for 
	// only 32 ports.
	.begin
	.reg	mask
		alu[--, in_port, OR , 0]
		alu_shf[mask, -- , B , 0x1, <<indirect]
		alu[--, $control_block[0], AND, mask]
		beq[fwding_disabled#]
	.end
#endif

	// Next check is to see if fragmentation required. If our packet size
	// is greater than the MTU, we send the packet to the core as we don't
	// support fragmentation in MEs.
	.begin
		.reg	mtu

		ld_field_w_clr[mtu, 0011, NH_DATA[NH_MTU_LW], >>16]
		alu[--, mtu, - , ip_total_len]
	.end
	blt[nh_fragment#]


	// check if the packet has to go out thru the same interface from
	// which it is received. This requires an ICMP redirect message to
	// be sent to the sender of the packet. The interface is identified
	// by a combination of blade ID and output interface on that blade.
	.begin
	.reg	tmp

	#ifdef	COMPRESSED_NEXTHOP_INFO
	// Compressed next hop id entry is 2 LW's.
		ld_field_w_clr[tmp, 1000, NH_DATA[NH_BLADE_LW], <<16]
	#else	// COMPRESSED_NEXTHOP_INFO
		ld_field_w_clr[tmp, 1000, NH_DATA[NH_BLADE_LW], 0]
	#endif	// COMPRESSED_NEXTHOP_INFO

		ld_field[tmp, 0011, NH_DATA[NH_OUTPORT_LW], 0]
		alu[--, in_port, - , tmp]
		beq[nh_icmp_redirect#]

	.end

	// If we execute this, the packet has passed all the checks for
	// next hop information, and we can forward the packet.
proc_nexthop_info_success#:
	// set the result code to IPV4_SUCCESS and exit macro.
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_SUCCESS]

	// below is exception packet processing. 
	
	// no route for the destination address.
no_route#:
	// update counter
	_ipv4_fwder_update_counter(OFFSET_PKTS_NO_ROUTE)

	// the packet is sent to core.
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_EXCP_NO_ROUTE]

	// check the flags in more detail. We have a particular
	// order in checking the flags. Also, currently these 
	// flags are mutually exclusive.
check_flags_for_stack#:

	// if local bit is set, send exception.with appropriate
	// exception code.
	br_bset[NH_DATA[NH_FLAGS_LW], IPV4_NH_FLAGS_LOCAL, nh_local_delivery#]

	// if broadcast flag is set, send appropriate exception
	br_bset[NH_DATA[NH_FLAGS_LW], IPV4_NH_FLAGS_BROADCAST, nh_broadcast#]

	// if multicast flag is set, 	
	br_bset[NH_DATA[NH_FLAGS_LW], IPV4_NH_FLAGS_MULTICAST, nh_multicast#]

	// NOTE : We will never fall through
	//	================================

check_flags_other#:
	// if port bit is set, send exception.with appropriate
	// exception code.
	br_bset[NH_DATA[NH_FLAGS_LW], IPV4_NH_FLAGS_DOWN, nh_down#]

	// Otherwise, simply drop the packet for unknown flag settings.

	// drop the packet.
nh_drop#:
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_FAILURE]

	// when the outport is down, an ICMP message will be sent to 
	// the sender of the packet. The core portion can do this.
nh_down#:
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_EXCP_DOWN]

	// the packet is for local applications.
nh_local_delivery#:
	dl_set_exception_priority[1]
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_EXCP_LOCAL_DELIVERY]

options#:
	dl_set_exception_priority[1]
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_EXCP_OPTIONS]

	// the packet is broadcast packet
nh_broadcast#:
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_EXCP_BROADCAST]

	// the packet is multicast.
nh_multicast#:
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_EXCP_MULTICAST]

	// fragmentation required.
nh_fragment#:
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_EXCP_FRAG_REQUIRED]

#ifdef	PROCESS_CONTROL_BLOCK
	// forwarding disabled for input port
fwding_disabled#:
	br[proc_nexthop_info_end#], defer[1]
	immed[out_result, IPV4_FAILURE]
#endif

	// ICMP redirect required.
nh_icmp_redirect#:
	immed[out_result, IPV4_EXCP_REDIRECT]

	// we are done processing nexthop information. Exit the macro.
proc_nexthop_info_end#:
	// free the buffer allocated.
	xbuf_free[NH_DATA]
.end

// cleanup name space
#undef	NH_DATA

#endm

/////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_update_header[]
//
//	 	Description: 
//		
//			Decrement TTL, update checksum and place the packet
//			in destination buffer.
//
//	 	Outputs:
//
//			result	-	result of this macro. Can be IPV4_SUCCESS
//				or IPV4_FAILURE or an exception code.
//			OUT_IP	-	Registers (GPR/Transfer) to store the IP
//				packet. The register buffer is assumed to be of
//				the format OUT_IP0, OUT_IP1, OUT_IP2 ...OUT_IP7
//
//			Tell if any global variables are set. (also called side effects)
//
//		Inputs:
//
//			WR_START	- The relavive start byte position 
//				of IP header w.r.t start of the register buffer.
//			IP	-	Registers (Transfer) containing the IP
//				packet. The register buffer is assumed to be of
//				the format IP0, IP1, IP2 ...IP7
//			RD_START	- The relavive start byte position 
//				of IP header w.r.t start of the register buffer.
//
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
//
//////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_update_header(out_result, out_ip, in_ip, WR_START, RD_START)

//	remove spaces from arguments passed
#define_eval	UH_WR_START	WR_START
#define_eval	UH_RD_START	RD_START

	// check the nexthop id type first.
.begin
	.reg nh_id_type
	dl_meta_get_nexthop_id_type[nh_id_type]
	br!=byte[nh_id_type, 0, 0, copy_header#]
.end

	// before forwarding, we need to check TTL to make sure that it is
	// atleast '2'. Otherwise, we will send the packet to core to send an
	// ICMP message.
	ipv4_ttl_verify(in_ip, UH_RD_START)
	br>0[bad_ttl#]

.begin
	.reg	prepend
	// ipv4_modify can put a prepend to the header. Right now, we don't
	// need any.
	immed[prepend, 0x0]

	// now we call IXP IPV4 macros to (1) decrement TTL (2) update
	// checksum (3) copy the packet header to destination address.
	ipv4_modify(out_ip, UH_WR_START, prepend, in_ip, UH_RD_START]
	br[ipv4_fwder_update_header_end#]
.end

copy_header#:
	// simply copy into destination. Here we copy IP header which is 20 bytes.
	// An important point to note is that we assume that the IP header is not
	// spread into multiple XBUFs. This means, eventhough the src XBUFs 
	// are linked,
	// The IP header can be found in the very first XBUF itself.
	xbuf_copy(out_ip, 0, WR_START, in_ip, RD_START, 0, 20, 0)
	br[ipv4_fwder_update_header_end#]

	// here, we set the exception code and exit the macro.
bad_ttl#:
	// update counter. 
	_ipv4_fwder_update_counter(OFFSET_PKTS_BAD_TTL)
	immed[out_result, IPV4_EXCP_BAD_TTL]

ipv4_fwder_update_header_end#:

// cleanup name space.
#undef	UH_WR_START
#undef	UH_RD_START


#endm

////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_is_lookup_needed[]
//
//	 	Description: 
//
//			Atomically increment a 32bit counter.
//
//			Note the order of parameters. This order, output, output/input,
//			input, input constants MUST be followed.
//
//	 	Outputs:
//			None.
//
//		Inputs:
//
//			addr	-	base address for counters. Should
//				be a GPR.
//			offset	- 	a constant offset (should be 7bits
//				maximum).
//
//		CONSTANTS		
//			None.
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_is_lookup_needed[SKIP_LABEL]

.begin
	.reg	nh_id
	
	dl_meta_get_nexthop_id[nh_id]
	// check if this is -1. Remember, this nh_id is 16 bits
	alu[nh_id, nh_id, + , 1]
	// bit 16 is reserved and will be zero, unless nhid == -1, in
	// which case bit 16 it set by above alu operation.
	br_bclr[nh_id, 16, SKIP_LABEL] 
.end	
#endm


/////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_lookup()
//
//	 	Description: 
//
//			Perform Longest Prefix Match (LPM) for the destination 
//			address. Uses Dual lookup as described in IXP1200 
//			Network Processor SRM.
//
//			Note the order of parameters. This order, output, output/input,
//			input, input constants MUST be followed.
//
//	 	Outputs:
//
//			nextHopPtr	-	Pointer (not index) to next hop 
//					information or IXP_IPV4_NO_ROUTE.
//
//		Inputs:
//
//			da			- destination address which is used for LPM.
//
//		CONSTANTS		
//			None.
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
//
/////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_lookup(out_nexthop_index, in_da)

	// perform route lookup. The value '0x5' indicates the
	// number of SRAM access in longest-path (32bit prefix)
	// only.
	ipv4_route_lookup(out_nexthop_index, in_da, TRIE_TABLE_SRAM_BASE, 0x5)


#endm


/////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_update_counter[]
//
//	 	Description: 
//
//			Atomically increment a 32bit counter.
//
//			Note the order of parameters. This order, output, output/input,
//			input, input constants MUST be followed.
//
//	 	Outputs:
//			None.
//
//		Inputs:
//
//			addr	-	base address for counters. Should
//				be a GPR.
//			offset	- 	a constant offset (should be 7bits
//				maximum).
//
//		CONSTANTS		
//			None.
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
/////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_update_counter(OFFSET)

#ifdef IPV4_COUNTERS

	#ifdef IPV4_COUNTERS_SCRATCH

		scratch[incr, --, OFFSET, counter_base]

	#else
		
		sram[incr, --, OFFSET, counter_base]
   
    #endif
				
#endif // IPV4_COUNTERS

#endm

#endif	// __IPV4_FWDER_UTIL_UC__

