/////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
/////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ipv4_fwder_init.uc
//
//      Purpose: Initializes global variables, kick starts ipv4_fwder.
//
////////////////////////////////////////////////////////////////////////////

#ifndef	__IPV4_FWDER_INIT_UC__
#define	__IPV4_FWDER_INIT_UC__

//	Contents
//	--------
//	_ipv4_fwder_kick_start()	: Macro to kickstart IPV4 Fwder microblock
//									chain.
//	_ipv4_fwder_init()			: Macro to initialize IPV4 Fwder globals
//

//	Include Files
// None.

//	Constants 
//	None.

//	Global Registers
//	None.

//	Global Signals
//	None.

/////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_kick_start()
//
//	 	Description: 
//
//		Kick starts IPV4 Forwarder blocks across the 4 MEs
//	(1) when the system initialization is complete.
//	(2) signal thread 0 on beginning ME after which the
//		thread order is maintained by DL macros.
//
//	All the MEs (and threads) running IPV4 Forwader call 
//	this macro. Only thread0 of first ME would execute 
//	this fully. Other threads of first ME and all the
//	threads on other MEs would continue and call the
//	dl_source[DL_THREAD_ORDER], and wait for signal
//	from previous thread. This would result in thread0
//	of first ME kickstarting the whole chain.
//
//
//	 	Outputs:
//			None.
//
//		Inputs:
//			None.
//
//		CONSTANTS
//			None.	
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
//
/////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_kick_start()

	// The IPV4 fwder code can run on single ME or multiple MEs. For
	// Point Reyes, we run this code on 4 MEs as a chain. These MEs
	// need to wait for the system initialization to complete which
	// is communicated by sending a signal. So, thread0 of the 
	// first ME waits for this signal. Once, this signal is received.
	// this thread kick starts the ME chain. (Actually any thread on
	// any ME can act as kick starter).

	// We need to know the ME number to get this right. If we pickup
	// different MEs, the following line has to be modified accordingly.
	// For Point Reyes, '0x1' is the first ME in the chain. 
#ifdef	IPV4_START_ME	

.begin
	// we need to manually allocate a signal on which we wait.
	.sig	sys_init_signal
	.addr	sys_init_signal	ME_INIT_SIGNAL
	.set_sig sys_init_signal

	br!=ctx[0, _ipv4_fwder_kick_start_end#]

	// this is code for thread0

	// along with the system initialization signal, we assert a wakeup
	// signal so that very first time around, we don't need to wait. This
	// helps in kick starting the chain.
	signal_first_ctx[DL_SIG_WAKE]

	// we wait for the signal here
	ctx_arb[sys_init_signal]

	// by exiting this block, we free up the signal
.end

_ipv4_fwder_kick_start_end#:

#endif	//IPV4_START_ME

	// here, initialize the thread ID.
#ifdef	IPV4_NEW_XBUF
.begin
	.reg	thread_id
	local_csr_rd[active_ctx_sts]
	immed[thread_id, 0]
	alu[thread_id, 0x7, AND, thread_id]
	dl_set_thread_id[thread_id]
.end
#endif

#endm


#ifdef	LM_DBCAST_TABLE
/////////////////////////////////////////////////////////////////////////////
//
// _lm_dbcast_populate()
//
//	 	Description: 
//
//		Reads in the Dbcast table in SRAM into Local memory.
//
//
//	 	Outputs:
//			None.
//
//		Inputs:
//			None.
//
//		CONSTANTS
//			LM_DBCAST_BASE, LM_DBCAST_BLOCK_COUNT, LM_DBCAST_BLOCK_SIZE, 
//			LM_DBCAST_INDEX, DBCAST_TABLE_SRAM_BASE
//
//		Global variables
//			None.
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
//
/////////////////////////////////////////////////////////////////////////////
#macro	_lm_dbcast_populate()

.begin
	.reg	lm_dbcast_sram_addr 
	.reg	num_trips
	.sig	dbcast_sram_sig

	// maximum number of longwords can be read from SRAM
#define	_LM_DBCAST_XFER_EACH_TRIP		8

	// allocate xfer register buffer
	xbuf_alloc($lm_dbcast_xfer, _LM_DBCAST_XFER_EACH_TRIP, read)

	// calculate some values
#define	_LM_DBCAST_XFER_TOTAL_LW	(\
					(LM_DBCAST_BLOCK_COUNT * LM_DBCAST_BLOCK_SIZE) + \
					(LM_DBCAST_BLOCK_COUNT/4))

#define	_LM_DBCAST_XFER_NUM_TRIPS	(_LM_DBCAST_XFER_TOTAL_LW / \
					_LM_DBCAST_XFER_EACH_TRIP)

#define	_LM_DBCAST_XFER_REMINDER	(_LM_DBCAST_XFER_TOTAL_LW % \
					_LM_DBCAST_XFER_EACH_TRIP)

	// initialize GPRs to avoid define_eval
	immed32(num_trips, _LM_DBCAST_XFER_NUM_TRIPS)

	.begin	
		.reg	tmp_base

		immed32[tmp_base, (LM_DBCAST_BASE * 4)]

		// set local memory pointer
		local_csr_wr[active_lm_addr_/**/LM_DBCAST_INDEX, tmp_base]
	.end

	// set the table base address
	immed32[lm_dbcast_sram_addr, DBCAST_TABLE_SRAM_BASE]


	// start loop
	//===============
make_trips#:
	alu[--, num_trips, - , 0]
	ble[check_reminder#]

	// issue read to sram
	sram[read, $lm_dbcast_xfer0, lm_dbcast_sram_addr, 0, \
			_LM_DBCAST_XFER_EACH_TRIP], ctx_swap[dbcast_sram_sig]

#define			_LOOP_CNT			_LM_DBCAST_XFER_EACH_TRIP
#define			_LOOP_INDEX			0

#while	(_LOOP_CNT > 0)

	// copy to local memory
	alu[*l$index/**/LM_DBCAST_INDEX++, -- , B, $lm_dbcast_xfer/**/_LOOP_INDEX]

#define_eval	_LOOP_INDEX		(_LOOP_INDEX + 1)
#define_eval	_LOOP_CNT		(_LOOP_CNT - 1)

#endloop

	// cleanup namespace
#undef	_LOOP_INDEX
#undef	_LOOP_CNT

	// for next iteration
	alu[lm_dbcast_sram_addr, lm_dbcast_sram_addr , + , \
			(_LM_DBCAST_XFER_EACH_TRIP * 4)]
	alu[num_trips, num_trips, - , 1]
	br[make_trips#]

	// transfer remaining bytes
	//==========================
check_reminder#:

#if	(_LM_DBCAST_XFER_REMINDER > 0)
	// issue read to sram
	sram[read, $lm_dbcast_xfer0, lm_dbcast_sram_addr, 0, \
			_LM_DBCAST_XFER_REMINDER], ctx_swap[dbcast_sram_sig]

#define			_LOOP_CNT			_LM_DBCAST_XFER_REMINDER
#define			_LOOP_INDEX			0

#while	(_LOOP_CNT > 0)

	// copy to local memory
	alu[*l$index/**/LM_DBCAST_INDEX++, -- , B, $lm_dbcast_xfer/**/_LOOP_INDEX]

#define_eval	_LOOP_INDEX		(_LOOP_INDEX + 1)
#define_eval	_LOOP_CNT		(_LOOP_CNT - 1)

#endloop

	// cleanup namespace
#undef	_LOOP_INDEX

#endif	// _LM_DBCAST_XFER_REMINDER

	// cleanup namespace
#undef	_LM_DBCAST_XFER_REMINDER
#undef	_LM_DBCAST_XFER_NUM_TRIPS

	xbuf_free($lm_dbcast_xfer)

.end
#endm


#endif

/////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder_init()
//
//	 	Description: 
//
//		Initializes the IPV4 Forwarder microblock variables
//
//
//	 	Outputs:
//			None.
//
//		Inputs:
//			None.
//
//		CONSTANTS
//			None.
//
//		Global variables
//			@ipv4_stats_base, nexthop_info_base, @limited_bcast
//
//		Size: 
//
//			?? instructions. (Worst case cycle count)
//
/////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder_init()

//	Global Registers
.reg	@ipv4_stats_base 		// base of the counters in SRAM/SCRATCH
.reg	nexthop_info_base 		// base of the next hop info table in SDRAM
.reg	ipv4_flags_mask_for_stack	// flags mask for local stack bound pkts

#ifdef	RFC2644_CHECKS
#ifndef	LM_DBCAST_TABLE
.reg	sram_dbcast_base
#endif
#endif


#ifdef	PROCESS_CONTROL_BLOCK
.reg	control_block_base
#endif

	br!=ctx[0, init_relative#]

init_absolute#:

	; initialize SRAM counter base, an import_var
	#ifdef IPV4_COUNTERS_SCRATCH

		immed32(@ipv4_stats_base, IPV4_STATS_TABLE_BASE)
    #else
		immed32(@ipv4_stats_base, STATS_TABLE_SRAM_BASE)

	#endif

#ifdef	LM_DBCAST_TABLE

	_lm_dbcast_populate()

#endif

init_relative#:

	immed32(ipv4_flags_mask_for_stack, IPV4_FLAGS_MASK_FOR_STACK_VAL)

	// for better DRAM bank utilization, and since the number of nexthop
	// entries are limited in number, the nexthop information could be
	// kept in SRAM which should improve the performance. If the 
	// compilation option is enabled, then the information is read from
	// SRAM. Otherwise, we read the information from DRAM.

#ifdef	NEXTHOP_INFO_SRAM
	// next hop base in SRAM
	immed32(nexthop_info_base, NEXTHOP_TABLE_SRAM_BASE)
#else
	// next hop base in DRAM
	immed32(nexthop_info_base, NEXTHOP_TABLE_SDRAM_BASE)
#endif

#ifdef	PROCESS_CONTROL_BLOCK

	#if IS_IXPTYPE(__IXP28XX)	
		immed32(control_block_base, CONTROL_BLOCK_SCRATCH_BASE)
	#else
		immed32(control_block_base, CONTROL_BLOCK_SRAM_BASE)
	#endif	// IXP2800
#endif

#ifdef	RFC2644_CHECKS
#ifndef	LM_DBCAST_TABLE
	immed32[sram_dbcast_base, DBCAST_TABLE_SRAM_BASE]
#endif
#endif

#endm



#endif	// __IPV4_FWDER_INIT_UC__

