////////////////////////////////////////////////////////////////////////////
//                                                                     
//                  I N T E L   P R O P R I E T A R Y                   
//                                                                      
//     COPYRIGHT (c)  2001-2002 BY  INTEL  CORPORATION.  ALL RIGHTS          
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY      
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A    
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER    
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,    
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT    
//     THE PRIOR WRITTEN PERMISSION OF :                                
//                                                                      
//                        INTEL  CORPORATION                            
//                                                                     
//                     2200 MISSION COLLEGE BLVD                        
//                                                                      
//               SANTA  CLARA,  CALIFORNIA  95052-8119                  
//                                                                      
////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: ipv4_fwder.uc
//
//      Purpose: This file contains the ipv4_fwder macro which implements the 
//		IPV4 fwding functionality. Calls in internal macros. 
//
//////////////////////////////////////////////////////////////////////////////

#ifndef	__IPV4_FWDER_UC__
#define	__IPV4_FWDER_UC__

//////////////////////////////////////////////////////////////////////////////

//	Contents
//	--------
//	_ipv4_fwder()	=	macro implementing IPV4 Fwding algorithm
//
//	No public API. Should be called in a dispatch loop.
//

//	Include Files

// include xbuf.uc from IXPblocks portable library
#include	"xbuf.uc"

// include ipv4.h in IXPblocks Portable library
#include	"ipv4.h"

// include ipv4.uc in IXPblocks Portable library to process IP header and
// LPM macro.
#include	"ipv4.uc"

// include IPV4 Forwarder files for initialization, utilities such as
// header verification, next hop info processing.  

#include	"ipv4_fwder_uc.h"
#include	"ipv4_fwder_init.uc"
#include	"ipv4_fwder_util.uc"

//////////////////////////////////////////////////////////////////////////////

//	Constants 
//	None.

//////////////////////////////////////////////////////////////////////////////

//	Global Registers
//	We use DL variables.

//////////////////////////////////////////////////////////////////////////////

//	Global Signals
//	None.

//////////////////////////////////////////////////////////////////////////////
//
// _ipv4_fwder()
//
//	 	Description: 
//
//			Verify IP header and forward the packet. 
//
//	 	Outputs:
//
//			out_result	-	IPV4_SUCCESS or IPV4_FAILURE or exception code
//			out_ip	-	Register buffer for modified IP header 
//
//		Inputs:
//
//			in_ip	-	buffer containing packet header
//
//			IPHDR_WR_START_BYTE	- byte address relavive to start of
//				out_ip
//
//			IPHDR_RD_START_BYTE	- byte address relative to start of
//				in_ip
//		Size: 
//
//			?? instructions. (Worst case cycle count)
//
//////////////////////////////////////////////////////////////////////////////
#macro	_ipv4_fwder(out_ip, in_ip, IPHDR_WR_START_BYTE, IPHDR_RD_START_BYTE)

// get rid of white spaces from constansts
#define_eval	WR_START	IPHDR_WR_START_BYTE
#define_eval	RD_START	IPHDR_RD_START_BYTE

.begin
	.reg	ip_total_len	// total length read from IP header 
	.reg	ver_hdr			// version and hdr length fields read from IP hdr
	.reg	in_port 		// port from which this packet is rxed
	.reg	counter_base 	// base where the stats are maintained
	.reg	nexthop_index	// index of next hop information.
	.reg	out_result		// result of macro calls.
	.sig	control_block_sig	// signal used to read control block
	.reg	da				// packet destination address

ipv4_fwder_start#:
 
#ifdef	IP_HDR_CACHE_LM

	.begin
		.reg	thread_id

		dl_get_thread_id(thread_id)

		xbuf_activate(in_ip, 0, thread_id, 0)
		xbuf_activate(out_ip, 1, thread_id, 0)

	.end
#endif
 	// check for our block id. If doesn't match, exit.

	br!=byte[dl_next_block, 0, BID_IPV4, ipv4_hdr_copy#]

	dl_meta_get_input_port[in_port]	; get the input port number.


#ifdef	IPV4_COUNTERS

	// build counter base for this port.
	alu[counter_base, -- , B, @ipv4_stats_base]
	alu_shf[counter_base, counter_base, OR , in_port, <<STATS_PER_PORT_SHF_VAL]

#endif


	// combine the inport with blade Id by loading bladeId
	// in most significant byte.
	alu_shf[in_port, in_port, OR , THIS_BLADE_ID, <<24]

	#if (!IS_IXPTYPE(__IXP28XX))
	// For IXP2800, we skip this counter to save sram utilization.
	// The counter can be computed using other counters maintainted.
	// update the number of packets received.
	_ipv4_fwder_update_counter(OFFSET_PKTS_RX)
	#endif

	// Extract some fields and initialize the state variables
	xbuf_extract(ver_hdr, in_ip, RD_START, IP_VERSION_LENGTH)
	xbuf_extract(ip_total_len, in_ip, RD_START, IP_TOTAL_LENGTH)
	xbuf_extract(da, in_ip, RD_START, IP_DESTINATION_ADDRESS)

	// Now perform header validation according to RFC1812. The result 
	// is IXP_IPV4_SUCCESS for a good packet.
	_ipv4_fwder_validate_header(out_result, in_ip, RD_START,perform_lookup#)
	br!=byte[out_result, 0, IXP_IPV4_SUCCESS, ipv4_fwder_macro_result#]


#ifdef	MULTICAST_SUPPORT_ENABLE

	// check if the dst. address is a multicast address. If so, send to
	// next block. Also note that in this case, we *don't* modify or copy 
	// the header as IPV4 Multicast Forwarder will do it for us.


	.begin
	
		.reg tmp
		
		alu[tmp, da, AND, 0xf, <<28]

		br=byte[tmp, 3, 224, multicast_pkt#]

	.end


#endif	// MULTICAST_SUPPORT_ENABLE

	// Update counter to indicate that we are attempting to forward a 
	// packet.
	_ipv4_fwder_update_counter(OFFSET_PKTS_FWD)

	/* check if we need to perform lookup or not */
	_ipv4_fwder_is_lookup_needed[update_ipv4_hdr#]

perform_lookup#:

#ifdef	PROCESS_CONTROL_BLOCK
	//	here, we read control block information.
	xbuf_alloc[$control_block, CONTROL_BLOCK_RD_SIZE, read]

	#if IS_IXPTYPE(__IXP28XX)
		scratch[read, $control_block[0], control_block_base, 0, // 
		CONTROL_BLOCK_RD_SIZE], sig_done[control_block_sig]
	#else
		sram[read, $control_block[0], control_block_base, 0, //
		CONTROL_BLOCK_RD_SIZE], sig_done[control_block_sig]
	#endif	// IXP2800

#endif

	// Perform the lookup on the destination address in our forwarding table.


	// perform lookup on destination address
	_ipv4_fwder_lookup(nexthop_index, da)

	// Using the nexthop index, read and process next hop info
	// from SDRAM. Outcome of this macro will be IPV4_SUCCESS (then the
	// packet will be forwarded) or IPV4_FAILURE (packet is dropped) or
	// an exception code (packet is sent to X-Scale).
	// This macro also processes control block.
	_ipv4_fwder_proc_nexthop_info(out_result)

#ifdef	PROCESS_CONTROL_BLOCK
	// doesn't generate code
	xbuf_free[$control_block]
#endif

	br!=byte[out_result, 0, IPV4_SUCCESS, ipv4_fwder_macro_result#]


	// If we execute this, the packet is good to forward. Now we check TTL,
	// decrement TTL and update checksum. Outcome of this macro will be 
	// IPV4_SUCCESS (then the packet will be forwarded) or IPV4_FAILURE 
	// (packet is dropped) or an exception code (packet is sent to X-Scale).
update_ipv4_hdr#:
	_ipv4_fwder_update_header(out_result, out_ip, in_ip, WR_START, RD_START)
	br!=byte[out_result, 0, IPV4_SUCCESS, ipv4_fwder_macro_result#]


	// we are done with ipv4 fwding. Exit this macro.
	br[ipv4_fwder_finish#], defer[1]
	immed[dl_next_block, IPV4_NEXT1]


#ifdef	MULTICAST_SUPPORT_ENABLE

multicast_pkt#:

	// send to multicast forwarder
	br[ipv4_fwder_finish#], defer[1]
	immed[dl_next_block, BID_IPV4_MCAST_NEXT1]

#endif


	// The following are Error and Exception conditions.


	// Below we look in more detail about the result code returned 
	// from macros. This code is not executed in critical path (:code
	// path taken by packets which are processed completely and 
	// successfully in data path.)

	// If the result is IPV4_FAILURE, then we will simply drop the
	// packet. Otherwise, the packet will be sent to X-scale. For 
	// debugging purposes, we simply increment a counter in both
	// cases. The caller should take care of actually dropping the
	// packet/sending an exception packet. If this code is found to
	// be unnecessary, then we can simply exit at this point.
ipv4_fwder_macro_result#:
	alu[ -- , out_result, - , IPV4_FAILURE]
	beq[ipv4_fwder_drop#]

ipv4_fwder_set_dl_exception#:
	// if here, we are in exception path
	_ipv4_fwder_update_counter(OFFSET_PKTS_EXCP)

	// here we have to be flexible on where to send the
	// packet. 
	alu[--, out_result, - , IPV4_EXCP_LOCAL_DELIVERY]
	bne[next_block_is_exception#], defer[2]
	dl_set_exception[BID_IPV4, out_result]

	// for local packet, target could be different.
	br[ipv4_hdr_copy#], defer[1]
	immed[dl_next_block, IPV4_LOCAL_NEXT1]

next_block_is_exception#:
	immed[dl_next_block, IX_EXCEPTION]

#ifndef	FLUSH_EXCP_PKT_HDR
	// don't need to copy header
	br[ipv4_fwder_finish#]
#endif

ipv4_hdr_copy#:
	// copy header into dst buffer
	xbuf_copy(out_ip, 0, WR_START, in_ip, RD_START, 0, 20, 0]

	br[ipv4_fwder_finish#]

ipv4_fwder_drop#:
	_ipv4_fwder_update_counter(OFFSET_PKTS_DROP)
	immed[dl_next_block, IX_DROP]
	
	// we simply fall thru ipv4_fwder_finish#:
	
ipv4_fwder_finish#:

#ifdef	IP_HDR_CACHE_LM

	xbuf_deactivate(in_ip)
	xbuf_deactivate(out_ip)

#endif
	// cleanup name space
#undef	WR_START
#undef	RD_START

.end
#endm

#endif	//	end __IPV4_FWDER_UC__

