/**
 * ============================================================================
 * = COPYRIGHT
 *              INTEL CORPORATION PROPRIETARY INFORMATION
 *   This software is supplied under the terms of a license agreement or
 *   nondisclosure agreement with Intel Corporation and may not be copied 
 *   or disclosed except in accordance with the terms in that agreement.
 *      Copyright (C) 2000-2001 Intel Corporation. All rights reserved.
 *
 * = PRODUCT
 *      Intel(r) IXA SDK 3.0 for the IXP2400 Network Processor, Release 5
 *
 * = LIBRARY
 *      
 *
 * = MODULE
 *      IPv4 Forwarder Core Component
 *
 * = FILENAME
 *      ix_cc_ipv4.h
 *
 * = DESCRIPTION
 *      This file defines allexported data types and function prototypes
 *
 * = AUTHOR
 *       Govindan Nair
 *       govindan.nair@intel.com
 *
 * = AKNOWLEDGEMENTS
 *      
 *
 * = CREATION TIME
 *      06/24/2002 5:30 PM
 *
 * = CHANGE HISTORY
 *
 * ============================================================================
 * $Id: ix_cc_ipv4.h,v 1.17 2003/04/22 01:12:09 ktseng Exp $
 */

#if !defined(__IX_CC_IPV4_H__)
#define __IX_CC_IPV4_H__

#include "ix_rm.h"
#include "ix_ossl.h"
#include "ix_cc.h"
#include "cc/ix_cc_rtmv4.h"
#include "ipv4_fwder_uc.h"

#if defined(__cplusplus)
extern "C"
{
#endif /* end defined(__cplusplus) */



#define IX_CC_IPV4_NH_BROADCAST 0xFFFFFFFB
#define IX_CC_IPV4_NH_MULTICAST 0xFFFFFFFC
#define IX_CC_IPV4_NH_DROP 0xFFFFFFFD
#define IX_CC_IPV4_NH_LOCAL 0xFFFFFFFE



/**
 * TYPENAME: ix_cc_ipv4_stats_data
 * 
 * DESCRIPTION: This structure defines format for statistics data.
 *
 */
typedef struct ix_s_cc_ipv4_stats_data
{
    ix_uint64 ipv4MbRcvdPkts;
    ix_uint64 ipv4MbFwdPkts;
    ix_uint64 ipv4MbDropPkts;
    ix_uint64 ipv4MbExcpPkts;
    ix_uint64 ipv4MbBadHeaderPkts;
    ix_uint64 ipv4MbBadTotalLengthPkts;
    ix_uint64 ipv4MbBadTTLPkts;
    ix_uint64 ipv4MbNoRoutePkts;
    ix_uint64 ipv4MbLengthTooSmallPkts;
    ix_uint32 ipv4CoreInvalidHeaderPkts;
    ix_uint32 ipv4CoreInvalidAddressPkts;
    ix_uint32 ipv4CoreRcvdPkts;
    ix_uint32 ipv4CoreFwdPkts;
    ix_uint32 ipv4CoreLocalDeliveryPkts;
    ix_uint32 ipv4CoreNoRoutePkts;
    ix_uint32 ipv4NumberOfRoutes;
    ix_uint32 ipv4NumberOfNextHops;
} ix_cc_ipv4_stats_data;






/**
 * TYPENAME: ix_cc_ipv4_messages
 * 
 * DESCRIPTION: This structure defines format for IPv4 messages
 *
 */
typedef enum ix_e_cc_ipv4_messages
{

    IX_CC_IPV4_MSG_ADD_ROUTE =IX_CC_COMMON_MSG_ID_LAST,

    /* Route Messages */
    IX_CC_IPV4_MSG_DELETE_ROUTE,
    IX_CC_IPV4_MSG_LOOKUP_ROUTE,
    IX_CC_IPV4_MSG_UPDATE_ROUTE,
    IX_CC_IPV4_MSG_PURGE_ROUTES,
    IX_CC_IPV4_MSG_DUMP_ROUTES,


    /* Next Hop Messages */
    IX_CC_IPV4_MSG_ADD_NEXT_HOP,
    IX_CC_IPV4_MSG_DELETE_NEXT_HOP,
    IX_CC_IPV4_MSG_UPDATE_NEXT_HOP,
    IX_CC_IPV4_MSG_GET_NEXT_HOP,
    IX_CC_IPV4_MSG_PURGE_RTM,
    IX_CC_IPV4_MSG_DUMP_NEXT_HOPS,
    IX_CC_IPV4_MSG_SET_MTU,
    IX_CC_IPV4_MSG_SET_FLAGS,


    /* ICMP Messages */
    IX_CC_IPV4_MSG_GET_SLEEP_TIME,
    IX_CC_IPV4_MSG_SET_SLEEP_TIME,
    IX_CC_IPV4_MSG_GET_QUEUE_DEPTH,
    IX_CC_IPV4_MSG_GET_PACKETS_TO_DRAIN,
    IX_CC_IPV4_MSG_SET_PACKETS_TO_DRAIN,

    /* Statistics message */
    IX_CC_IPV4_MSG_GET_STATISTICS,


    /* Add new messages before this */
    IX_CC_IPV4_MSG_LAST     
} ix_cc_ipv4_messages;



/**
 * TYPENAME: ix_cc_ipv4_error_code
 * 
 * DESCRIPTION: This enumerated type will describe ipv4 error codes
 *
 */
typedef enum ix_e_cc_ipv4_error_code
{
    IX_CC_IPV4_ERROR_FIRST = IX_ERROR_MAKE_GROUP(CC_IPV4),
    IX_CC_IPV4_ERROR_SUCCESS = IX_CC_IPV4_ERROR_FIRST,

    IX_CC_IPV4_ERROR_FAILED_PATCHING,
    IX_CC_IPV4_ERROR_REGISTRY,
    IX_CC_IPV4_ERROR_BUFFER_FREE,

    IX_CC_IPV4_ERROR_RTM,
    IX_CC_IPV4_ERROR_CCI,
    IX_CC_IPV4_ERROR_INVALID_CONTEXT,
    IX_CC_IPV4_ERROR_INVALID_INPUT_PARAM,
    IX_CC_IPV4_ERROR_INVALID_BUFFER,
    IX_CC_IPV4_ERROR_MSG_LIBRARY,
    IX_CC_IPV4_ERROR_INVALID_PACKET,
    IX_CC_IPV4_ERROR_INVALID_MASK,

    IX_CC_IPV4_ERROR_INVALID_NH_INFO,
    IX_CC_IPV4_ERROR_INVALID_FLAGS,
    IX_CC_IPV4_ERROR_INVALID_SLEEP_TIME,
    IX_CC_IPV4_ERROR_INVALID_PACKETS_TO_DRAIN,
    IX_CC_IPV4_ERROR_DUPLICATE_NEXT_HOP,
    IX_CC_IPV4_ERROR_NEXT_HOP_ID_IN_USE,

    IX_CC_IPV4_ERROR_INVALID_PROP_ID,
    IX_CC_IPV4_ERROR_FAILED_GET_PORT_STATUS, 
    IX_CC_IPV4_ERROR_CALLBACK,
    IX_CC_IPV4_ERROR_FAILED_ADD_SOURCE_BROADCAST,
    IX_CC_IPV4_ERROR_FAILED_DELETE_SOURCE_BROADCAST,
    
    /* unknown error code */
    IX_CC_IPV4_ERROR_UNKNOWN,
    IX_CC_IPV4_ERROR_LAST
} ix_cc_ipv4_error_code;




/**
 * TYPENAME: ix_cc_ipv4_dump_data
 * 
 * DESCRIPTION: This structure defines format for memory dump data .
 *
 */
typedef struct ix_s_cc_ipv4_dump_data
{
    char* pBuffer;
    ix_uint32 size;
} ix_cc_ipv4_dump_data;




/**
 * TYPENAME: ix_cc_ipv4_cb_route_op
 * 
 * DESCRIPTION: Type definition for general IPv4 Message callbacks.
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_route_op)(ix_error arg_Result, void* arg_pContext);



/**
 * TYPENAME: ix_cc_ipv4_cb_lookup_route
 * 
 * DESCRIPTION: Type definition for rtm route lookup callback
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_lookup_route)(ix_error arg_Result,
                                               void* arg_pContext,
                                               ix_cc_rtmv4_next_hop_info* arg_pNextHopInfo);

/**
 * TYPENAME: ix_cc_ipv4_cb_dump_data
 * 
 * DESCRIPTION: Type definition for rtm route and next hop lookup callback
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_dump_data)(ix_error arg_Result,
                                            void* arg_pContext,
                                            ix_cc_ipv4_dump_data* arg_pBuffer);



/**
 * TYPENAME: ix_cc_ipv4_cb_get_next_hop
 * 
 * DESCRIPTION: Type definition for next hop lookup callback.
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_get_next_hop)(ix_error arg_Result,
                                               void* arg_pContext,
                                               ix_cc_rtmv4_next_hop_info* arg_pNextHopInfo);



/**
 * TYPENAME: ix_cc_ipv4_cb_get_sleep_time
 * 
 * DESCRIPTION: Type definition for ICMP get sleep time callback.
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_get_sleep_time)(ix_error arg_Result,
                                               void* arg_pContext,
                                               ix_uint16 arg_Data);




/**
 * TYPENAME: ix_cc_ipv4_cb_get_queue_depth
 * 
 * DESCRIPTION: Type definition for ICMP get queue depth callback.
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_get_queue_depth)(ix_error arg_Result,
                                                  void* arg_pContext,
                                                  ix_uint16 arg_QueueDepth);





/**
 * TYPENAME: ix_cc_ipv4_cb_get_packets_to_drain
 * 
 * DESCRIPTION: Type definition for ICMP get packets to drain callback.
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_get_packets_to_drain)(ix_error arg_Result,
                                                       void* arg_pContext,
                                                       ix_uint16 arg_PacketsToDrain);



/**
 * TYPENAME: ix_cc_ipv4_cb_get_statistics
 * 
 * DESCRIPTION: Type definition for rtm route and next hop lookup callback
 *
 */

typedef ix_error (*ix_cc_ipv4_cb_get_statistics)(ix_error arg_Result,
                                                 void* arg_pContext,
                                                 ix_cc_ipv4_stats_data* arg_pBuffer);




/**
 * NAME: ix_cc_ipv4_async_add_route
 *
 * DESCRIPTION: This message helper primitive helps client to add a route in RTM. 
 * Client needs to successfully add nextHopId to RTM by calling ix_cc_ipv4_async_add_next_hop
 * messaging helper primitive before sending this message to IPv4 Forwarder core component. 
 * 
 * @Param:  - IN arg_IpAddr: network IP address identifying the route.
 * @Param:  - IN arg_NetMask: network mask for the route.
 * @Param:  - IN arg_NextHopId: next hop identifier for route.
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library
 * 
 */


ix_error ix_cc_ipv4_async_add_route(ix_uint32 arg_IpAddr,
                                    ix_uint32 arg_NetMask,
                                    ix_cc_rtmv4_nhid arg_NextHopId,
                                    ix_cc_ipv4_cb_route_op arg_Callback,
                                    void* arg_pUserContext
                                   );

/**
 * NAME: ix_cc_ipv4_async_delete_route
 *
 * DESCRIPTION:This message helper primitive helps a client to delete a route in RTM. 
 * Client needs to delete all routes associate with a next hop to delete the next hop
 * by calling ix_cc_ipv4_async_delete_next_hop message helper primitive. 
 * 
 * @Param:  - IN arg_IpAddr: network IP address identifying the route.
 * @Param:  - IN arg_NetMask: network mask for the route.
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT -none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */


ix_error ix_cc_ipv4_async_delete_route(ix_uint32 arg_IpAddr,
                                       ix_uint32 arg_NetMask,
                                       ix_cc_ipv4_cb_route_op arg_Callback,
                                       void* arg_pUserContext
                                      );


/**
 * NAME: ix_cc_ipv4_async_update_route
 *
 * DESCRIPTION: This message helper primitive helps client to update a route in RTM. 
 * Client needs to successfully add nextHopId to RTM by calling ix_cc_ipv4_async_add_next_hop
 * messaging helper primitive before sending this message to IPv4 Forwarder core component. 
 * 
 * @Param:  - IN arg_IpAddr: network IP address identifying the route.
 * @Param:  - IN arg_NetMask: network mask for the route.
 * @Param:  - IN arg_NextHopId: next hop identifier for the route.
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library
 * 
 */

ix_error ix_cc_ipv4_async_update_route(ix_uint32 arg_IpAddr,
                                    ix_uint32 arg_NetMask,
                                    ix_cc_rtmv4_nhid arg_NextHopId,
                                    ix_cc_ipv4_cb_route_op arg_Callback,
                                    void* arg_pUserContext
                                   );

/**
 * NAME: ix_cc_ipv4_async_lookup_route
 *
 * DESCRIPTION: This primitive helps a client to look up routing information 
 * for a given IP address. If there is a successful match, then the information will be 
 * given to client are nextHopId (including the blade id), portId, mtu and flags. 
 * If there is no match, then IX_CC_RTMV4_NHID_NO_ROUTE will be assigned to nextHopId
 * field of pNextHopData and all other information are invalid.
 * 
 * @Param:  - IN arg_IpAddr: network IP address identifying the route.
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT -none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 */

ix_error ix_cc_ipv4_async_lookup_route(ix_uint32 arg_IpAddr,
                                       ix_cc_ipv4_cb_lookup_route arg_Callback,
                                       void* arg_pUserContext
                                      );



/**
 * NAME: ix_cc_ipv4_async_purge_routes
 *
 * DESCRIPTION: This primitive helps a client to remove all routes from RTM.
 * 
 * @Param:  - IN arg_CcHandle - 
 *
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 */

ix_error ix_cc_ipv4_async_purge_routes(void);



/**
 * NAME: ix_cc_ipv4_async_dump_routes
 *
 * DESCRIPTION: This primitive helps to dump all routes of RTM in memory
 * 
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */

ix_error ix_cc_ipv4_async_dump_routes(ix_cc_ipv4_cb_dump_data arg_Callback,
                                      void* arg_pUserContext
                                     );


/**
 * NAME: ix_cc_ipv4_async_add_next_hop
 *
 * DESCRIPTION: This primitive helps a client to add next hop information to RTM database. 
 * Client needs to add next hop to RTM in order to add routes referring to that next hop.
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop to be added
 * @Param:  - IN arg_pNextHopData: pointer to next hop information
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined contex
 * 
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */

ix_error ix_cc_ipv4_async_add_next_hop(ix_cc_rtmv4_nhid arg_NextHopId,
                                       ix_cc_rtmv4_next_hop_info* arg_pNextHopData,
                                       ix_cc_ipv4_cb_route_op arg_Callback,
                                       void* arg_pUserContext
                                      );


/**
 * NAME: ix_cc_ipv4_async_delete_next_hop
 *
 * DESCRIPTION: This message helper primitive helps a client to delete next hop information 
 * from RTM database. Client needs to remove all routes associated with next hop before 
 * calling this primitive. If not, client gets an error.
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop to be removed.
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 */


ix_error ix_cc_ipv4_async_delete_next_hop(ix_cc_rtmv4_nhid arg_NextHopId,
                                          ix_cc_ipv4_cb_route_op arg_Callback,
                                          void* arg_pUserContext
                                          );


/**
 * NAME: ix_cc_ipv4_async_update_next_hop
 *
 * DESCRIPTION: This primitive helps a client to update next hop information to RTM database. 
 * Client needs to add next hop to RTM in order to add routes referring to that next hop.
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop to be updated
 * @Param:  - IN arg_pNextHopData: pointer to next hop information
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined contex
 * 
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */

ix_error ix_cc_ipv4_async_update_next_hop(ix_cc_rtmv4_nhid arg_NextHopId,
                                       ix_cc_rtmv4_next_hop_info* arg_pNextHopData,
                                       ix_cc_ipv4_cb_route_op arg_Callback,
                                       void* arg_pUserContext
                                      );


/**
 * NAME: ix_cc_ipv4_async_get_next_hop
 *
 * DESCRIPTION: This message helper primitive helps a client to retrieve 
 * next hop information from RTM database. 
 * 
 * @Param:  - IN arg_NextHopId: Identifier of the next hop information to be retrieved from RTM.
 * @Param:  - IN arg_Callback: pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 */

ix_error ix_cc_ipv4_async_get_next_hop(ix_cc_rtmv4_nhid arg_NextHopId,
                                       ix_cc_ipv4_cb_get_next_hop arg_Callback,
                                       void* arg_pUserContext
                                      );


/**
 * NAME: ix_cc_ipv4_async_purge_rtm
 *
 * DESCRIPTION: This primitive helps a client to removes all routes and next hops from RTM database
 * 
 * @Param:  - IN  - none
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */


ix_error ix_cc_ipv4_async_purge_rtm(void);


/**
 * NAME: ix_cc_ipv4_async_dump_next_hops
 *
 * DESCRIPTION:   This primitive helps to dump all next hops of RTM in memory.
 * 
 * @Param:  - IN arg_Callback : pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 */

ix_error ix_cc_ipv4_async_dump_next_hops(ix_cc_ipv4_cb_dump_data arg_Callback,
                                         void* arg_pUserContext
                                        );


/**
 * NAME: ix_cc_ipv4_async_set_mtu
 *
 * DESCRIPTION:This primitive helps client to update MTU for a given next hop. 
 * Before calling this message helper, client needs to call ix_cc_ipv4_async_add_next_hop to 
 * add next hop identifier in RTM database. The validity of MTU shall be done by the client.
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop.
 * @Param:  - IN arg_Mtu: new maximum transmission unit for the given next hop.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 */


ix_error ix_cc_ipv4_async_set_mtu(ix_cc_rtmv4_nhid arg_NextHopId,
                                  ix_uint32 arg_Mtu,
                                  ix_cc_ipv4_cb_route_op arg_Callback,
                                  void* arg_pUserContext
                                 );


/**
 * NAME: ix_cc_ipv4_async_set_flags
 *
 * DESCRIPTION:This primitive helps client to update flags for a given next hop.
 * Before calling this message helper, client needs to call ix_cc_ipv4_async_add_next_hop to add next hop 
 * identifier in RTM database. The supported flags[BLDBLK] are defined by IPv4 Forwarder microblock.
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop.
 * @Param:  - IN arg_Flags: new flags for the given next hop 
 * @Param:  - IN arg_pUserContext: pointer to client defined context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */
ix_error ix_cc_ipv4_async_set_flags(ix_cc_rtmv4_nhid arg_NextHopId,
                                    ix_uint32 arg_Flags,
                                    ix_cc_ipv4_cb_route_op arg_Callback,
                                    void* arg_pUserContext
                                   );



/**
 * NAME: ix_cc_ipv4_async_get_sleep_time
 *
 * DESCRIPTION:This message helper primitive helps client to retrieve the number of seconds 
 * between when the event will fire causing the ICMP messages to dequeue and send. 
 * 
 * @Param:  - IN arg_Callback: pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */

ix_error ix_cc_ipv4_async_get_sleep_time(ix_cc_ipv4_cb_get_sleep_time arg_Callback,
                                         void* arg_pUserContext
                                        );


/**
 * NAME: ix_cc_ipv4_async_set_sleep_time
 *
 * DESCRIPTION: This message helper primitive helps client to set the number of seconds between 
 * when the event will fire causing the ICMP messages to dequeue and send.  The lower the value of arg_SleepTime, 
 * the more frequent the ICMP messages will be dequeued and sent.
 * 
 * @Param:  - IN arg_SleepTime: minimum value is 1 second and maximum value is 32 seconds.
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_IPV4_ERROR_INVALID_SLEEP_TIME - invalid sleep time
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */

ix_error ix_cc_ipv4_async_set_sleep_time(ix_uint16 arg_SleepTime);



/**
 * NAME: ix_cc_ipv4_async_get_queue_depth
 *
 * DESCRIPTION:This message helper primitive helps client to retrieves the depth of the ICMP error message queue. 
 * Client retrieves the maximum number of ICMP messages that can fit in the queue, not the number of messages
 * currently in the queue.
 * 
 * @Param:  - IN arg_Callback: pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */
ix_error ix_cc_ipv4_async_get_queue_depth(ix_cc_ipv4_cb_get_queue_depth arg_Callback,
                                          void* arg_pUserContext
                                         );

/**
 * NAME: ix_cc_ipv4_async_get_packets_to_drain
 *
 * DESCRIPTION:This message helper primitive helps client to retrieve the maximum number of ICMP error messages 
 * to send each time the queue processing event fires. The processing event will send up to this many ICMP error
 * packets from the queue before going back to sleep. 
 * 
 * @Param:  - IN arg_Callback: pointer to client provided callback function.
 * @Param:  - IN arg_pUserContext: pointer to client defined context.
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */
ix_error ix_cc_ipv4_async_get_packets_to_drain(ix_cc_ipv4_cb_get_packets_to_drain arg_Callback,
                                               void* arg_pUserContext
                                              );

/**
 * NAME: ix_cc_ipv4_async_set_packets_to_drain
 *
 * DESCRIPTION:This message helper primitive helps client to set the maximum number of ICMP error messages to 
 * send each time the queue processing event fires. The value to arg_PacketsToDrain must not exceed the depth of 
 * the queue.
 * 
 * @Param:  - IN arg_PacketsToDrain: The maximum number of packets to send each time the queue processing event fires.
 *
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_IPV4_ERROR_INVALID_PACKETS_TO_DRAIN - invalid value for packets to drain
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 * 
 */
ix_error ix_cc_ipv4_async_set_packets_to_drain(ix_uint16 arg_PacketsToDrain);




/**
 * NAME: ix_cc_ipv4_async_get_statistics
 *
 * DESCRIPTION:This message helper primitive helps client to retrieve statistics report from IPv4 Forwarder core component.
 * This primitive will give the statistics from IPv4 microblock, RTM and IPV4 Forwarder core component.  
 * 
 * @Param:  - IN arg_Callback: pointer to client provided callback.
 * @Param:  - IN arg_pUserContext: pointer to client defined context. 
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_MSG_LIBRARY- failure from Message Support Library 
 */

ix_error ix_cc_ipv4_async_get_statistics(ix_cc_ipv4_cb_get_statistics arg_Callback,
                                         void* arg_pUserContext
                                        );




/**
 * NAME: ix_cc_ipv4_init
 *
 * DESCRIPTION: This primitive shall be called and returned successfully before requesting any service
 *              from IPv4 Forwarder component. This primitive should be called only once to initialize 
 *              IPv4 Forwarder component.This function performs allocation of memory for symbols to be patched,
 *              Creation of 64 bit counters, registration of packet and message handlers,
 *              initialization and configuration of RTM, allocation and initialization of internal data structures
 *              and creation of event handler for ICMP.
 * 
 * @Param:  - IN arg_hCcHandle - handle to IPV4 core component, created by core component infrastructure; 
 *            this shall be used later to get other (to add event handler) services from core component infrastructure.
 *
 * @Param:  - OUT arg_ppContext - location where the pointer to the control block allocated by 
 *            IPv4 Forwarder core component will be stored. The control block is internal to IPv4 
 *            Forwarder core component and it contains information about IPv4 internal data structures, 
 *            allocated memory and other relevant information. Later this pointer to be passed into  
 *            ix_cc_ipv4_fini function for freeing memory and other house keeping operations 
 *            when IPv4 component is being destroyed.  
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_OOM   - memory allocation failure      
 * IX_CC_IPV4_ERROR_FAILED_PATCHING   - patching failures                      
 * IX_CC_ERROR_OOM_64BIT_COUNTER - 64 bit counter creation failure
 * IX_CC_IPV4_ERROR_REGISTRY  - registry access failures
 * IX_CC_IPV4_ERROR_RTM - failure from RTM core component
 * IX_CC_IPV4_ERROR_CCI - failure from CCI 
 * 
 */

ix_error ix_cc_ipv4_init(
                         ix_cc_handle arg_hCcHandle,
                         void** arg_ppContext
                        );



/**
 * NAME: ix_cc_ipv4_fini
 *
 * DESCRIPTION: This primitive shall be called to terminate services from IPv4 forwarder 
 * component. This primitive frees memory allocated during initialization, shut down RTM, 
 * free all the created resources: delete 64 bit counter. If client claims for any services 
 * from IPv4 after scalling this function, then the behavior is undefined.
 * 
 * @Param:  - IN arg_hCcHandle - handle to IPv4 Forwader core component
 *
 * @Param:  - IN arg_pContext - ppointer to the control block memory allocated 
 * earlier in ix_cc_ipv4_init function. 
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_INVALID_INPUT_PARAM - invalid input parameter(s)
 * IX_CC_ERROR _OOM   - failure of memory deletion                
 * IX_CC_ERROR_OOM_64BIT_COUNTER - failure of  64 bit counter deletion 
 * IX_CC_IPV4_ERROR _RTM - failure from RTM
 * IX_CC_IPV4_ERROR_CCI - failure from CCI
 * 
 */
ix_error ix_cc_ipv4_fini(
                         ix_cc_handle arg_hCcHandle,
                         void* arg_pContext
                        );


/**
 * NAME: ix_cc_ipv4_msg_handler
 *
 * DESCRIPTION: This primitive is the message handler function for IPv4 Forwarder
 * core component. IPV4 Forwarder core component receives messages from other core component
 * through this message handler function and internally it calls appropriate library function
 * to process the message. This message handler shall be used to update dynamic properties.
 * 
 * @Param:  - IN arg_hDataToken - buffer handle embedding information for the message
 * passed in arg_UserData.
 *
 * arg_UserData - message type
 *
 * @Param:  - OUT - arg_pComponentContext: pointer to IPv4 Forwarder core component context 
 * that will be passed to the core component when a message arrives. This context was defined
 * by the core component and passed to core components infrastructure through 
 * ix_cc_ipv4_init function.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - null argument
 * IX_CC_IPV4_ERROR_INVALID_CONTEXT - context is unusable
 * IX_CC_ERROR_UNDEFINED_MSG - un supported message
 * IX_CC_IPV4_ERROR_MSG_LIBRARY - error from message support library
 * 
 */
ix_error ix_cc_ipv4_msg_handler(
                                ix_buffer_handle arg_hDataToken,
                                ix_uint32 arg_UserData,
                                void* arg_pContext
                                );



/**
 * NAME: ix_cc_ipv4_microblock_high_priority_pkt_handler
 *
 * DESCRIPTION: This is the only registered function to receive exception packets from high priority
 * queue of IPv4 microblock. This function sends the packet to stack driver
 * (ie, recieved only local delivery packets). 
 * 
 * @Param:  - IN arg_hDataToken: handle to a buffer which contains exception packets
 *  from IPv4 microblock
 * @Param: - IN arg_ExceptionCode: will be ignored.
 * @Param: - IN arg_pContext: pointer to IPV4 Forwarder core component context that 
 * will be passed to the core component when a packet arrives. This context was defined by
 * the core component and passed to core components infrastructure through 
 * the ix_cc_ipv4_init function.
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_CCI - error from CCI
 * 
 */

ix_error ix_cc_ipv4_microblock_high_priority_pkt_handler(
                                                         ix_buffer_handle arg_hDataToken,  
                                                         ix_uint32 arg_ExceptionCode,
                                                         void* arg_pContext
                                                        );




/**
 * NAME: ix_cc_ipv4_microblock_low_priority_pkt_handler
 *
 * DESCRIPTION: TThis is the registered function to receive exception packets from 
 * IPv4 microblock. This function internally calls/performs different functions/operations
 * based on exception code for the given packet.
 * 
 * @Param:  - IN arg_hDataToken: handle to a buffer which contains exception packets
 * from IPv4 microblock
 * @Param: - IN arg_ExceptionCode: Exception codes generated by IPv4 forwarder microblock.
 * @Param: - IN arg_pContext: pointer to IPV4 Forwarder core component context that 
 * will be passed to the core component when a packet arrives. This context was defined by
 * the core component and passed to core components infrastructure through 
 * the ix_cc_ipv4_init function.
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_CCI - error from CCI
 * IX_CC_ERROR_UNDEFINED_EXCEP - unsupported exception code
 * IX_CC_ERROR_INTERNAL - an internal error
 * IX_CC_ERROR_ALIGN - pointer not aligned properly
 * IX_CC_ERROR_NULL - null pointer
 * IX_CC_IPV4_ERROR_RTM - error from RTM
 *
 *
 */
ix_error ix_cc_ipv4_microblock_low_priority_pkt_handler(
                                                        ix_buffer_handle arg_hDataToken,  
                                                        ix_uint32 arg_ExceptionCode,
                                                        void* arg_pContext
                                                       );





/**
 * NAME: ix_cc_ipv4_stkdrv_pkt_handler
 *
 * DESCRIPTION: This is the registered function to receive packets from stack driver 
 * core component. Packet comes from stack driver need special processing by 
 * IPv4 core component: no TTL decrement, no IP header validation.
 * 
 * @Param:  - IN arg_hDataToken: handle to a buffer which contains packet
 * from stack driver core component. - 
 *
 * @Param:  - IN arg_ExceptionCode:  will be Ignored 
 *
 * @Param:  - IN arg_pComponnetContext: pointer to IPV4 Forwarder core component context
 * that will be passed to the core component when a packet arrives. This context was 
 * defined by the core component and passed to core components infrastructure through 
 * the ix_cc_ipv4_init function. 
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_CCI - error from CCI
 * IX_CC_ERROR_INTERNAL - an internal error
 * IX_CC_ERROR_ALIGN - pointer not aligned properly
 * IX_CC_ERROR_NULL - null pointer
 * 
 */
ix_error ix_cc_ipv4_stkdrv_pkt_handler(
                                           ix_buffer_handle arg_hDataToken,
                                           ix_uint32 arg_ExceptionCode,
                                           void* arg_pContext
                                           );




/**
 * NAME: ix_cc_ipv4_common_pkt_handler
 *
 * DESCRIPTION: This is the registered function to receive packets 
 * from any core component other than stack driver: interface Rx core component.
 * 
 * @Param:  - IN arg_hDataToken: handle to a buffer which contains packet from any core component other than stack driver.
 * @Param:  - INarg_ExceptionCode:  will be Ignored
 * @Param:  - INarg_pComponnetContext: pointer to IPV4 Forwarder core component context 
 * that will be passed to the core component when a packet arrives. 
 * This context was defined by the core component and passed to core components infrastructure
 * through the ix_cc_ipv4_init function.
 *
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_INTERNAL - an internal error
 * IX_CC_ERROR_ALIGN - pointer not aligned properly
 * IX_CC_ERROR_NULL - null pointer
 * 
 */
ix_error ix_cc_ipv4_common_pkt_handler(
                                           ix_buffer_handle arg_hDataToken,
                                           ix_uint32 arg_ExceptionCode,
                                           void* arg_pContext
                                           );



/**
 * NAME: ix_cc_ipv4_add_route
 *
 * DESCRIPTION:This library primitive helps to add a route in RTM. 
 * This API internally calls ix_cc_rtmv4_add_route API of RTM core component to add route in RTM database.
 * 
 * @Param:  - IN arg_IpAddr: network IP address identifying the route.
 * @Param:  - IN arg_NetMask: network mask for the route.
 * @Param:  - IN arg_NextHopId: next hop identifier for route.
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_ENTRY_NOT_FOUND - invalid next hop identifier
 * IX_CC_ERROR_DUPLICATE_ROUTE - route was already added to this NHID
 * IX_CC_ERROR_CONFLICTING_ROUTE - route was already added to another NHID
 * IX_CC_IPV4_ERROR_RTM - RTM internal failure
 * 
 */

ix_error ix_cc_ipv4_add_route(
                              ix_uint32 arg_IpAddr,
                              ix_uint32 arg_NetMask,
                              ix_cc_rtmv4_nhid arg_NextHopId, 
                              void* arg_pContext
                              );



/**
 * NAME: ix_cc_ipv4_delete_route
 *
 * DESCRIPTION: This library primitive helps to delete a route from RTM. 
 * This API internally calls ix_cc_rtmv4_del_route API of RTM core component to delete route from RTM database.
 * 
 * @Param:  - IN arg_IpAddr: network IP address identifying the route
 *
 * @Param:  - IN arg_NetMask: network mask for the route
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_ENTRY_NOT_FOUND - entry not found in RTM
 * IX_CC_IPV4_ERROR_RTM - RTM internal failure
 * 
 */

ix_error ix_cc_ipv4_delete_route(
                                 ix_uint32 arg_IpAddr,
                                 ix_uint32 arg_NetMask, 
                                 void* arg_pContext
                                );


/**
 * NAME: ix_cc_ipv4_update_route
 *
 * DESCRIPTION:This library primitive helps to add a route in RTM. 
 * This API internally calls ix_cc_rtmv4_update_route API of RTM core component to 
 * add route in RTM database. 
 * 
 * @Param:  - IN arg_IpAddr: network IP address the route
 * @Param:  - IN arg_NetMask: network mask for the route
 * @Param:  - IN arg_NextHopId: hop identifier for route to be updated
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_ENTRY_NOT_FOUND - invalid next hop identifier
 * IX_CC_IPV4_ERROR_INVALID_MASK - invalid netmask
 * IX_CC_ERROR_CONFLICTING_ROUTE - route was already added to another NHID
 * IX_CC_IPV4_ERROR_RTM - RTM internal failure
 * 
 */

ix_error ix_cc_ipv4_update_route(ix_uint32 arg_IpAddr,
                              ix_uint32 arg_NetMask,
                              ix_cc_rtmv4_nhid arg_NextHopId, 
                              void* arg_pContext
                             );

/**
 * NAME: ix_cc_ipv4_lookup_route
 *
 * DESCRIPTION: This primitive helps a client to look up routing information for a given IP address.
 * If there is a successful match, then the information will be given to client are nextHopId (including the blade id),
 * portId, mtu and flags. 
 * 
 * @Param:  - IN arg_IpAddr: network IP address identifying the route to be updated.
 * @Param:  - INarg_NetMask: network mask for the route to be updated.
 * @Param:  - INarg_NextHopId: next hop identifier for the route to be updated.
 * @Param:  - IN arg_pContext: pointer to IPv4 context 
 *
 * @Param:  - OUT
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - null pinter
 * IX_CC_IPV4_ERROR_RTM - RTM internal error
 * 
 */

ix_error ix_cc_ipv4_lookup_route(
                                 ix_uint32 arg_IpAddr,
                                 ix_cc_rtmv4_next_hop_info *arg_pNextHopInfo, 
                                 void* arg_pContext);



/**
 * NAME: ix_cc_ipv4_purge_routes
 *
 * DESCRIPTION: This primitive helps a client to remove all routes from RTM.
 * 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - null pointer
 * IX_CC_ERROR_ALIGN - pointer not aligned properly
 * 
 */
ix_error ix_cc_ipv4_purge_routes(void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_dump_routes
 *
 * DESCRIPTION: This primitive helps to dump all routes of RTM in memory
 * 
 * @Param:  - IN arg_pBuffer - pointer to a block of memory where routes are to be stored.
 * @Param:  - IN arg_Size - size of buffer in bytes.
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - null pointer
 * IX_CC_ERROR_ALIGN - pointer not aligned properly
 * IX_CC_ERROR_FULL -  buffer is too small
 */
ix_error ix_cc_ipv4_dump_routes(
                                char* arg_pBuffer,
                                ix_uint32 arg_Size, 
                                void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_add_next_hop
 *
 * DESCRIPTION: This primitive helps a client to add next hop information to RTM database.
 * Client needs to add next hop to RTM in order to add routes referring to that next hop. 
 * 
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop to be added.
 * @Param:  - IN arg_pNextHopData: defined by RTM
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT  none 
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_ERROR_DUPLICATE_ROUTE - route was already added to this NHID
 * IX_CC_ERROR_CONFLICTING_ROUTE - route was already added to another NHID
 * IX_CC_IPV4_ERROR_RTM - RTM internal failure
 * IX_CC_IPV4_ERROR_INVALID_NH_INFO - invalid next hop information
 */

ix_error ix_cc_ipv4_add_next_hop(
                               ix_cc_rtmv4_nhid arg_NextHopId,
                               ix_cc_rtmv4_next_hop_info* arg_pNextHopInfo, 
                              void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_delete_next_hop
 *
 * DESCRIPTION: This primitive helps a client to delete next hop information from RTM database. 
 * Client needs to remove all routes associated with next hop before calling this primitive. 
 * If not, client gets an error.This API internally calls ix_cc_rtmv4_delete_next_hop API of RTM
 * to delete next hop information from RTM.
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop to be removed.
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT -none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_RTM - RTM internal failure
 * IX_CC_ERROR_ENTRY_NOT_FOUND - invalid next hop id
 * IX_CC_IPV4_ERROR_NEXT_HOP_ID_IN_USE -next hop id is in use
 */

ix_error ix_cc_ipv4_delete_next_hop(ix_cc_rtmv4_nhid arg_pNextHopId, 
                                    void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_update_next_hop
 *
 * DESCRIPTION: This primitive helps a client to add next hop information to RTM database.
 * Client needs to add next hop to RTM in order to add routes referring to that next hop. 
 * 
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop to be added.
 * @Param:  - IN arg_pNextHopData: defined by RTM
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT  none 
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - invalid pointer
 * IX_CC_IPV4_ERROR_RTM - RTM internal failure
 * IX_CC_IPV4_ERROR_INVALID_NH_INFO - invalid next hop information
 * IX_CC_ERROR_ENTRY_NOT_FOUND - invalid next hop id
 */

ix_error ix_cc_ipv4_update_next_hop(
                               ix_cc_rtmv4_nhid arg_NextHopId,
                               ix_cc_rtmv4_next_hop_info* arg_pNextHopInfo, 
                               void* arg_pContext);



/**
 * NAME: ix_cc_ipv4_get_next_hop
 *
 * DESCRIPTION:This library primitive helps to retrieve next hop information from RTM.
 * This API internally calls ix_cc_rtmv4_get_next_hop API of RTM core component to 
 * retrieve next hop information. 
 * 
 * @Param:  - IN arg_NextHopId: Identifier of the next hop information to be retrieved from RTM.
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT arg_pNextHopInfo: Result will be placed into this structure
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_RTM - RTM internal failure
 * IX_CC_ERROR_ENTRY_NOT_FOUND - invalid next hop id
 * 
 */
ix_error ix_cc_ipv4_get_next_hop(
                                 ix_cc_rtmv4_nhid arg_NextHopId,
                                 ix_cc_rtmv4_next_hop_info* arg_pNextHopInfo, 
                                 void* arg_pContext
                                );




/**
 * NAME: ix_cc_ipv4_dump_next_hops
 *
 * DESCRIPTION: This API internally calls ix_cc_rtmv4_dump_next_hop API of RTM core component. 
 * Client shall use IX_CC_RTMV4_DUMP_NEXT_HOP_SIZE to determine how many bytes to allocate.
 * 
 * @Param:  - IN arg_pBuffer - pointer to a block of memory where next hops are to be stored.
 * @Param:  - IN arg_Size - size of buffer in bytes
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_RTM - RTM failure
 * 
 */

ix_error ix_cc_ipv4_dump_next_hops(
                                char* arg_pBuffer,
                                ix_uint32 arg_Size, 
                                void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_purge_rtm
 *
 * DESCRIPTION: This primitive helps a client to removes all routes and next hops from RTM database.
 * 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_RTM - RTM failure
 * 
 */
ix_error ix_cc_ipv4_purge_rtm(void* arg_pContext);



/**
 * NAME: ix_cc_ipv4_set_mtu
 *
 * DESCRIPTION: This primitive helps client to update MTU for a given next hop. 
 * Before calling this, client needs to call ix_cc_ipv4_add_next_hop to add next hop 
 * identifier in RTM database. The validity of MTU shall be done by the client.
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop.
 * @Param:  - IN arg_Mtu: new maximum transmission unit for the given next hop.
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 *
 * IX_CC_IPV4_ERROR_RTM - RTM failure
 * IX_CC_ERROR_ENTRY_NOT_FOUND - invalid next hop id 
 * IX_CC_IPV4_ERROR_INVALID_NH_INFO - invalid mtu
 */

ix_error ix_cc_ipv4_set_mtu(
                            ix_cc_rtmv4_nhid arg_NextHopId,
                            ix_uint32 arg_Mtu, 
                            void* arg_pContext);



/**
 * NAME: ix_cc_ipv4_set_flags
 *
 * DESCRIPTION:This primitive helps client to update flags for a given next hop.
 * Before calling this primitive, client needs to call ix_cc_ipv4_add_next_hop to 
 * add next hop identifier in RTM database. 
 * 
 * @Param:  - IN arg_NextHopId: identifier of next hop.
 * @Param:  - IN arg_Flags: new flags for the given next hop 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_ERROR_RTM - RTM failure
 * IX_CC_ERROR_ENTRY_NOT_FOUND - invalid next hop id 
 * IX_CC_IPV4_ERROR_INVALID_FLAGS - invalid flags 
 * 
 */
ix_error ix_cc_ipv4_set_flags(
                              ix_cc_rtmv4_nhid arg_NextHopId,
                              ix_uint32 arg_Flags, 
                              void* arg_pContext);



/**
 * NAME: ix_cc_ipv4_get_sleep_time
 *
 * DESCRIPTION: This library primitive helps to retrieve the number of seconds 
 * between when the event will fire causing the ICMP messages to be dequeued and sent. 
 * The lower the value of arg_pSleepTime, the more frequently the ICMP messages will be 
 * dequeued and sent.  
 * 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT arg_pSleepTime: ICMP sleep period 
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - null pointer
 * 
 */
ix_error ix_cc_ipv4_get_sleep_time(ix_uint16* arg_pSleepTime, 
                                   void* arg_pContext);



/**
 * NAME: ix_cc_ipv4_set_sleep_time
 *
 * DESCRIPTION: This library primitive helps to set the period between events firing to process 
 * the ICMP error message queue. The lower the value of arg_SleepTime, the more frequently
 * the output queue will be processed. The minimum value of arg_SleepTime is 1 sec and 
 * maximum value can be 32sec.
 * 
 * @Param:  - IN arg_SleepTime: the number of seconds to wait between processing
 * the ICMP error message queue.
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_INVALID_SLEEP_TIME - invalid sleep time
 * 
 */

ix_error ix_cc_ipv4_set_sleep_time(ix_uint16 arg_pSleepTime, 
                                   void* arg_pContext);




/**
 * NAME: ix_cc_ipv4_get_queue_depth
 *
 * DESCRIPTION: This library primitive helps to retrieves the depth of the ICMP error message 
 * queue. That is, the maximum number of ICMP messages that can fit in the queue, not the
 * number of messages currently in the queue.
 * 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT arg_pQueueDepth: the current number of entries that the ICMP error message queue can hold.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - null pointer
 * 
 */
ix_error ix_cc_ipv4_get_queue_depth(ix_uint16* arg_pQueueDepth, 
                                    void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_get_packets_to_drain
 *
 * DESCRIPTION:This library primitive helps to retrieve the maximum number of ICMP error messages
 * to send (from the ICMP message queue ) each time the queue processing event fires. 
 * 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT arg_pPacketsToDrain: the maximum number of packets to send each time the 
 * event fires for processing the queue.  
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - invalid pointer
 * 
 */
ix_error ix_cc_ipv4_get_packets_to_drain(ix_uint16* arg_pPacketsToDrain, 
                                         void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_set_packets_to_drain
 *
 * DESCRIPTION: This library primitive helps to set the maximum number of ICMP error messages 
 * to send each time the queue processing event fires. The value of arg_PacketsToDrain
 * must not exceed the depth of the queue.
 * 
 * @Param:  - IN arg_PacketsToDrain: The maximum number of packets to send each time the queue processing
 * event fires. 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_IPV4_INVALID_PACKETS_TO_DRAIN - invalid value
 * 
 */
ix_error ix_cc_ipv4_set_packets_to_drain(ix_uint16 arg_pPacketsToDrain, 
                                         void* arg_pContext);



/**
 * NAME: ix_cc_ipv4_get_statistics
 *
 * DESCRIPTION: This library primitive helps to retrieve statistics report from IPV4 Forwarder 
 * core component. This primitive will give the statistics of IPV4 microblock, RTM and 
 * IPV4 Forwarder core component. Client needs to allocate memory to hold data.
 * 
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT arg_pBuffer: pointer to memory to hold statistics data.
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * 
 * IX_CC_ERROR_NULL - null pointer
 * 
 */
ix_error ix_cc_ipv4_get_statistics(ix_cc_ipv4_stats_data* arg_pBuffer,
                                   void* arg_pContext);


/**
 * NAME: ix_cc_ipv4_set_property
 *
 * DESCRIPTION: This library primitive helps to set dynamic properties of 
 * IPv4 Forwarder core component. 
 * 
 * @Param:  - IN arg_PropId: Identifier of a property or properties(ORed) to be updated 
 * for a port
 * @Param:  - IN arg_pPropData: pointer to dynamic property data structure
 * @Param:  - IN arg_pContext: pointer to IPv4 context
 *
 * @Param:  - OUT - none
 * 
 * @Return: IX_SUCCESS if successful or a valid error token for failure.
 *
 * Error Codes:
 * IX_CC_ERROR_NUL - invalid pointer
 * IX_CC_IPV4_ERROR_INVALID_INPUT_PARAM - invalid input param
 * IX_ERROR_FULL - table is full
 * IX_CC_ERROR_ENTRY_NOT_FOUND - entry not found in table
 *
 */
ix_error ix_cc_ipv4_set_property(ix_uint32 arg_PropId,
                                 ix_cc_properties* arg_pPropData, 
                                 void* arg_pContext);



#if defined(__cplusplus)
}
#endif /* end defined(__cplusplus) */

#endif /* end !defined(__IX_CC_IPV4_H__) */
