///////////////////////////////////////////////////////////////////////////////
//
//
//                  I N T E L   P R O P R I E T A R Y
//
//     COPYRIGHT [c]  2002 BY  INTEL  CORPORATION.  ALL RIGHTS
//     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY
//     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A
//     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER
//     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,
//     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT
//     THE PRIOR WRITTEN PERMISSION OF :
//
//                        INTEL  CORPORATION
//
//                     2200 MISSION COLLEGE BLVD
//
//               SANTA  CLARA,  CALIFORNIA  95052-8119
//
///////////////////////////////////////////////////////////////////////////////
//
//
//      File Name: packet_rx_two_me_util.uc
//
//      Purpose: Contains macros used when packet rx runs on two ME's.
//
//      History:
//
//      Date            Comment                         By
//      ---------------------------------------------------------------------
//
//      11/11/2002      Created                         asv
//
/////////////////////////////////////////////////////////////////////////////

#ifndef	__PACKET_RX_TWO_ME_UTIL_UC__
#define	__PACKET_RX_TWO_ME_UTIL_UC__

#include <dl_system.h>
#include <stdmac.uc>
#include <dispatch_loop.uc>
#include <hardware.h>
#include <packet_rx.h>
#include <packet_rx_context.h>
#include <packet_rx_common_util.uc>


//--------------------------------
//	First Packet RX ME - Begin
//--------------------------------
#ifdef FIRST_PACKET_RX_ME

//	Some Local Definitions

//	From context size, compute the power of 2. Needed for later computation.
#define_eval 	PO2_CTXSIZE		log2(CONTEXT_SIZE)

//	Global Signals

.sig volatile sig_next_thread					// next thread signal
.sig volatile rx_event							// receive wakeup signal

//	needed because <<&rx_event is not recognised by assembler.
.addr rx_event RX_EVENT_SIG

//	Global Regsiters

//	Following registers are used as constants to save instructions
//	If we run out of GPRs, these are prime candidates for removal.
//	If you run out of GPRs use the following criteria to decide
//	which register to remove
//
//					Instructions	where is it
//					   saved			saved
// ----------------------------------------------------------------------------
//	free_list_reg		4 			all paths
//	nt_sig_reg			2 + 2 		all paths (twice per iteration)
//	dram_rbuf_ov		2			all paths
//	rbuf_base			1			critical path (i.e min pkt case)
//	@rand_offset_mask	1			all paths 
//	@max_buf_size		1 			(not in critical path)
//	protocol_mask		1			all paths (but only when using 4 Byte PPP hdr)

.reg	@rand_offset_mask						// mask for wrapping around @ 512
.reg	rbuf_base								// base address of rbuf
.reg 	thread_num								// Current thread number
.reg	max_buf_size							// The max buf data size
.reg	freelist_reg							// to put a thread in free list.
.reg	nt_sig_reg								// used in signalling next thread
.reg 	dram_rbuf_ov							// for moving data from rbuf to dram
.reg 	protocol_mask							// PPP hdr has 16bit protocol field
.reg	rx_freelist								// RX free list gpr

#ifdef	RFC_PPP_COUNTERS
.reg	counter_base							// base addr for counters.
#endif

//	Following regsiters are used as global variables

.reg		@rand_buf_offset					// offset to use in next(SOP) buffer.
.reg read 	$prefetch_buf_handle				// buffer in which next pkt stored.
												// "read" for protection
.reg		prefetch_buf_handle_gpr				// gpr that holds $prefetch_buf_handle
.reg		prefetch_data_handle				// dram addr of above buffer
.reg		byte_count							// bytes in current mpacket.
.reg		sink_sig_mask						// mask to tell what I/Os were done
.reg 		$rsw1, $rsw2						// rbuf read status words
.xfer_order $rsw1, $rsw2

///////////////////////////////////////////////////////////////////////////////
//
// packet_pkt_rx
//
//	 	Description: 
//
// 			Packet processing entry point for packet microblock. In each iteration
//			one mpacket will be handled. Implements a state machine to receive
//			SOP, MOP and EOP mpackets.
//
//	 	Outputs:
//
//			All global variables will be affected.
//
//		Inputs:
//
//			Will use global varaibles.
//
///////////////////////////////////////////////////////////////////////////////
#macro packet_pkt_rx()
.begin

	.reg 	ctx_st								// keeps the machine state
	.reg 	$protocol							// protocol in PPP hdr
	.reg	protocol_reg						// protocol in PPP hdr
	.reg 	drop_sop_handle, drop_eop_handle	// SOP & EOP buffers to drop
	.reg 	port_num							// inpur port number
	.reg 	ele_num								// rbuf element number
	.reg 	lm_ctx_base							// base addr of reassembly context

	.set 	$rsw1, $rsw2						// to avoid warnings

	// Signal used while pre-fetching a buffer.
	.sig sig_buf_alloc

comn_rx_phase_1#:

	//	The thread autmatically wakes up on receiving an mpacket.
	//	The receive status words are available in $rsw1 and $rsw2.
	//	If no packet had been recieved within the timeout, then
	//	the status word will indicate that a NULL packet was received.

	_packet_rx_rsw_get_port_num(port_num, $rsw1)					; From which port pkt received.

	//	locate the context in local memory and setup local memory. 
	//	There are totally 16 contexts, one per port. Each conext 
	//	is 64 bytes (16 long words)

	alu_shf[lm_ctx_base, CONTEXT_DESC_BASE, or, port_num, <<PO2_CTXSIZE]
	local_csr_wr[active_lm_addr_0, lm_ctx_base]			; setup local memory for this port

	//	check if error in receviing

	alu[--, $rsw1, AND, ALL_ERROR_CODE, <<8] 			; 0x2E00. XXX - should it be 0x2a00?

	bne[rx_error_processing#],defer[1]					; goto err processing
	alu[sink_sig_mask, --, B, 1, <<RX_EVENT_SIG]     	; set sink_sig_mask	

	//	a proper mpacket was received. Depending on the State of the 
	//	Receive State Machine and the kind of mpacket (SOP, EOP, MOP)
	//	received branch accordingly.

	_packet_rx_get_machine_state(ctx_st, $rsw1)					; state will be returned in ctx_st

	_packet_rx_debug_incr_counter(@pkt_rx_num_proper_mpkts_rxed)

	//	There are only two states: START and PROCessing. There are four
	//	kinds of mpackets: SOP, EOP, MOP and SOP&EOP. MOP is (!SOP and !EOP).
	//	In START state SOP or SOP&EOP mpackets are expected.
	//	In Processing state MOP or EOP mpackets are expected.

	//	ctx_st is interpreted as follows.

	//	000 START 	state, 	MOP mpacket	--	Error case	--	msp
	//	001 PROC 	state, 	MOP mpacket	--	Normalcase	--	mpp
	//	010 START 	state, 	EOP mpacket	--	Error case	--	esp
	//	011 PROC 	state, 	EOP mpacket	--	Normal case	--	epp
	//	100 START 	state, 	SOP mpacket	--	Normal case	--	ssp
	//	101 PROC 	state, 	SOP mpacket	--	Error case	-- 	spp (+ normal case)
	//	110 START 	state, 	SOP&EOP 	--	Normal case	-- 	sesp
	//	111 PROC 	state, 	SOP&EOP 	--	Error case	--	sepp (+ normal case)

	//	jump to the corresponding state processing

	jump[ctx_st, jump_table#], defer[2], targets[msp_j#, mpp_j#, esp_j#, \
												epp_j#, ssp_j#, spp_j#, sesp_j#, sepp_j#] 
		_packet_rx_rsw_get_element_num(ele_num, $rsw1)			; get rbuf element number
		_packet_rx_rsw_get_byte_count(byte_count, $rsw1) 			; get number of bytes received

//	Start of jump table. 

jump_table#:
msp_j#:
	br[msp#]    ; goto mop_start_processing		--	Error case

mpp_j#:
	br[mpp#]	; goto mop_proc_processing		--	Normal case

esp_j#:
	br[esp#]	; goto eop_start_processing		--	Error case

epp_j#:
	br[epp#]	; goto eop_proc_processing		--	Normal case

ssp_j#:
	br[ssp#]	; goto sop_start_processing		-- 	Normal case

spp_j#:
	br[spp#]	; goto sop_proc_processing		--	Error + Normal case

sesp_j#:
	br[sesp#]	; goto sop_eop_start_processing	--	Normal case

sepp_j#:
	br[sepp#]	; goto sop_eop_proc_processing	--	Error + Normal case


//	Error processing code

rx_error_processing#: 

	//	We are here because receive status word indicated an error.

	//	check for NULL (i.e timeout)

	alu[--, $rsw1, AND, 1, <<9]							;[9] indicates NULL receive
	bne[null_error_processing#], defer[1]
		_packet_rx_rsw_get_element_num(ele_num, $rsw1)		; get rbuf element number

	//	Some other error. We need to drop any partially assembled packets.
	//	If we are in START state then we do not have any partially assembled packets.
	//	lm_state will have the current state. Only two states are used. So checking
	//	Bit[0] is enough.

	br_bclr[lm_state, 0, nothing_to_drop_in_error#]	; check if we are already in START state (0)

	//	Prepare to drop partially assembled packet. For large packets 
	//	drop_eop_handle will be the last buffer in the chain

	alu[drop_sop_handle, --, B, lm_sop_buf]
	alu[drop_eop_handle, --, B, lm_cur_buf]

	//	Reset State Machine

	_packet_rx_set_state(STATE_START)						; reset to start state

	//	End of Phase 1. Wait next thread signal to synchronize

	ctx_arb[sig_next_thread]

	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	//	Drop the mpacket.

	_packet_rx_drop_current_pkt(drop_sop_handle, drop_eop_handle)

	_packet_rx_incr_counter(port_num, PACKET_PKTS_DROPPED)

	//	Even though we dropped the packet, we'll propagate a NULL
	//	packet thro' the pipeline to keep the pipeline moving.

	//	Continue with Phase2 processing.

	// Free RBUF element.
	_packet_rx_free_rbuf_element(ele_num)

	br[comn_rx_phase_2#]

nothing_to_drop_in_error#:

	_packet_rx_debug_incr_counter(@pkt_rx_num_err_mpkts_rxed)

	//	End of Phase 1. Wait next thread signal to synchronize

	ctx_arb[sig_next_thread]

	br[comn_rx_phase_2#], defer[2]
		alu[dl_next_block, --, B, BID_NULL] 			; NULL (0xFF)
		alu[dl_buf_handle, --, B, DL_BUF_NULL]     		; NULL (0xFF)

null_error_processing#: 

	//	Null mpkt processing. (i.e receive logic timed out for this thread)
	
	//	At this point, the receiving logic has not received a mpkt for certain
	//	amount of time. In order to keep the pipeline moving, the receiving logic
	//	removes the thread from freelist and wakes up the thread by sending the
	//	thread a (rx_event) signal.  

	//	Nothing to do. Just synchronize with rest of the threads
	//	and go to Phase2.

	ctx_arb[sig_next_thread]

	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	br[comn_rx_phase_2#]

msp#:	
	//	Got MOP mpacket while expecting SOP packet - Error case 

esp#:
	//	Got EOP mpacket while expecting SOP packet - Error case

	//	Both the above error cases are handled here.
	//	Being in START state, we have absolutely nothing to do here.
	//	Just synchronize with rest of the threads and go to Phase2.

	ctx_arb[sig_next_thread]

	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	// Free RBUF element.
	_packet_rx_free_rbuf_element(ele_num)

	br[comn_rx_phase_2#]

mpp#:
	//	Got MOP mpacket while expecting a MOP/EOP mpacket: Normal case

	_packet_rx_mpp_processing()

	br[comn_rx_phase_2#]
				
sepp#:

	//	Got a SOP&EOP (both in same mpacket) while expecting MOP/EOP mpacket - Error
	
	//	We do two things. 
	//	[1] We discard the partially (so far) assembled packet.
	//	[2] And since SOP&EOP is a new but complete packet we can process it.
		
	//	[1]
	//	Reset State Machine

	_packet_rx_set_state(STATE_START)								; reset to start state
	 
	//	Prepare to drop the mpacket For large packets 
	//	drop_eop_handle will be the last buffer in the chain
	//	XXX - Why can't drop packet done befor sesp_processing?

	alu[drop_sop_handle, --, B, lm_sop_buf]
	alu[drop_eop_handle, --, B, lm_cur_buf]

	//	[2]	Process SOP&EOP mpacket

	_packet_rx_sesp_processing()

	//	drop the partially assembled pkt

	_packet_rx_drop_current_pkt(drop_sop_handle, drop_eop_handle)	

	_packet_rx_incr_counter(port_num, PACKET_PKTS_DROPPED)

	br[comn_rx_phase_2#]

epp#:
	//	Got EOP mapacket while expecting MOP/EOP - Normal case

	_packet_rx_epp_processing()

	br[comn_rx_phase_2#]

spp#:
	//	Got SOP mpacket while expecting MOP/EOP -  Error case

	//	We do two things. 
	//	[1] We discard the partially (so far) assembled packet.
	//	[2] And since SOP is a new packet we can process it.

	//	Reset State Machine. 

	_packet_rx_set_state(STATE_START)								; reset to start state

	//	Prepare to drop the packet. For large packets 
	//	drop_eop_handle will be the last buffer in the chain
	//	XXX - Why can't drop packet done befor sesp_processing? .

	alu[drop_sop_handle, --, B, lm_sop_buf]
	alu[drop_eop_handle, --, B, lm_cur_buf]

	//	Do the SOP processing for the newly received mpacket

	_packet_rx_ssp_processing()

	//	drop the partially assembled pkt

	_packet_rx_drop_current_pkt(drop_sop_handle, drop_eop_handle)	

	_packet_rx_incr_counter(port_num, PACKET_PKTS_DROPPED)

	br[comn_rx_phase_2#]

ssp#:
	//	Got SOP mpacket while expecting a SOP mpacket -  normal case

	//	Do the SOP processing for the newly received mpacket

	_packet_rx_ssp_processing()

	br[comn_rx_phase_2#]

sesp#:
	//	Got SOP mpacket while expecting a SOP mpacket -  normal case

	//	This is the critical path, where min packets are handled. 

	_packet_rx_sesp_processing()
	
comn_rx_phase_2#:

	// add current thread to freelist 0

	msf[fast_wr, --, rx_freelist, freelist_reg]			; add thread to freelist

	// Wait to receive next m-packet signal and possibly prefetch buffer signal.
	ctx_arb[--], defer[1]
	local_csr_wr[active_ctx_wakeup_events, sink_sig_mask]

	.io_completed sig_buf_alloc

.end
#endm


///////////////////////////////////////////////////////////////////////////////
//
// _packet_rx_write_to_nn_ring
//
//	 	Description: Write information worth 5 LW's to the nn ring so that
//      the next me in the rx context pipeline can read it and continue
//      processing.
//
//      The format is
//          LW0 (31:0) : in_dram_handle
//          LW1 (31:0) : in_curr_buf_handle
//          LW2 (31:0) : in_sop_buf_handle
//          LW3 (31:16): in_sop_buf_size
//          LW3 (15:15): in_eop_flag
//          LW3 (14:8) : in_rbuf_elem
//          LW3 (7:0)  : in_byte_count
//          LW4 (31:16): in_inport
//          LW4 (15:0) : in_packet_len
//
//	 	Outputs: None.
//
//		Inputs:
//			in_dram_handle: DRAM address where the m-packet will be written.
//          in_curr_buf_handle: current buffer handle of packet (will be eop
//            buf handle when called by eop macro. (only valid if 
//            in_eop_flag is 1).
//          in_sop_buf_handle: sop buffer handle of packet. (only valid if 
//            in_eop_flag is 1).
//          in_byte_count: number of bytes to transfer from RBUF to DRAM.
//          in_rbuf_elem: RBUF element number containing the m-packet.
//          in_eop_flag: bit indicating if this is the last m-packet for the
//            current packet.
//          in_sop_buf_size: Size in bytes of sop buffer.(only valid if 
//            in_eop_flag is 1).
//          in_packet_len: Size in bytes of entire packet.(only valid if 
//            in_eop_flag is 1).
//          in_inport: Input port number of the packet.(only valid if 
//            in_eop_flag is 1).
//
///////////////////////////////////////////////////////////////////////////////
#macro _packet_rx_write_to_nn_ring(in_dram_handle, in_curr_buf_handle, in_sop_buf_handle,\
			in_byte_count, in_rbuf_elem, in_eop_flag, in_sop_buf_size, \
			in_packet_len, in_inport)
.begin

	.reg tmp

writing_to_nn_ring#:
	br_inp_state[NN_FULL, writing_to_nn_ring#]

	alu[*n$index++, --, b, in_dram_handle]
	alu[*n$index++, --, b, in_curr_buf_handle]
	alu[*n$index++, --, b, in_sop_buf_handle]

	alu[tmp, --, b, in_byte_count]
	alu[tmp, tmp, or, in_rbuf_elem, <<8]
	alu[tmp, tmp, or, in_eop_flag, <<15]
	alu[*n$index++, tmp, or, in_sop_buf_size, <<16]

	alu[tmp, --, b, in_packet_len]
	alu[*n$index++, tmp, or, in_inport, <<16]

.end
#endm


///////////////////////////////////////////////////////////////////////////////
//
// _packet_rx_sesp_processing
//
//	 	Description: 
//
//			Process a mpacket containing both SOP&EOP. i.e packets size
//			ranging from 42(min) - 128 are handled here. Min packet case 
//			is the critical path, so it will be optimised wherever possible.
//
//	 	Outputs:
//
//		Inputs:
//
//		Size: 
//
///////////////////////////////////////////////////////////////////////////////

#macro _packet_rx_sesp_processing()

.begin

	.reg	buf_offset
	.reg	header_type
	.reg	dram_handle
	
	//	SOP and EOP are in the same mpacket. This is the critical path
	//	i.e min packets are handled here. 

	; Set 'prefetch_buf_handle_gpr' to the pre-fetched buffer handle.
	; See macro _packet_rx_buf_alloc() for why we use this instruction.
	alu[prefetch_buf_handle_gpr, prefetch_buf_handle_gpr, xor, $prefetch_buf_handle]

	//	check if prefetch buffer handle is zero

	br=byte[prefetch_buf_handle_gpr, 3, 0, sesp_prf_buf_zero_proc#]

	//	We got a valid buffer here.

	//	Get a random offset (headroom) for this buffer. 

	_packet_rx_get_rand_buf_offset(buf_offset)

	//	prefetch_data_handle is where the buffer begins.
	//	+ buf_offset will specify where we should store data 
	//	(including PPP hdr)

	alu[dl_buf_handle, --, B, prefetch_buf_handle_gpr] 	; set dl_buf_handle 
	dl_buf_get_data(prefetch_data_handle, prefetch_buf_handle_gpr)	

	//	Prefetch a new buffer so that it can be used next time around.

	_packet_rx_buf_alloc(BUF_FREE_LIST0, sig_buf_alloc, SIG_NONE, sink_sig_mask)

	//	Find out if the received mpacket can be contained in 1 cell 
	//	(on the transmit side). Remember that for every packet there is
	//	a header ( 8 bytes) that gets prepended. (This is apart from per 
	//	cell header (8 bytes) that gets added) This Per pkt header is added
	//	on the fly. Hence it is not counted in buffer/packet size.

	#define	FCELL_SIZE_WO_PPHDR		(CELL_SIZE - PER_PKT_PREPEND_BYTES)

	alu[--, byte_count, -, FCELL_SIZE_WO_PPHDR]			; > cell size?

	//	Remember if the packet can be contained in one cell, 
	//	then the cell count is 0. If it can be contained in 
	//	two cells, then cell count is 1 etc. 
	//	(less 1 than the actual) 
	ble[continue_with_sesp#],defer[1]		; yes, 1 cell is enough.
		alu[dram_handle, buf_offset, +, prefetch_data_handle]

	//	The packet cannot be contained in one cell. Calculate the number
	//	of cells in the packet. The cell count calculation is generic 
	//	with no hard coded values but is expensive. Watch out if
	//	you run out of budget.

	.reg	cell_count
	.reg	bytes_remaining
	.reg	temp_byte_count

	alu[temp_byte_count, byte_count, +, PER_PKT_PREPEND_BYTES] 	; cell includes per pkt hdr
	_packet_rx_set_first_cell_count(cell_count, bytes_remaining, temp_byte_count)
	alu[dl_buf_handle, dl_buf_handle, OR, cell_count, <<24]

	br[continue_with_sesp#]

	//	Error Handling.

sesp_prf_buf_zero_proc#:

	_packet_rx_incr_counter(port_num, PACKET_PKTS_DROPPED)

	_packet_rx_debug_incr_counter[@pkt_rx_buf_alloc_fail]

	//	We got a null buffer. So nothing to do. We still need to keep the pipeline
	//	moving. Get a new prefetch buffer so that it can used next time around.

	_packet_rx_buf_alloc(BUF_FREE_LIST0, sig_buf_alloc, SIG_NONE, sink_sig_mask)

	ctx_arb[sig_next_thread]

	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	// Free RBUF element.
	_packet_rx_free_rbuf_element(ele_num)
 
	br[end#]

continue_with_sesp#:

	//	increment packets received and bytes received counter
	_packet_rx_incr_counter(port_num, PACKET_PKTS_RECEIVED)			; pkts rcvd++

	ctx_arb[sig_next_thread]

	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	// Write information to nn ring so that the 2nd me in the rx context 
	// pipeline can continue processing.
	_packet_rx_write_to_nn_ring(dram_handle, DL_BUF_NULL, dl_buf_handle,
		byte_count, ele_num, 1, byte_count,
		byte_count, port_num)
  
end#:

#undef	FCELL_SIZE_WO_PPHDR

.end
#endm


///////////////////////////////////////////////////////////////////////////////
//
// _packet_rx_ssp_processing
//
//	 	Description: 
//
//			Process a mpacket containing only SOP. Even though we have
//			3 times the budget (3 * 88) as min (42 byte) packets
//			we should be careful with 129 byte packet, where the budget is 
//			reduced to 1.5 * 88 (because we now have to handle two mpackets
//			in 3 * 88 budget. The second mpacket with EOP will be handled in
//			epp_processing). 
//
//	 	Outputs:
//
//		Inputs:
//
//		Size: 
//
///////////////////////////////////////////////////////////////////////////////

#macro _packet_rx_ssp_processing()

.begin

	.reg 	temp_byte_count
	.reg	buf_offset
	.reg	header_type
	.sig	sig_msf_rd

	; Set 'prefetch_buf_handle_gpr' to the pre-fetched buffer handle.
	; See macro _packet_rx_buf_alloc() for why we use this instruction.
	alu[prefetch_buf_handle_gpr, prefetch_buf_handle_gpr, xor, $prefetch_buf_handle]

	dl_buf_get_data(prefetch_data_handle, prefetch_buf_handle_gpr)
	
	//	check if prefetch buffer handle is zero

	alu[--, --, B, prefetch_buf_handle_gpr, <<8]			; [24:0] contains buf handle
	beq[ssp_prf_buf_zero_proc#]							; goto error handling
	alu[header_type, lm_header_type,and,0xFF]			; get header type

	//	This is the first SOP mpacket within the packet

	_packet_rx_get_rand_buf_offset(buf_offset)						; get a random offset for sop

	//	prefetch_data_handle is where the buffer begins.
	//	+ buf_offset will specify where we should store data 
	//	(including PPP hdr)

	alu[lm_cur_data, buf_offset, +, prefetch_data_handle]	; add offset to buf begining

	.reg lm_cur_data_gpr
	alu[lm_cur_data_gpr, --, b, lm_cur_data]

	//	Update various reassembly context data.

	alu[lm_cur_buf, --, B, prefetch_buf_handle_gpr]		; prefetch to current 
	alu[lm_sop_buf, --, B, prefetch_buf_handle_gpr] 		; also make it sop buffer
	alu[lm_sop_buf_offset, --, B, buf_offset]			; remember this offset as
	alu[lm_buf_offset, --, B, buf_offset]				; sop buf and cur buf offset 
	alu[lm_buf_size, --, B, byte_count]					; buf size excludes hdr lenth 
	alu[lm_pkt_size, --, B, byte_count]					; pkt size excludes hdr length

	#ifdef IXP2800

	; In IXP2800 reference designs, we write 3 LW's on the outgoing scratch ring
	; instead of 5 LW's. The additional 2 LW's contain sop buffer meta-data. We
	; need to write this meta-data to SRAM now.

	xbuf_alloc($sop_meta, 4, write)

	.sig sig_sop_meta

	; Flush 2 LW's of SOP buffer meta-data to SRAM.
	dl_meta_set_header_type(header_type)				; set header type for the port
	dl_meta_set_rx_stat(0)								; set rx stat. Zero for now
	dl_meta_set_free_list(FREE_LIST_ID)					; set free list. 
	dl_meta_set_input_port(port_num)        			; set the input port
	
	dl_meta_flush_cache($sop_meta, prefetch_buf_handle_gpr, sig_sop_meta, SIG_NONE, 2, 2)

	#endif	// IXP2800

	//	Cell Count Calculation.
	//	On the transmit side, for every packet we add a prepend hdr
	//	(apart from a cell hdr for every cell). Include this hdr in
	//	cell count calculation. (For a given packet, we'll hit SOP
	//	condition only once so this place fits well for this 
	//	work)

	alu[temp_byte_count, byte_count, +, PER_PKT_PREPEND_BYTES]
	_packet_rx_set_first_cell_count(lm_cell_count, lm_bytes_remaining, temp_byte_count)

	//	Advance the current data pointer (where next mpacket will
	//	be stored)

	alu[lm_cur_data, lm_cur_data, +, byte_count]		; move the cur data ptr

	//	Prefetch a new buffer so that it can be used next time around.

	_packet_rx_buf_alloc(BUF_FREE_LIST0, sig_buf_alloc, SIG_NONE, sink_sig_mask)

	//	Move the Receive Machine to PROCessing State

	_packet_rx_set_state(STATE_PROC)

	//	End of Phase 1
	#ifdef IXP2800
	ctx_arb[sig_next_thread, sig_sop_meta]
	xbuf_free($sop_meta)
	#else
	ctx_arb[sig_next_thread]
	#endif	// IXP2800

	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	// Write information to nn ring so that the 2nd me in the rx context 
	// pipeline can continue processing.
	_packet_rx_write_to_nn_ring(lm_cur_data_gpr, 0, 0,\
			byte_count, ele_num, 0, 0, \
			0, 0)

	//	Phase 2

	br[ssp_end#]

	//	Null Buffer error condition

ssp_prf_buf_zero_proc#:

	_packet_rx_incr_counter(port_num, PACKET_PKTS_DROPPED)

	_packet_rx_debug_incr_counter[@pkt_rx_buf_alloc_fail]

	//	We got a null buffer. So nothing to do. We still need to keep the pipeline
	//	moving. Get a new prefetch buffer so that it can used next time around.

	_packet_rx_buf_alloc(BUF_FREE_LIST0, sig_buf_alloc, SIG_NONE, sink_sig_mask)

	ctx_arb[sig_next_thread]
	
	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	// Free RBUF element.
	_packet_rx_free_rbuf_element(ele_num)

ssp_end#:

.end
#endm

///////////////////////////////////////////////////////////////////////////////
//
// _packet_rx_epp_processing
//
//	 	Description: 
//
//			Process a mpacket containing only EOP. 
//
//	 	Outputs:
//
//		Inputs:
//
//		Size: 
//
///////////////////////////////////////////////////////////////////////////////

#macro _packet_rx_epp_processing()

.begin

	.reg	cur_buf
	.reg	prev_buf
	.reg 	cell_count
	.reg	buf_offset
	.reg	header_type
	.sig	sig_prev_meta
	.sig	sig_cur_meta

	#ifdef IXP2800
	; Flush meta-data belonging to channel 0 and BUF_SRAM_CHANNEL using
	; separate macro calls.
	.sig	sig_prev_meta1 sig_cur_meta1
	#endif

	.reg lm_cur_data_gpr \
		 lm_sop_buf_gpr \
		 lm_pkt_size_gpr \
		 lm_buf_size_gpr

	//	Got EOP mpacket while expecting MOP/EOP - Normal case
	//	Receive State Machine is in PROCessing State.

	alu[header_type, lm_header_type,and,0xFF]
		
	//	increment cell count depending on the number of bytes received.
	//	embed this cell count in the buffer handle.

	_packet_rx_incr_cell_count(lm_cell_count, lm_bytes_remaining, byte_count)

	alu[cell_count, --, B, lm_cell_count]
	alu[cur_buf, --, B, lm_cur_buf] 					; get curr buffer handle
	alu[cur_buf, cur_buf, OR, cell_count, <<24] 		; embed cell count
														; by default EOP bit is set.
	//	update buffer size, packet size and reset
	//	state machine to START state.
	//	[See defer slots]

	//	Check if current buffer is a sop buffer

	br_bclr[cur_buf, 30, epp_not_sop_buffer#], defer[3]	; [30] is SOP bit.
		alu[lm_buf_size, lm_buf_size, +, byte_count]	; update buffer size
		alu[lm_pkt_size, lm_pkt_size, +, byte_count]	; update packet size
		_packet_rx_set_state(STATE_START)					; Reset State Machine

epp_sop_buffer#:

	_packet_rx_incr_counter(port_num, PACKET_PKTS_RECEIVED)			; pkts rcvd++

	//	Yes, this buffer has SOP. i.e this buffer now contains
	//	a complete packet (SOP ...EOP). So we need to handle only
	//	a single buffer. Set all META data fields and write the 
	//	meta data to SRAM.
		
	//	buffer offset is where data begins in the buffer.
	// 	For subsequent blocks buffer offset should point to 
	//	where IP header (or rather whatever follows PPP header)
	//	begins.
	alu[buf_offset, --, B, lm_sop_buf_offset]			; get buffer offset 
	dl_meta_set_offset(buf_offset)
	dl_meta_set_buffer_next(IX_NULL)               		; there is no next buffer
	dl_meta_set_buffer_size(lm_buf_size)  				; set buffer size
	dl_meta_set_packet_size(lm_pkt_size)				; set packet size (= buf size)
	dl_meta_set_input_port(port_num)        			; set the input port
	dl_meta_set_header_type(header_type)				; set header type for the port
	dl_meta_set_free_list(FREE_LIST_ID)					; set free list. 
	dl_meta_set_rx_stat(0)								; set rx stat. Zero for now

	//	Write the meta data to SRAM.

	xbuf_alloc($cur_meta,4 , write) 					; alloc xfer regs for flush 

	dl_meta_flush_cache($cur_meta, cur_buf, sig_cur_meta, SIG_NONE, 0, 4)

	// Copy registers that will be used during nn transfer into gpr's.
	alu[lm_cur_data_gpr, --, b, lm_cur_data]
	alu[lm_buf_size_gpr, --, b, lm_buf_size]
	alu[lm_pkt_size_gpr, --, b, lm_pkt_size]

	//	End of Phase 1.
	//	Wait on the all the previous I/O plus the next thread signal.

	#ifdef IXP2800
	; Flush meta-data belonging to channel 0 and BUF_SRAM_CHANNEL using
	; separate macro calls.
	ctx_arb[sig_next_thread, sig_cur_meta, sig_cur_meta1]
	#else
	ctx_arb[sig_next_thread, sig_cur_meta]
	#endif	// IXP2800

	xbuf_free($cur_meta) 								; free sram tranfer regs

	// Write information to nn ring so that the 2nd me in the rx context 
	// pipeline can continue processing.
	_packet_rx_write_to_nn_ring(lm_cur_data_gpr, DL_BUF_NULL, cur_buf,\
			byte_count, ele_num, 1, lm_buf_size_gpr, \
			lm_pkt_size_gpr, port_num)

	br[end#]
	

epp_not_sop_buffer#:

	//	At this point, EOP and SOP are in different buffers i.e large packet.
	//  So we need to write previous buffer's meta data & current buffer's (EOP) 
	//	meta data to SRAM.
	//	There are two cases to handle.

	//	[1] Previous Buffer is also SOP Buffer.
	//	[2] Previous Buffer is not SOP Buffer.

	//  [Recap] cur_buf contains current buffer handle. cell count
	//	had already been updated in the buffer handle.

	//	compose meta data for the previous buffer. (defer slots)

	//	Check if the previous buffer is a SOP buffer. If it is not a SOP buffer,
	//	the offset is set as NON_SOP_OFFSET(0) (in the defer slot) i.e no headroom
	//	If it is a SOP buffer we have stored the offset in lm. Use that.

	alu[prev_buf, --, B, lm_prev_buf] 

	br_bclr[prev_buf, 30, epp_prev_buffer_not_sop#], defer[3]	; [30] is SOP bit
		dl_meta_set_buffer_next(cur_buf)    			; cur buf is next for prev
		dl_meta_set_buffer_size(lm_prev_buf_size)  		; set buffer size
		dl_meta_set_offset(NON_SOP_OFFSET)				; saves an extra branch

epp_prev_buffer_sop#:

	//	Previous buffer is a SOP buffer. Buffer offset should point to where 
	//	IP header (or rather whatever follows PPP header) begins.
	alu[buf_offset,--, B, lm_sop_buf_offset]			; strip header len
	dl_meta_set_offset(buf_offset)						; set offset
	dl_meta_set_header_type(header_type)				; set header type for the port

epp_prev_buffer_not_sop#:

	xbuf_alloc($prev_meta1, 2, write) 					; allocate xfer regs
	xbuf_alloc($cur_meta1, 2, write) 					; allocate xfer regs

	//	Write META data of previous buffer to SRAM
	dl_meta_flush_cache($prev_meta1, prev_buf, sig_prev_meta, SIG_NONE, 0, 2)

	//	Set current buffer meta data

	dl_meta_set_buffer_next(IX_NULL)  					; this is the last buffer
	dl_meta_set_offset(NON_SOP_OFFSET) 					; this buffer !SOP buffer
	dl_meta_set_buffer_size(lm_buf_size)    			; set buffer size
	
	//	Write META data of current buffer to SRAM
	dl_meta_flush_cache($cur_meta1, cur_buf, sig_cur_meta, SIG_NONE, 0, 2)

	//	End of Phase 1.
	//	Wait on the all the previous I/O plus the next thread signal.

	// Copy registers that will be used during nn transfer into gpr's.
	alu[lm_cur_data_gpr, --, b, lm_cur_data]
	alu[lm_sop_buf_gpr, --, b, lm_sop_buf]
	alu[lm_pkt_size_gpr, --, b, lm_pkt_size]

	#ifdef IXP2800
	; Flush meta-data belonging to channel 0 and BUF_SRAM_CHANNEL using
	; separate macro calls.
	ctx_arb[sig_next_thread, sig_prev_meta, sig_cur_meta, \
			sig_prev_meta1, sig_cur_meta1]
	#else
	ctx_arb[sig_next_thread, sig_prev_meta, sig_cur_meta]
	#endif	// IXP2800

	xbuf_free($prev_meta1)  							; free $prev_meta1
	xbuf_free($cur_meta1)								; free $cur_meta1

	// Write information to nn ring so that the 2nd me in the rx context 
	// pipeline can continue processing.
	_packet_rx_write_to_nn_ring(lm_cur_data_gpr, cur_buf, lm_sop_buf_gpr,\
			byte_count, ele_num, 1, lm_sop_buf_size, \
			lm_pkt_size_gpr, port_num)

end#:
	
	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

.end
#endm


///////////////////////////////////////////////////////////////////////////////
//
// _packet_rx_mpp_processing
//
//	 	Description: 
//
//			Process a mpacket containing neither SOP nor EOP. i.e this is a 
//			MOP (middle of packet).
//
//	 	Outputs:
//
//		Inputs:
//
//		Size: 
//
///////////////////////////////////////////////////////////////////////////////

#macro _packet_rx_mpp_processing()

.begin

	.reg 	buf_size
	.reg	cur_buf
	.reg	cell_count
	.reg 	buf_offset
	.reg	cur_data
	.reg 	temp
	.reg 	header_type
	.reg	buf_hndl
	.sig	sig_meta

	#ifdef IXP2800
	; Flush meta-data belonging to channel 0 and BUF_SRAM_CHANNEL using
	; separate macro calls.
	.sig	sig_meta1
	#endif

	//	Expect a MOP and get a MOP: Normal case

	//	Increment buffer size by number of bytes received. Buffer size
	//	is the amount of the data currently present in the buffer. We are
	//	guaranteed to have room in the buffer for this mpacket.
	//	[ See the defer slots below]

	//	Find out if buffer is full (for next mpacket). 
	//	Buffer size + headroom in the buffer will show 
	//	if the buffer is full.
	
	alu[buf_size, byte_count, +, lm_buf_size]			; buf size after THIS mpacket.
	alu[buf_size, buf_size, +, lm_buf_offset] 			; + headroom
	
	alu[--, buf_size, -, max_buf_size]					; is buffer full? for next time

	blt[mpp_buffer_not_full#],defer[2] 					; no, goto buffer not full
		alu[lm_buf_size, byte_count, +, lm_buf_size]	; increment buffer size						
		alu[lm_pkt_size, byte_count, +, lm_pkt_size]	; increment packet size

	//	At this point buffer will be full after storing the current
	//	mpacket. See if a new buffer (prefetch buffer) is available .

	; Set 'prefetch_buf_handle_gpr' to the pre-fetched buffer handle.
	; See macro _packet_rx_buf_alloc() for why we use this instruction.
	alu[prefetch_buf_handle_gpr, prefetch_buf_handle_gpr, xor, $prefetch_buf_handle]

	dl_buf_get_data(prefetch_data_handle, prefetch_buf_handle_gpr)

	alu[--, --, B, prefetch_buf_handle_gpr, <<8]			; [23:0] contains buf handle
	beq[mpp_null_buffer#]								; null buffer error processing	

	//	New buffer is available. Normal processing.

	// Write information to nn ring so that the 2nd me in the rx context 
	// pipeline can continue processing.
	_packet_rx_write_to_nn_ring(lm_cur_data, 0, 0,\
			byte_count, ele_num, 0, 0, \
			0, 0)

	//	check if current buffer is a SOP buffer. (i.e a buffer containing
	//	start of packet.)

	alu[cur_buf, --, B, lm_cur_buf] 					; lm cannot be used for bit ops
	br_bclr[cur_buf, 30, mpp_buffer_full_not_sop#], defer[1]	; [30] is SOP bit
		alu_shf[cur_buf, cur_buf, AND~, 0x1, <<31]		; anyway, clear EOP [31] bit.
														; because this buffer will not
														; contain EOP.
mpp_buffer_full_sop#:

	//	[Recap] This buffer is full. This buffer contains SOP. 
	//	A new buffer is available.

	//	increment cell count depending on the number of bytes received.
	//	embed this cell count in the buffer handle.

	_packet_rx_incr_cell_count(lm_cell_count, lm_bytes_remaining, byte_count)

	alu[cell_count, --, B, lm_cell_count]				; lm can't be used with <<
	alu[cur_buf, cur_buf, OR, cell_count, <<24]			; [29:24] are cell count

	//	At this point, no meta data needs to be flushed to SRAM
	//	keep the SOP buffer offset, buffer size and move current
	//	buffer handle to the previous buffer handle in the context data
	//	cur_buf contains the current buffer handle

	//	We are done with current buffer. Make it previous buffer as well as
	//	SOP Buffer

	alu[lm_sop_buf, --, B, cur_buf]						; cur buf is SOP buf
	alu[lm_prev_buf, --, B, cur_buf]					; cur buf to prev buf
	alu[lm_prev_buf_size, --, B, lm_buf_size]			; buf size to prev

	//	Since this buffer is also SOP buffer, remember
	//	a few more fields.

	alu[lm_sop_buf_size,--, B, lm_buf_size]				; buf size to sop buf size.

	//	Ok, now make the prefetch buffer as current buffer. Any new
	//	mpacket received will be stored in this new buffer.

	alu[temp, prefetch_buf_handle_gpr, AND~, 0x1, <<30]	; clear sop bit
	alu[lm_cur_buf, --, B, temp]						; prefetch buffer is current
	alu[lm_cur_data, --, B, prefetch_data_handle]		; prefetch data ptr is current

	//	clear cell count and byte reminder.

	alu[lm_cell_count, --, B, 0x0] 						; clear cell count
	alu[lm_bytes_remaining, --, B, 0x0] 				; clear bytes remaining

	//	Prefetch a new buffer so that it can be used next time around.

	_packet_rx_buf_alloc(BUF_FREE_LIST0, sig_buf_alloc, SIG_NONE, sink_sig_mask)

	//	End of Phase 1.
	//	Wait on the all the previous I/O plus the next thread signal.

	ctx_arb[sig_next_thread], defer[2]
		alu[lm_buf_size, --, B, 0] 						; set buffer size as 0
		alu[lm_buf_offset, --, B, NON_SOP_OFFSET]		; set offset as 0
	
	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	// 	go to phase 2 processing

	//	Remember that we have received only a part of the packet so far.

	br[end#]
									
mpp_buffer_full_not_sop#:
	
	//	[Recap]: Current buffer is full. But it does not contain SOP.
	//	cur_buf contains the current buffer handle.

	//	Two slightly different cases to handle here. 
	//	[1] Previous Buffer is SOP. (this will have headroom)
	//	[2] Previous Buffer is not SOP. (this will not)

	//	increment cell count depending on the number of bytes received.
	//	embed this cell count in the buffer handle.
	//	XXX - Why can't this cell count calc. be done in the common path 
	//	up above.

	_packet_rx_incr_cell_count(lm_cell_count, lm_bytes_remaining, byte_count)

	alu[cell_count, --, B, lm_cell_count]				; lm can't be used with <<
	alu[cur_buf, cur_buf, OR, cell_count, <<24]			; [29:24] are cell count

	//	Now we are ready to flush META data of previous buffer to SRAM.
	//	Remember the META data contains a hw_next pointer which, in this case, 
	//	should point to current buffer. [Various META data fields are set in
	//	the defer slots below]

	alu[dl_buf_handle,--, B, lm_prev_buf]				; prepare to set meta data

	//	Check if the previous buffer is a SOP buffer
	//	If it is not a SOP buffer, the offset is set 
	//	as NON_SOP_OFFSET(0) (in the defer slot) i.e no headroom
	//	If it is a SOP buffer we have stored the offset in
	//	lm. Use that.

	br_bclr[dl_buf_handle,30,mpp_prev_buffer_not_sop#], defer[3]
		dl_meta_set_buffer_next(cur_buf)    			; cur buf is next for prev.
		dl_meta_set_buffer_size(lm_prev_buf_size)  		; set previous buffer size
		dl_meta_set_offset(NON_SOP_OFFSET)				; saves an extra branch

	//	Here, previous buffer is a SOP buffer
	//	set header type as stored in LM.
	alu[header_type,lm_header_type, and, 0xFF]			; get header type
	alu[buf_offset, --, B, lm_sop_buf_offset]			; Get SOP buffer offset
	dl_meta_set_offset(buf_offset)						; set SOP buffer offset
	dl_meta_set_header_type(header_type)				; set header type for the port

mpp_prev_buffer_not_sop#:

	//	All the required fields of META data have been set
	//	for the previous buffer. Write it to SRAM. The META
	//	Data for !SOP buffers is only a subset of SOP META
	//	and contains only the first 2 longwords of META.

	xbuf_alloc($x_0,2, write) 							; allocate xfer registers

	dl_meta_flush_cache($x_0, dl_buf_handle, sig_meta, SIG_NONE, 0, 2)	; write 2 LWs

	//	Ok, prepare for our next trip down this lane.

	//	We are done with current buffer. Make it previous buffer 

	alu_shf[lm_prev_buf, --, B, cur_buf]
	alu[lm_prev_buf_size,--, B, lm_buf_size]
	
	//	Ok, now make the prefetch buffer as current buffer. Any new
	//	mpacket received will be stored in this new buffer.

	alu[temp, prefetch_buf_handle_gpr, AND~, 0x1, <<30]	; clear sop bit
	alu[lm_cur_buf, --, B, temp]						; prefetch buffer is current
	alu[lm_cur_data, --, B, prefetch_data_handle]		; prefetch data ptr is current

	//	clear cell count and byte reminder.

	alu[lm_cell_count, --, B, 0x0]  					; clear cell count
	alu[lm_bytes_remaining, --, B, 0x0] 				; clear bytes remaining

	//	Prefetch a new buffer so that it can be used next time around.

	_packet_rx_buf_alloc(BUF_FREE_LIST0, sig_buf_alloc, SIG_NONE, sink_sig_mask)

	//	End of Phase 1.
	//	Wait on the all the previous I/O plus the next thread signal.

	#ifdef IXP2800
	; Flush meta-data belonging to channel 0 and BUF_SRAM_CHANNEL using
	; separate macro calls.
	ctx_arb[sig_next_thread, sig_meta, sig_meta1], defer[2]
		alu[lm_buf_size, --, B, 0] 						; set buffer size as 0
		alu[lm_buf_offset, --, B, NON_SOP_OFFSET]		; set offset as 0
	#else
	ctx_arb[sig_next_thread, sig_meta], defer[2]
		alu[lm_buf_size, --, B, 0] 						; set buffer size as 0
		alu[lm_buf_offset, --, B, NON_SOP_OFFSET]		; set offset as 0
	#endif	// IXP2800
	
	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	xbuf_free($x_0) 									; free sram trnsfer regs 

	// 	go to phase 2 processing

	//	Remember that we have received only a part of the packet so far.

	br[end#]

mpp_buffer_not_full#:

	//	MOP received and Current buffer is not FULL. (i.e it can
	//	take current mpacket + atleast one more mpacket)

	// Write information to nn ring so that the 2nd me in the rx context 
	// pipeline can continue processing.
	_packet_rx_write_to_nn_ring(lm_cur_data, 0, 0,\
			byte_count, ele_num, 0, 0, \
			0, 0)

	//	increment cell count depending on the number of bytes received.

	_packet_rx_incr_cell_count(lm_cell_count, lm_bytes_remaining, byte_count)

	alu[lm_cur_data, byte_count, +, lm_cur_data]

	//	End of Phase 1.
	//	Wait on the all the previous I/O plus the next thread signal.
	//	Remember that we have received only a part of the packet so far.

	ctx_arb[sig_next_thread]
	
	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	br[end#]

mpp_null_buffer#:

	//	NULL Buffer error handling. 

	_packet_rx_incr_counter(port_num, PACKET_PKTS_DROPPED)
	
	_packet_rx_debug_incr_counter[@pkt_rx_buf_alloc_fail]

	//	If buffer not available drop the current (so far assembled) 
	//	packet as well as the just received mpacket. Reset State 
	//	Machine (to	START State)

	_packet_rx_set_state(STATE_START)								; reset to start state.
	
	//	Prefetch a new buffer so that it can be used next time around.

	_packet_rx_buf_alloc(BUF_FREE_LIST0, sig_buf_alloc, SIG_NONE, sink_sig_mask)

	ctx_arb[sig_next_thread], defer[2]
		alu[drop_sop_handle, --, B, lm_sop_buf]	
		alu[drop_eop_handle, --, B, lm_cur_buf]
	
	; Signal next thread
	_packet_rx_signal_next_thread(nt_sig_reg)

	// Free RBUF element.
	_packet_rx_free_rbuf_element(ele_num)

	//	drop current (so far assembled) packet

	_packet_rx_drop_current_pkt(drop_sop_handle, drop_eop_handle)

end#:

.end
#endm

#endif // FIRST_PACKET_RX_ME
//--------------------------------
//	First Packet RX ME - End
//--------------------------------


//--------------------------------
//	Second Packet RX ME - Begin
//--------------------------------
#ifdef SECOND_PACKET_RX_ME

// Some local definitions.

//	From RBUF size, compute the power of 2 (ln[rbuf_size]).
_LN(RBUF_SIZE)
#define_eval	PO2_RSIZE		LN_RESULT	// LN_RESULT contains result of ln[...]
#undef			LN_RESULT	// cleanup namespace

//	There is a difference between IXP2400 and IXP2800 on how rbuf to rsw 
//	mapping is done. Pls refer to IXP2800 HRM.
#ifdef	IXP2800
#define_eval	PO2_RSIZE		6
#endif

//	Global Signals

.sig volatile sig_next_thread					// next thread signal

// Global Registers

.reg	rbuf_base								// base address of rbuf
.reg	nt_sig_reg								// used in signalling next thread
.reg 	dram_rbuf_ov							// for moving data from rbuf to dram

#ifdef	RFC_PPP_COUNTERS
.reg	counter_base							// base addr for counters.
#endif

///////////////////////////////////////////////////////////////////////////////
//
// packet_rx_2_init
//
//	 	Description: 
//
//			One time initialisation code for the second me of Packet RX.
//
//	 	Outputs:
//
//			Global variables will be affected.
//
//		Inputs:
//
//			Will use global variables.
//
///////////////////////////////////////////////////////////////////////////////
#macro packet_rx_2_init()
.begin

	move(dram_rbuf_ov,((1 << 25)|(1<<4)))	   	; [25] ref_cnt, [4] xfer regs overwrite

	move(rbuf_base, RBUF_TBUF_BASE)            	; set up RBUF base

	; Get current me and thread number.
	.reg me_num thread_num
	_packet_rx_get_current_me_thread_number[me_num, thread_num]

	; Based on RX_PHY_MODE, setup thread order.
	_packet_rx_setup_next_thread_sig[nt_sig_reg, thread_num]

	#ifdef	RFC_PPP_COUNTERS
	; Initialize counters base address.
	immed32(counter_base, PACKET_COUNTERS_SRAM_BASE)
	#endif

	; Init meta-data gpr's. Some of these will be used to write to the scratch ring.
	.reg zero
	immed[zero, 0]
	dl_meta_init_cache(zero, zero, zero, zero, zero, zero, zero, zero)

	; For next iteration, set meta-data fields that remain constant
	; and will be sent over the scratch ring.

	; Set buf_offset. Constant in this design (set to 384).
	.reg buf_offset
	_packet_rx_get_rand_buf_offset(buf_offset)
	dl_meta_set_offset(buf_offset)
	; Set freelist id.
	dl_meta_set_free_list(FREE_LIST_ID)
	; Set rx stat to 0 for now.
	dl_meta_set_rx_stat(0)
	; Set header type to packet type coming in on port 0 for now.
	; Will need to change if different ports handle different packet types.
	; For ex: If port 0 handles PPP and port 1 handles Ethernet.
	dl_meta_set_header_type(PORT_0_TYPE)	

	_dl_pos_sink_init()

	.if(ctx()==0)
		; Init nn ring.
		.reg ctx_enables_reg
		immed32[ctx_enables_reg, 0xcff00]
		local_csr_wr[ctx_enables, ctx_enables_reg]
		local_csr_wr[nn_get, 0]
		local_csr_wr[nn_put, 0]	

		; Wait for system initialization complete signal.
		.sig volatile system_init_sig
		.addr system_init_sig ME_INIT_SIGNAL
		ctx_arb[system_init_sig]

	.else
		ctx_arb[sig_next_thread]

	.endif

.end
#endm

///////////////////////////////////////////////////////////////////////////////
//
// packet_pkt_rx_2
//
//	 	Description: 
//
// 			Packet processing entry point for second me of packet rx.
//			In each iteration one mpacket will be handled.
//
//	 	Outputs:
//
//			Global variables will be affected.
//
//		Inputs:
//
//			Will use global variables.
//
///////////////////////////////////////////////////////////////////////////////
#macro packet_pkt_rx_2()
.begin

	.reg dram_handle \
		 byte_count \
		 ele_num \
		 ele_addr \
		 sop_buf_size \
		 packet_size \
		 input_port

	.sig sig_dram

	; Read from the nn ring. The first me of the context pipeline
	; puts data on the ring. For the format refer to macro description
	; _packet_rx_write_to_nn_ring()
	br_inp_state[nn_empty, null_request#]

	; Copy data from nn ring into gpr's.
	alu[dram_handle, --, b, *n$index++]

reading_from_nn_ring#:
	br_inp_state[nn_empty, reading_from_nn_ring#]

	.reg tmp0 tmp1

	; Copy data from nn ring into gpr's.
	alu[dl_eop_buf_handle, --, b, *n$index++]
	alu[dl_buf_handle, --, b, *n$index++]
	alu[tmp0, --, b, *n$index++]
	alu[tmp1, --, b, *n$index++]

	ld_field_w_clr[byte_count, 0001, tmp0]
	alu[ele_num, 0x7F, and, tmp0, >>8]
	ld_field_w_clr[sop_buf_size, 0011, tmp0, >>16]

	ld_field_w_clr[packet_size, 0011, tmp1]
	ld_field_w_clr[input_port, 0011, tmp1, >>16]

	; Check if 'eop_flag' is set.
	br_bclr[tmp0, 15, dram_write#]

	; If yes, set dl_next_block.
	alu[dl_next_block, --, b, BID_NEXT_BLOCK]	

dram_write#:

	; Convert RBUF element number to RBUF element address.
	alu_shf[ele_addr, --, B, ele_num, <<PO2_RSIZE]	; ele_num * size of rbuf element

	; Write the RBUF element to DRAM.
	_packet_rx_move_rbuf_to_pkt_buf(dram_handle, ele_addr, byte_count, rbuf_base, sig_dram)

	ctx_arb[sig_next_thread, sig_dram], defer[2]

	; Set meta-data fields that will be sent over the scratch ring.
	dl_meta_set_buffer_size(sop_buf_size)
	dl_meta_set_packet_size(packet_size)

	; Free RBUF element.
	_packet_rx_free_rbuf_element(ele_num)

	br[end#], defer[2]

	; Set meta-data fields that will be sent over the scratch ring.
	dl_meta_set_input_port(input_port)

	; Signal next thread.
	_packet_rx_signal_next_thread(nt_sig_reg)

null_request#:

	; For next iteration, initialize meta-data cache.
	.reg zero
	immed[zero, 0]
	dl_meta_init_cache(zero, zero, zero, zero, zero, zero, zero, zero)

	; For next iteration, set meta-data fields that remain constant
	; and will be sent over the scratch ring.

	; Set buf_offset. Constant in this design (set to 384).
	.reg buf_offset
	_packet_rx_get_rand_buf_offset(buf_offset)
	dl_meta_set_offset(buf_offset)

	; Set freelist id.
	dl_meta_set_free_list(FREE_LIST_ID)

	; Set rx stat to 0 for now.
	dl_meta_set_rx_stat(0)

	; Set header type to packet type coming in on port 0 for now.
	; Will need to change if different ports handle different packet types.
	; For ex: If port 0 handles PPP and port 1 handles Ethernet.
	dl_meta_set_header_type(PORT_0_TYPE)	

	ctx_arb[sig_next_thread], defer[2]

	alu[dl_buf_handle, --, B, DL_BUF_NULL]
	alu[dl_eop_buf_handle, --, B, DL_BUF_NULL]

	; Signal next thread.
	_packet_rx_signal_next_thread(nt_sig_reg)

end#:

.end
#endm

#endif // SECOND_PACKET_RX_ME
//--------------------------------
//	Second Packet RX ME - End
//--------------------------------

#endif	//	__PACKET_RX_TWO_ME_UTIL_UC__
