// Wanderer.c
// ****************************************************
// File: Wanderer.c
// Date: 11/20/2001
// Last modified: 02/16/2002
// Author: Karl McEntire
// Program: Wander about avoiding obstructions
// Description: run bot toward an clear view sensor
// take counter measures if approaching an obstacle 
// ****************************************************

//Setup ID numbers
#define wanderFORM 			5000
#define wanderStartBUTTON 	5001
#define wanderExitBUTTON 	5002
#define wanderInstructLBL1 	5030
#define wanderInstructLBL2 	5031
#define wanderInstructLBL3 	5032

int newState(float view) {

	// create binary state number
	// a zero is an obstacle
	// 							 sensor
	//	state	obstacles	3		2		1		Action
	//	-----	---------	-		-		-		------
	//	  0	  	  1 2 3		0		0		0		Spin
	//	  1	  	    2 3 	0		0		1		to 1
	//	  2		  1   3		0		1		0		to 2
	//	  3			  3		0		1		1		1 or 2*
	//	  4		  1 2		1		0		0		to 3
	//	  5			2 		1		0		1		1* or 3
	//	  6		  1			1		1		0		2 or 3*
	//	  7		  none		1		1		1		to any sensor 1*

	int i, sensor[4] ;
			
	for( i = 1 ; i < 4 ; i++ ) 
		// store a 1 (true) if clear ahead
		// use IrVal rather than IrDist for faster processing?
		sensor[i] = IrVal( i )  < view ;  
	return ( sensor[3] * 4 + sensor[2] * 2 + sensor[1] ) ;
}

Wander() {

	float vel = - 6.0 ;    	// drive away from servo toward sensor
	float obstacle = 50.0 ;   	// obstruction within this range 50=30cm
	float clearView = 40.0	;	// clear view ahead  40=40cm
	float view ;				// how far ahead are we looking now?
	float omega = 0.5 ;  	// bot body rotational velocity (in/sec)
	int oldstate = 0, state = 0, mask = 7 ;

	view = clearView ;		// lets make sure we have a clear path to start
	
	while( 2 != event(2) ) {  // continue until pen down event
		oldstate = state ;
		state = newState( view ) ;
		
		if( ( mask & state ) == ( mask & oldstate ) ) { // did the oldstate sensor change?
			view = obstacle ; 		// change view to look for obstacles
			continue ;  			// loop until state changes
		} 
		else {							// state changed 
			view = clearView ; 			// make sure we have a clear path
		}

		switch( state ) {
			case 0: // all sensors obstructed - spin until cleared
				Turn ( omega ) ;
				mask = 7 ;  // look for any sensor to be clear
				break ;	
			case 1: // drive toward sensor 1
				headingDriver( vel, 0, 0 ) ;
				mask = 1 ;  // see if sensor 1 changes
				break ;
			case 2: // drive toward sensor 2
				headingDriver( vel, 2 * PI / 3, 0 ) ;
				mask = 2 ;  // see if sensor 2 changes
				break ;
			case 3: // sensor 1 or 2 is clear
				headingDriver( vel, 2 * PI / 3, 0 ) ; // pick 2
				state = 2 ; mask = 2 ; // see if sensor 2 changes
				break ;
			case 4: // drive toward sensor 3
				headingDriver( vel, 4* PI / 3, 0 ) ;
				mask = 4 ; // monitor changes to sensor 3 (binary 100)
				break ;
			case 5: // sensor 1 or 3 is clear
				headingDriver( vel, 0, 0 ) ;  // pick 1 for now
				state = 1 ; mask = 1 ;
				break ;		
			case 6: // sensor 2 or 3 is clear
				headingDriver( vel, 4*PI / 3, 0 ) ; // pick 3 for now
				state = 3 ; mask = 4 ;
				break ;				
			case 7:  // all sensors are clear
				headingDriver( vel,  0, 0 ) ; // pick 1
				state = 1 ; mask = 1 ;
				break ;
		}
	}
}

Wanderer() {
	int e;
		
	Form(wanderFORM,"Wanderer           ");
	
	//buttons	
	Button(wanderExitBUTTON,0x10,120,140,0,0,"Exit");
	Button(wanderStartBUTTON,0x00,40,140,0,0,"Start");
	Label(wanderInstructLBL1,10,60,"Bot will wander avoiding obstacles.");
	Label(wanderInstructLBL2,20,70,"Tap Start button to begin.");
	Label(wanderInstructLBL3,30,80,"Tap screen to stop.");

	Fctl(DRAW,wanderFORM); //draw objects

	while(1) {
		e=Fevent(1);
		switch(e) {
			case wanderExitBUTTON: 
				StopRobot();
				return;
			case wanderStartBUTTON:
				portOpen(); 
				Wander();
				StopRobot() ;
				portClose();
				break;
		}
	}
}
