/*
 * ras: subroutine package to read and write Sun raster file format
 *
 * Paul Heckbert	16 Dec 1988
 */

static char rcsid[] = "$Header: ras.c,v 1.3 89/05/19 13:21:10 ph Locked $";

#include <simple.h>
#include "ras.h"
#include "dump.h"

#ifdef vax		/* swap bytes in header if little-endian */
#   define BYTESWAP
#endif

#define UNDEF PIXEL_UNDEFINED

#define CHECK_HEAD_UNWRITTEN(p, subrname) {	\
    if ((p)->headwritten) {			\
	fprintf(stderr, "%s: can't change state once writing commences\n", \
	    subrname);				\
	exit(1);				\
    }						\
}
#define CHECK_HEAD_WRITTEN(p, subrname) \
    if (!(p)->headwritten) ras_write_head(p, subrname); else

Ras *ras_open_read(), *ras_open_write();

Ras *ras_open(file, mode)
char *file, *mode;
{
         if (str_eq(mode, "r")) return ras_open_read(file);
    else if (str_eq(mode, "w")) return ras_open_write(file);
    fprintf(stderr, "ras_open: can't do mode %s\n", mode);
    exit(1); /*NOTREACHED*/
}

static Ras *ras_open_write(file)
char *file;
{
    Dump *p;

    ALLOC(p, Dump, 1);
    if ((p->fp = fopen(file, "w")) == NULL) {
	fprintf(stderr, "ras_open_write: can't write %s\n", file);
	free(p);
	return 0;
    }
    p->h.nchan = 1;
    p->h.dx = UNDEF;
    strcpy(p->name, file);
    p->headsize = sizeof(Ras_head);
    p->headwritten = 0;
    p->curx = p->cury = 0;
    return p;
}

static void ras_write_head(p, subrname)
Ras *p;
char *subrname;
{
    Ras_head h;
    int v, i;
    unsigned char cm[3][256];

    if (p->h.dx==UNDEF) {
	fprintf(stderr, "%s: size of %s is uninitialized\n", p->name);
	exit(1);
    }
    h.magic = RAS_MAGIC;
    h.dx = p->h.dx;
    h.dy = p->h.dy;
    h.nbit = p->h.nchan == 3 ? 24 : 8;
    h.len = p->h.dx*p->h.dy*p->h.nchan;
    h.type = RAS_TYPE_STANDARD;
    h.cmtype = RAS_CM_EQUAL_RGB;
    h.cmlen = 3*256;
#   ifdef BYTESWAP
	headswap(&h);
#   endif
    if (fwrite(&h, sizeof(Ras_head), 1, p->fp) != 1) {
	fprintf(stderr, "%s: write error on %s\n", subrname, p->name);
	exit(1);
    }
    printf("%s: %dx%d %d-chan\n", p->name, p->h.dx, p->h.dy, p->h.nchan);
    for (i=0; i<3; i++)
	for (v=0; v<256; v++)
	    cm[i][v] = v;
    if (fwrite(cm, 256, 3, p->fp) != 3) {
	fprintf(stderr, "ras_open_write: write error on colormap of %s\n",
	    p->name);
	exit(1);
    }
    p->headwritten = 1;
}

static Ras *ras_open_read(file)
char *file;
{
    int badhead;
    Ras_head h;
    Dump *p;

    ALLOC_ZERO(p, Dump, 1);
    if ((p->fp = fopen(file, "r")) == NULL) {
	fprintf(stderr, "ras_open_read: can't find %s\n", file);
	free(p);
	return 0;
    }

    badhead = fread(&h, sizeof(Ras_head), 1, p->fp) != 1;
#   ifdef BYTESWAP
	headswap(&h);
#   endif
    if (badhead || h.magic!=RAS_MAGIC) {
	fprintf(stderr, "ras_open_read: %s is not a ras file\n", file);
	free(p);
	return 0;
    }
    printf("%s: %dx%d %d-bit, ",
	file, h.dx, h.dy, h.nbit);
    printf("len=%d type=%d cmtype=%d cmlen=%d\n",
	h.len, h.type, h.cmtype, h.cmlen);
    if (!(h.type==RAS_TYPE_STANDARD && (h.nbit==8 || h.nbit==24) &&
	(h.cmtype==RAS_CM_NONE && h.cmlen==0 || h.cmtype==RAS_CM_EQUAL_RGB))) {
	    fprintf(stderr,
		"ras_open_read: don't know type %d, %d bpp, cmtype %d\n",
		    h.type, h.nbit, h.cmtype);
	    exit(1);
    }
    if (h.len!=h.dx*h.dy*h.nbit/8)
	fprintf(stderr, "ras_open_read: surprised length isn't %d\n",
	    h.dx*h.dy*h.nbit/8);

    if (h.cmlen>0) {
	Pixel1 *cm[3], cmb[3*256];
	int ncolor, v;
	ncolor = h.cmlen/3;
	if (fread(cmb, ncolor, 3, p->fp) != 3) {
	    fprintf(stderr, "ras_open_read: read error on colormap of %s\n",
		p->name);
	    exit(1);
	}
	cm[0] = &cmb[0];
	cm[1] = &cmb[ncolor];
	cm[2] = &cmb[2*ncolor];
	/* print colormap if it's not an identity */
	for (v=0; v<ncolor; v++)
	    if (cm[0][v]!=v || cm[1][v]!=v || cm[2][v]!=v) {
		for (v=0; v<ncolor; v++)
		    printf("cm[%3d] = (%3d,%3d,%3d)\n",
			v, cm[0][v], cm[1][v], cm[2][v]);
		break;
	    }
    }
    /* else no colormap */

    strcpy(p->name, file);
    p->h.dx = h.dx;
    p->h.dy = h.dy;
    p->h.nchan = h.nbit == 24 ? 3 : 1;
    p->headsize = sizeof(Ras_head)+h.cmlen;
    p->headwritten = 1;
    p->curx = p->cury = 0;
    return p;
}

#ifdef BYTESWAP

static headswap(h)
Ras_head *h;
{
    swap_long(&h->magic);
    swap_long(&h->dx);
    swap_long(&h->dy);
    swap_long(&h->nbit);
    swap_long(&h->len);
    swap_long(&h->type);
    swap_long(&h->cmtype);
    swap_long(&h->cmlen);
}

#endif

/*-------------------- file writing routines --------------------*/

void ras_set_nchan(p, nchan)
Dump *p;
int nchan;
{
    CHECK_HEAD_UNWRITTEN(p, "ras_set_nchan");
    if (nchan!=1 && nchan!=3) {
	fprintf(stderr, "ras_set_nchan: can't handle nchan=%d\n", nchan);
	exit(1);
    }
    p->h.nchan = nchan;
}

void ras_set_box(p, ox, oy, dx, dy)
Dump *p;
int ox, oy, dx, dy;
{
    CHECK_HEAD_UNWRITTEN(p, "ras_set_box");
    /* ignore ox, oy */
    p->h.dx = dx;
    p->h.dy = dy;
}

void ras_write_pixel(p, x, y, pv)
Dump *p;
int x, y;
Pixel1 pv;
{
    fprintf(stderr, "ras_write_pixel: unimplemented\n");
}

void ras_write_pixel_rgba(p, x, y, r, g, b, a)
Dump *p;
int x, y;
Pixel1 r, g, b, a;
{
    fprintf(stderr, "ras_write_pixel_rgba: unimplemented\n");
}

void ras_write_row(p, y, x0, nx, buf)
register Dump *p;
int y, x0, nx;
Pixel1 *buf;
{
    CHECK_HEAD_WRITTEN(p, "ras_write_row");
    if (x0!=p->curx || y!=p->cury)
	dump_jump_to_pixel(p, x0, y);
    if (fwrite(buf, nx*sizeof(Pixel1), 1, p->fp) != 1) {
	fprintf(stderr, "ras_write_row: write error on %s\n", p->name);
	exit(1);
    }
    dump_advance(p, nx);
}

void ras_write_row_rgba(p, y, x0, nx, buf)
register Dump *p;
int y, x0, nx;
register Pixel1_rgba *buf;
{
    register int x;

    CHECK_HEAD_WRITTEN(p, "ras_write_row_rgba");
    if (x0!=p->curx || y!=p->cury)
	dump_jump_to_pixel(p, x0, y);
    for (x=nx; --x>=0; buf++) {
	putc(buf->r, p->fp);
	putc(buf->g, p->fp);
	putc(buf->b, p->fp);
    }
    dump_advance(p, nx);
}
