/*
	File:			Vec.cc

	Function:		See header file

	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott

	Notes:			

*/

#include "Vec.h"
#include <iomanip.h>
#include <math.h>
#include <string.h>
#include <stdarg.h>
#include "CopyVec.h"
#include "Array.h"		// for the >> operator...


#pragma mark -
// --- Vec Constructors -------------------------------------------------------


TMPLVec TVec::Vec() : elts(0), data(0), isRef(0)
{
}

TMPLVec TVec::Vec(Int n) : elts(n), isRef(0)
{
	Assert(n > 0,"(Vec) illegal vector size");

	data = new TReal[n];
	Assert(data != 0, "(Vec) Out of memory");	
}

TMPLVec TVec::Vec(Int n, ZeroOrOne k) : elts(n), isRef(0)
{
	Assert(n > 0,"(Vec) illegal vector size");

	data = new TReal[n];
	Assert(data != 0, "(Vec) Out of memory");	
	
	MakeBlock(k);
}

TMPLVec TVec::Vec(Int n, Axis a) : elts(n), isRef(0)
{
	Assert(n > 0,"(Vec) illegal vector size");

	data = new TReal[n];
	Assert(data != 0, "(Vec) Out of memory");	
	
	MakeUnit(a);
}

TMPLVec TVec::Vec(Int n, TReal *data) : elts(n), isRef(1), data(data)
{
}

TMPLVec TVec::Vec(const TVec &v) : elts(v.elts), isRef(v.isRef)
{
	if (isRef || v.data == 0)
		data = v.data;
	else
	{
		data = new TReal[elts];	
		Assert(data != 0, "(Vec) Out of memory");
		memcpy(data, v.data, v.elts * sizeof(TReal));
	}
}

TMPLVec TVec::Vec(const TSGVec &v) : elts(v.Elts()), isRef(0)
{
	data = new TReal[elts];	
	Assert(data != 0, "(Vec) Out of memory");
	
	CopyVec(SELF, v);
}

TMPLVec TVec::Vec(const TVec2 &v) : elts(v.Elts()), isRef(1), data(v.Ref())
{
}

TMPLVec TVec::Vec(const TVec3 &v) : elts(v.Elts()), isRef(1), data(v.Ref())
{
}

TMPLVec TVec::Vec(const TVec4 &v) : elts(v.Elts()), isRef(1), data(v.Ref())
{
}

TMPLVec TVec::Vec(Int n, double elt0, ...) : elts(n), isRef(0)
{
	Assert(n > 0,"(Vec) illegal vector size");

	va_list ap;
	Int 	i = 1;
		
	data = new TReal[n];

	va_start(ap, elt0);
		
	data[0] = elt0;
	
	while (--n)
		data[i++] = va_arg(ap, double);

	va_end(ap);
}

TMPLVec TVec::~Vec()
{
	if (!isRef)
		delete[] data;
}


#pragma mark -
// --- Vec Assignment Operators -----------------------------------------------


TMPLVec TVec &TVec::operator = (const TVec &v)
{
	if (elts == 0)
		SetSize(v.Elts());

	return(CopyVec(SELF, v));
}

TMPLVec TVec &TVec::operator = (const TSGVec &v)
{
	if (elts == 0)
		SetSize(v.Elts());

	return(CopyVec(SELF, v));
}

TMPLVec TVec &TVec::operator = (const TVec2 &v)
{
	if (elts == 0)
		SetSize(v.Elts());

	return(CopyVec((SELF), v));
}

TMPLVec TVec &TVec::operator = (const TVec3 &v)
{
	if (elts == 0)
		SetSize(v.Elts());

	return(CopyVec((SELF), v));
}

TMPLVec TVec &TVec::operator = (const TVec4 &v)
{
	if (elts == 0)
		SetSize(v.Elts());

	return(CopyVec((SELF), v));
}

TMPLVec TVec &TVec::operator >> (Action<TReal> &a)
{
	Int i;
	
	a.Start();
	
	for (i = 0; i < elts; i++)
		a.Process(data[i]);
	
	a.Stop();
	
	return(SELF);
}

TMPLVec void TVec::SetSize(Int n)
{	
	if (!isRef)
		delete[] data;
		
	elts = n;
	data = new TReal[elts];
	
	Assert(data != 0, "(Vec::SetSize) Out of memory");
}

TMPLVec void TVec::MakeZero()
{
	Int j;

	for (j = 0; j < elts; j++)
		data[j] = vl_zero;	
}

TMPLVec void TVec::MakeUnit(Int i, TReal k)
{
	Int j;

	for (j = 0; j < elts; j++)
		data[j] = vl_zero;	
		
	data[i] = k;
}

TMPLVec void TVec::MakeBlock(TReal k)
{
	Int i;
	
	for (i = 0; i < elts; i++)
		data[i] = k;
}


#pragma mark -
// --- Vec In-Place operators -------------------------------------------------


TMPLVec TVec &operator += (TVec &a, const TVec &b)
{
	Assert(a.Elts() == b.Elts(), "(Vec::+=) vector sizes don't match");	

	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		a[i] += b[i];
	
	return(a);
}

TMPLVec TVec &operator -= (TVec &a, const TVec &b)
{
	Assert(a.Elts() == b.Elts(), "(Vec::-=) vector sizes don't match");	

	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		a[i] -= b[i];
		
	return(a);
}

TMPLVec TVec &operator *= (TVec &a, const TVec &b)
{
	Assert(a.Elts() == b.Elts(), "(Vec::*=) Vec sizes don't match");

	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		a[i] *= b[i];
	
	return(a);
}

TMPLVec TVec &operator *= (TVec &v, TReal s)
{
	Int		i;
	
	for (i = 0; i < v.Elts(); i++) 
		v[i] *= s;
	
	return(v);
}

TMPLVec TVec &operator /= (TVec &a, const TVec &b)
{
	Assert(a.Elts() == b.Elts(), "(Vec::/=) Vec sizes don't match");

	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		a[i] /= b[i];
	
	return(a);
}

TMPLVec TVec &operator /= (TVec &v, TReal s)
{
	Int		i;
	
	for (i = 0; i < v.Elts(); i++) 
		v[i] /= s;
	
	return(v);
}


#pragma mark -
// --- Vec Comparison Operators -----------------------------------------------


TMPLVec Bool operator == (const TVec &a, const TVec &b)
{
	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		if (a[i] != b[i])
			return(0);
	
	return(1);
}

TMPLVec Bool operator != (const TVec &a, const TVec &b)
{
	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		if (a[i] != b[i])
			return(1);
	
	return(0);
}


#pragma mark -
// --- Vec Arithmetic Operators -----------------------------------------------



TMPLVec TVec operator + (const TVec &a, const TVec &b)
{
	Assert(a.Elts() == b.Elts(), "(Vec::+) Vec sizes don't match");

	TVec	result(a.Elts());
	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		result[i] = a[i] + b[i];
	
	return(result);
}

TMPLVec TVec operator - (const TVec &a, const TVec &b) 
{
	Assert(a.Elts() == b.Elts(), "(Vec::-) Vec sizes don't match");
	
	TVec	result(a.Elts());
	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		result[i] = a[i] - b[i];
	
	return(result);
}

TMPLVec TVec operator - (const TVec &v)
{
	TVec	result(v.Elts());
	Int		i;
	
	for (i = 0; i < v.Elts(); i++) 
		result[i] = - v[i];
	
	return(result);
}

TMPLVec TVec operator * (const TVec &a, const TVec &b)			
{
	Assert(a.Elts() == b.Elts(), "(Vec::*) Vec sizes don't match");
	
	TVec	result(a.Elts());
	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		result[i] = a[i] * b[i];
	
	return(result);
}

TMPLVec TVec operator * (const TVec &v, TReal s) 
{
	TVec	result(v.Elts());
	Int		i;
	
	for (i = 0; i < v.Elts(); i++) 
		result[i] = v[i] * s;
	
	return(result);
}

TMPLVec TVec operator / (const TVec &a, const TVec &b)			
{
	Assert(a.Elts() == b.Elts(), "(Vec::/) Vec sizes don't match");
	
	TVec	result(a.Elts());
	Int		i;
	
	for (i = 0; i < a.Elts(); i++) 
		result[i] = a[i] / b[i];
	
	return(result);
}

TMPLVec TVec operator / (const TVec &v, TReal s) 
{
	TVec	result(v.Elts());
	Int		i;
	
	for (i = 0; i < v.Elts(); i++) 
		result[i] = v[i] / s;
	
	return(result);
}

TMPLMat TReal operator dot (const TMVec &a, const TVec &b) 
{
	Assert(a.Elts() == b.Elts(), "(Vec::dot) Vec sizes don't match");

	TMReal	sum = 0;
	Int		i;
		
	for (i = 0; i < a.Elts(); i++) 
		sum += a[i] * b[i];
	
	return(sum);
}

TMPLVec TVec operator * (TReal s, const TVec &v)
{
	TVec	result(v.Elts());
	Int		i;
	
	for (i = 0; i < v.Elts(); i++) 
		result[i] = v[i] * s;
	
	return(result);
}

TMPLVec TVec sub(const TVec &v, Int start, Int length)
{
	Assert(start >= 0 && length > 0 && start + length <= v.Elts(),
		"(sub(Vec)) illegal subset of vector");

	return(TVec(length, v.Ref() + start));
}

TMPLVec TReal len(const TVec &v)
{
	return(sqrt(v dot v));
}

TMPLVec TReal sqrlen(const TVec &v)
{
	return(v dot v);
}

TMPLVec TVec norm(const TVec &v)	
{
	return(v / len(v));
}


#pragma mark -
// --- Vec Input & Output -----------------------------------------------------


TMPLVec ostream &operator << (ostream &s, const TVec &v)
{
	Int i, w;

	s << '[';
	
	if (v.Elts() > 0)
	{
    	w = s.width();
		s << v[0];
	
		for (i = 1; i < v.Elts(); i++)
			s << ' ' << setw(w) << v[i];
	}
	
	s << ']';
	
	return(s);
}

TMPLVec istream &operator >> (istream &s, TVec &v)
{
	Array<TReal> array;
	
	// Expected format: [1 2 3 4 ...]

	s >> array;									// Read input into variable-sized array

	v = TVec(array.NumItems(), array.Ref());	// Copy input into vector
	
    return(s);
}

